/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.schema;

import com.google.common.base.Function;
import com.google.common.collect.Iterables;
import com.google.common.collect.MapDifference;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.cache.CachingOptions;
import org.apache.cassandra.config.CFMetaData;
import org.apache.cassandra.config.ColumnDefinition;
import org.apache.cassandra.config.IndexType;
import org.apache.cassandra.config.KSMetaData;
import org.apache.cassandra.config.Schema;
import org.apache.cassandra.config.TriggerDefinition;
import org.apache.cassandra.config.UTMetaData;
import org.apache.cassandra.cql3.ColumnIdentifier;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.cql3.functions.AbstractFunction;
import org.apache.cassandra.cql3.functions.FunctionName;
import org.apache.cassandra.cql3.functions.Functions;
import org.apache.cassandra.cql3.functions.UDAggregate;
import org.apache.cassandra.cql3.functions.UDFunction;
import org.apache.cassandra.db.CFRowAdder;
import org.apache.cassandra.db.ColumnFamily;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.ColumnFamilyType;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.RangeTombstone;
import org.apache.cassandra.db.Row;
import org.apache.cassandra.db.RowPosition;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.columniterator.IdentityQueryFilter;
import org.apache.cassandra.db.composites.CellNameType;
import org.apache.cassandra.db.composites.CellNames;
import org.apache.cassandra.db.composites.Composite;
import org.apache.cassandra.db.filter.QueryFilter;
import org.apache.cassandra.db.index.SecondaryIndexManager;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.AsciiType;
import org.apache.cassandra.db.marshal.CompositeType;
import org.apache.cassandra.db.marshal.ListType;
import org.apache.cassandra.db.marshal.LongType;
import org.apache.cassandra.db.marshal.TypeParser;
import org.apache.cassandra.db.marshal.UTF8Type;
import org.apache.cassandra.db.marshal.UserType;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.exceptions.ConfigurationException;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.io.compress.CompressionParameters;
import org.apache.cassandra.locator.AbstractReplicationStrategy;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LegacySchemaTables {
    private static final Logger logger = LoggerFactory.getLogger(LegacySchemaTables.class);
    public static final String KEYSPACES = "schema_keyspaces";
    public static final String COLUMNFAMILIES = "schema_columnfamilies";
    public static final String COLUMNS = "schema_columns";
    public static final String TRIGGERS = "schema_triggers";
    public static final String USERTYPES = "schema_usertypes";
    public static final String FUNCTIONS = "schema_functions";
    public static final String AGGREGATES = "schema_aggregates";
    public static final List<String> ALL = Arrays.asList("schema_keyspaces", "schema_columnfamilies", "schema_columns", "schema_triggers", "schema_usertypes", "schema_functions", "schema_aggregates");
    private static final CFMetaData Keyspaces = LegacySchemaTables.compile("schema_keyspaces", "keyspace definitions", "CREATE TABLE %s (keyspace_name text,durable_writes boolean,strategy_class text,strategy_options text,PRIMARY KEY ((keyspace_name))) WITH COMPACT STORAGE");
    private static final CFMetaData Columnfamilies = LegacySchemaTables.compile("schema_columnfamilies", "table definitions", "CREATE TABLE %s (keyspace_name text,columnfamily_name text,bloom_filter_fp_chance double,caching text,cf_id uuid,comment text,compaction_strategy_class text,compaction_strategy_options text,comparator text,compression_parameters text,default_time_to_live int,default_validator text,dropped_columns map<text, bigint>,gc_grace_seconds int,is_dense boolean,key_validator text,local_read_repair_chance double,max_compaction_threshold int,max_index_interval int,memtable_flush_period_in_ms int,min_compaction_threshold int,min_index_interval int,read_repair_chance double,speculative_retry text,subcomparator text,type text,PRIMARY KEY ((keyspace_name), columnfamily_name))");
    private static final CFMetaData Columns = LegacySchemaTables.compile("schema_columns", "column definitions", "CREATE TABLE %s (keyspace_name text,columnfamily_name text,column_name text,component_index int,index_name text,index_options text,index_type text,type text,validator text,PRIMARY KEY ((keyspace_name), columnfamily_name, column_name))");
    private static final CFMetaData Triggers = LegacySchemaTables.compile("schema_triggers", "trigger definitions", "CREATE TABLE %s (keyspace_name text,columnfamily_name text,trigger_name text,trigger_options map<text, text>,PRIMARY KEY ((keyspace_name), columnfamily_name, trigger_name))");
    private static final CFMetaData Usertypes = LegacySchemaTables.compile("schema_usertypes", "user defined type definitions", "CREATE TABLE %s (keyspace_name text,type_name text,field_names list<text>,field_types list<text>,PRIMARY KEY ((keyspace_name), type_name))");
    private static final CFMetaData Functions = LegacySchemaTables.compile("schema_functions", "user defined function definitions", "CREATE TABLE %s (keyspace_name text,function_name text,signature frozen<list<text>>,argument_names list<text>,argument_types list<text>,body text,language text,return_type text,called_on_null_input boolean,PRIMARY KEY ((keyspace_name), function_name, signature))");
    private static final CFMetaData Aggregates = LegacySchemaTables.compile("schema_aggregates", "user defined aggregate definitions", "CREATE TABLE %s (keyspace_name text,aggregate_name text,signature frozen<list<text>>,argument_types list<text>,final_func text,initcond blob,return_type text,state_func text,state_type text,PRIMARY KEY ((keyspace_name), aggregate_name, signature))");
    public static final List<CFMetaData> All = Arrays.asList(Keyspaces, Columnfamilies, Columns, Triggers, Usertypes, Functions, Aggregates);

    private static CFMetaData compile(String name, String description, String schema) {
        return CFMetaData.compile(String.format(schema, name), "system").comment(description).gcGraceSeconds((int)TimeUnit.DAYS.toSeconds(7L));
    }

    public static void saveSystemKeyspaceSchema() {
        KSMetaData keyspace = Schema.instance.getKSMetaData("system");
        for (String table : ALL) {
            QueryProcessor.executeOnceInternal(String.format("DELETE FROM system.%s WHERE keyspace_name = ?", table), keyspace.name);
        }
        LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, FBUtilities.timestampMicros() + 1L).apply();
    }

    public static Collection<KSMetaData> readSchemaFromSystemTables() {
        List<Row> serializedSchema = LegacySchemaTables.getSchemaPartitionsForTable(KEYSPACES);
        ArrayList<KSMetaData> keyspaces = new ArrayList<KSMetaData>(serializedSchema.size());
        for (Row partition : serializedSchema) {
            if (LegacySchemaTables.isEmptySchemaPartition(partition) || LegacySchemaTables.isSystemKeyspaceSchemaPartition(partition)) continue;
            keyspaces.add(LegacySchemaTables.createKeyspaceFromSchemaPartitions(partition, LegacySchemaTables.readSchemaPartitionForKeyspace(COLUMNFAMILIES, partition.key), LegacySchemaTables.readSchemaPartitionForKeyspace(USERTYPES, partition.key)));
            for (UDFunction function : LegacySchemaTables.createFunctionsFromFunctionsPartition(LegacySchemaTables.readSchemaPartitionForKeyspace(FUNCTIONS, partition.key)).values()) {
                org.apache.cassandra.cql3.functions.Functions.addFunction(function);
            }
            for (UDAggregate aggregate : LegacySchemaTables.createAggregatesFromAggregatesPartition(LegacySchemaTables.readSchemaPartitionForKeyspace(AGGREGATES, partition.key)).values()) {
                org.apache.cassandra.cql3.functions.Functions.addFunction(aggregate);
            }
        }
        return keyspaces;
    }

    public static void truncateSchemaTables() {
        for (String table : ALL) {
            LegacySchemaTables.getSchemaCFS(table).truncateBlocking();
        }
    }

    private static void flushSchemaTables() {
        for (String table : ALL) {
            SystemKeyspace.forceBlockingFlush(table);
        }
    }

    public static UUID calculateSchemaDigest() {
        MessageDigest digest;
        try {
            digest = MessageDigest.getInstance("MD5");
        }
        catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
        for (String table : ALL) {
            for (Row partition : LegacySchemaTables.getSchemaPartitionsForTable(table)) {
                if (LegacySchemaTables.isEmptySchemaPartition(partition) || LegacySchemaTables.isSystemKeyspaceSchemaPartition(partition)) continue;
                ColumnFamilyStore.removeDeletedColumnsOnly(partition.cf, Integer.MAX_VALUE, SecondaryIndexManager.nullUpdater);
                partition.cf.purgeTombstones(Integer.MAX_VALUE);
                partition.cf.updateDigest(digest);
            }
        }
        return UUID.nameUUIDFromBytes(digest.digest());
    }

    private static ColumnFamilyStore getSchemaCFS(String schemaTableName) {
        return Keyspace.open("system").getColumnFamilyStore(schemaTableName);
    }

    private static List<Row> getSchemaPartitionsForTable(String schemaTableName) {
        Token minToken = StorageService.getPartitioner().getMinimumToken();
        return LegacySchemaTables.getSchemaCFS(schemaTableName).getRangeSlice(new Range<RowPosition>(minToken.minKeyBound(), minToken.maxKeyBound()), null, new IdentityQueryFilter(), Integer.MAX_VALUE, System.currentTimeMillis());
    }

    public static Collection<Mutation> convertSchemaToMutations() {
        HashMap<DecoratedKey, Mutation> mutationMap = new HashMap<DecoratedKey, Mutation>();
        for (String table : ALL) {
            LegacySchemaTables.convertSchemaToMutations(mutationMap, table);
        }
        return mutationMap.values();
    }

    private static void convertSchemaToMutations(Map<DecoratedKey, Mutation> mutationMap, String schemaTableName) {
        for (Row partition : LegacySchemaTables.getSchemaPartitionsForTable(schemaTableName)) {
            if (LegacySchemaTables.isSystemKeyspaceSchemaPartition(partition)) continue;
            Mutation mutation = mutationMap.get(partition.key);
            if (mutation == null) {
                mutation = new Mutation("system", partition.key.getKey());
                mutationMap.put(partition.key, mutation);
            }
            mutation.add(partition.cf);
        }
    }

    private static Map<DecoratedKey, ColumnFamily> readSchemaForKeyspaces(String schemaTableName, Set<String> keyspaceNames) {
        HashMap<DecoratedKey, ColumnFamily> schema = new HashMap<DecoratedKey, ColumnFamily>();
        for (String keyspaceName : keyspaceNames) {
            Row schemaEntity = LegacySchemaTables.readSchemaPartitionForKeyspace(schemaTableName, keyspaceName);
            if (schemaEntity.cf == null) continue;
            schema.put(schemaEntity.key, schemaEntity.cf);
        }
        return schema;
    }

    private static ByteBuffer getSchemaKSKey(String ksName) {
        return AsciiType.instance.fromString(ksName);
    }

    private static Row readSchemaPartitionForKeyspace(String schemaTableName, String keyspaceName) {
        DecoratedKey keyspaceKey = StorageService.getPartitioner().decorateKey(LegacySchemaTables.getSchemaKSKey(keyspaceName));
        return LegacySchemaTables.readSchemaPartitionForKeyspace(schemaTableName, keyspaceKey);
    }

    private static Row readSchemaPartitionForKeyspace(String schemaTableName, DecoratedKey keyspaceKey) {
        QueryFilter filter = QueryFilter.getIdentityFilter(keyspaceKey, schemaTableName, System.currentTimeMillis());
        return new Row(keyspaceKey, LegacySchemaTables.getSchemaCFS(schemaTableName).getColumnFamily(filter));
    }

    private static Row readSchemaPartitionForTable(String schemaTableName, String keyspaceName, String tableName) {
        DecoratedKey key = StorageService.getPartitioner().decorateKey(LegacySchemaTables.getSchemaKSKey(keyspaceName));
        ColumnFamilyStore store = LegacySchemaTables.getSchemaCFS(schemaTableName);
        Composite prefix = store.getComparator().make(tableName);
        ColumnFamily cells = store.getColumnFamily(key, prefix, prefix.end(), false, Integer.MAX_VALUE, System.currentTimeMillis());
        return new Row(key, cells);
    }

    private static boolean isEmptySchemaPartition(Row partition) {
        return partition.cf == null || partition.cf.isMarkedForDelete() && !partition.cf.hasColumns();
    }

    private static boolean isSystemKeyspaceSchemaPartition(Row partition) {
        return LegacySchemaTables.getSchemaKSKey("system").equals(partition.key.getKey());
    }

    public static synchronized void mergeSchema(Collection<Mutation> mutations) throws ConfigurationException, IOException {
        LegacySchemaTables.mergeSchema(mutations, true);
        Schema.instance.updateVersionAndAnnounce();
    }

    public static synchronized void mergeSchema(Collection<Mutation> mutations, boolean doFlush) throws IOException {
        HashSet<String> keyspaces = new HashSet<String>(mutations.size());
        for (Mutation mutation : mutations) {
            keyspaces.add(ByteBufferUtil.string(mutation.key()));
        }
        Map<DecoratedKey, ColumnFamily> oldKeyspaces = LegacySchemaTables.readSchemaForKeyspaces(KEYSPACES, keyspaces);
        Map<DecoratedKey, ColumnFamily> oldColumnFamilies = LegacySchemaTables.readSchemaForKeyspaces(COLUMNFAMILIES, keyspaces);
        Map<DecoratedKey, ColumnFamily> oldTypes = LegacySchemaTables.readSchemaForKeyspaces(USERTYPES, keyspaces);
        Map<DecoratedKey, ColumnFamily> oldFunctions = LegacySchemaTables.readSchemaForKeyspaces(FUNCTIONS, keyspaces);
        Map<DecoratedKey, ColumnFamily> oldAggregates = LegacySchemaTables.readSchemaForKeyspaces(AGGREGATES, keyspaces);
        for (Mutation mutation : mutations) {
            mutation.apply();
        }
        if (doFlush) {
            LegacySchemaTables.flushSchemaTables();
        }
        Map<DecoratedKey, ColumnFamily> newKeyspaces = LegacySchemaTables.readSchemaForKeyspaces(KEYSPACES, keyspaces);
        Map<DecoratedKey, ColumnFamily> newColumnFamilies = LegacySchemaTables.readSchemaForKeyspaces(COLUMNFAMILIES, keyspaces);
        Map<DecoratedKey, ColumnFamily> newTypes = LegacySchemaTables.readSchemaForKeyspaces(USERTYPES, keyspaces);
        Map<DecoratedKey, ColumnFamily> newFunctions = LegacySchemaTables.readSchemaForKeyspaces(FUNCTIONS, keyspaces);
        Map<DecoratedKey, ColumnFamily> newAggregates = LegacySchemaTables.readSchemaForKeyspaces(AGGREGATES, keyspaces);
        Set<String> keyspacesToDrop = LegacySchemaTables.mergeKeyspaces(oldKeyspaces, newKeyspaces);
        LegacySchemaTables.mergeTables(oldColumnFamilies, newColumnFamilies);
        LegacySchemaTables.mergeTypes(oldTypes, newTypes);
        LegacySchemaTables.mergeFunctions(oldFunctions, newFunctions);
        LegacySchemaTables.mergeAggregates(oldAggregates, newAggregates);
        for (String keyspaceToDrop : keyspacesToDrop) {
            Schema.instance.dropKeyspace(keyspaceToDrop);
        }
    }

    private static Set<String> mergeKeyspaces(Map<DecoratedKey, ColumnFamily> before, Map<DecoratedKey, ColumnFamily> after) {
        ArrayList<Row> created = new ArrayList<Row>();
        ArrayList<String> altered = new ArrayList<String>();
        HashSet<String> dropped = new HashSet<String>();
        MapDifference diff = Maps.difference(before, after);
        for (Map.Entry entry : diff.entriesOnlyOnRight().entrySet()) {
            if (!((ColumnFamily)entry.getValue()).hasColumns()) continue;
            created.add(new Row((DecoratedKey)entry.getKey(), (ColumnFamily)entry.getValue()));
        }
        for (Map.Entry entry : diff.entriesDiffering().entrySet()) {
            String keyspaceName = (String)AsciiType.instance.compose(((DecoratedKey)entry.getKey()).getKey());
            ColumnFamily pre = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).leftValue();
            ColumnFamily post = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).rightValue();
            if (pre.hasColumns() && post.hasColumns()) {
                altered.add(keyspaceName);
                continue;
            }
            if (pre.hasColumns()) {
                dropped.add(keyspaceName);
                continue;
            }
            if (!post.hasColumns()) continue;
            created.add(new Row((DecoratedKey)entry.getKey(), post));
        }
        for (Row row : created) {
            Schema.instance.addKeyspace(LegacySchemaTables.createKeyspaceFromSchemaPartition(row));
        }
        for (String name : altered) {
            Schema.instance.updateKeyspace(name);
        }
        return dropped;
    }

    private static void mergeTables(Map<DecoratedKey, ColumnFamily> before, Map<DecoratedKey, ColumnFamily> after) {
        ArrayList<CFMetaData> created = new ArrayList<CFMetaData>();
        ArrayList altered = new ArrayList();
        ArrayList<Object> dropped = new ArrayList<Object>();
        MapDifference diff = Maps.difference(before, after);
        for (Map.Entry entry : diff.entriesOnlyOnRight().entrySet()) {
            if (!((ColumnFamily)entry.getValue()).hasColumns()) continue;
            created.addAll(LegacySchemaTables.createTablesFromTablesPartition(new Row((DecoratedKey)entry.getKey(), (ColumnFamily)entry.getValue())).values());
        }
        for (Map.Entry entry : diff.entriesDiffering().entrySet()) {
            String keyspaceName = (String)AsciiType.instance.compose(((DecoratedKey)entry.getKey()).getKey());
            ColumnFamily pre = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).leftValue();
            ColumnFamily post = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).rightValue();
            if (pre.hasColumns() && post.hasColumns()) {
                MapDifference delta = Maps.difference(Schema.instance.getKSMetaData(keyspaceName).cfMetaData(), LegacySchemaTables.createTablesFromTablesPartition(new Row((DecoratedKey)entry.getKey(), post)));
                dropped.addAll(delta.entriesOnlyOnLeft().values());
                created.addAll(delta.entriesOnlyOnRight().values());
                Iterables.addAll(altered, (Iterable)Iterables.transform(delta.entriesDiffering().values(), (Function)new Function<MapDifference.ValueDifference<CFMetaData>, CFMetaData>(){

                    public CFMetaData apply(MapDifference.ValueDifference<CFMetaData> pair) {
                        return (CFMetaData)pair.rightValue();
                    }
                }));
                continue;
            }
            if (pre.hasColumns()) {
                dropped.addAll(Schema.instance.getKSMetaData(keyspaceName).cfMetaData().values());
                continue;
            }
            if (!post.hasColumns()) continue;
            created.addAll(LegacySchemaTables.createTablesFromTablesPartition(new Row((DecoratedKey)entry.getKey(), post)).values());
        }
        for (CFMetaData cFMetaData : created) {
            Schema.instance.addTable(cFMetaData);
        }
        for (CFMetaData cFMetaData : altered) {
            Schema.instance.updateTable(cFMetaData.ksName, cFMetaData.cfName);
        }
        for (CFMetaData cFMetaData : dropped) {
            Schema.instance.dropTable(cFMetaData.ksName, cFMetaData.cfName);
        }
    }

    private static void mergeTypes(Map<DecoratedKey, ColumnFamily> before, Map<DecoratedKey, ColumnFamily> after) {
        ArrayList<UserType> created = new ArrayList<UserType>();
        ArrayList altered = new ArrayList();
        ArrayList<Object> dropped = new ArrayList<Object>();
        MapDifference diff = Maps.difference(before, after);
        for (Map.Entry entry : diff.entriesOnlyOnRight().entrySet()) {
            if (!((ColumnFamily)entry.getValue()).hasColumns()) continue;
            created.addAll(LegacySchemaTables.createTypesFromPartition(new Row((DecoratedKey)entry.getKey(), (ColumnFamily)entry.getValue())).values());
        }
        for (Map.Entry entry : diff.entriesDiffering().entrySet()) {
            String keyspaceName = (String)AsciiType.instance.compose(((DecoratedKey)entry.getKey()).getKey());
            ColumnFamily pre = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).leftValue();
            ColumnFamily post = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).rightValue();
            if (pre.hasColumns() && post.hasColumns()) {
                MapDifference delta = Maps.difference(Schema.instance.getKSMetaData((String)keyspaceName).userTypes.getAllTypes(), LegacySchemaTables.createTypesFromPartition(new Row((DecoratedKey)entry.getKey(), post)));
                dropped.addAll(delta.entriesOnlyOnLeft().values());
                created.addAll(delta.entriesOnlyOnRight().values());
                Iterables.addAll(altered, (Iterable)Iterables.transform(delta.entriesDiffering().values(), (Function)new Function<MapDifference.ValueDifference<UserType>, UserType>(){

                    public UserType apply(MapDifference.ValueDifference<UserType> pair) {
                        return (UserType)pair.rightValue();
                    }
                }));
                continue;
            }
            if (pre.hasColumns()) {
                dropped.addAll(Schema.instance.getKSMetaData((String)keyspaceName).userTypes.getAllTypes().values());
                continue;
            }
            if (!post.hasColumns()) continue;
            created.addAll(LegacySchemaTables.createTypesFromPartition(new Row((DecoratedKey)entry.getKey(), post)).values());
        }
        for (UserType userType : created) {
            Schema.instance.addType(userType);
        }
        for (UserType userType : altered) {
            Schema.instance.updateType(userType);
        }
        for (UserType userType : dropped) {
            Schema.instance.dropType(userType);
        }
    }

    private static void mergeFunctions(Map<DecoratedKey, ColumnFamily> before, Map<DecoratedKey, ColumnFamily> after) {
        ArrayList<UDFunction> created = new ArrayList<UDFunction>();
        ArrayList altered = new ArrayList();
        ArrayList<Object> dropped = new ArrayList<Object>();
        MapDifference diff = Maps.difference(before, after);
        for (Map.Entry entry : diff.entriesOnlyOnRight().entrySet()) {
            if (!((ColumnFamily)entry.getValue()).hasColumns()) continue;
            created.addAll(LegacySchemaTables.createFunctionsFromFunctionsPartition(new Row((DecoratedKey)entry.getKey(), (ColumnFamily)entry.getValue())).values());
        }
        for (Map.Entry entry : diff.entriesDiffering().entrySet()) {
            ColumnFamily pre = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).leftValue();
            ColumnFamily post = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).rightValue();
            if (pre.hasColumns() && post.hasColumns()) {
                MapDifference delta = Maps.difference(LegacySchemaTables.createFunctionsFromFunctionsPartition(new Row((DecoratedKey)entry.getKey(), pre)), LegacySchemaTables.createFunctionsFromFunctionsPartition(new Row((DecoratedKey)entry.getKey(), post)));
                dropped.addAll(delta.entriesOnlyOnLeft().values());
                created.addAll(delta.entriesOnlyOnRight().values());
                Iterables.addAll(altered, (Iterable)Iterables.transform(delta.entriesDiffering().values(), (Function)new Function<MapDifference.ValueDifference<UDFunction>, UDFunction>(){

                    public UDFunction apply(MapDifference.ValueDifference<UDFunction> pair) {
                        return (UDFunction)pair.rightValue();
                    }
                }));
                continue;
            }
            if (pre.hasColumns()) {
                dropped.addAll(LegacySchemaTables.createFunctionsFromFunctionsPartition(new Row((DecoratedKey)entry.getKey(), pre)).values());
                continue;
            }
            if (!post.hasColumns()) continue;
            created.addAll(LegacySchemaTables.createFunctionsFromFunctionsPartition(new Row((DecoratedKey)entry.getKey(), post)).values());
        }
        for (UDFunction uDFunction : created) {
            Schema.instance.addFunction(uDFunction);
        }
        for (UDFunction uDFunction : altered) {
            Schema.instance.updateFunction(uDFunction);
        }
        for (UDFunction uDFunction : dropped) {
            Schema.instance.dropFunction(uDFunction);
        }
    }

    private static void mergeAggregates(Map<DecoratedKey, ColumnFamily> before, Map<DecoratedKey, ColumnFamily> after) {
        ArrayList<UDAggregate> created = new ArrayList<UDAggregate>();
        ArrayList altered = new ArrayList();
        ArrayList<Object> dropped = new ArrayList<Object>();
        MapDifference diff = Maps.difference(before, after);
        for (Map.Entry entry : diff.entriesOnlyOnRight().entrySet()) {
            if (!((ColumnFamily)entry.getValue()).hasColumns()) continue;
            created.addAll(LegacySchemaTables.createAggregatesFromAggregatesPartition(new Row((DecoratedKey)entry.getKey(), (ColumnFamily)entry.getValue())).values());
        }
        for (Map.Entry entry : diff.entriesDiffering().entrySet()) {
            ColumnFamily pre = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).leftValue();
            ColumnFamily post = (ColumnFamily)((MapDifference.ValueDifference)entry.getValue()).rightValue();
            if (pre.hasColumns() && post.hasColumns()) {
                MapDifference delta = Maps.difference(LegacySchemaTables.createAggregatesFromAggregatesPartition(new Row((DecoratedKey)entry.getKey(), pre)), LegacySchemaTables.createAggregatesFromAggregatesPartition(new Row((DecoratedKey)entry.getKey(), post)));
                dropped.addAll(delta.entriesOnlyOnLeft().values());
                created.addAll(delta.entriesOnlyOnRight().values());
                Iterables.addAll(altered, (Iterable)Iterables.transform(delta.entriesDiffering().values(), (Function)new Function<MapDifference.ValueDifference<UDAggregate>, UDAggregate>(){

                    public UDAggregate apply(MapDifference.ValueDifference<UDAggregate> pair) {
                        return (UDAggregate)pair.rightValue();
                    }
                }));
                continue;
            }
            if (pre.hasColumns()) {
                dropped.addAll(LegacySchemaTables.createAggregatesFromAggregatesPartition(new Row((DecoratedKey)entry.getKey(), pre)).values());
                continue;
            }
            if (!post.hasColumns()) continue;
            created.addAll(LegacySchemaTables.createAggregatesFromAggregatesPartition(new Row((DecoratedKey)entry.getKey(), post)).values());
        }
        for (UDAggregate uDAggregate : created) {
            Schema.instance.addAggregate(uDAggregate);
        }
        for (UDAggregate uDAggregate : altered) {
            Schema.instance.updateAggregate(uDAggregate);
        }
        for (UDAggregate uDAggregate : dropped) {
            Schema.instance.dropAggregate(uDAggregate);
        }
    }

    public static Mutation makeCreateKeyspaceMutation(KSMetaData keyspace, long timestamp) {
        return LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, true);
    }

    private static Mutation makeCreateKeyspaceMutation(KSMetaData keyspace, long timestamp, boolean withTablesAndTypesAndFunctions) {
        Mutation mutation = new Mutation("system", LegacySchemaTables.getSchemaKSKey(keyspace.name));
        ColumnFamily cells = mutation.addOrGet(Keyspaces);
        CFRowAdder adder = new CFRowAdder(cells, LegacySchemaTables.Keyspaces.comparator.builder().build(), timestamp);
        adder.add("durable_writes", keyspace.durableWrites);
        adder.add("strategy_class", keyspace.strategyClass.getName());
        adder.add("strategy_options", FBUtilities.json(keyspace.strategyOptions));
        if (withTablesAndTypesAndFunctions) {
            for (UserType type : keyspace.userTypes.getAllTypes().values()) {
                LegacySchemaTables.addTypeToSchemaMutation(type, timestamp, mutation);
            }
            for (CFMetaData table : keyspace.cfMetaData().values()) {
                LegacySchemaTables.addTableToSchemaMutation(table, timestamp, true, mutation);
            }
        }
        return mutation;
    }

    public static Mutation makeDropKeyspaceMutation(KSMetaData keyspace, long timestamp) {
        Mutation mutation = new Mutation("system", LegacySchemaTables.getSchemaKSKey(keyspace.name));
        for (String schemaTable : ALL) {
            mutation.delete(schemaTable, timestamp);
        }
        mutation.delete("IndexInfo", timestamp);
        return mutation;
    }

    private static KSMetaData createKeyspaceFromSchemaPartitions(Row serializedKeyspace, Row serializedTables, Row serializedTypes) {
        Collection<CFMetaData> tables = LegacySchemaTables.createTablesFromTablesPartition(serializedTables).values();
        UTMetaData types = new UTMetaData(LegacySchemaTables.createTypesFromPartition(serializedTypes));
        return LegacySchemaTables.createKeyspaceFromSchemaPartition(serializedKeyspace).cloneWith(tables, types);
    }

    public static KSMetaData createKeyspaceFromName(String keyspace) {
        Row partition = LegacySchemaTables.readSchemaPartitionForKeyspace(KEYSPACES, keyspace);
        if (LegacySchemaTables.isEmptySchemaPartition(partition)) {
            throw new RuntimeException(String.format("%s not found in the schema definitions keyspaceName (%s).", keyspace, KEYSPACES));
        }
        return LegacySchemaTables.createKeyspaceFromSchemaPartition(partition);
    }

    private static KSMetaData createKeyspaceFromSchemaPartition(Row partition) {
        String query = String.format("SELECT * FROM %s.%s", "system", KEYSPACES);
        UntypedResultSet.Row row = QueryProcessor.resultify(query, partition).one();
        return new KSMetaData(row.getString("keyspace_name"), AbstractReplicationStrategy.getClass(row.getString("strategy_class")), FBUtilities.fromJsonMap(row.getString("strategy_options")), row.getBoolean("durable_writes"));
    }

    public static Mutation makeCreateTypeMutation(KSMetaData keyspace, UserType type, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        LegacySchemaTables.addTypeToSchemaMutation(type, timestamp, mutation);
        return mutation;
    }

    private static void addTypeToSchemaMutation(UserType type, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Usertypes);
        Composite prefix = LegacySchemaTables.Usertypes.comparator.make(type.name);
        CFRowAdder adder = new CFRowAdder(cells, prefix, timestamp);
        adder.resetCollection("field_names");
        adder.resetCollection("field_types");
        for (int i = 0; i < type.size(); ++i) {
            adder.addListEntry("field_names", type.fieldName(i));
            adder.addListEntry("field_types", type.fieldType(i).toString());
        }
    }

    public static Mutation dropTypeFromSchemaMutation(KSMetaData keyspace, UserType type, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        ColumnFamily cells = mutation.addOrGet(Usertypes);
        int ldt = (int)(System.currentTimeMillis() / 1000L);
        Composite prefix = LegacySchemaTables.Usertypes.comparator.make(type.name);
        cells.addAtom(new RangeTombstone(prefix, prefix.end(), timestamp, ldt));
        return mutation;
    }

    private static Map<ByteBuffer, UserType> createTypesFromPartition(Row partition) {
        String query = String.format("SELECT * FROM %s.%s", "system", USERTYPES);
        HashMap<ByteBuffer, UserType> types = new HashMap<ByteBuffer, UserType>();
        for (UntypedResultSet.Row row : QueryProcessor.resultify(query, partition)) {
            UserType type = LegacySchemaTables.createTypeFromRow(row);
            types.put(type.name, type);
        }
        return types;
    }

    private static UserType createTypeFromRow(UntypedResultSet.Row row) {
        String keyspace = row.getString("keyspace_name");
        ByteBuffer name = ByteBufferUtil.bytes(row.getString("type_name"));
        List<String> rawColumns = row.getList("field_names", UTF8Type.instance);
        List<String> rawTypes = row.getList("field_types", UTF8Type.instance);
        ArrayList<ByteBuffer> columns = new ArrayList<ByteBuffer>(rawColumns.size());
        for (String rawColumn : rawColumns) {
            columns.add(ByteBufferUtil.bytes(rawColumn));
        }
        ArrayList types = new ArrayList(rawTypes.size());
        for (String rawType : rawTypes) {
            types.add(LegacySchemaTables.parseType(rawType));
        }
        return new UserType(keyspace, name, columns, types);
    }

    public static Mutation makeCreateTableMutation(KSMetaData keyspace, CFMetaData table, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        LegacySchemaTables.addTableToSchemaMutation(table, timestamp, true, mutation);
        return mutation;
    }

    private static void addTableToSchemaMutation(CFMetaData table, long timestamp, boolean withColumnsAndTriggers, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Columnfamilies);
        Composite prefix = LegacySchemaTables.Columnfamilies.comparator.make(table.cfName);
        CFRowAdder adder = new CFRowAdder(cells, prefix, timestamp);
        adder.add("cf_id", table.cfId);
        adder.add("type", table.cfType.toString());
        if (table.isSuper()) {
            adder.add("comparator", table.comparator.subtype(0).toString());
            adder.add("subcomparator", table.comparator.subtype(1).toString());
        } else {
            adder.add("comparator", table.comparator.toString());
        }
        adder.add("bloom_filter_fp_chance", table.getBloomFilterFpChance());
        adder.add("caching", table.getCaching().toString());
        adder.add("comment", table.getComment());
        adder.add("compaction_strategy_class", table.compactionStrategyClass.getName());
        adder.add("compaction_strategy_options", FBUtilities.json(table.compactionStrategyOptions));
        adder.add("compression_parameters", FBUtilities.json(table.compressionParameters.asThriftOptions()));
        adder.add("default_time_to_live", table.getDefaultTimeToLive());
        adder.add("default_validator", table.getDefaultValidator().toString());
        adder.add("gc_grace_seconds", table.getGcGraceSeconds());
        adder.add("key_validator", table.getKeyValidator().toString());
        adder.add("local_read_repair_chance", table.getDcLocalReadRepairChance());
        adder.add("max_compaction_threshold", table.getMaxCompactionThreshold());
        adder.add("max_index_interval", table.getMaxIndexInterval());
        adder.add("memtable_flush_period_in_ms", table.getMemtableFlushPeriod());
        adder.add("min_compaction_threshold", table.getMinCompactionThreshold());
        adder.add("min_index_interval", table.getMinIndexInterval());
        adder.add("read_repair_chance", table.getReadRepairChance());
        adder.add("speculative_retry", table.getSpeculativeRetry().toString());
        for (Map.Entry<ColumnIdentifier, Long> entry : table.getDroppedColumns().entrySet()) {
            adder.addMapEntry("dropped_columns", entry.getKey().toString(), entry.getValue());
        }
        adder.add("is_dense", table.getIsDense());
        if (withColumnsAndTriggers) {
            for (ColumnDefinition column : table.allColumns()) {
                LegacySchemaTables.addColumnToSchemaMutation(table, column, timestamp, mutation);
            }
            for (TriggerDefinition trigger : table.getTriggers().values()) {
                LegacySchemaTables.addTriggerToSchemaMutation(table, trigger, timestamp, mutation);
            }
        }
    }

    public static Mutation makeUpdateTableMutation(KSMetaData keyspace, CFMetaData oldTable, CFMetaData newTable, long timestamp, boolean fromThrift) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        LegacySchemaTables.addTableToSchemaMutation(newTable, timestamp, false, mutation);
        MapDifference columnDiff = Maps.difference(oldTable.getColumnMetadata(), newTable.getColumnMetadata());
        for (ColumnDefinition column : columnDiff.entriesOnlyOnLeft().values()) {
            if (fromThrift && column.kind != ColumnDefinition.Kind.REGULAR) continue;
            LegacySchemaTables.dropColumnFromSchemaMutation(oldTable, column, timestamp, mutation);
        }
        for (ColumnDefinition column : columnDiff.entriesOnlyOnRight().values()) {
            LegacySchemaTables.addColumnToSchemaMutation(newTable, column, timestamp, mutation);
        }
        for (ByteBuffer name : columnDiff.entriesDiffering().keySet()) {
            LegacySchemaTables.addColumnToSchemaMutation(newTable, newTable.getColumnDefinition(name), timestamp, mutation);
        }
        MapDifference triggerDiff = Maps.difference(oldTable.getTriggers(), newTable.getTriggers());
        for (TriggerDefinition trigger : triggerDiff.entriesOnlyOnLeft().values()) {
            LegacySchemaTables.dropTriggerFromSchemaMutation(oldTable, trigger, timestamp, mutation);
        }
        for (TriggerDefinition trigger : triggerDiff.entriesOnlyOnRight().values()) {
            LegacySchemaTables.addTriggerToSchemaMutation(newTable, trigger, timestamp, mutation);
        }
        return mutation;
    }

    public static Mutation makeDropTableMutation(KSMetaData keyspace, CFMetaData table, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        ColumnFamily cells = mutation.addOrGet(Columnfamilies);
        int ldt = (int)(System.currentTimeMillis() / 1000L);
        Composite prefix = LegacySchemaTables.Columnfamilies.comparator.make(table.cfName);
        cells.addAtom(new RangeTombstone(prefix, prefix.end(), timestamp, ldt));
        for (ColumnDefinition column : table.allColumns()) {
            LegacySchemaTables.dropColumnFromSchemaMutation(table, column, timestamp, mutation);
        }
        for (TriggerDefinition trigger : table.getTriggers().values()) {
            LegacySchemaTables.dropTriggerFromSchemaMutation(table, trigger, timestamp, mutation);
        }
        ColumnFamily indexCells = mutation.addOrGet(SystemKeyspace.BuiltIndexes);
        for (String indexName : Keyspace.open(keyspace.name).getColumnFamilyStore(table.cfName).getBuiltIndexes()) {
            indexCells.addTombstone(indexCells.getComparator().makeCellName(indexName), ldt, timestamp);
        }
        return mutation;
    }

    public static CFMetaData createTableFromName(String keyspace, String table) {
        Row partition = LegacySchemaTables.readSchemaPartitionForTable(COLUMNFAMILIES, keyspace, table);
        if (LegacySchemaTables.isEmptySchemaPartition(partition)) {
            throw new RuntimeException(String.format("%s:%s not found in the schema definitions keyspace.", keyspace, table));
        }
        return LegacySchemaTables.createTableFromTablePartition(partition);
    }

    private static Map<String, CFMetaData> createTablesFromTablesPartition(Row partition) {
        if (partition.cf == null) {
            return Collections.emptyMap();
        }
        String query = String.format("SELECT * FROM %s.%s", "system", COLUMNFAMILIES);
        HashMap<String, CFMetaData> tables = new HashMap<String, CFMetaData>();
        for (UntypedResultSet.Row row : QueryProcessor.resultify(query, partition)) {
            CFMetaData cfm = LegacySchemaTables.createTableFromTableRow(row);
            tables.put(cfm.cfName, cfm);
        }
        return tables;
    }

    public static CFMetaData createTableFromTablePartitionAndColumnsPartition(Row serializedTable, Row serializedColumns) {
        String query = String.format("SELECT * FROM %s.%s", "system", COLUMNFAMILIES);
        return LegacySchemaTables.createTableFromTableRowAndColumnsPartition(QueryProcessor.resultify(query, serializedTable).one(), serializedColumns);
    }

    private static CFMetaData createTableFromTableRowAndColumnsPartition(UntypedResultSet.Row tableRow, Row serializedColumns) {
        String query = String.format("SELECT * FROM %s.%s", "system", COLUMNS);
        return LegacySchemaTables.createTableFromTableRowAndColumnRows(tableRow, QueryProcessor.resultify(query, serializedColumns));
    }

    private static CFMetaData createTableFromTablePartition(Row row) {
        String query = String.format("SELECT * FROM %s.%s", "system", COLUMNFAMILIES);
        return LegacySchemaTables.createTableFromTableRow(QueryProcessor.resultify(query, row).one());
    }

    private static CFMetaData createTableFromTableRow(UntypedResultSet.Row result) {
        String ksName = result.getString("keyspace_name");
        String cfName = result.getString("columnfamily_name");
        Row serializedColumns = LegacySchemaTables.readSchemaPartitionForTable(COLUMNS, ksName, cfName);
        CFMetaData cfm = LegacySchemaTables.createTableFromTableRowAndColumnsPartition(result, serializedColumns);
        Row serializedTriggers = LegacySchemaTables.readSchemaPartitionForTable(TRIGGERS, ksName, cfName);
        for (TriggerDefinition trigger : LegacySchemaTables.createTriggersFromTriggersPartition(serializedTriggers)) {
            cfm.addTriggerDefinition(trigger);
        }
        return cfm;
    }

    public static CFMetaData createTableFromTableRowAndColumnRows(UntypedResultSet.Row result, UntypedResultSet serializedColumnDefinitions) {
        String ksName = result.getString("keyspace_name");
        String cfName = result.getString("columnfamily_name");
        AbstractType<?> rawComparator = TypeParser.parse(result.getString("comparator"));
        AbstractType<?> subComparator = result.has("subcomparator") ? TypeParser.parse(result.getString("subcomparator")) : null;
        ColumnFamilyType cfType = ColumnFamilyType.valueOf(result.getString("type"));
        AbstractType<?> fullRawComparator = CFMetaData.makeRawAbstractType(rawComparator, subComparator);
        List<ColumnDefinition> columnDefs = LegacySchemaTables.createColumnsFromColumnRows(serializedColumnDefinitions, ksName, cfName, fullRawComparator, cfType == ColumnFamilyType.Super);
        boolean isDense = result.has("is_dense") ? result.getBoolean("is_dense") : CFMetaData.calculateIsDense(fullRawComparator, columnDefs);
        CellNameType comparator = CellNames.fromAbstractType(fullRawComparator, isDense);
        UUID cfId = result.has("cf_id") ? result.getUUID("cf_id") : CFMetaData.generateLegacyCfId(ksName, cfName);
        CFMetaData cfm = new CFMetaData(ksName, cfName, cfType, comparator, cfId);
        cfm.isDense(isDense);
        cfm.readRepairChance(result.getDouble("read_repair_chance"));
        cfm.dcLocalReadRepairChance(result.getDouble("local_read_repair_chance"));
        cfm.gcGraceSeconds(result.getInt("gc_grace_seconds"));
        cfm.defaultValidator(TypeParser.parse(result.getString("default_validator")));
        cfm.keyValidator(TypeParser.parse(result.getString("key_validator")));
        cfm.minCompactionThreshold(result.getInt("min_compaction_threshold"));
        cfm.maxCompactionThreshold(result.getInt("max_compaction_threshold"));
        if (result.has("comment")) {
            cfm.comment(result.getString("comment"));
        }
        if (result.has("memtable_flush_period_in_ms")) {
            cfm.memtableFlushPeriod(result.getInt("memtable_flush_period_in_ms"));
        }
        cfm.caching(CachingOptions.fromString(result.getString("caching")));
        if (result.has("default_time_to_live")) {
            cfm.defaultTimeToLive(result.getInt("default_time_to_live"));
        }
        if (result.has("speculative_retry")) {
            cfm.speculativeRetry(CFMetaData.SpeculativeRetry.fromString(result.getString("speculative_retry")));
        }
        cfm.compactionStrategyClass(CFMetaData.createCompactionStrategy(result.getString("compaction_strategy_class")));
        cfm.compressionParameters(CompressionParameters.create(FBUtilities.fromJsonMap(result.getString("compression_parameters"))));
        cfm.compactionStrategyOptions(FBUtilities.fromJsonMap(result.getString("compaction_strategy_options")));
        if (result.has("min_index_interval")) {
            cfm.minIndexInterval(result.getInt("min_index_interval"));
        }
        if (result.has("max_index_interval")) {
            cfm.maxIndexInterval(result.getInt("max_index_interval"));
        }
        if (result.has("bloom_filter_fp_chance")) {
            cfm.bloomFilterFpChance(result.getDouble("bloom_filter_fp_chance"));
        } else {
            cfm.bloomFilterFpChance(cfm.getBloomFilterFpChance());
        }
        if (result.has("dropped_columns")) {
            cfm.droppedColumns(LegacySchemaTables.convertDroppedColumns(result.getMap("dropped_columns", UTF8Type.instance, LongType.instance)));
        }
        for (ColumnDefinition cd : columnDefs) {
            cfm.addOrReplaceColumnDefinition(cd);
        }
        return cfm.rebuild();
    }

    private static Map<ColumnIdentifier, Long> convertDroppedColumns(Map<String, Long> raw) {
        HashMap converted = Maps.newHashMap();
        for (Map.Entry<String, Long> entry : raw.entrySet()) {
            converted.put(new ColumnIdentifier(entry.getKey(), true), entry.getValue());
        }
        return converted;
    }

    private static void addColumnToSchemaMutation(CFMetaData table, ColumnDefinition column, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Columns);
        Composite prefix = LegacySchemaTables.Columns.comparator.make(table.cfName, column.name.toString());
        CFRowAdder adder = new CFRowAdder(cells, prefix, timestamp);
        adder.add("validator", column.type.toString());
        adder.add("type", LegacySchemaTables.serializeKind(column.kind));
        adder.add("component_index", column.isOnAllComponents() ? null : Integer.valueOf(column.position()));
        adder.add("index_name", column.getIndexName());
        adder.add("index_type", column.getIndexType() == null ? null : column.getIndexType().toString());
        adder.add("index_options", FBUtilities.json(column.getIndexOptions()));
    }

    private static String serializeKind(ColumnDefinition.Kind kind) {
        return kind == ColumnDefinition.Kind.CLUSTERING_COLUMN ? "clustering_key" : kind.toString().toLowerCase();
    }

    private static ColumnDefinition.Kind deserializeKind(String kind) {
        if (kind.equalsIgnoreCase("clustering_key")) {
            return ColumnDefinition.Kind.CLUSTERING_COLUMN;
        }
        return Enum.valueOf(ColumnDefinition.Kind.class, kind.toUpperCase());
    }

    private static void dropColumnFromSchemaMutation(CFMetaData table, ColumnDefinition column, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Columns);
        int ldt = (int)(System.currentTimeMillis() / 1000L);
        Composite prefix = LegacySchemaTables.Columns.comparator.make(table.cfName, column.name.toString());
        cells.addAtom(new RangeTombstone(prefix, prefix.end(), timestamp, ldt));
    }

    private static List<ColumnDefinition> createColumnsFromColumnRows(UntypedResultSet rows, String keyspace, String table, AbstractType<?> rawComparator, boolean isSuper) {
        ArrayList<ColumnDefinition> columns = new ArrayList<ColumnDefinition>();
        for (UntypedResultSet.Row row : rows) {
            columns.add(LegacySchemaTables.createColumnFromColumnRow(row, keyspace, table, rawComparator, isSuper));
        }
        return columns;
    }

    private static ColumnDefinition createColumnFromColumnRow(UntypedResultSet.Row row, String keyspace, String table, AbstractType<?> rawComparator, boolean isSuper) {
        ColumnDefinition.Kind kind = LegacySchemaTables.deserializeKind(row.getString("type"));
        Integer componentIndex = null;
        if (row.has("component_index")) {
            componentIndex = row.getInt("component_index");
        } else if (kind == ColumnDefinition.Kind.CLUSTERING_COLUMN && isSuper) {
            componentIndex = 1;
        }
        UTF8Type comparator = kind == ColumnDefinition.Kind.REGULAR ? LegacySchemaTables.getComponentComparator(rawComparator, componentIndex) : UTF8Type.instance;
        ColumnIdentifier name = new ColumnIdentifier(((AbstractType)comparator).fromString(row.getString("column_name")), comparator);
        AbstractType<?> validator = LegacySchemaTables.parseType(row.getString("validator"));
        IndexType indexType = null;
        if (row.has("index_type")) {
            indexType = IndexType.valueOf(row.getString("index_type"));
        }
        Map<String, String> indexOptions = null;
        if (row.has("index_options")) {
            indexOptions = FBUtilities.fromJsonMap(row.getString("index_options"));
        }
        String indexName = null;
        if (row.has("index_name")) {
            indexName = row.getString("index_name");
        }
        return new ColumnDefinition(keyspace, table, name, validator, indexType, indexOptions, indexName, componentIndex, kind);
    }

    private static AbstractType<?> getComponentComparator(AbstractType<?> rawComparator, Integer componentIndex) {
        return componentIndex == null || componentIndex == 0 && !(rawComparator instanceof CompositeType) ? rawComparator : ((CompositeType)rawComparator).types.get(componentIndex);
    }

    private static void addTriggerToSchemaMutation(CFMetaData table, TriggerDefinition trigger, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Triggers);
        Composite prefix = LegacySchemaTables.Triggers.comparator.make(table.cfName, trigger.name);
        CFRowAdder adder = new CFRowAdder(cells, prefix, timestamp);
        adder.addMapEntry("trigger_options", "class", trigger.classOption);
    }

    private static void dropTriggerFromSchemaMutation(CFMetaData table, TriggerDefinition trigger, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Triggers);
        int ldt = (int)(System.currentTimeMillis() / 1000L);
        Composite prefix = LegacySchemaTables.Triggers.comparator.make(table.cfName, trigger.name);
        cells.addAtom(new RangeTombstone(prefix, prefix.end(), timestamp, ldt));
    }

    private static List<TriggerDefinition> createTriggersFromTriggersPartition(Row partition) {
        ArrayList<TriggerDefinition> triggers = new ArrayList<TriggerDefinition>();
        String query = String.format("SELECT * FROM %s.%s", "system", TRIGGERS);
        for (UntypedResultSet.Row row : QueryProcessor.resultify(query, partition)) {
            String name = row.getString("trigger_name");
            String classOption = row.getMap("trigger_options", UTF8Type.instance, UTF8Type.instance).get("class");
            triggers.add(new TriggerDefinition(name, classOption));
        }
        return triggers;
    }

    public static Mutation makeCreateFunctionMutation(KSMetaData keyspace, UDFunction function, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        LegacySchemaTables.addFunctionToSchemaMutation(function, timestamp, mutation);
        return mutation;
    }

    private static void addFunctionToSchemaMutation(UDFunction function, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Functions);
        Composite prefix = LegacySchemaTables.Functions.comparator.make(function.name().name, LegacySchemaTables.functionSignatureWithTypes(function));
        CFRowAdder adder = new CFRowAdder(cells, prefix, timestamp);
        adder.resetCollection("argument_names");
        adder.resetCollection("argument_types");
        for (int i = 0; i < function.argNames().size(); ++i) {
            adder.addListEntry("argument_names", function.argNames().get((int)i).bytes);
            adder.addListEntry("argument_types", function.argTypes().get(i).toString());
        }
        adder.add("body", function.body());
        adder.add("language", function.language());
        adder.add("return_type", function.returnType().toString());
        adder.add("called_on_null_input", function.isCalledOnNullInput());
    }

    public static Mutation makeDropFunctionMutation(KSMetaData keyspace, UDFunction function, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        ColumnFamily cells = mutation.addOrGet(Functions);
        int ldt = (int)(System.currentTimeMillis() / 1000L);
        Composite prefix = LegacySchemaTables.Functions.comparator.make(function.name().name, LegacySchemaTables.functionSignatureWithTypes(function));
        cells.addAtom(new RangeTombstone(prefix, prefix.end(), timestamp, ldt));
        return mutation;
    }

    private static Map<ByteBuffer, UDFunction> createFunctionsFromFunctionsPartition(Row partition) {
        HashMap<ByteBuffer, UDFunction> functions = new HashMap<ByteBuffer, UDFunction>();
        String query = String.format("SELECT * FROM %s.%s", "system", FUNCTIONS);
        for (UntypedResultSet.Row row : QueryProcessor.resultify(query, partition)) {
            UDFunction function = LegacySchemaTables.createFunctionFromFunctionRow(row);
            functions.put(LegacySchemaTables.functionSignatureWithNameAndTypes(function), function);
        }
        return functions;
    }

    private static UDFunction createFunctionFromFunctionRow(UntypedResultSet.Row row) {
        String ksName = row.getString("keyspace_name");
        String functionName = row.getString("function_name");
        FunctionName name = new FunctionName(ksName, functionName);
        ArrayList<ColumnIdentifier> argNames = new ArrayList<ColumnIdentifier>();
        if (row.has("argument_names")) {
            for (String string : row.getList("argument_names", UTF8Type.instance)) {
                argNames.add(new ColumnIdentifier(string, true));
            }
        }
        ArrayList argTypes = new ArrayList();
        if (row.has("argument_types")) {
            for (String type : row.getList("argument_types", UTF8Type.instance)) {
                argTypes.add(LegacySchemaTables.parseType(type));
            }
        }
        AbstractType<?> abstractType = LegacySchemaTables.parseType(row.getString("return_type"));
        String language = row.getString("language");
        String body = row.getString("body");
        boolean calledOnNullInput = row.getBoolean("called_on_null_input");
        try {
            return UDFunction.create(name, argNames, argTypes, abstractType, calledOnNullInput, language, body);
        }
        catch (InvalidRequestException e) {
            logger.error(String.format("Cannot load function '%s' from schema: this function won't be available (on this node)", name), (Throwable)e);
            return UDFunction.createBrokenFunction(name, argNames, argTypes, abstractType, calledOnNullInput, language, body, e);
        }
    }

    public static Mutation makeCreateAggregateMutation(KSMetaData keyspace, UDAggregate aggregate, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        LegacySchemaTables.addAggregateToSchemaMutation(aggregate, timestamp, mutation);
        return mutation;
    }

    private static void addAggregateToSchemaMutation(UDAggregate aggregate, long timestamp, Mutation mutation) {
        ColumnFamily cells = mutation.addOrGet(Aggregates);
        Composite prefix = LegacySchemaTables.Aggregates.comparator.make(aggregate.name().name, LegacySchemaTables.functionSignatureWithTypes(aggregate));
        CFRowAdder adder = new CFRowAdder(cells, prefix, timestamp);
        adder.resetCollection("argument_types");
        adder.add("return_type", aggregate.returnType().toString());
        adder.add("state_func", aggregate.stateFunction().name().name);
        if (aggregate.stateType() != null) {
            adder.add("state_type", aggregate.stateType().toString());
        }
        if (aggregate.finalFunction() != null) {
            adder.add("final_func", aggregate.finalFunction().name().name);
        }
        if (aggregate.initialCondition() != null) {
            adder.add("initcond", aggregate.initialCondition());
        }
        for (AbstractType<?> argType : aggregate.argTypes()) {
            adder.addListEntry("argument_types", argType.toString());
        }
    }

    private static Map<ByteBuffer, UDAggregate> createAggregatesFromAggregatesPartition(Row partition) {
        HashMap<ByteBuffer, UDAggregate> aggregates = new HashMap<ByteBuffer, UDAggregate>();
        String query = String.format("SELECT * FROM %s.%s", "system", AGGREGATES);
        for (UntypedResultSet.Row row : QueryProcessor.resultify(query, partition)) {
            UDAggregate aggregate = LegacySchemaTables.createAggregateFromAggregateRow(row);
            aggregates.put(LegacySchemaTables.functionSignatureWithNameAndTypes(aggregate), aggregate);
        }
        return aggregates;
    }

    private static UDAggregate createAggregateFromAggregateRow(UntypedResultSet.Row row) {
        List<AbstractType<?>> argTypes;
        String ksName = row.getString("keyspace_name");
        String functionName = row.getString("aggregate_name");
        FunctionName name = new FunctionName(ksName, functionName);
        List<String> types = row.getList("argument_types", UTF8Type.instance);
        if (types == null) {
            argTypes = Collections.emptyList();
        } else {
            argTypes = new ArrayList(types.size());
            for (String type : types) {
                argTypes.add(LegacySchemaTables.parseType(type));
            }
        }
        AbstractType<?> returnType = LegacySchemaTables.parseType(row.getString("return_type"));
        FunctionName stateFunc = new FunctionName(ksName, row.getString("state_func"));
        FunctionName finalFunc = row.has("final_func") ? new FunctionName(ksName, row.getString("final_func")) : null;
        AbstractType<?> stateType = row.has("state_type") ? LegacySchemaTables.parseType(row.getString("state_type")) : null;
        ByteBuffer initcond = row.has("initcond") ? row.getBytes("initcond") : null;
        try {
            return UDAggregate.create(name, argTypes, returnType, stateFunc, finalFunc, stateType, initcond);
        }
        catch (InvalidRequestException reason) {
            return UDAggregate.createBroken(name, argTypes, returnType, initcond, reason);
        }
    }

    public static Mutation makeDropAggregateMutation(KSMetaData keyspace, UDAggregate aggregate, long timestamp) {
        Mutation mutation = LegacySchemaTables.makeCreateKeyspaceMutation(keyspace, timestamp, false);
        ColumnFamily cells = mutation.addOrGet(Aggregates);
        int ldt = (int)(System.currentTimeMillis() / 1000L);
        Composite prefix = LegacySchemaTables.Aggregates.comparator.make(aggregate.name().name, LegacySchemaTables.functionSignatureWithTypes(aggregate));
        cells.addAtom(new RangeTombstone(prefix, prefix.end(), timestamp, ldt));
        return mutation;
    }

    private static AbstractType<?> parseType(String str) {
        return TypeParser.parse(str);
    }

    public static ByteBuffer functionSignatureWithTypes(AbstractFunction fun) {
        ListType<String> list = ListType.getInstance(UTF8Type.instance, false);
        ArrayList<String> strList = new ArrayList<String>(fun.argTypes().size());
        for (AbstractType<?> argType : fun.argTypes()) {
            strList.add(argType.asCQL3Type().toString());
        }
        return list.decompose((String)((Object)strList));
    }

    public static ByteBuffer functionSignatureWithNameAndTypes(AbstractFunction fun) {
        ListType<String> list = ListType.getInstance(UTF8Type.instance, false);
        ArrayList<String> strList = new ArrayList<String>(fun.argTypes().size() + 2);
        strList.add(fun.name().keyspace);
        strList.add(fun.name().name);
        for (AbstractType<?> argType : fun.argTypes()) {
            strList.add(argType.asCQL3Type().toString());
        }
        return list.decompose((String)((Object)strList));
    }
}

