/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.lifecycle;

import java.io.File;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.zip.CRC32;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.utils.FBUtilities;

final class LogRecord {
    public final Type type;
    public final String relativeFilePath;
    public final long updateTime;
    public final int numFiles;
    public final String raw;
    public final long checksum;
    public String error;
    public LogRecord onDiskRecord;
    static Pattern REGEX = Pattern.compile("^(add|remove|commit|abort):\\[([^,]*),?([^,]*),?([^,]*)\\]\\[(\\d*)\\]$", 2);

    public static LogRecord make(String line) {
        try {
            Matcher matcher = REGEX.matcher(line);
            if (!matcher.matches()) {
                return new LogRecord(Type.UNKNOWN, "", 0L, 0, 0L, line).error(String.format("Failed to parse [%s]", line));
            }
            Type type = Type.fromPrefix(matcher.group(1));
            return new LogRecord(type, matcher.group(2), Long.valueOf(matcher.group(3)), Integer.valueOf(matcher.group(4)), Long.valueOf(matcher.group(5)), line);
        }
        catch (Throwable t) {
            return new LogRecord(Type.UNKNOWN, "", 0L, 0, 0L, line).error(t);
        }
    }

    public static LogRecord makeCommit(long updateTime) {
        return new LogRecord(Type.COMMIT, "", updateTime, 0);
    }

    public static LogRecord makeAbort(long updateTime) {
        return new LogRecord(Type.ABORT, "", updateTime, 0);
    }

    public static LogRecord make(Type type, File parentFolder, SSTable table) {
        String relativePath = FileUtils.getRelativePath(parentFolder.getPath(), table.descriptor.baseFilename());
        return LogRecord.make(type, LogRecord.getExistingFiles(parentFolder, relativePath), table.getAllFilePaths().size(), relativePath);
    }

    public static LogRecord make(Type type, List<File> files, int minFiles, String relativeFilePath) {
        long lastModified = files.stream().map(File::lastModified).reduce(0L, Long::max);
        return new LogRecord(type, relativeFilePath, lastModified, Math.max(minFiles, files.size()));
    }

    private LogRecord(Type type, String relativeFilePath, long updateTime, int numFiles) {
        this(type, relativeFilePath, updateTime, numFiles, 0L, null);
    }

    private LogRecord(Type type, String relativeFilePath, long updateTime, int numFiles, long checksum, String raw) {
        this.type = type;
        this.relativeFilePath = type.hasFile() ? relativeFilePath : "";
        this.updateTime = type == Type.REMOVE ? updateTime : 0L;
        int n = this.numFiles = type.hasFile() ? numFiles : 0;
        if (raw == null) {
            assert (checksum == 0L);
            this.checksum = this.computeChecksum();
            this.raw = this.format();
        } else {
            this.checksum = checksum;
            this.raw = raw;
        }
        this.error = "";
    }

    public LogRecord error(Throwable t) {
        return this.error(t.getMessage());
    }

    public LogRecord error(String error) {
        this.error = error;
        return this;
    }

    public boolean isValid() {
        return this.error.isEmpty();
    }

    private String format() {
        return String.format("%s:[%s,%d,%d][%d]", this.type.toString(), this.relativeFilePath, this.updateTime, this.numFiles, this.checksum);
    }

    public List<File> getExistingFiles(File folder) {
        if (!this.type.hasFile()) {
            return Collections.emptyList();
        }
        return LogRecord.getExistingFiles(folder, this.relativeFilePath);
    }

    public static List<File> getExistingFiles(File parentFolder, String relativeFilePath) {
        return Arrays.asList(parentFolder.listFiles((dir, name) -> name.startsWith(relativeFilePath)));
    }

    public int hashCode() {
        return Objects.hash(new Object[]{this.type, this.relativeFilePath, this.error});
    }

    public boolean equals(Object obj) {
        if (!(obj instanceof LogRecord)) {
            return false;
        }
        LogRecord other = (LogRecord)obj;
        return this.type == other.type && this.relativeFilePath.equals(other.relativeFilePath) && this.error.equals(other.error);
    }

    public String toString() {
        return this.raw;
    }

    long computeChecksum() {
        CRC32 crc32 = new CRC32();
        crc32.update(this.relativeFilePath.getBytes(FileUtils.CHARSET));
        crc32.update(this.type.toString().getBytes(FileUtils.CHARSET));
        FBUtilities.updateChecksumInt(crc32, (int)this.updateTime);
        FBUtilities.updateChecksumInt(crc32, (int)(this.updateTime >>> 32));
        FBUtilities.updateChecksumInt(crc32, this.numFiles);
        return crc32.getValue() & Long.MAX_VALUE;
    }

    public static enum Type {
        UNKNOWN,
        ADD,
        REMOVE,
        COMMIT,
        ABORT;


        public static Type fromPrefix(String prefix) {
            return Type.valueOf(prefix.toUpperCase());
        }

        public boolean hasFile() {
            return this == ADD || this == REMOVE;
        }

        public boolean matches(LogRecord record) {
            return this == record.type;
        }
    }
}

