/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.io.util;

import com.google.common.base.Throwables;
import java.io.Closeable;
import java.io.DataInput;
import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.AtomicMoveNotSupportedException;
import java.nio.file.DirectoryIteratorException;
import java.nio.file.DirectoryStream;
import java.nio.file.FileStore;
import java.nio.file.Files;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttributeView;
import java.nio.file.attribute.FileStoreAttributeView;
import java.text.DecimalFormat;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Predicate;
import java.util.stream.StreamSupport;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.io.FSError;
import org.apache.cassandra.io.FSErrorHandler;
import org.apache.cassandra.io.FSReadError;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.CorruptSSTableException;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import sun.nio.ch.DirectBuffer;

public final class FileUtils {
    public static final Charset CHARSET = StandardCharsets.UTF_8;
    private static final Logger logger = LoggerFactory.getLogger(FileUtils.class);
    public static final long ONE_KB = 1024L;
    public static final long ONE_MB = 0x100000L;
    public static final long ONE_GB = 0x40000000L;
    public static final long ONE_TB = 0x10000000000L;
    private static final DecimalFormat df = new DecimalFormat("#.##");
    private static final boolean canCleanDirectBuffers;
    private static final AtomicReference<FSErrorHandler> fsErrorHandler;

    public static void createHardLink(String from, String to) {
        FileUtils.createHardLink(new File(from), new File(to));
    }

    public static void createHardLink(File from, File to) {
        if (to.exists()) {
            throw new RuntimeException("Tried to create duplicate hard link to " + to);
        }
        if (!from.exists()) {
            throw new RuntimeException("Tried to hard link to file that does not exist " + from);
        }
        try {
            Files.createLink(to.toPath(), from.toPath());
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, to);
        }
    }

    public static File createTempFile(String prefix, String suffix, File directory) {
        try {
            return File.createTempFile(prefix, suffix, directory);
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, directory);
        }
    }

    public static File createTempFile(String prefix, String suffix) {
        return FileUtils.createTempFile(prefix, suffix, new File(System.getProperty("java.io.tmpdir")));
    }

    public static Throwable deleteWithConfirm(String filePath, boolean expect, Throwable accumulate) {
        return FileUtils.deleteWithConfirm(new File(filePath), expect, accumulate);
    }

    public static Throwable deleteWithConfirm(File file, boolean expect, Throwable accumulate) {
        boolean exists = file.exists();
        assert (exists || !expect) : "attempted to delete non-existing file " + file.getName();
        try {
            if (exists) {
                Files.delete(file.toPath());
            }
        }
        catch (Throwable t) {
            try {
                throw new FSWriteError(t, file);
            }
            catch (Throwable t2) {
                accumulate = org.apache.cassandra.utils.Throwables.merge(accumulate, t2);
            }
        }
        return accumulate;
    }

    public static void deleteWithConfirm(String file) {
        FileUtils.deleteWithConfirm(new File(file));
    }

    public static void deleteWithConfirm(File file) {
        org.apache.cassandra.utils.Throwables.maybeFail(FileUtils.deleteWithConfirm(file, true, null));
    }

    public static void renameWithOutConfirm(String from, String to) {
        block2: {
            try {
                FileUtils.atomicMoveWithFallback(new File(from).toPath(), new File(to).toPath());
            }
            catch (IOException e) {
                if (!logger.isTraceEnabled()) break block2;
                logger.trace("Could not move file " + from + " to " + to, (Throwable)e);
            }
        }
    }

    public static void renameWithConfirm(String from, String to) {
        FileUtils.renameWithConfirm(new File(from), new File(to));
    }

    public static void renameWithConfirm(File from, File to) {
        assert (from.exists());
        if (logger.isTraceEnabled()) {
            logger.trace(String.format("Renaming %s to %s", from.getPath(), to.getPath()));
        }
        try {
            FileUtils.atomicMoveWithFallback(from.toPath(), to.toPath());
        }
        catch (IOException e) {
            throw new RuntimeException(String.format("Failed to rename %s to %s", from.getPath(), to.getPath()), e);
        }
    }

    private static void atomicMoveWithFallback(Path from, Path to) throws IOException {
        try {
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING, StandardCopyOption.ATOMIC_MOVE);
        }
        catch (AtomicMoveNotSupportedException e) {
            logger.trace("Could not do an atomic move", (Throwable)e);
            Files.move(from, to, StandardCopyOption.REPLACE_EXISTING);
        }
    }

    public static void truncate(String path, long size) {
        try (FileChannel channel = FileChannel.open(Paths.get(path, new String[0]), StandardOpenOption.READ, StandardOpenOption.WRITE);){
            channel.truncate(size);
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static void closeQuietly(Closeable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void closeQuietly(AutoCloseable c) {
        try {
            if (c != null) {
                c.close();
            }
        }
        catch (Exception e) {
            logger.warn("Failed closing {}", (Object)c, (Object)e);
        }
    }

    public static void close(Closeable ... cs) throws IOException {
        FileUtils.close(Arrays.asList(cs));
    }

    public static void close(Iterable<? extends Closeable> cs) throws IOException {
        Throwable e = null;
        for (Closeable closeable : cs) {
            try {
                if (closeable == null) continue;
                closeable.close();
            }
            catch (Throwable ex) {
                if (e == null) {
                    e = ex;
                } else {
                    e.addSuppressed(ex);
                }
                logger.warn("Failed closing stream {}", (Object)closeable, (Object)ex);
            }
        }
        org.apache.cassandra.utils.Throwables.maybeFail(e, IOException.class);
    }

    public static void closeQuietly(Iterable<? extends AutoCloseable> cs) {
        for (AutoCloseable autoCloseable : cs) {
            try {
                if (autoCloseable == null) continue;
                autoCloseable.close();
            }
            catch (Exception ex) {
                logger.warn("Failed closing {}", (Object)autoCloseable, (Object)ex);
            }
        }
    }

    public static String getCanonicalPath(String filename) {
        try {
            return new File(filename).getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, filename);
        }
    }

    public static String getCanonicalPath(File file) {
        try {
            return file.getCanonicalPath();
        }
        catch (IOException e) {
            throw new FSReadError((Throwable)e, file);
        }
    }

    public static boolean isContained(File folder, File file) {
        Path folderPath = Paths.get(FileUtils.getCanonicalPath(folder), new String[0]);
        Path filePath = Paths.get(FileUtils.getCanonicalPath(file), new String[0]);
        return filePath.startsWith(folderPath);
    }

    public static String getRelativePath(String basePath, String path) {
        try {
            return Paths.get(basePath, new String[0]).relativize(Paths.get(path, new String[0])).toString();
        }
        catch (Exception ex) {
            String absDataPath = FileUtils.getCanonicalPath(basePath);
            return Paths.get(absDataPath, new String[0]).relativize(Paths.get(path, new String[0])).toString();
        }
    }

    public static boolean isCleanerAvailable() {
        return canCleanDirectBuffers;
    }

    public static void clean(ByteBuffer buffer) {
        DirectBuffer db;
        if (FileUtils.isCleanerAvailable() && buffer.isDirect() && (db = (DirectBuffer)((Object)buffer)).cleaner() != null) {
            db.cleaner().clean();
        }
    }

    public static void createDirectory(String directory) {
        FileUtils.createDirectory(new File(directory));
    }

    public static void createDirectory(File directory) {
        if (!directory.exists() && !directory.mkdirs()) {
            throw new FSWriteError((Throwable)new IOException("Failed to mkdirs " + directory), directory);
        }
    }

    public static boolean delete(String file) {
        File f = new File(file);
        return f.delete();
    }

    public static void delete(File ... files) {
        if (files == null) {
            logger.debug("Received null list of files to delete");
            return;
        }
        for (File file : files) {
            file.delete();
        }
    }

    public static void deleteAsync(final String file) {
        Runnable runnable = new Runnable(){

            @Override
            public void run() {
                FileUtils.deleteWithConfirm(new File(file));
            }
        };
        ScheduledExecutors.nonPeriodicTasks.execute(runnable);
    }

    public static void visitDirectory(Path dir, Predicate<? super File> filter, Consumer<? super File> consumer) {
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(dir);){
            StreamSupport.stream(stream.spliterator(), false).map(Path::toFile).filter(f -> f.exists() && (filter == null || filter.test((File)f))).forEach(consumer);
        }
        catch (IOException | DirectoryIteratorException ex) {
            logger.error("Failed to list files in {} with exception: {}", new Object[]{dir, ex.getMessage(), ex});
        }
    }

    public static String stringifyFileSize(double value) {
        if (value >= 1.099511627776E12) {
            double d = value / 1.099511627776E12;
            String val = df.format(d);
            return val + " TB";
        }
        if (value >= 1.073741824E9) {
            double d = value / 1.073741824E9;
            String val = df.format(d);
            return val + " GB";
        }
        if (value >= 1048576.0) {
            double d = value / 1048576.0;
            String val = df.format(d);
            return val + " MB";
        }
        if (value >= 1024.0) {
            double d = value / 1024.0;
            String val = df.format(d);
            return val + " KB";
        }
        String val = df.format(value);
        return val + " bytes";
    }

    public static void deleteRecursive(File dir) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursive(new File(dir, child));
            }
        }
        FileUtils.deleteWithConfirm(dir);
    }

    public static void deleteRecursiveOnExit(File dir) {
        if (dir.isDirectory()) {
            String[] children;
            for (String child : children = dir.list()) {
                FileUtils.deleteRecursiveOnExit(new File(dir, child));
            }
        }
        logger.trace("Scheduling deferred deletion of file: " + dir);
        dir.deleteOnExit();
    }

    public static void handleCorruptSSTable(CorruptSSTableException e) {
        FSErrorHandler handler = fsErrorHandler.get();
        if (handler != null) {
            handler.handleCorruptSSTable(e);
        }
    }

    public static void handleFSError(FSError e) {
        FSErrorHandler handler = fsErrorHandler.get();
        if (handler != null) {
            handler.handleFSError(e);
        }
    }

    public static void handleFSErrorAndPropagate(FSError e) {
        FileUtils.handleFSError(e);
        throw Throwables.propagate((Throwable)e);
    }

    public static long folderSize(File directory) {
        long length = 0L;
        for (File file : directory.listFiles()) {
            if (file.isFile()) {
                length += file.length();
                continue;
            }
            length += FileUtils.folderSize(file);
        }
        return length;
    }

    public static void copyTo(DataInput in, OutputStream out, int length) throws IOException {
        byte[] buffer = new byte[65536];
        int copiedBytes = 0;
        while (copiedBytes + buffer.length < length) {
            in.readFully(buffer);
            out.write(buffer);
            copiedBytes += buffer.length;
        }
        if (copiedBytes < length) {
            int left = length - copiedBytes;
            in.readFully(buffer, 0, left);
            out.write(buffer, 0, left);
        }
    }

    public static boolean isSubDirectory(File parent, File child) throws IOException {
        parent = parent.getCanonicalFile();
        for (File toCheck = child = child.getCanonicalFile(); toCheck != null; toCheck = toCheck.getParentFile()) {
            if (!parent.equals(toCheck)) continue;
            return true;
        }
        return false;
    }

    public static void append(File file, String ... lines) {
        if (file.exists()) {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.APPEND);
        } else {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.CREATE);
        }
    }

    public static void appendAndSync(File file, String ... lines) {
        if (file.exists()) {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.APPEND, StandardOpenOption.SYNC);
        } else {
            FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.CREATE, StandardOpenOption.SYNC);
        }
    }

    public static void replace(File file, String ... lines) {
        FileUtils.write(file, Arrays.asList(lines), StandardOpenOption.TRUNCATE_EXISTING);
    }

    public static void write(File file, List<String> lines, StandardOpenOption ... options) {
        try {
            Files.write(file.toPath(), lines, CHARSET, (OpenOption[])options);
        }
        catch (IOException ex) {
            throw new RuntimeException(ex);
        }
    }

    public static List<String> readLines(File file) {
        try {
            return Files.readAllLines(file.toPath(), CHARSET);
        }
        catch (IOException ex) {
            if (ex instanceof NoSuchFileException) {
                return Collections.emptyList();
            }
            throw new RuntimeException(ex);
        }
    }

    public static void setFSErrorHandler(FSErrorHandler handler) {
        fsErrorHandler.getAndSet(handler);
    }

    public static long getTotalSpace(File file) {
        return FileUtils.handleLargeFileSystem(file.getTotalSpace());
    }

    public static long getFreeSpace(File file) {
        return FileUtils.handleLargeFileSystem(file.getFreeSpace());
    }

    public static long getUsableSpace(File file) {
        return FileUtils.handleLargeFileSystem(file.getUsableSpace());
    }

    public static FileStore getFileStore(Path path) throws IOException {
        return new SafeFileStore(Files.getFileStore(path));
    }

    private static long handleLargeFileSystem(long size) {
        return size < 0L ? Long.MAX_VALUE : size;
    }

    private FileUtils() {
    }

    static {
        fsErrorHandler = new AtomicReference();
        boolean canClean = false;
        try {
            ByteBuffer buf = ByteBuffer.allocateDirect(1);
            ((DirectBuffer)((Object)buf)).cleaner().clean();
            canClean = true;
        }
        catch (Throwable t) {
            JVMStabilityInspector.inspectThrowable(t);
            logger.info("Cannot initialize un-mmaper.  (Are you using a non-Oracle JVM?)  Compacted data files will not be removed promptly.  Consider using an Oracle JVM or using standard disk access mode");
        }
        canCleanDirectBuffers = canClean;
    }

    private static final class SafeFileStore
    extends FileStore {
        private final FileStore fileStore;

        public SafeFileStore(FileStore fileStore) {
            this.fileStore = fileStore;
        }

        @Override
        public String name() {
            return this.fileStore.name();
        }

        @Override
        public String type() {
            return this.fileStore.type();
        }

        @Override
        public boolean isReadOnly() {
            return this.fileStore.isReadOnly();
        }

        @Override
        public long getTotalSpace() throws IOException {
            return FileUtils.handleLargeFileSystem(this.fileStore.getTotalSpace());
        }

        @Override
        public long getUsableSpace() throws IOException {
            return FileUtils.handleLargeFileSystem(this.fileStore.getUsableSpace());
        }

        @Override
        public long getUnallocatedSpace() throws IOException {
            return FileUtils.handleLargeFileSystem(this.fileStore.getUnallocatedSpace());
        }

        @Override
        public boolean supportsFileAttributeView(Class<? extends FileAttributeView> type) {
            return this.fileStore.supportsFileAttributeView(type);
        }

        @Override
        public boolean supportsFileAttributeView(String name) {
            return this.fileStore.supportsFileAttributeView(name);
        }

        @Override
        public <V extends FileStoreAttributeView> V getFileStoreAttributeView(Class<V> type) {
            return this.fileStore.getFileStoreAttributeView(type);
        }

        @Override
        public Object getAttribute(String attribute) throws IOException {
            return this.fileStore.getAttribute(attribute);
        }
    }
}

