/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.lifecycle;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.Iterables;
import java.io.File;
import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.cassandra.db.compaction.OperationType;
import org.apache.cassandra.db.lifecycle.LogRecord;
import org.apache.cassandra.db.lifecycle.LogReplicaSet;
import org.apache.cassandra.db.lifecycle.LogTransaction;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.big.BigFormat;
import org.apache.cassandra.utils.Throwables;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class LogFile {
    private static final Logger logger = LoggerFactory.getLogger(LogFile.class);
    static String EXT = ".log";
    static char SEP = (char)95;
    static Pattern FILE_REGEX = Pattern.compile(String.format("^(.{2})_txn_(.*)_(.*)%s$", EXT));
    private final LogReplicaSet replicas = new LogReplicaSet();
    private final LinkedHashSet<LogRecord> records = new LinkedHashSet();
    private final OperationType type;
    private final UUID id;

    static LogFile make(File logReplica) {
        return LogFile.make(logReplica.getName(), Collections.singletonList(logReplica));
    }

    static LogFile make(String fileName, List<File> logReplicas) {
        Matcher matcher = FILE_REGEX.matcher(fileName);
        boolean matched = matcher.matches();
        assert (matched && matcher.groupCount() == 3);
        OperationType operationType = OperationType.fromFileName(matcher.group(2));
        UUID id = UUID.fromString(matcher.group(3));
        return new LogFile(operationType, id, logReplicas);
    }

    Throwable syncFolder(Throwable accumulate) {
        return this.replicas.syncFolder(accumulate);
    }

    OperationType type() {
        return this.type;
    }

    UUID id() {
        return this.id;
    }

    Throwable removeUnfinishedLeftovers(Throwable accumulate) {
        try {
            this.deleteFilesForRecordsOfType(this.committed() ? LogRecord.Type.REMOVE : LogRecord.Type.ADD);
            Throwables.maybeFail(this.syncFolder(accumulate));
            accumulate = this.replicas.delete(accumulate);
        }
        catch (Throwable t) {
            accumulate = Throwables.merge(accumulate, t);
        }
        return accumulate;
    }

    static boolean isLogFile(File file) {
        return FILE_REGEX.matcher(file.getName()).matches();
    }

    LogFile(OperationType type, UUID id, List<File> replicas) {
        this(type, id);
        this.replicas.addReplicas(replicas);
    }

    LogFile(OperationType type, UUID id) {
        this.type = type;
        this.id = id;
    }

    boolean verify() {
        assert (this.records.isEmpty());
        if (!this.replicas.readRecords(this.records)) {
            logger.error("Failed to read records from {}", (Object)this.replicas);
            return false;
        }
        this.records.forEach(LogFile::verifyRecord);
        Optional<LogRecord> firstInvalid = this.records.stream().filter(LogRecord::isInvalidOrPartial).findFirst();
        if (!firstInvalid.isPresent()) {
            return true;
        }
        LogRecord failedOn = firstInvalid.get();
        if (this.getLastRecord() != failedOn) {
            LogFile.logError(failedOn);
            return false;
        }
        this.records.stream().filter(r -> r != failedOn).forEach(LogFile::verifyRecordWithCorruptedLastRecord);
        if (this.records.stream().filter(r -> r != failedOn).filter(LogRecord::isInvalid).map(LogFile::logError).findFirst().isPresent()) {
            LogFile.logError(failedOn);
            return false;
        }
        logger.warn(String.format("Last record of transaction %s is corrupt or incomplete [%s], but all previous records match state on disk; continuing", this.id, failedOn.error()));
        return true;
    }

    static LogRecord logError(LogRecord record) {
        logger.error("{}", (Object)record.error());
        return record;
    }

    static void verifyRecord(LogRecord record) {
        if (record.checksum != record.computeChecksum()) {
            record.setError(String.format("Invalid checksum for sstable [%s], record [%s]: [%d] should have been [%d]", record.fileName(), record, record.checksum, record.computeChecksum()));
            return;
        }
        if (record.type != LogRecord.Type.REMOVE) {
            return;
        }
        record.status.onDiskRecord = record.withExistingFiles();
        if (record.updateTime != record.status.onDiskRecord.updateTime && record.status.onDiskRecord.numFiles > 0) {
            record.setError(String.format("Unexpected files detected for sstable [%s], record [%s]: last update time [%tT] should have been [%tT]", record.fileName(), record, record.status.onDiskRecord.updateTime, record.updateTime));
        }
    }

    static void verifyRecordWithCorruptedLastRecord(LogRecord record) {
        if (record.type == LogRecord.Type.REMOVE && record.status.onDiskRecord.numFiles < record.numFiles) {
            record.setError(String.format("Incomplete fileset detected for sstable [%s], record [%s]: number of files [%d] should have been [%d]. Treating as unrecoverable due to corruption of the final record.", record.fileName(), record.raw, record.status.onDiskRecord.numFiles, record.numFiles));
        }
    }

    void commit() {
        assert (!this.completed()) : "Already completed!";
        this.addRecord(LogRecord.makeCommit(System.currentTimeMillis()));
    }

    void abort() {
        assert (!this.completed()) : "Already completed!";
        this.addRecord(LogRecord.makeAbort(System.currentTimeMillis()));
    }

    private boolean isLastRecordValidWithType(LogRecord.Type type) {
        LogRecord lastRecord = this.getLastRecord();
        return lastRecord != null && lastRecord.type == type && lastRecord.isValid();
    }

    boolean committed() {
        return this.isLastRecordValidWithType(LogRecord.Type.COMMIT);
    }

    boolean aborted() {
        return this.isLastRecordValidWithType(LogRecord.Type.ABORT);
    }

    boolean completed() {
        return this.committed() || this.aborted();
    }

    void add(LogRecord.Type type, SSTable table) {
        if (!this.addRecord(this.makeRecord(type, table))) {
            throw new IllegalStateException();
        }
    }

    private LogRecord makeRecord(LogRecord.Type type, SSTable table) {
        assert (type == LogRecord.Type.ADD || type == LogRecord.Type.REMOVE);
        File folder = table.descriptor.directory;
        this.replicas.maybeCreateReplica(folder, this.getFileName(folder), this.records);
        return LogRecord.make(type, table);
    }

    private boolean addRecord(LogRecord record) {
        if (this.records.contains(record)) {
            return false;
        }
        this.replicas.append(record);
        return this.records.add(record);
    }

    void remove(LogRecord.Type type, SSTable table) {
        LogRecord record = this.makeRecord(type, table);
        assert (this.records.contains(record)) : String.format("[%s] is not tracked by %s", record, this.id);
        LogFile.deleteRecordFiles(record);
        this.records.remove(record);
    }

    boolean contains(LogRecord.Type type, SSTable table) {
        return this.records.contains(this.makeRecord(type, table));
    }

    void deleteFilesForRecordsOfType(LogRecord.Type type) {
        this.records.stream().filter(type::matches).forEach(LogFile::deleteRecordFiles);
        this.records.clear();
    }

    private static void deleteRecordFiles(LogRecord record) {
        List<File> files = record.getExistingFiles();
        files.sort((f1, f2) -> Long.compare(f1.lastModified(), f2.lastModified()));
        files.forEach(LogTransaction::delete);
    }

    Map<LogRecord, Set<File>> getFilesOfType(NavigableSet<File> files, LogRecord.Type type) {
        HashMap<LogRecord, Set<File>> ret = new HashMap<LogRecord, Set<File>>();
        this.records.stream().filter(type::matches).filter(LogRecord::isValid).forEach(r -> ret.put((LogRecord)r, LogFile.getRecordFiles(files, r)));
        return ret;
    }

    LogRecord getLastRecord() {
        return (LogRecord)Iterables.getLast(this.records, null);
    }

    private static Set<File> getRecordFiles(NavigableSet<File> files, LogRecord record) {
        String fileName = record.fileName();
        return files.stream().filter(f -> f.getName().startsWith(fileName)).collect(Collectors.toSet());
    }

    boolean exists() {
        return this.replicas.exists();
    }

    void close() {
        this.replicas.close();
    }

    public String toString() {
        return this.replicas.toString();
    }

    @VisibleForTesting
    List<File> getFiles() {
        return this.replicas.getFiles();
    }

    @VisibleForTesting
    List<String> getFilePaths() {
        return this.replicas.getFilePaths();
    }

    private String getFileName(File folder) {
        String fileName = StringUtils.join((Object[])new Object[]{BigFormat.latestVersion, Character.valueOf(SEP), "txn", Character.valueOf(SEP), this.type.fileName, Character.valueOf(SEP), this.id.toString(), EXT});
        return StringUtils.join((Object[])new Serializable[]{folder, File.separator, fileName});
    }
}

