/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cql3;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Iterators;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.MoreExecutors;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.antlr.runtime.RecognitionException;
import org.apache.cassandra.concurrent.ScheduledExecutors;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.cql3.BatchQueryOptions;
import org.apache.cassandra.cql3.CQLFragmentParser;
import org.apache.cassandra.cql3.CQLStatement;
import org.apache.cassandra.cql3.CqlParser;
import org.apache.cassandra.cql3.QueryHandler;
import org.apache.cassandra.cql3.QueryOptions;
import org.apache.cassandra.cql3.ResultSet;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.cql3.functions.FunctionName;
import org.apache.cassandra.cql3.statements.BatchStatement;
import org.apache.cassandra.cql3.statements.ModificationStatement;
import org.apache.cassandra.cql3.statements.QualifiedStatement;
import org.apache.cassandra.cql3.statements.RequestValidations;
import org.apache.cassandra.cql3.statements.SelectStatement;
import org.apache.cassandra.db.ConsistencyLevel;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.partitions.PartitionIterator;
import org.apache.cassandra.db.partitions.PartitionIterators;
import org.apache.cassandra.db.rows.RowIterator;
import org.apache.cassandra.exceptions.CassandraException;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.exceptions.RequestExecutionException;
import org.apache.cassandra.exceptions.RequestValidationException;
import org.apache.cassandra.exceptions.SyntaxException;
import org.apache.cassandra.metrics.CQLMetrics;
import org.apache.cassandra.schema.Schema;
import org.apache.cassandra.schema.SchemaChangeListener;
import org.apache.cassandra.service.ClientState;
import org.apache.cassandra.service.QueryState;
import org.apache.cassandra.service.pager.QueryPager;
import org.apache.cassandra.tracing.Tracing;
import org.apache.cassandra.transport.ProtocolVersion;
import org.apache.cassandra.transport.messages.ResultMessage;
import org.apache.cassandra.utils.ByteBufferUtil;
import org.apache.cassandra.utils.CassandraVersion;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.MD5Digest;
import org.apache.cassandra.utils.ObjectSizes;
import org.apache.cassandra.utils.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class QueryProcessor
implements QueryHandler {
    public static final CassandraVersion CQL_VERSION = new CassandraVersion("3.4.5");
    public static final QueryProcessor instance = new QueryProcessor();
    private static final Logger logger = LoggerFactory.getLogger(QueryProcessor.class);
    private static final Cache<MD5Digest, QueryHandler.Prepared> preparedStatements;
    private static final ConcurrentMap<String, QueryHandler.Prepared> internalStatements;
    public static final CQLMetrics metrics;
    private static final AtomicInteger lastMinuteEvictionsCount;

    private static long capacityToBytes(long cacheSizeMB) {
        return cacheSizeMB * 1024L * 1024L;
    }

    public static int preparedStatementsCount() {
        return preparedStatements.asMap().size();
    }

    public static void preloadPreparedStatement() {
        ClientState clientState = ClientState.forInternalCalls();
        int count = 0;
        for (Pair<String, String> useKeyspaceAndCQL : SystemKeyspace.loadPreparedStatements()) {
            try {
                clientState.setKeyspace((String)useKeyspaceAndCQL.left);
                QueryProcessor.prepare((String)useKeyspaceAndCQL.right, clientState);
                ++count;
            }
            catch (RequestValidationException e) {
                logger.warn("prepared statement recreation error: {}", useKeyspaceAndCQL.right, (Object)e);
            }
        }
        logger.info("Preloaded {} prepared statements", (Object)count);
    }

    @VisibleForTesting
    public static void clearPreparedStatements(boolean memoryOnly) {
        preparedStatements.invalidateAll();
        if (!memoryOnly) {
            SystemKeyspace.resetPreparedStatements();
        }
    }

    @VisibleForTesting
    public static ConcurrentMap<String, QueryHandler.Prepared> getInternalStatements() {
        return internalStatements;
    }

    @VisibleForTesting
    public static QueryState internalQueryState() {
        return new QueryState(InternalStateInstance.INSTANCE.clientState);
    }

    private QueryProcessor() {
        Schema.instance.registerListener(new StatementInvalidatingListener());
    }

    @Override
    public QueryHandler.Prepared getPrepared(MD5Digest id) {
        return (QueryHandler.Prepared)preparedStatements.getIfPresent((Object)id);
    }

    public static void validateKey(ByteBuffer key) throws InvalidRequestException {
        if (key == null || key.remaining() == 0) {
            throw new InvalidRequestException("Key may not be empty");
        }
        if (key == ByteBufferUtil.UNSET_BYTE_BUFFER) {
            throw new InvalidRequestException("Key may not be unset");
        }
        if (key.remaining() > 65535) {
            throw new InvalidRequestException("Key length of " + key.remaining() + " is longer than maximum of " + 65535);
        }
    }

    public ResultMessage processStatement(CQLStatement statement, QueryState queryState, QueryOptions options, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        ResultMessage result;
        logger.trace("Process {} @CL.{}", (Object)statement, (Object)options.getConsistency());
        ClientState clientState = queryState.getClientState();
        statement.authorize(clientState);
        statement.validate(clientState);
        if (options.getConsistency() == ConsistencyLevel.NODE_LOCAL) {
            assert (Boolean.getBoolean("cassandra.enable_nodelocal_queries")) : "Node local consistency level is highly dangerous and should be used only for debugging purposes";
            assert (statement instanceof SelectStatement) : "Only SELECT statements are permitted for node-local execution";
            logger.info("Statement {} executed with NODE_LOCAL consistency level.", (Object)statement);
            result = statement.executeLocally(queryState, options);
        } else {
            result = statement.execute(queryState, options, queryStartNanoTime);
        }
        return result == null ? new ResultMessage.Void() : result;
    }

    public static ResultMessage process(String queryString, ConsistencyLevel cl, QueryState queryState, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        QueryOptions options = QueryOptions.forInternalCalls(cl, Collections.emptyList());
        CQLStatement statement = instance.parse(queryString, queryState, options);
        return instance.process(statement, queryState, options, queryStartNanoTime);
    }

    @Override
    public CQLStatement parse(String queryString, QueryState queryState, QueryOptions options) {
        return QueryProcessor.getStatement(queryString, queryState.getClientState().cloneWithKeyspaceIfSet(options.getKeyspace()));
    }

    @Override
    public ResultMessage process(CQLStatement statement, QueryState state, QueryOptions options, Map<String, ByteBuffer> customPayload, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        return this.process(statement, state, options, queryStartNanoTime);
    }

    public ResultMessage process(CQLStatement prepared, QueryState queryState, QueryOptions options, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        options.prepare(prepared.getBindVariables());
        if (prepared.getBindVariables().size() != options.getValues().size()) {
            throw new InvalidRequestException("Invalid amount of bind variables");
        }
        if (!queryState.getClientState().isInternal) {
            QueryProcessor.metrics.regularStatementsExecuted.inc();
        }
        return this.processStatement(prepared, queryState, options, queryStartNanoTime);
    }

    public static CQLStatement parseStatement(String queryStr, ClientState clientState) throws RequestValidationException {
        return QueryProcessor.getStatement(queryStr, clientState);
    }

    public static UntypedResultSet process(String query, ConsistencyLevel cl) throws RequestExecutionException {
        return QueryProcessor.process(query, cl, Collections.emptyList());
    }

    public static UntypedResultSet process(String query, ConsistencyLevel cl, List<ByteBuffer> values) throws RequestExecutionException {
        QueryOptions options;
        QueryState queryState = QueryState.forInternalCalls();
        CQLStatement statement = instance.parse(query, queryState, options = QueryOptions.forInternalCalls(cl, values));
        ResultMessage result = instance.process(statement, queryState, options, System.nanoTime());
        if (result instanceof ResultMessage.Rows) {
            return UntypedResultSet.create(((ResultMessage.Rows)result).result);
        }
        return null;
    }

    @VisibleForTesting
    public static QueryOptions makeInternalOptions(CQLStatement prepared, Object[] values) {
        return QueryProcessor.makeInternalOptions(prepared, values, ConsistencyLevel.ONE);
    }

    private static QueryOptions makeInternalOptions(CQLStatement prepared, Object[] values, ConsistencyLevel cl) {
        if (prepared.getBindVariables().size() != values.length) {
            throw new IllegalArgumentException(String.format("Invalid number of values. Expecting %d but got %d", prepared.getBindVariables().size(), values.length));
        }
        ArrayList<ByteBuffer> boundValues = new ArrayList<ByteBuffer>(values.length);
        for (int i = 0; i < values.length; ++i) {
            Object value = values[i];
            AbstractType<?> type = prepared.getBindVariables().get((int)i).type;
            boundValues.add(value instanceof ByteBuffer || value == null ? (ByteBuffer)value : type.decompose(value));
        }
        return QueryOptions.forInternalCalls(cl, boundValues);
    }

    public static QueryHandler.Prepared prepareInternal(String query) throws RequestValidationException {
        QueryHandler.Prepared prepared = (QueryHandler.Prepared)internalStatements.get(query);
        if (prepared != null) {
            return prepared;
        }
        CQLStatement statement = QueryProcessor.parseStatement(query, QueryProcessor.internalQueryState().getClientState());
        statement.validate(QueryProcessor.internalQueryState().getClientState());
        prepared = new QueryHandler.Prepared(statement);
        internalStatements.put(query, prepared);
        return prepared;
    }

    public static UntypedResultSet executeInternal(String query, Object ... values) {
        QueryHandler.Prepared prepared = QueryProcessor.prepareInternal(query);
        ResultMessage result = prepared.statement.executeLocally(QueryProcessor.internalQueryState(), QueryProcessor.makeInternalOptions(prepared.statement, values));
        if (result instanceof ResultMessage.Rows) {
            return UntypedResultSet.create(((ResultMessage.Rows)result).result);
        }
        return null;
    }

    public static UntypedResultSet execute(String query, ConsistencyLevel cl, Object ... values) throws RequestExecutionException {
        return QueryProcessor.execute(query, cl, QueryProcessor.internalQueryState(), values);
    }

    public static UntypedResultSet execute(String query, ConsistencyLevel cl, QueryState state, Object ... values) throws RequestExecutionException {
        try {
            QueryHandler.Prepared prepared = QueryProcessor.prepareInternal(query);
            ResultMessage result = prepared.statement.execute(state, QueryProcessor.makeInternalOptions(prepared.statement, values, cl), System.nanoTime());
            if (result instanceof ResultMessage.Rows) {
                return UntypedResultSet.create(((ResultMessage.Rows)result).result);
            }
            return null;
        }
        catch (RequestValidationException e) {
            throw new RuntimeException("Error validating " + query, e);
        }
    }

    public static UntypedResultSet executeInternalWithPaging(String query, int pageSize, Object ... values) {
        QueryHandler.Prepared prepared = QueryProcessor.prepareInternal(query);
        if (!(prepared.statement instanceof SelectStatement)) {
            throw new IllegalArgumentException("Only SELECTs can be paged");
        }
        SelectStatement select = (SelectStatement)prepared.statement;
        QueryPager pager = select.getQuery(QueryProcessor.makeInternalOptions(prepared.statement, values), FBUtilities.nowInSeconds()).getPager(null, ProtocolVersion.CURRENT);
        return UntypedResultSet.create(select, pager, pageSize);
    }

    public static UntypedResultSet executeOnceInternal(String query, Object ... values) {
        CQLStatement statement = QueryProcessor.parseStatement(query, QueryProcessor.internalQueryState().getClientState());
        statement.validate(QueryProcessor.internalQueryState().getClientState());
        ResultMessage result = statement.executeLocally(QueryProcessor.internalQueryState(), QueryProcessor.makeInternalOptions(statement, values));
        if (result instanceof ResultMessage.Rows) {
            return UntypedResultSet.create(((ResultMessage.Rows)result).result);
        }
        return null;
    }

    public static UntypedResultSet executeInternalWithNow(int nowInSec, long queryStartNanoTime, String query, Object ... values) {
        QueryHandler.Prepared prepared = QueryProcessor.prepareInternal(query);
        assert (prepared.statement instanceof SelectStatement);
        SelectStatement select = (SelectStatement)prepared.statement;
        ResultMessage.Rows result = select.executeInternal(QueryProcessor.internalQueryState(), QueryProcessor.makeInternalOptions(prepared.statement, values), nowInSec, queryStartNanoTime);
        assert (result instanceof ResultMessage.Rows);
        return UntypedResultSet.create(result.result);
    }

    public static UntypedResultSet resultify(String query, RowIterator partition) {
        return QueryProcessor.resultify(query, PartitionIterators.singletonIterator(partition));
    }

    public static UntypedResultSet resultify(String query, PartitionIterator partitions) {
        try (PartitionIterator iter = partitions;){
            SelectStatement ss = (SelectStatement)QueryProcessor.getStatement(query, null);
            ResultSet cqlRows = ss.process(iter, FBUtilities.nowInSeconds());
            UntypedResultSet untypedResultSet = UntypedResultSet.create(cqlRows);
            return untypedResultSet;
        }
    }

    @Override
    public ResultMessage.Prepared prepare(String query, ClientState clientState, Map<String, ByteBuffer> customPayload) throws RequestValidationException {
        return QueryProcessor.prepare(query, clientState);
    }

    public static ResultMessage.Prepared prepare(String queryString, ClientState clientState) {
        ResultMessage.Prepared existing = QueryProcessor.getStoredPreparedStatement(queryString, clientState.getRawKeyspace());
        if (existing != null) {
            return existing;
        }
        CQLStatement statement = QueryProcessor.getStatement(queryString, clientState);
        QueryHandler.Prepared prepared = new QueryHandler.Prepared(statement, queryString);
        int boundTerms = statement.getBindVariables().size();
        if (boundTerms > 65535) {
            throw new InvalidRequestException(String.format("Too many markers(?). %d markers exceed the allowed maximum of %d", boundTerms, 65535));
        }
        return QueryProcessor.storePreparedStatement(queryString, clientState.getRawKeyspace(), prepared);
    }

    private static MD5Digest computeId(String queryString, String keyspace) {
        String toHash = keyspace == null ? queryString : keyspace + queryString;
        return MD5Digest.compute(toHash);
    }

    private static ResultMessage.Prepared getStoredPreparedStatement(String queryString, String keyspace) throws InvalidRequestException {
        MD5Digest statementId = QueryProcessor.computeId(queryString, keyspace);
        QueryHandler.Prepared existing = (QueryHandler.Prepared)preparedStatements.getIfPresent((Object)statementId);
        if (existing == null) {
            return null;
        }
        RequestValidations.checkTrue(queryString.equals(existing.rawCQLStatement), String.format("MD5 hash collision: query with the same MD5 hash was already prepared. \n Existing: '%s'", existing.rawCQLStatement));
        ResultSet.PreparedMetadata preparedMetadata = ResultSet.PreparedMetadata.fromPrepared(existing.statement);
        ResultSet.ResultMetadata resultMetadata = ResultSet.ResultMetadata.fromPrepared(existing.statement);
        return new ResultMessage.Prepared(statementId, resultMetadata.getResultMetadataId(), preparedMetadata, resultMetadata);
    }

    private static ResultMessage.Prepared storePreparedStatement(String queryString, String keyspace, QueryHandler.Prepared prepared) throws InvalidRequestException {
        long statementSize = ObjectSizes.measureDeep(prepared.statement);
        if (statementSize > QueryProcessor.capacityToBytes(DatabaseDescriptor.getPreparedStatementsCacheSizeMB())) {
            throw new InvalidRequestException(String.format("Prepared statement of size %d bytes is larger than allowed maximum of %d MB: %s...", statementSize, DatabaseDescriptor.getPreparedStatementsCacheSizeMB(), queryString.substring(0, 200)));
        }
        MD5Digest statementId = QueryProcessor.computeId(queryString, keyspace);
        preparedStatements.put((Object)statementId, (Object)prepared);
        SystemKeyspace.writePreparedStatement(keyspace, statementId, queryString);
        ResultSet.PreparedMetadata preparedMetadata = ResultSet.PreparedMetadata.fromPrepared(prepared.statement);
        ResultSet.ResultMetadata resultMetadata = ResultSet.ResultMetadata.fromPrepared(prepared.statement);
        return new ResultMessage.Prepared(statementId, resultMetadata.getResultMetadataId(), preparedMetadata, resultMetadata);
    }

    @Override
    public ResultMessage processPrepared(CQLStatement statement, QueryState state, QueryOptions options, Map<String, ByteBuffer> customPayload, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        return this.processPrepared(statement, state, options, queryStartNanoTime);
    }

    public ResultMessage processPrepared(CQLStatement statement, QueryState queryState, QueryOptions options, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        List<ByteBuffer> variables = options.getValues();
        if (!variables.isEmpty() || !statement.getBindVariables().isEmpty()) {
            if (variables.size() != statement.getBindVariables().size()) {
                throw new InvalidRequestException(String.format("there were %d markers(?) in CQL but %d bound variables", statement.getBindVariables().size(), variables.size()));
            }
            if (logger.isTraceEnabled()) {
                for (int i = 0; i < variables.size(); ++i) {
                    logger.trace("[{}] '{}'", (Object)(i + 1), (Object)variables.get(i));
                }
            }
        }
        QueryProcessor.metrics.preparedStatementsExecuted.inc();
        return this.processStatement(statement, queryState, options, queryStartNanoTime);
    }

    @Override
    public ResultMessage processBatch(BatchStatement statement, QueryState state, BatchQueryOptions options, Map<String, ByteBuffer> customPayload, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        return this.processBatch(statement, state, options, queryStartNanoTime);
    }

    public ResultMessage processBatch(BatchStatement batch, QueryState queryState, BatchQueryOptions options, long queryStartNanoTime) throws RequestExecutionException, RequestValidationException {
        ClientState clientState = queryState.getClientState().cloneWithKeyspaceIfSet(options.getKeyspace());
        batch.authorize(clientState);
        batch.validate();
        batch.validate(clientState);
        return batch.execute(queryState, options, queryStartNanoTime);
    }

    public static CQLStatement getStatement(String queryStr, ClientState clientState) throws RequestValidationException {
        Tracing.trace("Parsing {}", (Object)queryStr);
        CQLStatement.Raw statement = QueryProcessor.parseStatement(queryStr);
        if (statement instanceof QualifiedStatement) {
            ((QualifiedStatement)statement).setKeyspace(clientState);
        }
        Tracing.trace("Preparing statement");
        return statement.prepare(clientState);
    }

    public static <T extends CQLStatement.Raw> T parseStatement(String queryStr, Class<T> klass, String type) throws SyntaxException {
        try {
            CQLStatement.Raw stmt = QueryProcessor.parseStatement(queryStr);
            if (!klass.isAssignableFrom(stmt.getClass())) {
                throw new IllegalArgumentException("Invalid query, must be a " + type + " statement but was: " + stmt.getClass());
            }
            return (T)((CQLStatement.Raw)klass.cast(stmt));
        }
        catch (RequestValidationException e) {
            throw new IllegalArgumentException(e.getMessage(), e);
        }
    }

    public static CQLStatement.Raw parseStatement(String queryStr) throws SyntaxException {
        try {
            return CQLFragmentParser.parseAnyUnhandled(CqlParser::query, queryStr);
        }
        catch (CassandraException ce) {
            throw ce;
        }
        catch (RuntimeException re) {
            logger.error(String.format("The statement: [%s] could not be parsed.", queryStr), (Throwable)re);
            throw new SyntaxException(String.format("Failed parsing statement: [%s] reason: %s %s", queryStr, re.getClass().getSimpleName(), re.getMessage()));
        }
        catch (RecognitionException e) {
            throw new SyntaxException("Invalid or malformed CQL query string: " + e.getMessage());
        }
    }

    private static int measure(Object key, QueryHandler.Prepared value) {
        return Ints.checkedCast((long)(ObjectSizes.measureDeep(key) + ObjectSizes.measureDeep(value)));
    }

    @VisibleForTesting
    public static void clearInternalStatementsCache() {
        internalStatements.clear();
    }

    static {
        internalStatements = new ConcurrentHashMap<String, QueryHandler.Prepared>();
        metrics = new CQLMetrics();
        lastMinuteEvictionsCount = new AtomicInteger(0);
        preparedStatements = Caffeine.newBuilder().executor(MoreExecutors.directExecutor()).maximumWeight(QueryProcessor.capacityToBytes(DatabaseDescriptor.getPreparedStatementsCacheSizeMB())).weigher(QueryProcessor::measure).removalListener((key, prepared, cause) -> {
            MD5Digest md5Digest = (MD5Digest)key;
            if (cause.wasEvicted()) {
                QueryProcessor.metrics.preparedStatementsEvicted.inc();
                lastMinuteEvictionsCount.incrementAndGet();
                SystemKeyspace.removePreparedStatement(md5Digest);
            }
        }).build();
        ScheduledExecutors.scheduledTasks.scheduleAtFixedRate(() -> {
            long count = lastMinuteEvictionsCount.getAndSet(0);
            if (count > 0L) {
                logger.warn("{} prepared statements discarded in the last minute because cache limit reached ({} MB)", (Object)count, (Object)DatabaseDescriptor.getPreparedStatementsCacheSizeMB());
            }
        }, 1L, 1L, TimeUnit.MINUTES);
        logger.info("Initialized prepared statement caches with {} MB", (Object)DatabaseDescriptor.getPreparedStatementsCacheSizeMB());
    }

    private static class StatementInvalidatingListener
    extends SchemaChangeListener {
        private StatementInvalidatingListener() {
        }

        private static void removeInvalidPreparedStatements(String ksName, String cfName) {
            StatementInvalidatingListener.removeInvalidPreparedStatements(internalStatements.values().iterator(), ksName, cfName);
            StatementInvalidatingListener.removeInvalidPersistentPreparedStatements(preparedStatements.asMap().entrySet().iterator(), ksName, cfName);
        }

        private static void removeInvalidPreparedStatementsForFunction(String ksName, String functionName) {
            Predicate matchesFunction = f -> ksName.equals(f.name().keyspace) && functionName.equals(f.name().name);
            Iterator iter = preparedStatements.asMap().entrySet().iterator();
            while (iter.hasNext()) {
                Map.Entry pstmt = iter.next();
                if (!Iterables.any(((QueryHandler.Prepared)pstmt.getValue()).statement.getFunctions(), (Predicate)matchesFunction)) continue;
                SystemKeyspace.removePreparedStatement((MD5Digest)pstmt.getKey());
                iter.remove();
            }
            Iterators.removeIf(internalStatements.values().iterator(), statement -> Iterables.any(statement.statement.getFunctions(), (Predicate)matchesFunction));
        }

        private static void removeInvalidPersistentPreparedStatements(Iterator<Map.Entry<MD5Digest, QueryHandler.Prepared>> iterator, String ksName, String cfName) {
            while (iterator.hasNext()) {
                Map.Entry<MD5Digest, QueryHandler.Prepared> entry = iterator.next();
                if (!StatementInvalidatingListener.shouldInvalidate(ksName, cfName, entry.getValue().statement)) continue;
                SystemKeyspace.removePreparedStatement(entry.getKey());
                iterator.remove();
            }
        }

        private static void removeInvalidPreparedStatements(Iterator<QueryHandler.Prepared> iterator, String ksName, String cfName) {
            while (iterator.hasNext()) {
                if (!StatementInvalidatingListener.shouldInvalidate(ksName, cfName, iterator.next().statement)) continue;
                iterator.remove();
            }
        }

        private static boolean shouldInvalidate(String ksName, String cfName, CQLStatement statement) {
            String statementCfName;
            String statementKsName;
            if (statement instanceof ModificationStatement) {
                ModificationStatement modificationStatement = (ModificationStatement)statement;
                statementKsName = modificationStatement.keyspace();
                statementCfName = modificationStatement.columnFamily();
            } else if (statement instanceof SelectStatement) {
                SelectStatement selectStatement = (SelectStatement)statement;
                statementKsName = selectStatement.keyspace();
                statementCfName = selectStatement.columnFamily();
            } else {
                if (statement instanceof BatchStatement) {
                    BatchStatement batchStatement = (BatchStatement)statement;
                    for (ModificationStatement stmt : batchStatement.getStatements()) {
                        if (!StatementInvalidatingListener.shouldInvalidate(ksName, cfName, stmt)) continue;
                        return true;
                    }
                    return false;
                }
                return false;
            }
            return ksName.equals(statementKsName) && (cfName == null || cfName.equals(statementCfName));
        }

        @Override
        public void onCreateFunction(String ksName, String functionName, List<AbstractType<?>> argTypes) {
            StatementInvalidatingListener.onCreateFunctionInternal(ksName, functionName, argTypes);
        }

        @Override
        public void onCreateAggregate(String ksName, String aggregateName, List<AbstractType<?>> argTypes) {
            StatementInvalidatingListener.onCreateFunctionInternal(ksName, aggregateName, argTypes);
        }

        private static void onCreateFunctionInternal(String ksName, String functionName, List<AbstractType<?>> argTypes) {
            if (Schema.instance.getKeyspaceMetadata((String)ksName).functions.get(new FunctionName(ksName, functionName)).size() > 1) {
                StatementInvalidatingListener.removeInvalidPreparedStatementsForFunction(ksName, functionName);
            }
        }

        @Override
        public void onAlterTable(String ksName, String cfName, boolean affectsStatements) {
            logger.trace("Column definitions for {}.{} changed, invalidating related prepared statements", (Object)ksName, (Object)cfName);
            if (affectsStatements) {
                StatementInvalidatingListener.removeInvalidPreparedStatements(ksName, cfName);
            }
        }

        @Override
        public void onAlterFunction(String ksName, String functionName, List<AbstractType<?>> argTypes) {
            StatementInvalidatingListener.removeInvalidPreparedStatementsForFunction(ksName, functionName);
        }

        @Override
        public void onAlterAggregate(String ksName, String aggregateName, List<AbstractType<?>> argTypes) {
            StatementInvalidatingListener.removeInvalidPreparedStatementsForFunction(ksName, aggregateName);
        }

        @Override
        public void onDropKeyspace(String ksName) {
            logger.trace("Keyspace {} was dropped, invalidating related prepared statements", (Object)ksName);
            StatementInvalidatingListener.removeInvalidPreparedStatements(ksName, null);
        }

        @Override
        public void onDropTable(String ksName, String cfName) {
            logger.trace("Table {}.{} was dropped, invalidating related prepared statements", (Object)ksName, (Object)cfName);
            StatementInvalidatingListener.removeInvalidPreparedStatements(ksName, cfName);
        }

        @Override
        public void onDropFunction(String ksName, String functionName, List<AbstractType<?>> argTypes) {
            StatementInvalidatingListener.removeInvalidPreparedStatementsForFunction(ksName, functionName);
        }

        @Override
        public void onDropAggregate(String ksName, String aggregateName, List<AbstractType<?>> argTypes) {
            StatementInvalidatingListener.removeInvalidPreparedStatementsForFunction(ksName, aggregateName);
        }
    }

    private static enum InternalStateInstance {
        INSTANCE;

        private final ClientState clientState = ClientState.forInternalCalls("system");
    }
}

