/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.index.internal;

import com.google.common.collect.ImmutableSet;
import com.google.common.util.concurrent.ListenableFuture;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.function.BiFunction;
import java.util.stream.Collectors;
import java.util.stream.StreamSupport;
import org.apache.cassandra.cql3.Operator;
import org.apache.cassandra.cql3.statements.RequestValidations;
import org.apache.cassandra.cql3.statements.schema.IndexTarget;
import org.apache.cassandra.db.CBuilder;
import org.apache.cassandra.db.Clustering;
import org.apache.cassandra.db.ClusteringComparator;
import org.apache.cassandra.db.ClusteringPrefix;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.DeletionTime;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.LivenessInfo;
import org.apache.cassandra.db.RangeTombstone;
import org.apache.cassandra.db.ReadCommand;
import org.apache.cassandra.db.RegularAndStaticColumns;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.WriteContext;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.filter.RowFilter;
import org.apache.cassandra.db.lifecycle.SSTableSet;
import org.apache.cassandra.db.lifecycle.View;
import org.apache.cassandra.db.marshal.AbstractType;
import org.apache.cassandra.db.marshal.CollectionType;
import org.apache.cassandra.db.partitions.PartitionIterator;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.db.rows.BTreeRow;
import org.apache.cassandra.db.rows.Cell;
import org.apache.cassandra.db.rows.CellPath;
import org.apache.cassandra.db.rows.ComplexColumnData;
import org.apache.cassandra.db.rows.Row;
import org.apache.cassandra.dht.LocalPartitioner;
import org.apache.cassandra.exceptions.InvalidRequestException;
import org.apache.cassandra.index.Index;
import org.apache.cassandra.index.IndexRegistry;
import org.apache.cassandra.index.TargetParser;
import org.apache.cassandra.index.internal.CassandraIndexFunctions;
import org.apache.cassandra.index.internal.CollatedViewIndexBuilder;
import org.apache.cassandra.index.internal.IndexEntry;
import org.apache.cassandra.index.internal.composites.CompositesSearcher;
import org.apache.cassandra.index.internal.keys.KeysSearcher;
import org.apache.cassandra.index.transactions.IndexTransaction;
import org.apache.cassandra.index.transactions.UpdateTransaction;
import org.apache.cassandra.io.sstable.ReducingKeyIterator;
import org.apache.cassandra.io.sstable.SSTable;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.schema.ColumnMetadata;
import org.apache.cassandra.schema.IndexMetadata;
import org.apache.cassandra.schema.TableMetadata;
import org.apache.cassandra.schema.TableMetadataRef;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.Pair;
import org.apache.cassandra.utils.concurrent.Refs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CassandraIndex
implements Index {
    private static final Logger logger = LoggerFactory.getLogger(CassandraIndex.class);
    public final ColumnFamilyStore baseCfs;
    protected IndexMetadata metadata;
    protected ColumnFamilyStore indexCfs;
    protected ColumnMetadata indexedColumn;
    protected CassandraIndexFunctions functions;

    protected CassandraIndex(ColumnFamilyStore baseCfs, IndexMetadata indexDef) {
        this.baseCfs = baseCfs;
        this.setMetadata(indexDef);
    }

    protected boolean supportsOperator(ColumnMetadata indexedColumn, Operator operator) {
        return operator == Operator.EQ;
    }

    protected abstract <T> CBuilder buildIndexClusteringPrefix(ByteBuffer var1, ClusteringPrefix<T> var2, CellPath var3);

    public abstract IndexEntry decodeEntry(DecoratedKey var1, Row var2);

    public abstract boolean isStale(Row var1, ByteBuffer var2, int var3);

    protected abstract ByteBuffer getIndexedValue(ByteBuffer var1, Clustering<?> var2, CellPath var3, ByteBuffer var4);

    public ColumnMetadata getIndexedColumn() {
        return this.indexedColumn;
    }

    public ClusteringComparator getIndexComparator() {
        return this.indexCfs.metadata().comparator;
    }

    public ColumnFamilyStore getIndexCfs() {
        return this.indexCfs;
    }

    @Override
    public void register(IndexRegistry registry) {
        registry.registerIndex(this);
    }

    @Override
    public Callable<?> getInitializationTask() {
        return this.isBuilt() || this.baseCfs.isEmpty() ? null : this.getBuildIndexTask();
    }

    @Override
    public IndexMetadata getIndexMetadata() {
        return this.metadata;
    }

    @Override
    public Optional<ColumnFamilyStore> getBackingTable() {
        return this.indexCfs == null ? Optional.empty() : Optional.of(this.indexCfs);
    }

    public Callable<Void> getBlockingFlushTask() {
        return () -> {
            this.indexCfs.forceBlockingFlush();
            return null;
        };
    }

    @Override
    public Callable<?> getInvalidateTask() {
        return () -> {
            this.invalidate();
            return null;
        };
    }

    @Override
    public Callable<?> getMetadataReloadTask(IndexMetadata indexDef) {
        return () -> {
            this.indexCfs.reload();
            return null;
        };
    }

    @Override
    public void validate(ReadCommand command) throws InvalidRequestException {
        Optional<RowFilter.Expression> target = this.getTargetExpression(command.rowFilter().getExpressions());
        if (target.isPresent()) {
            ByteBuffer indexValue = target.get().getIndexValue();
            RequestValidations.checkFalse(indexValue.remaining() > 65535, "Index expression values may not be larger than 64K");
        }
    }

    private void setMetadata(IndexMetadata indexDef) {
        this.metadata = indexDef;
        Pair<ColumnMetadata, IndexTarget.Type> target = TargetParser.parse(this.baseCfs.metadata(), indexDef);
        this.functions = CassandraIndex.getFunctions(indexDef, target);
        TableMetadataRef tableRef = TableMetadataRef.forOfflineTools(CassandraIndex.indexCfsMetadata(this.baseCfs.metadata(), indexDef));
        this.indexCfs = ColumnFamilyStore.createColumnFamilyStore(this.baseCfs.keyspace, tableRef.name, tableRef, this.baseCfs.getTracker().loadsstables);
        this.indexedColumn = (ColumnMetadata)target.left;
    }

    @Override
    public Callable<?> getTruncateTask(long truncatedAt) {
        return () -> {
            this.indexCfs.discardSSTables(truncatedAt);
            return null;
        };
    }

    @Override
    public boolean shouldBuildBlocking() {
        return true;
    }

    @Override
    public boolean dependsOn(ColumnMetadata column) {
        return this.indexedColumn.name.equals(column.name);
    }

    @Override
    public boolean supportsExpression(ColumnMetadata column, Operator operator) {
        return this.indexedColumn.name.equals(column.name) && this.supportsOperator(this.indexedColumn, operator);
    }

    private boolean supportsExpression(RowFilter.Expression expression) {
        return this.supportsExpression(expression.column(), expression.operator());
    }

    @Override
    public AbstractType<?> customExpressionValueType() {
        return null;
    }

    @Override
    public long getEstimatedResultRows() {
        return this.indexCfs.getMeanRowCount();
    }

    @Override
    public BiFunction<PartitionIterator, ReadCommand, PartitionIterator> postProcessorFor(ReadCommand command) {
        return (partitionIterator, readCommand) -> partitionIterator;
    }

    @Override
    public RowFilter getPostIndexQueryFilter(RowFilter filter) {
        return this.getTargetExpression(filter.getExpressions()).map(filter::without).orElse(filter);
    }

    private Optional<RowFilter.Expression> getTargetExpression(List<RowFilter.Expression> expressions) {
        return expressions.stream().filter(this::supportsExpression).findFirst();
    }

    @Override
    public Index.Searcher searcherFor(ReadCommand command) {
        Optional<RowFilter.Expression> target = this.getTargetExpression(command.rowFilter().getExpressions());
        if (target.isPresent()) {
            switch (this.getIndexMetadata().kind) {
                case COMPOSITES: {
                    return new CompositesSearcher(command, target.get(), this);
                }
                case KEYS: {
                    return new KeysSearcher(command, target.get(), this);
                }
            }
            throw new IllegalStateException(String.format("Unsupported index type %s for index %s on %s", new Object[]{this.metadata.kind, this.metadata.name, this.indexedColumn.name.toString()}));
        }
        return null;
    }

    @Override
    public void validate(PartitionUpdate update) throws InvalidRequestException {
        switch (this.indexedColumn.kind) {
            case PARTITION_KEY: {
                this.validatePartitionKey(update.partitionKey());
                break;
            }
            case CLUSTERING: {
                this.validateClusterings(update);
                break;
            }
            case REGULAR: {
                if (!update.columns().regulars.contains(this.indexedColumn)) break;
                this.validateRows(update);
                break;
            }
            case STATIC: {
                if (!update.columns().statics.contains(this.indexedColumn)) break;
                this.validateRows(Collections.singleton(update.staticRow()));
            }
        }
    }

    @Override
    public Index.Indexer indexerFor(final DecoratedKey key, RegularAndStaticColumns columns, final int nowInSec, final WriteContext ctx, IndexTransaction.Type transactionType) {
        if (!this.isPrimaryKeyIndex() && !columns.contains(this.indexedColumn)) {
            return null;
        }
        return new Index.Indexer(){

            @Override
            public void begin() {
            }

            @Override
            public void partitionDelete(DeletionTime deletionTime) {
            }

            @Override
            public void rangeTombstone(RangeTombstone tombstone) {
            }

            @Override
            public void insertRow(Row row) {
                if (row.isStatic() && !CassandraIndex.this.indexedColumn.isStatic() && !CassandraIndex.this.indexedColumn.isPartitionKey()) {
                    return;
                }
                if (CassandraIndex.this.isPrimaryKeyIndex()) {
                    this.indexPrimaryKey((Clustering<?>)row.clustering(), this.getPrimaryKeyIndexLiveness(row), row.deletion());
                } else if (CassandraIndex.this.indexedColumn.isComplex()) {
                    this.indexCells((Clustering<?>)row.clustering(), row.getComplexColumnData(CassandraIndex.this.indexedColumn));
                } else {
                    this.indexCell((Clustering<?>)row.clustering(), row.getCell(CassandraIndex.this.indexedColumn));
                }
            }

            @Override
            public void removeRow(Row row) {
                if (CassandraIndex.this.isPrimaryKeyIndex()) {
                    return;
                }
                if (CassandraIndex.this.indexedColumn.isComplex()) {
                    this.removeCells((Clustering<?>)row.clustering(), row.getComplexColumnData(CassandraIndex.this.indexedColumn));
                } else {
                    this.removeCell((Clustering<?>)row.clustering(), row.getCell(CassandraIndex.this.indexedColumn));
                }
            }

            @Override
            public void updateRow(Row oldRow, Row newRow) {
                assert (oldRow.isStatic() == newRow.isStatic());
                if (newRow.isStatic() != CassandraIndex.this.indexedColumn.isStatic()) {
                    return;
                }
                if (CassandraIndex.this.isPrimaryKeyIndex()) {
                    this.indexPrimaryKey((Clustering<?>)newRow.clustering(), newRow.primaryKeyLivenessInfo(), newRow.deletion());
                }
                if (CassandraIndex.this.indexedColumn.isComplex()) {
                    this.indexCells((Clustering<?>)newRow.clustering(), newRow.getComplexColumnData(CassandraIndex.this.indexedColumn));
                    this.removeCells((Clustering<?>)oldRow.clustering(), oldRow.getComplexColumnData(CassandraIndex.this.indexedColumn));
                } else {
                    this.indexCell((Clustering<?>)newRow.clustering(), newRow.getCell(CassandraIndex.this.indexedColumn));
                    this.removeCell((Clustering<?>)oldRow.clustering(), oldRow.getCell(CassandraIndex.this.indexedColumn));
                }
            }

            @Override
            public void finish() {
            }

            private void indexCells(Clustering<?> clustering, Iterable<Cell<?>> cells) {
                if (cells == null) {
                    return;
                }
                for (Cell<?> cell : cells) {
                    this.indexCell(clustering, cell);
                }
            }

            private void indexCell(Clustering<?> clustering, Cell<?> cell) {
                if (cell == null || !cell.isLive(nowInSec)) {
                    return;
                }
                CassandraIndex.this.insert(key.getKey(), clustering, cell, LivenessInfo.withExpirationTime(cell.timestamp(), cell.ttl(), cell.localDeletionTime()), ctx);
            }

            private void removeCells(Clustering<?> clustering, Iterable<Cell<?>> cells) {
                if (cells == null) {
                    return;
                }
                for (Cell<?> cell : cells) {
                    this.removeCell(clustering, cell);
                }
            }

            private void removeCell(Clustering<?> clustering, Cell<?> cell) {
                if (cell == null || !cell.isLive(nowInSec)) {
                    return;
                }
                CassandraIndex.this.delete(key.getKey(), clustering, cell, ctx, nowInSec);
            }

            private void indexPrimaryKey(Clustering<?> clustering, LivenessInfo liveness, Row.Deletion deletion) {
                if (liveness.timestamp() != Long.MIN_VALUE) {
                    CassandraIndex.this.insert(key.getKey(), clustering, null, liveness, ctx);
                }
                if (!deletion.isLive()) {
                    CassandraIndex.this.delete(key.getKey(), clustering, deletion.time(), ctx);
                }
            }

            private LivenessInfo getPrimaryKeyIndexLiveness(Row row) {
                long timestamp = row.primaryKeyLivenessInfo().timestamp();
                int ttl = row.primaryKeyLivenessInfo().ttl();
                for (Cell<?> cell : row.cells()) {
                    long cellTimestamp = cell.timestamp();
                    if (!cell.isLive(nowInSec) || cellTimestamp <= timestamp) continue;
                    timestamp = cellTimestamp;
                    ttl = cell.ttl();
                }
                return LivenessInfo.create(timestamp, ttl, nowInSec);
            }
        };
    }

    public void deleteStaleEntry(DecoratedKey indexKey, Clustering<?> indexClustering, DeletionTime deletion, WriteContext ctx) {
        this.doDelete(indexKey, indexClustering, deletion, ctx);
        logger.trace("Removed index entry for stale value {}", (Object)indexKey);
    }

    private void insert(ByteBuffer rowKey, Clustering<?> clustering, Cell<?> cell, LivenessInfo info, WriteContext ctx) {
        DecoratedKey valueKey = this.getIndexKeyFor(this.getIndexedValue(rowKey, clustering, cell));
        BTreeRow row = BTreeRow.noCellLiveRow(this.buildIndexClustering(rowKey, clustering, cell), info);
        PartitionUpdate upd = this.partitionUpdate(valueKey, row);
        this.indexCfs.getWriteHandler().write(upd, ctx, UpdateTransaction.NO_OP);
        logger.trace("Inserted entry into index for value {}", (Object)valueKey);
    }

    private void delete(ByteBuffer rowKey, Clustering<?> clustering, Cell<?> cell, WriteContext ctx, int nowInSec) {
        DecoratedKey valueKey = this.getIndexKeyFor(this.getIndexedValue(rowKey, clustering, cell));
        this.doDelete(valueKey, this.buildIndexClustering(rowKey, clustering, cell), new DeletionTime(cell.timestamp(), nowInSec), ctx);
    }

    private void delete(ByteBuffer rowKey, Clustering<?> clustering, DeletionTime deletion, WriteContext ctx) {
        DecoratedKey valueKey = this.getIndexKeyFor(this.getIndexedValue(rowKey, clustering, null));
        this.doDelete(valueKey, this.buildIndexClustering(rowKey, clustering, null), deletion, ctx);
    }

    private void doDelete(DecoratedKey indexKey, Clustering<?> indexClustering, DeletionTime deletion, WriteContext ctx) {
        BTreeRow row = BTreeRow.emptyDeletedRow(indexClustering, Row.Deletion.regular(deletion));
        PartitionUpdate upd = this.partitionUpdate(indexKey, row);
        this.indexCfs.getWriteHandler().write(upd, ctx, UpdateTransaction.NO_OP);
        logger.trace("Removed index entry for value {}", (Object)indexKey);
    }

    private void validatePartitionKey(DecoratedKey partitionKey) throws InvalidRequestException {
        assert (this.indexedColumn.isPartitionKey());
        this.validateIndexedValue(this.getIndexedValue(partitionKey.getKey(), null, null));
    }

    private void validateClusterings(PartitionUpdate update) throws InvalidRequestException {
        assert (this.indexedColumn.isClusteringColumn());
        for (Row row : update) {
            this.validateIndexedValue(this.getIndexedValue(null, (Clustering<?>)row.clustering(), null));
        }
    }

    private void validateRows(Iterable<Row> rows) {
        assert (!this.indexedColumn.isPrimaryKeyColumn());
        for (Row row : rows) {
            if (this.indexedColumn.isComplex()) {
                ComplexColumnData data = row.getComplexColumnData(this.indexedColumn);
                if (data == null) continue;
                for (Cell<?> cell : data) {
                    this.validateIndexedValue(this.getIndexedValue(null, null, cell.path(), cell.buffer()));
                }
                continue;
            }
            this.validateIndexedValue(this.getIndexedValue(null, null, row.getCell(this.indexedColumn)));
        }
    }

    private void validateIndexedValue(ByteBuffer value) {
        if (value != null && value.remaining() >= 65535) {
            throw new InvalidRequestException(String.format("Cannot index value of size %d for index %s on %s(%s) (maximum allowed size=%d)", value.remaining(), this.metadata.name, this.baseCfs.metadata, this.indexedColumn.name.toString(), 65535));
        }
    }

    private ByteBuffer getIndexedValue(ByteBuffer rowKey, Clustering<?> clustering, Cell<?> cell) {
        return this.getIndexedValue(rowKey, clustering, cell == null ? null : cell.path(), cell == null ? null : cell.buffer());
    }

    private Clustering<?> buildIndexClustering(ByteBuffer rowKey, Clustering<?> clustering, Cell<?> cell) {
        return this.buildIndexClusteringPrefix(rowKey, clustering, cell == null ? null : cell.path()).build();
    }

    private DecoratedKey getIndexKeyFor(ByteBuffer value) {
        return this.indexCfs.decorateKey(value);
    }

    private PartitionUpdate partitionUpdate(DecoratedKey valueKey, Row row) {
        return PartitionUpdate.singleRowUpdate(this.indexCfs.metadata(), valueKey, row);
    }

    private void invalidate() {
        Set<ColumnFamilyStore> cfss = Collections.singleton(this.indexCfs);
        CompactionManager.instance.interruptCompactionForCFs(cfss, sstable -> true, true);
        CompactionManager.instance.waitForCessation(cfss, sstable -> true);
        Keyspace.writeOrder.awaitNewBarrier();
        this.indexCfs.forceBlockingFlush();
        this.indexCfs.readOrdering.awaitNewBarrier();
        this.indexCfs.invalidate();
    }

    private boolean isBuilt() {
        return SystemKeyspace.isIndexBuilt(this.baseCfs.keyspace.getName(), this.metadata.name);
    }

    private boolean isPrimaryKeyIndex() {
        return this.indexedColumn.isPrimaryKeyColumn();
    }

    private Callable<?> getBuildIndexTask() {
        return () -> {
            this.buildBlocking();
            return null;
        };
    }

    private void buildBlocking() {
        this.baseCfs.forceBlockingFlush();
        try (ColumnFamilyStore.RefViewFragment viewFragment = this.baseCfs.selectAndReference(View.selectFunction(SSTableSet.CANONICAL));
             Refs<SSTableReader> sstables = viewFragment.refs;){
            if (sstables.isEmpty()) {
                logger.info("No SSTable data for {}.{} to build index {} from, marking empty index as built", new Object[]{this.baseCfs.metadata.keyspace, this.baseCfs.metadata.name, this.metadata.name});
                return;
            }
            logger.info("Submitting index build of {} for data in {}", (Object)this.metadata.name, (Object)CassandraIndex.getSSTableNames(sstables));
            CollatedViewIndexBuilder builder = new CollatedViewIndexBuilder(this.baseCfs, Collections.singleton(this), new ReducingKeyIterator(sstables), (Collection<SSTableReader>)ImmutableSet.copyOf(sstables));
            ListenableFuture<?> future = CompactionManager.instance.submitIndexBuild(builder);
            FBUtilities.waitOnFuture(future);
            this.indexCfs.forceBlockingFlush();
        }
        logger.info("Index build of {} complete", (Object)this.metadata.name);
    }

    private static String getSSTableNames(Collection<SSTableReader> sstables) {
        return StreamSupport.stream(sstables.spliterator(), false).map(SSTable::toString).collect(Collectors.joining(", "));
    }

    public static TableMetadata indexCfsMetadata(TableMetadata baseCfsMetadata, IndexMetadata indexMetadata) {
        Pair<ColumnMetadata, IndexTarget.Type> target = TargetParser.parse(baseCfsMetadata, indexMetadata);
        CassandraIndexFunctions utils = CassandraIndex.getFunctions(indexMetadata, target);
        ColumnMetadata indexedColumn = (ColumnMetadata)target.left;
        AbstractType<?> indexedValueType = utils.getIndexedValueType(indexedColumn);
        TableMetadata.Builder builder = TableMetadata.builder(baseCfsMetadata.keyspace, baseCfsMetadata.indexTableName(indexMetadata), baseCfsMetadata.id).kind(TableMetadata.Kind.INDEX).partitioner(new LocalPartitioner(indexedValueType)).addPartitionKeyColumn(indexedColumn.name, indexedColumn.type).addClusteringColumn("partition_key", baseCfsMetadata.partitioner.partitionOrdering());
        builder = utils.addIndexClusteringColumns(builder, baseCfsMetadata, indexedColumn);
        return builder.build().updateIndexTableMetadata(baseCfsMetadata.params);
    }

    public static CassandraIndex newIndex(ColumnFamilyStore baseCfs, IndexMetadata indexMetadata) {
        return CassandraIndex.getFunctions(indexMetadata, TargetParser.parse(baseCfs.metadata(), indexMetadata)).newIndexInstance(baseCfs, indexMetadata);
    }

    static CassandraIndexFunctions getFunctions(IndexMetadata indexDef, Pair<ColumnMetadata, IndexTarget.Type> target) {
        if (indexDef.isKeys()) {
            return CassandraIndexFunctions.KEYS_INDEX_FUNCTIONS;
        }
        ColumnMetadata indexedColumn = (ColumnMetadata)target.left;
        if (indexedColumn.type.isCollection() && indexedColumn.type.isMultiCell()) {
            switch (((CollectionType)indexedColumn.type).kind) {
                case LIST: {
                    return CassandraIndexFunctions.COLLECTION_VALUE_INDEX_FUNCTIONS;
                }
                case SET: {
                    return CassandraIndexFunctions.COLLECTION_KEY_INDEX_FUNCTIONS;
                }
                case MAP: {
                    switch ((IndexTarget.Type)((Object)target.right)) {
                        case KEYS: {
                            return CassandraIndexFunctions.COLLECTION_KEY_INDEX_FUNCTIONS;
                        }
                        case KEYS_AND_VALUES: {
                            return CassandraIndexFunctions.COLLECTION_ENTRY_INDEX_FUNCTIONS;
                        }
                        case VALUES: {
                            return CassandraIndexFunctions.COLLECTION_VALUE_INDEX_FUNCTIONS;
                        }
                    }
                    throw new AssertionError();
                }
            }
        }
        switch (indexedColumn.kind) {
            case CLUSTERING: {
                return CassandraIndexFunctions.CLUSTERING_COLUMN_INDEX_FUNCTIONS;
            }
            case REGULAR: 
            case STATIC: {
                return CassandraIndexFunctions.REGULAR_COLUMN_INDEX_FUNCTIONS;
            }
            case PARTITION_KEY: {
                return CassandraIndexFunctions.PARTITION_KEY_INDEX_FUNCTIONS;
            }
        }
        throw new AssertionError();
    }
}

