/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.commitlog;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.file.FileStore;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.UUID;
import java.util.concurrent.TimeUnit;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.zip.CRC32;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.config.ParameterizedClass;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.commitlog.AbstractCommitLogSegmentManager;
import org.apache.cassandra.db.commitlog.AbstractCommitLogService;
import org.apache.cassandra.db.commitlog.BatchCommitLogService;
import org.apache.cassandra.db.commitlog.CommitLogArchiver;
import org.apache.cassandra.db.commitlog.CommitLogDescriptor;
import org.apache.cassandra.db.commitlog.CommitLogMBean;
import org.apache.cassandra.db.commitlog.CommitLogPosition;
import org.apache.cassandra.db.commitlog.CommitLogReplayer;
import org.apache.cassandra.db.commitlog.CommitLogSegment;
import org.apache.cassandra.db.commitlog.CommitLogSegmentManagerCDC;
import org.apache.cassandra.db.commitlog.GroupCommitLogService;
import org.apache.cassandra.db.commitlog.PeriodicCommitLogService;
import org.apache.cassandra.exceptions.CDCWriteException;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.compress.ICompressor;
import org.apache.cassandra.io.util.DataOutputBuffer;
import org.apache.cassandra.io.util.DataOutputBufferFixed;
import org.apache.cassandra.io.util.DataOutputPlus;
import org.apache.cassandra.io.util.File;
import org.apache.cassandra.io.util.PathUtils;
import org.apache.cassandra.metrics.CommitLogMetrics;
import org.apache.cassandra.schema.CompressionParams;
import org.apache.cassandra.schema.TableId;
import org.apache.cassandra.security.EncryptionContext;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.apache.cassandra.utils.JVMStabilityInspector;
import org.apache.cassandra.utils.MBeanWrapper;
import org.apache.cassandra.utils.concurrent.UncheckedInterruptedException;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CommitLog
implements CommitLogMBean {
    private static final Logger logger = LoggerFactory.getLogger(CommitLog.class);
    public static final CommitLog instance = CommitLog.construct();
    private static final BiPredicate<File, String> unmanagedFilesFilter = (dir, name) -> CommitLogDescriptor.isValid(name) && CommitLogSegment.shouldReplay(name);
    public final AbstractCommitLogSegmentManager segmentManager;
    public CommitLogArchiver archiver;
    public final CommitLogMetrics metrics;
    final AbstractCommitLogService executor;
    volatile Configuration configuration = new Configuration(DatabaseDescriptor.getCommitLogCompression(), DatabaseDescriptor.getEncryptionContext());
    private boolean started = false;

    private static CommitLog construct() {
        CommitLog log = new CommitLog(CommitLogArchiver.construct(), DatabaseDescriptor.getCommitLogSegmentMgrProvider());
        MBeanWrapper.instance.registerMBean((Object)log, "org.apache.cassandra.db:type=Commitlog");
        return log;
    }

    @VisibleForTesting
    CommitLog(CommitLogArchiver archiver) {
        this(archiver, DatabaseDescriptor.getCommitLogSegmentMgrProvider());
    }

    @VisibleForTesting
    CommitLog(CommitLogArchiver archiver, Function<CommitLog, AbstractCommitLogSegmentManager> segmentManagerProvider) {
        DatabaseDescriptor.createAllDirectories();
        this.archiver = archiver;
        this.metrics = new CommitLogMetrics();
        switch (DatabaseDescriptor.getCommitLogSync()) {
            case periodic: {
                this.executor = new PeriodicCommitLogService(this);
                break;
            }
            case batch: {
                this.executor = new BatchCommitLogService(this);
                break;
            }
            case group: {
                this.executor = new GroupCommitLogService(this);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unknown commitlog service type: " + (Object)((Object)DatabaseDescriptor.getCommitLogSync()));
            }
        }
        this.segmentManager = segmentManagerProvider.apply(this);
        this.metrics.attach(this.executor, this.segmentManager);
    }

    public synchronized CommitLog start() {
        if (this.started) {
            return this;
        }
        try {
            this.segmentManager.start();
            this.executor.start();
            this.started = true;
        }
        catch (Throwable t) {
            this.started = false;
            throw t;
        }
        return this;
    }

    public boolean isStarted() {
        return this.started;
    }

    public boolean hasFilesToReplay() {
        return this.getUnmanagedFiles().length > 0;
    }

    public File[] getUnmanagedFiles() {
        File[] files = new File(this.segmentManager.storageDirectory).tryList(unmanagedFilesFilter);
        if (files == null) {
            return new File[0];
        }
        return files;
    }

    public int recoverSegmentsOnDisk() throws IOException {
        for (File file : this.getUnmanagedFiles()) {
            this.archiver.maybeArchive(file.path(), file.name());
            this.archiver.maybeWaitForArchiving(file.name());
        }
        assert (this.archiver.archivePending.isEmpty()) : "Not all commit log archive tasks were completed before restore";
        this.archiver.maybeRestoreArchive();
        Object[] files = this.getUnmanagedFiles();
        int replayed = 0;
        if (files.length == 0) {
            logger.info("No commitlog files found; skipping replay");
        } else {
            Arrays.sort(files, new CommitLogSegment.CommitLogSegmentFileComparator());
            logger.info("Replaying {}", (Object)StringUtils.join((Object[])files, (String)", "));
            replayed = this.recoverFiles((File[])files);
            logger.info("Log replay complete, {} replayed mutations", (Object)replayed);
            for (Object f : files) {
                this.segmentManager.handleReplayedSegment((File)f);
            }
        }
        return replayed;
    }

    public int recoverFiles(File ... clogs) throws IOException {
        CommitLogReplayer replayer = CommitLogReplayer.construct(this, CommitLog.getLocalHostId());
        replayer.replayFiles(clogs);
        return replayer.blockForWrites();
    }

    public void recoverPath(String path) throws IOException {
        CommitLogReplayer replayer = CommitLogReplayer.construct(this, CommitLog.getLocalHostId());
        replayer.replayPath(new File(path), false);
        replayer.blockForWrites();
    }

    private static UUID getLocalHostId() {
        return StorageService.instance.getLocalHostUUID();
    }

    @Override
    public void recover(String path) throws IOException {
        this.recoverPath(path);
    }

    public CommitLogPosition getCurrentPosition() {
        return this.segmentManager.getCurrentPosition();
    }

    public void forceRecycleAllSegments(Collection<TableId> droppedTables) {
        this.segmentManager.forceRecycleAll(droppedTables);
    }

    public void forceRecycleAllSegments() {
        this.segmentManager.forceRecycleAll(Collections.emptyList());
    }

    public void sync(boolean flush) throws IOException {
        this.segmentManager.sync(flush);
    }

    public void requestExtraSync() {
        this.executor.requestExtraSync();
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public CommitLogPosition add(Mutation mutation) throws CDCWriteException {
        assert (mutation != null);
        mutation.validateSize(12, 12);
        try (DataOutputBuffer dob = (DataOutputBuffer)DataOutputBuffer.scratchBuffer.get();){
            Mutation.serializer.serialize(mutation, (DataOutputPlus)dob, 12);
            int size = dob.getLength();
            int totalSize = size + 12;
            CommitLogSegment.Allocation alloc = this.segmentManager.allocate(mutation, totalSize);
            CRC32 checksum = new CRC32();
            ByteBuffer buffer = alloc.getBuffer();
            try (DataOutputBufferFixed dos = new DataOutputBufferFixed(buffer);){
                dos.writeInt(size);
                FBUtilities.updateChecksumInt(checksum, size);
                buffer.putInt((int)checksum.getValue());
                dos.write(dob.getData(), 0, size);
                FBUtilities.updateChecksum(checksum, buffer, buffer.position() - size, size);
                buffer.putInt((int)checksum.getValue());
            }
            catch (IOException e) {
                throw new FSWriteError((Throwable)e, alloc.getSegment().getPath());
            }
            finally {
                alloc.markWritten();
            }
            this.executor.finishWriteFor(alloc);
            CommitLogPosition commitLogPosition = alloc.getCommitLogPosition();
            return commitLogPosition;
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, this.segmentManager.allocatingFrom().getPath());
        }
    }

    public void discardCompletedSegments(TableId id, CommitLogPosition lowerBound, CommitLogPosition upperBound) {
        logger.trace("discard completed log segments for {}-{}, table {}", new Object[]{lowerBound, upperBound, id});
        Iterator<CommitLogSegment> iter = this.segmentManager.getActiveSegments().iterator();
        while (iter.hasNext()) {
            CommitLogSegment segment = iter.next();
            segment.markClean(id, lowerBound, upperBound);
            if (segment.isUnused()) {
                logger.debug("Commit log segment {} is unused", (Object)segment);
                this.segmentManager.archiveAndDiscard(segment);
            } else if (logger.isTraceEnabled()) {
                logger.trace("Not safe to delete{} commit log segment {}; dirty is {}", new Object[]{iter.hasNext() ? "" : " active", segment, segment.dirtyString()});
            }
            if (!segment.contains(upperBound)) continue;
            break;
        }
    }

    @Override
    public String getArchiveCommand() {
        return this.archiver.archiveCommand;
    }

    @Override
    public String getRestoreCommand() {
        return this.archiver.restoreCommand;
    }

    @Override
    public String getRestoreDirectories() {
        return this.archiver.restoreDirectories;
    }

    @Override
    public long getRestorePointInTime() {
        return this.archiver.restorePointInTimeInMicroseconds;
    }

    @Override
    public String getRestorePrecision() {
        return this.archiver.precision.toString();
    }

    @VisibleForTesting
    public void setCommitlogArchiver(CommitLogArchiver archiver) {
        this.archiver = archiver;
    }

    @Override
    public List<String> getActiveSegmentNames() {
        Collection<CommitLogSegment> segments = this.segmentManager.getActiveSegments();
        ArrayList<String> segmentNames = new ArrayList<String>(segments.size());
        for (CommitLogSegment seg : segments) {
            segmentNames.add(seg.getName());
        }
        return segmentNames;
    }

    @Override
    public List<String> getArchivingSegmentNames() {
        return new ArrayList<String>(this.archiver.archivePending.keySet());
    }

    @Override
    public long getActiveContentSize() {
        long size = 0L;
        for (CommitLogSegment seg : this.segmentManager.getActiveSegments()) {
            size += seg.contentSize();
        }
        return size;
    }

    @Override
    public long getActiveOnDiskSize() {
        return this.segmentManager.onDiskSize();
    }

    @Override
    public Map<String, Double> getActiveSegmentCompressionRatios() {
        TreeMap<String, Double> segmentRatios = new TreeMap<String, Double>();
        for (CommitLogSegment seg : this.segmentManager.getActiveSegments()) {
            segmentRatios.put(seg.getName(), 1.0 * (double)seg.onDiskSize() / (double)seg.contentSize());
        }
        return segmentRatios;
    }

    @Override
    public boolean getCDCBlockWrites() {
        return DatabaseDescriptor.getCDCBlockWrites();
    }

    @Override
    public void setCDCBlockWrites(boolean val) {
        Preconditions.checkState((boolean)DatabaseDescriptor.isCDCEnabled(), (String)"Unable to set block_writes (%s): CDC is not enabled.", (Object)val);
        Preconditions.checkState((boolean)(this.segmentManager instanceof CommitLogSegmentManagerCDC), (String)"CDC is enabled but we have the wrong CommitLogSegmentManager type: %s. Please report this as bug.", (Object)this.segmentManager.getClass().getName());
        boolean oldVal = DatabaseDescriptor.getCDCBlockWrites();
        CommitLogSegment currentSegment = this.segmentManager.allocatingFrom();
        if (!val && currentSegment.getCDCState() == CommitLogSegment.CDCState.FORBIDDEN) {
            currentSegment.setCDCState(CommitLogSegment.CDCState.PERMITTED);
        }
        DatabaseDescriptor.setCDCBlockWrites(val);
        logger.info("Updated CDC block_writes from {} to {}", (Object)oldVal, (Object)val);
    }

    public synchronized void shutdownBlocking() throws InterruptedException {
        if (!this.started) {
            return;
        }
        this.started = false;
        this.executor.shutdown();
        this.executor.awaitTermination();
        this.segmentManager.shutdown();
        this.segmentManager.awaitTermination(1L, TimeUnit.MINUTES);
    }

    @VisibleForTesting
    public synchronized int resetUnsafe(boolean deleteSegments) throws IOException {
        this.stopUnsafe(deleteSegments);
        this.resetConfiguration();
        return this.restartUnsafe();
    }

    @VisibleForTesting
    public synchronized void resetConfiguration() {
        this.configuration = new Configuration(DatabaseDescriptor.getCommitLogCompression(), DatabaseDescriptor.getEncryptionContext());
    }

    @VisibleForTesting
    public synchronized void stopUnsafe(boolean deleteSegments) {
        if (!this.started) {
            return;
        }
        this.started = false;
        this.executor.shutdown();
        try {
            this.executor.awaitTermination();
        }
        catch (InterruptedException e) {
            throw new UncheckedInterruptedException(e);
        }
        this.segmentManager.stopUnsafe(deleteSegments);
        CommitLogSegment.resetReplayLimit();
        if (DatabaseDescriptor.isCDCEnabled() && deleteSegments) {
            for (File f : new File(DatabaseDescriptor.getCDCLogLocation()).tryList()) {
                f.delete();
            }
        }
    }

    @VisibleForTesting
    public synchronized int restartUnsafe() throws IOException {
        this.started = false;
        return this.start().recoverSegmentsOnDisk();
    }

    public static long freeDiskSpace() {
        return PathUtils.tryGetSpace(new File(DatabaseDescriptor.getCommitLogLocation()).toPath(), FileStore::getTotalSpace);
    }

    @VisibleForTesting
    public static boolean handleCommitError(String message, Throwable t) {
        JVMStabilityInspector.inspectCommitLogThrowable(t);
        switch (DatabaseDescriptor.getCommitFailurePolicy()) {
            case die: 
            case stop: {
                StorageService.instance.stopTransports();
            }
            case stop_commit: {
                String errorMsg = String.format("%s. Commit disk failure policy is %s; terminating thread.", new Object[]{message, DatabaseDescriptor.getCommitFailurePolicy()});
                logger.error(CommitLog.addAdditionalInformationIfPossible(errorMsg), t);
                return false;
            }
            case ignore: {
                logger.error(CommitLog.addAdditionalInformationIfPossible(message), t);
                return true;
            }
        }
        throw new AssertionError((Object)DatabaseDescriptor.getCommitFailurePolicy());
    }

    private static String addAdditionalInformationIfPossible(String msg) {
        int segmentSize;
        long unallocatedSpace = CommitLog.freeDiskSpace();
        if (unallocatedSpace < (long)(segmentSize = DatabaseDescriptor.getCommitLogSegmentSize())) {
            return String.format("%s. %d bytes required for next commitlog segment but only %d bytes available. Check %s to see if not enough free space is the reason for this error.", msg, segmentSize, unallocatedSpace, DatabaseDescriptor.getCommitLogLocation());
        }
        return msg;
    }

    public static final class Configuration {
        private final ParameterizedClass compressorClass;
        private final ICompressor compressor;
        private EncryptionContext encryptionContext;

        public Configuration(ParameterizedClass compressorClass, EncryptionContext encryptionContext) {
            this.compressorClass = compressorClass;
            this.compressor = compressorClass != null ? CompressionParams.createCompressor(compressorClass) : null;
            this.encryptionContext = encryptionContext;
        }

        public boolean useCompression() {
            return this.compressor != null;
        }

        public boolean useEncryption() {
            return this.encryptionContext != null && this.encryptionContext.isEnabled();
        }

        public ICompressor getCompressor() {
            return this.compressor;
        }

        public ParameterizedClass getCompressorClass() {
            return this.compressorClass;
        }

        public String getCompressorName() {
            return this.useCompression() ? this.compressor.getClass().getSimpleName() : "none";
        }

        public EncryptionContext getEncryptionContext() {
            return this.encryptionContext;
        }
    }
}

