/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.utils.btree;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.Iterators;
import com.google.common.collect.Ordering;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.Objects;
import java.util.SortedSet;
import java.util.function.BiConsumer;
import java.util.function.BiFunction;
import java.util.function.Consumer;
import java.util.function.Function;
import org.apache.cassandra.config.CassandraRelevantProperties;
import org.apache.cassandra.utils.BiLongAccumulator;
import org.apache.cassandra.utils.BulkIterator;
import org.apache.cassandra.utils.LongAccumulator;
import org.apache.cassandra.utils.ObjectSizes;
import org.apache.cassandra.utils.btree.BTreeSearchIterator;
import org.apache.cassandra.utils.btree.BTreeSet;
import org.apache.cassandra.utils.btree.FullBTreeSearchIterator;
import org.apache.cassandra.utils.btree.LeafBTreeSearchIterator;
import org.apache.cassandra.utils.btree.UpdateFunction;
import org.apache.cassandra.utils.caching.TinyThreadLocalPool;

public class BTree {
    public static final int BRANCH_SHIFT = CassandraRelevantProperties.BTREE_BRANCH_SHIFT.getInt();
    private static final int BRANCH_FACTOR = 1 << BRANCH_SHIFT;
    public static final int MIN_KEYS = BRANCH_FACTOR / 2 - 1;
    public static final int MAX_KEYS = BRANCH_FACTOR - 1;
    private static final Object[] EMPTY_LEAF = new Object[1];
    private static final int[][] DENSE_SIZE_MAPS = BTree.buildBalancedSizeMaps(BRANCH_SHIFT);
    private static final long[] PERFECT_DENSE_SIZE_ON_HEAP = BTree.sizeOnHeapOfPerfectTrees(BRANCH_SHIFT);
    private static Object POSITIVE_INFINITY = new Object();
    private static Object NEGATIVE_INFINITY = new Object();

    public static Object[] empty() {
        return EMPTY_LEAF;
    }

    public static Object[] singleton(Object value) {
        return new Object[]{value};
    }

    @Deprecated
    public static <C, K extends C, V extends C> Object[] build(Collection<K> source) {
        return BTree.build(source, UpdateFunction.noOp());
    }

    @Deprecated
    public static <C, K extends C, V extends C> Object[] build(Collection<K> source, UpdateFunction<K, V> updateF) {
        return BTree.build(BulkIterator.of(source.iterator()), source.size(), updateF);
    }

    public static <C, I extends C, O extends C> Object[] build(BulkIterator<I> source, int size, UpdateFunction<I, O> updateF) {
        assert (size >= 0);
        if (size == 0) {
            return EMPTY_LEAF;
        }
        if (size <= MAX_KEYS) {
            return BTree.buildLeaf(source, size, updateF);
        }
        return BTree.buildRoot(source, size, updateF);
    }

    private static <C, I extends C, O extends C> Object[] buildLeaf(BulkIterator<I> insert, int size, UpdateFunction<I, O> updateF) {
        Object[] values = new Object[size | 1];
        insert.fetch(values, 0, size);
        if (!BTree.isSimple(updateF)) {
            updateF.onAllocatedOnHeap(ObjectSizes.sizeOfReferenceArray(values.length));
            for (int i = 0; i < size; ++i) {
                values[i] = updateF.insert(values[i]);
            }
        }
        return values;
    }

    private static <C, I extends C, O extends C> Object[] buildLeafWithoutSizeTracking(BulkIterator<I> insert, int size, UpdateFunction<I, O> updateF) {
        Object[] values = new Object[size | 1];
        insert.fetch(values, 0, size);
        if (!BTree.isSimple(updateF)) {
            for (int i = 0; i < size; ++i) {
                values[i] = updateF.insert(values[i]);
            }
        }
        return values;
    }

    private static <C, I extends C, O extends C> Object[] buildRoot(BulkIterator<I> source, int size, UpdateFunction<I, O> updateF) {
        int height = BTree.minHeight(size);
        assert (height > 1);
        assert (height * BRANCH_SHIFT < 32);
        int denseChildSize = BTree.denseSize(height - 1);
        int childCount = size / (denseChildSize + 1) + 1;
        return BTree.buildMaximallyDense(source, childCount, size, height, updateF);
    }

    private static <C, I extends C, O extends C> Object[] buildMaximallyDense(BulkIterator<I> source, int childCount, int size, int height, UpdateFunction<I, O> updateF) {
        assert (childCount <= MAX_KEYS + 1);
        int keyCount = childCount - 1;
        int[] sizeMap = new int[childCount];
        Object[] branch = new Object[childCount * 2];
        if (height == 2) {
            int remaining = size;
            int threshold = MAX_KEYS + 1 + MIN_KEYS;
            int i = 0;
            while (remaining >= threshold) {
                branch[keyCount + i] = BTree.buildLeaf(source, MAX_KEYS, updateF);
                branch[i] = BTree.isSimple(updateF) ? source.next() : updateF.insert(source.next());
                sizeMap[i++] = size - (remaining -= MAX_KEYS + 1) - 1;
            }
            if (remaining > MAX_KEYS) {
                int childSize = remaining / 2;
                branch[keyCount + i] = BTree.buildLeaf(source, childSize, updateF);
                branch[i] = BTree.isSimple(updateF) ? source.next() : updateF.insert(source.next());
                sizeMap[i++] = size - (remaining -= childSize + 1) - 1;
            }
            branch[keyCount + i] = BTree.buildLeaf(source, remaining, updateF);
            sizeMap[i++] = size;
            assert (i == childCount);
        } else {
            int childSize;
            int grandChildCount;
            int denseChildSize = BTree.denseSize(--height);
            int denseGrandChildSize = BTree.denseSize(height - 1);
            int threshold = denseChildSize + 1 + MIN_KEYS * (denseGrandChildSize + 1);
            int remaining = size;
            int i = 0;
            while (remaining >= threshold) {
                branch[keyCount + i] = BTree.buildPerfectDense(source, height, updateF);
                branch[i] = BTree.isSimple(updateF) ? source.next() : updateF.insert(source.next());
                sizeMap[i++] = size - (remaining -= denseChildSize + 1) - 1;
            }
            if (remaining > denseChildSize) {
                grandChildCount = remaining / ((denseGrandChildSize + 1) * 2);
                assert (grandChildCount >= MIN_KEYS + 1);
                childSize = grandChildCount * (denseGrandChildSize + 1) - 1;
                branch[keyCount + i] = BTree.buildMaximallyDense(source, grandChildCount, childSize, height, updateF);
                branch[i] = BTree.isSimple(updateF) ? source.next() : updateF.insert(source.next());
                sizeMap[i++] = size - (remaining -= childSize + 1) - 1;
            }
            grandChildCount = remaining / (denseGrandChildSize + 1) + 1;
            assert (grandChildCount >= MIN_KEYS + 1);
            childSize = remaining;
            branch[keyCount + i] = BTree.buildMaximallyDense(source, grandChildCount, childSize, height, updateF);
            sizeMap[i++] = size;
            assert (i == childCount);
        }
        branch[2 * keyCount + 1] = sizeMap;
        if (!BTree.isSimple(updateF)) {
            updateF.onAllocatedOnHeap(ObjectSizes.sizeOfArray(branch) + ObjectSizes.sizeOfArray(sizeMap));
        }
        return branch;
    }

    private static <C, I extends C, O extends C> Object[] buildPerfectDense(BulkIterator<I> source, int height, UpdateFunction<I, O> updateF) {
        Object[] result = BTree.buildPerfectDenseWithoutSizeTracking(source, height, updateF);
        updateF.onAllocatedOnHeap(PERFECT_DENSE_SIZE_ON_HEAP[height]);
        return result;
    }

    private static <C, I extends C, O extends C> Object[] buildPerfectDenseWithoutSizeTracking(BulkIterator<I> source, int height, UpdateFunction<I, O> updateF) {
        int keyCount = (1 << BRANCH_SHIFT) - 1;
        Object[] node = new Object[(1 << BRANCH_SHIFT) * 2];
        if (height == 2) {
            int childSize = BTree.treeSize2n(1, BRANCH_SHIFT);
            for (int i = 0; i < keyCount; ++i) {
                node[keyCount + i] = BTree.buildLeafWithoutSizeTracking(source, childSize, updateF);
                node[i] = BTree.isSimple(updateF) ? source.next() : updateF.insert(source.next());
            }
            node[2 * keyCount] = BTree.buildLeafWithoutSizeTracking(source, childSize, updateF);
        } else {
            for (int i = 0; i < keyCount; ++i) {
                Object[] child;
                node[keyCount + i] = child = BTree.buildPerfectDenseWithoutSizeTracking(source, height - 1, updateF);
                node[i] = BTree.isSimple(updateF) ? source.next() : updateF.insert(source.next());
            }
            node[2 * keyCount] = BTree.buildPerfectDenseWithoutSizeTracking(source, height - 1, updateF);
        }
        node[keyCount * 2 + 1] = DENSE_SIZE_MAPS[height - 2];
        return node;
    }

    public static <Compare> Object[] update(Object[] toUpdate, Object[] insert, Comparator<? super Compare> comparator) {
        return BTree.update(toUpdate, insert, comparator, UpdateFunction.noOp());
    }

    public static <Compare, Existing extends Compare, Insert extends Compare> Object[] update(Object[] toUpdate, Object[] insert, Comparator<? super Compare> comparator, UpdateFunction<Insert, Existing> updateF) {
        if (BTree.isEmpty(insert)) {
            return toUpdate;
        }
        if (BTree.isEmpty(toUpdate)) {
            if (BTree.isSimple(updateF)) {
                return insert;
            }
            insert = BTree.transform(insert, updateF::insert);
            updateF.onAllocatedOnHeap(BTree.sizeOnHeapOf(insert));
            return insert;
        }
        if (BTree.isLeaf(toUpdate) && BTree.isLeaf(insert)) {
            if (updateF == UpdateFunction.noOp && toUpdate.length < insert.length) {
                Object[] tmp = toUpdate;
                toUpdate = insert;
                insert = tmp;
            }
            Object[] merged = BTree.updateLeaves(toUpdate, insert, comparator, updateF);
            updateF.onAllocatedOnHeap(BTree.sizeOnHeapOf(merged) - BTree.sizeOnHeapOf(toUpdate));
            return merged;
        }
        if (!BTree.isLeaf(insert) && BTree.isSimple(updateF)) {
            int updateSize = BTree.size(toUpdate);
            int insertSize = BTree.size(insert);
            int scale = Integer.numberOfLeadingZeros(updateSize) - Integer.numberOfLeadingZeros(insertSize);
            if (scale >= 4) {
                Object[] tmp = insert;
                insert = toUpdate;
                toUpdate = tmp;
                if (updateF != UpdateFunction.noOp) {
                    updateF = ((UpdateFunction.Simple)updateF).flip();
                }
            }
        }
        try (Updater<? super Compare, Existing, Insert> updater = Updater.get();){
            Object[] objectArray = updater.update(toUpdate, insert, comparator, updateF);
            return objectArray;
        }
    }

    public static <Compare, Existing extends Compare, Insert extends Compare> Object[] updateLeaves(Object[] unode, Object[] inode, Comparator<? super Compare> comparator, UpdateFunction<Insert, Existing> updateF) {
        int upos = -1;
        int usz = BTree.sizeOfLeaf(unode);
        Object uk = unode[0];
        int ipos = 0;
        int isz = BTree.sizeOfLeaf(inode);
        Object ik = inode[0];
        Object merged = null;
        int c = -1;
        while (c <= 0) {
            if (c < 0) {
                int n = c = (upos = BTree.exponentialSearch(comparator, unode, upos + 1, usz, ik)) < 0 ? 1 : 0;
                if (upos < 0) {
                    upos = -(1 + upos);
                }
                if (upos == usz) break;
                uk = unode[upos];
                continue;
            }
            merged = updateF.merge(uk, ik);
            if (merged != uk) break;
            if (++ipos == isz) {
                return unode;
            }
            if (++upos == usz) break;
            uk = unode[upos];
            ik = inode[ipos];
            c = comparator.compare(uk, ik);
        }
        try (FastBuilder<Object> builder = BTree.fastBuilder();){
            if (upos > 0) {
                builder.leaf().copy(unode, 0, upos);
            }
            if (upos < usz) {
                if (c == 0) {
                    builder.add(merged);
                    if (++upos < usz) {
                        uk = unode[upos];
                    }
                } else {
                    builder.add(updateF.insert(ik));
                }
                if (++ipos < isz) {
                    ik = inode[ipos];
                }
                if (upos < usz && ipos < isz) {
                    c = comparator.compare(uk, ik);
                    while (true) {
                        int until;
                        if (c == 0) {
                            builder.leaf().addKey(updateF.merge(uk, ik));
                            if (++upos == usz || ++ipos == isz) break;
                            uk = unode[upos];
                            ik = inode[ipos];
                            c = comparator.compare(uk, ik);
                            continue;
                        }
                        if (c < 0) {
                            until = BTree.exponentialSearch(comparator, unode, upos + 1, usz, ik);
                            int n = c = until < 0 ? 1 : 0;
                            if (until < 0) {
                                until = -(1 + until);
                            }
                            builder.leaf().copy(unode, upos, until - upos);
                            upos = until;
                            if (upos == usz) break;
                            uk = unode[upos];
                            continue;
                        }
                        until = BTree.exponentialSearch(comparator, inode, ipos + 1, isz, uk);
                        c = until & Integer.MIN_VALUE;
                        if (until < 0) {
                            until = -(1 + until);
                        }
                        builder.leaf().copy(inode, ipos, until - ipos, updateF);
                        ipos = until;
                        if (ipos == isz) break;
                        ik = inode[ipos];
                    }
                }
                if (upos < usz) {
                    builder.leaf().copy(unode, upos, usz - upos);
                }
            }
            if (ipos < isz) {
                builder.leaf().copy(inode, ipos, isz - ipos, updateF);
            }
            Object[] objectArray = builder.build();
            return objectArray;
        }
    }

    public static void reverseInSitu(Object[] tree) {
        BTree.reverseInSitu(tree, BTree.height(tree), true);
    }

    private static void reverseInSitu(Object[] tree, int height, boolean copySizeMaps) {
        if (BTree.isLeaf(tree)) {
            BTree.reverse(tree, 0, BTree.sizeOfLeaf(tree));
        } else {
            int keyCount = BTree.shallowSizeOfBranch(tree);
            BTree.reverse(tree, 0, keyCount);
            BTree.reverse(tree, keyCount, keyCount * 2 + 1);
            for (int i = keyCount; i <= keyCount * 2; ++i) {
                BTree.reverseInSitu((Object[])tree[i], height - 1, copySizeMaps);
            }
            int[] sizeMap = (int[])tree[2 * keyCount + 1];
            if (sizeMap != DENSE_SIZE_MAPS[height - 2]) {
                if (copySizeMaps) {
                    sizeMap = (int[])sizeMap.clone();
                }
                BTree.sizeMapToSizes(sizeMap);
                BTree.reverse(sizeMap, 0, sizeMap.length);
                BTree.sizesToSizeMap(sizeMap);
            }
        }
    }

    public static <V> Iterator<V> iterator(Object[] btree) {
        return BTree.iterator(btree, Dir.ASC);
    }

    public static <V> Iterator<V> iterator(Object[] btree, Dir dir) {
        return BTree.isLeaf(btree) ? new LeafBTreeSearchIterator(btree, null, dir) : new FullBTreeSearchIterator(btree, null, dir);
    }

    public static <V> Iterator<V> iterator(Object[] btree, int lb, int ub, Dir dir) {
        return BTree.isLeaf(btree) ? new LeafBTreeSearchIterator(btree, null, dir, lb, ub) : new FullBTreeSearchIterator(btree, null, dir, lb, ub);
    }

    public static <V> Iterable<V> iterable(Object[] btree) {
        return BTree.iterable(btree, Dir.ASC);
    }

    public static <V> Iterable<V> iterable(Object[] btree, Dir dir) {
        return () -> BTree.iterator(btree, dir);
    }

    public static <V> Iterable<V> iterable(Object[] btree, int lb, int ub, Dir dir) {
        return () -> BTree.iterator(btree, lb, ub, dir);
    }

    public static <K, V> BTreeSearchIterator<K, V> slice(Object[] btree, Comparator<? super K> comparator, Dir dir) {
        return BTree.isLeaf(btree) ? new LeafBTreeSearchIterator(btree, comparator, dir) : new FullBTreeSearchIterator(btree, comparator, dir);
    }

    public static <K, V extends K> BTreeSearchIterator<K, V> slice(Object[] btree, Comparator<? super K> comparator, K start, K end, Dir dir) {
        return BTree.slice(btree, comparator, start, true, end, false, dir);
    }

    public static <K, V extends K> BTreeSearchIterator<K, V> slice(Object[] btree, Comparator<? super K> comparator, int startIndex, int endIndex, Dir dir) {
        return BTree.isLeaf(btree) ? new LeafBTreeSearchIterator(btree, comparator, dir, startIndex, endIndex) : new FullBTreeSearchIterator(btree, comparator, dir, startIndex, endIndex);
    }

    public static <K, V extends K> BTreeSearchIterator<K, V> slice(Object[] btree, Comparator<? super K> comparator, K start, boolean startInclusive, K end, boolean endInclusive, Dir dir) {
        int inclusiveLowerBound = Math.max(0, start == null ? Integer.MIN_VALUE : (startInclusive ? BTree.ceilIndex(btree, comparator, start) : BTree.higherIndex(btree, comparator, start)));
        int inclusiveUpperBound = Math.min(BTree.size(btree) - 1, end == null ? Integer.MAX_VALUE : (endInclusive ? BTree.floorIndex(btree, comparator, end) : BTree.lowerIndex(btree, comparator, end)));
        return BTree.isLeaf(btree) ? new LeafBTreeSearchIterator(btree, comparator, dir, inclusiveLowerBound, inclusiveUpperBound) : new FullBTreeSearchIterator(btree, comparator, dir, inclusiveLowerBound, inclusiveUpperBound);
    }

    public static <V> V find(Object[] node, Comparator<? super V> comparator, V find) {
        int keyEnd;
        int i;
        while ((i = Arrays.binarySearch(node, 0, keyEnd = BTree.getKeyEnd(node), find, comparator)) < 0) {
            if (BTree.isLeaf(node)) {
                return null;
            }
            i = -1 - i;
            node = (Object[])node[keyEnd + i];
        }
        return (V)node[i];
    }

    public static <V> void replaceInSitu(Object[] tree, int index, V replace) {
        if (index < 0 | index >= BTree.size(tree)) {
            throw new IndexOutOfBoundsException(index + " not in range [0.." + BTree.size(tree) + ")");
        }
        while (!BTree.isLeaf(tree)) {
            int[] sizeMap = BTree.getSizeMap(tree);
            int boundary = Arrays.binarySearch(sizeMap, index);
            if (boundary >= 0) {
                assert (boundary < sizeMap.length - 1);
                tree[boundary] = replace;
                return;
            }
            if ((boundary = -1 - boundary) > 0) {
                assert (boundary < sizeMap.length);
                index -= 1 + sizeMap[boundary - 1];
            }
            tree = (Object[])tree[BTree.getChildStart(tree) + boundary];
        }
        assert (index < BTree.getLeafKeyEnd(tree));
        tree[index] = replace;
    }

    public static <V> void replaceInSitu(Object[] node, Comparator<? super V> comparator, V find, V replace) {
        while (true) {
            int keyEnd;
            int i;
            if ((i = Arrays.binarySearch(node, 0, keyEnd = BTree.getKeyEnd(node), find, comparator)) >= 0) {
                assert (find == node[i]);
                node[i] = replace;
                return;
            }
            if (BTree.isLeaf(node)) {
                throw new NoSuchElementException();
            }
            i = -1 - i;
            node = (Object[])node[keyEnd + i];
        }
    }

    public static <V> int findIndex(Object[] node, Comparator<? super V> comparator, V find) {
        int lb = 0;
        while (true) {
            int keyEnd;
            int i;
            boolean exact;
            boolean bl = exact = (i = Arrays.binarySearch(node, 0, keyEnd = BTree.getKeyEnd(node), find, comparator)) >= 0;
            if (BTree.isLeaf(node)) {
                return exact ? lb + i : i - lb;
            }
            if (!exact) {
                i = -1 - i;
            }
            int[] sizeMap = BTree.getSizeMap(node);
            if (exact) {
                return lb + sizeMap[i];
            }
            if (i > 0) {
                lb += sizeMap[i - 1] + 1;
            }
            node = (Object[])node[keyEnd + i];
        }
    }

    public static <V> V findByIndex(Object[] tree, int index) {
        if (index < 0 | index >= BTree.size(tree)) {
            throw new IndexOutOfBoundsException(index + " not in range [0.." + BTree.size(tree) + ")");
        }
        Object[] node = tree;
        while (true) {
            if (BTree.isLeaf(node)) {
                int keyEnd = BTree.getLeafKeyEnd(node);
                assert (index < keyEnd);
                return (V)node[index];
            }
            int[] sizeMap = BTree.getSizeMap(node);
            int boundary = Arrays.binarySearch(sizeMap, index);
            if (boundary >= 0) {
                assert (boundary < sizeMap.length - 1);
                return (V)node[boundary];
            }
            if ((boundary = -1 - boundary) > 0) {
                assert (boundary < sizeMap.length);
                index -= 1 + sizeMap[boundary - 1];
            }
            node = (Object[])node[BTree.getChildStart(node) + boundary];
        }
    }

    public static <V> int lowerIndex(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.findIndex(btree, comparator, find);
        if (i < 0) {
            i = -1 - i;
        }
        return i - 1;
    }

    public static <V> V lower(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.lowerIndex(btree, comparator, find);
        return i >= 0 ? (V)BTree.findByIndex(btree, i) : null;
    }

    public static <V> int floorIndex(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.findIndex(btree, comparator, find);
        if (i < 0) {
            i = -2 - i;
        }
        return i;
    }

    public static <V> V floor(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.floorIndex(btree, comparator, find);
        return i >= 0 ? (V)BTree.findByIndex(btree, i) : null;
    }

    public static <V> int higherIndex(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.findIndex(btree, comparator, find);
        i = i < 0 ? -1 - i : ++i;
        return i;
    }

    public static <V> V higher(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.higherIndex(btree, comparator, find);
        return i < BTree.size(btree) ? (V)BTree.findByIndex(btree, i) : null;
    }

    public static <V> int ceilIndex(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.findIndex(btree, comparator, find);
        if (i < 0) {
            i = -1 - i;
        }
        return i;
    }

    public static <V> V ceil(Object[] btree, Comparator<? super V> comparator, V find) {
        int i = BTree.ceilIndex(btree, comparator, find);
        return i < BTree.size(btree) ? (V)BTree.findByIndex(btree, i) : null;
    }

    static int getKeyEnd(Object[] node) {
        if (BTree.isLeaf(node)) {
            return BTree.getLeafKeyEnd(node);
        }
        return BTree.getBranchKeyEnd(node);
    }

    static int getLeafKeyEnd(Object[] node) {
        int len = node.length;
        return node[len - 1] == null ? len - 1 : len;
    }

    static int getBranchKeyEnd(Object[] branchNode) {
        return branchNode.length / 2 - 1;
    }

    static int getChildStart(Object[] branchNode) {
        return BTree.getBranchKeyEnd(branchNode);
    }

    static int getChildEnd(Object[] branchNode) {
        return branchNode.length - 1;
    }

    static int getChildCount(Object[] branchNode) {
        return branchNode.length / 2;
    }

    static int[] getSizeMap(Object[] branchNode) {
        return (int[])branchNode[BTree.getChildEnd(branchNode)];
    }

    static int lookupSizeMap(Object[] branchNode, int index) {
        return BTree.getSizeMap(branchNode)[index];
    }

    public static int size(Object[] tree) {
        if (BTree.isLeaf(tree)) {
            return BTree.getLeafKeyEnd(tree);
        }
        int length = tree.length;
        return ((int[])tree[length - 1])[length / 2 - 1];
    }

    public static long sizeOfStructureOnHeap(Object[] tree) {
        if (tree == EMPTY_LEAF) {
            return 0L;
        }
        long size = ObjectSizes.sizeOfArray(tree);
        if (BTree.isLeaf(tree)) {
            return size;
        }
        for (int i = BTree.getChildStart(tree); i < BTree.getChildEnd(tree); ++i) {
            size += BTree.sizeOfStructureOnHeap((Object[])tree[i]);
        }
        return size;
    }

    public static boolean isLeaf(Object[] node) {
        return (node.length & 1) == 1;
    }

    public static boolean isEmpty(Object[] tree) {
        return tree == EMPTY_LEAF;
    }

    static int shallowSize(Object[] node) {
        if (BTree.isLeaf(node)) {
            return BTree.sizeOfLeaf(node);
        }
        return BTree.shallowSizeOfBranch(node);
    }

    static int sizeOfLeaf(Object[] leaf) {
        int len = leaf.length;
        return leaf[len - 1] == null ? len - 1 : len;
    }

    static int shallowSizeOfBranch(Object[] branch) {
        return branch.length / 2 - 1;
    }

    static int childOffset(Object[] branch) {
        return BTree.shallowSizeOfBranch(branch);
    }

    static int childEndOffset(Object[] branch) {
        return branch.length - 1;
    }

    public static int depth(Object[] tree) {
        int depth = 1;
        while (!BTree.isLeaf(tree)) {
            ++depth;
            tree = (Object[])tree[BTree.getKeyEnd(tree)];
        }
        return depth;
    }

    public static int toArray(Object[] tree, Object[] target, int targetOffset) {
        return BTree.toArray(tree, 0, BTree.size(tree), target, targetOffset);
    }

    public static int toArray(Object[] tree, int treeStart, int treeEnd, Object[] target, int targetOffset) {
        if (BTree.isLeaf(tree)) {
            int count = treeEnd - treeStart;
            System.arraycopy(tree, treeStart, target, targetOffset, count);
            return count;
        }
        int newTargetOffset = targetOffset;
        int childCount = BTree.getChildCount(tree);
        int childOffset = BTree.getChildStart(tree);
        for (int i = 0; i < childCount; ++i) {
            int childStart = BTree.treeIndexOffsetOfChild(tree, i);
            int childEnd = BTree.treeIndexOfBranchKey(tree, i);
            if (childStart > treeEnd || childEnd < treeStart) continue;
            newTargetOffset += BTree.toArray((Object[])tree[childOffset + i], Math.max(0, treeStart - childStart), Math.min(childEnd, treeEnd) - childStart, target, newTargetOffset);
            if (treeStart > childEnd || treeEnd <= childEnd) continue;
            target[newTargetOffset++] = tree[i];
        }
        return newTargetOffset - targetOffset;
    }

    private static <I, O> Object[] transformAndFilterLeaf(Object[] leaf, Function<? super I, ? extends O> apply) {
        O out;
        Object in;
        int i = 0;
        int sz = BTree.sizeOfLeaf(leaf);
        while ((in = leaf[i]) == (out = apply.apply(in)) && ++i < sz) {
        }
        int identicalUntil = i++;
        if (out == null && i < sz) {
            while (null == (out = apply.apply(in = leaf[i])) && ++i < sz) {
            }
        }
        if (i == sz) {
            if (identicalUntil == sz) {
                return leaf;
            }
            if (identicalUntil == 0) {
                return BTree.empty();
            }
            Object[] copy = new Object[identicalUntil | 1];
            System.arraycopy(leaf, 0, copy, 0, identicalUntil);
            return copy;
        }
        try (FastBuilder builder = BTree.fastBuilder();){
            if (identicalUntil > 0) {
                builder.leaf().copyNoOverflow(leaf, 0, identicalUntil);
            }
            builder.leaf().addKeyNoOverflow(out);
            while (++i < sz) {
                in = leaf[i];
                out = apply.apply(in);
                if (out == null) continue;
                builder.leaf().addKeyNoOverflow(out);
            }
            Object[] objectArray = builder.build();
            return objectArray;
        }
    }

    public static <I, I2, O> Object[] transformAndFilter(Object[] tree, BiFunction<? super I, ? super I2, ? extends O> apply, I2 param) {
        if (BTree.isEmpty(tree)) {
            return tree;
        }
        if (BTree.isLeaf(tree)) {
            return BTree.transformAndFilterLeaf(tree, apply, param);
        }
        try (BiTransformer<Object[], I2, O> transformer = BiTransformer.get(apply, param);){
            O o = transformer.apply(tree);
            return o;
        }
    }

    public static <I, O> Object[] transformAndFilter(Object[] tree, Function<? super I, ? extends O> apply) {
        if (BTree.isEmpty(tree)) {
            return tree;
        }
        if (BTree.isLeaf(tree)) {
            return BTree.transformAndFilterLeaf(tree, apply);
        }
        try (Transformer<Object[], O> transformer = Transformer.get(apply);){
            O o = transformer.apply(tree);
            return o;
        }
    }

    private static <I, I2, O> Object[] transformAndFilterLeaf(Object[] leaf, BiFunction<? super I, ? super I2, ? extends O> apply, I2 param) {
        O out;
        Object in;
        int i = 0;
        int sz = BTree.sizeOfLeaf(leaf);
        while ((in = leaf[i]) == (out = apply.apply(in, param)) && ++i < sz) {
        }
        int identicalUntil = i++;
        if (out == null && i < sz) {
            while (null == (out = apply.apply(in = leaf[i], param)) && ++i < sz) {
            }
        }
        if (i == sz) {
            if (identicalUntil == sz) {
                return leaf;
            }
            if (identicalUntil == 0) {
                return BTree.empty();
            }
            Object[] copy = new Object[identicalUntil | 1];
            System.arraycopy(leaf, 0, copy, 0, identicalUntil);
            return copy;
        }
        try (FastBuilder builder = BTree.fastBuilder();){
            if (identicalUntil > 0) {
                builder.leaf().copyNoOverflow(leaf, 0, identicalUntil);
            }
            builder.leaf().addKeyNoOverflow(out);
            while (++i < sz) {
                in = leaf[i];
                out = apply.apply(in, param);
                if (out == null) continue;
                builder.leaf().addKeyNoOverflow(out);
            }
            Object[] objectArray = builder.build();
            return objectArray;
        }
    }

    public static <I, O> Object[] transform(Object[] tree, Function<? super I, ? extends O> function) {
        if (BTree.isEmpty(tree)) {
            return tree;
        }
        if (BTree.isLeaf(tree)) {
            return BTree.transformLeaf(tree, function);
        }
        Object[] result = tree;
        int keyCount = BTree.shallowSizeOfBranch(tree);
        for (int i = 0; i < keyCount; ++i) {
            Object[] curChild = (Object[])tree[keyCount + i];
            Object[] updChild = BTree.transform(curChild, function);
            Object curKey = tree[i];
            O updKey = function.apply(curKey);
            if (result == tree) {
                if (curChild == updChild && curKey == updKey) continue;
                result = BTree.transformCopyBranchHelper(tree, keyCount, i, i);
            }
            result[keyCount + i] = updChild;
            result[i] = updKey;
        }
        Object[] curChild = (Object[])tree[2 * keyCount];
        Object[] updChild = BTree.transform(curChild, function);
        if (result == tree) {
            if (curChild == updChild) {
                return tree;
            }
            result = BTree.transformCopyBranchHelper(tree, keyCount, keyCount, keyCount);
        }
        result[2 * keyCount] = updChild;
        result[2 * keyCount + 1] = tree[2 * keyCount + 1];
        return result;
    }

    private static Object[] transformCopyBranchHelper(Object[] branch, int keyCount, int copyKeyCount, int copyChildCount) {
        Object[] result = new Object[branch.length];
        System.arraycopy(branch, 0, result, 0, copyKeyCount);
        System.arraycopy(branch, keyCount, result, keyCount, copyChildCount);
        return result;
    }

    private static <I, O> Object[] transformLeaf(Object[] leaf, Function<? super I, ? extends O> apply) {
        Object[] result = leaf;
        int size = BTree.sizeOfLeaf(leaf);
        for (int i = 0; i < size; ++i) {
            Object current = leaf[i];
            O updated = apply.apply(current);
            if (result == leaf) {
                if (current == updated) continue;
                result = new Object[leaf.length];
                System.arraycopy(leaf, 0, result, 0, i);
            }
            result[i] = updated;
        }
        return result;
    }

    public static boolean equals(Object[] a, Object[] b) {
        return BTree.size(a) == BTree.size(b) && Iterators.elementsEqual(BTree.iterator(a), BTree.iterator(b));
    }

    public static int hashCode(Object[] btree) {
        int result = 1;
        for (Object v : BTree.iterable(btree)) {
            result = 31 * result + Objects.hashCode(v);
        }
        return result;
    }

    public static String toString(Object[] btree) {
        return BTree.appendBranchOrLeaf(new StringBuilder().append('['), btree).append(']').toString();
    }

    private static StringBuilder appendBranchOrLeaf(StringBuilder builder, Object[] node) {
        return BTree.isLeaf(node) ? BTree.appendLeaf(builder, node) : BTree.appendBranch(builder, node);
    }

    private static StringBuilder appendBranch(StringBuilder builder, Object[] branch) {
        int i;
        int childCount = branch.length / 2;
        int keyCount = childCount - 1;
        for (i = 0; i < keyCount; ++i) {
            if (i != 0) {
                builder.append(", ");
            }
            builder.append(branch[i]);
        }
        int m = branch.length - 1;
        for (i = keyCount; i < m; ++i) {
            builder.append(", ");
            BTree.appendBranchOrLeaf(builder, (Object[])branch[i]);
        }
        builder.append(", ").append(Arrays.toString((int[])branch[branch.length - 1]));
        return builder;
    }

    private static StringBuilder appendLeaf(StringBuilder builder, Object[] leaf) {
        return builder.append(Arrays.toString(leaf));
    }

    public static int treeIndexOfKey(Object[] root, int keyIndex) {
        int[] sizeMap;
        if (BTree.isLeaf(root)) {
            return keyIndex;
        }
        if (keyIndex >= 0 & keyIndex < (sizeMap = BTree.getSizeMap(root)).length) {
            return sizeMap[keyIndex];
        }
        if (keyIndex < 0) {
            return -1;
        }
        return sizeMap[keyIndex - 1] + 1;
    }

    public static int treeIndexOfLeafKey(int keyIndex) {
        return keyIndex;
    }

    public static int treeIndexOfBranchKey(Object[] root, int keyIndex) {
        return BTree.lookupSizeMap(root, keyIndex);
    }

    public static int treeIndexOffsetOfChild(Object[] root, int childIndex) {
        if (childIndex == 0) {
            return 0;
        }
        return 1 + BTree.lookupSizeMap(root, childIndex - 1);
    }

    public static <V> Builder<V> builder(Comparator<? super V> comparator) {
        return new Builder<V>(comparator);
    }

    public static <V> Builder<V> builder(Comparator<? super V> comparator, int initialCapacity) {
        return new Builder<V>(comparator, initialCapacity);
    }

    private static <V, A> void applyValue(V value, BiConsumer<A, V> function, A argument) {
        function.accept(argument, value);
    }

    public static <V, A> void applyLeaf(Object[] btree, BiConsumer<A, V> function, A argument) {
        Preconditions.checkArgument((boolean)BTree.isLeaf(btree));
        int limit = BTree.getLeafKeyEnd(btree);
        for (int i = 0; i < limit; ++i) {
            BTree.applyValue(btree[i], function, argument);
        }
    }

    public static <V, A> void apply(Object[] btree, BiConsumer<A, V> function, A argument) {
        if (BTree.isLeaf(btree)) {
            BTree.applyLeaf(btree, function, argument);
            return;
        }
        int childOffset = BTree.getChildStart(btree);
        int limit = btree.length - 1 - childOffset;
        for (int i = 0; i < limit; ++i) {
            BTree.apply((Object[])btree[childOffset + i], function, argument);
            if (i >= childOffset) continue;
            BTree.applyValue(btree[i], function, argument);
        }
    }

    public static <V> void apply(Object[] btree, Consumer<V> function) {
        BTree.apply(btree, Consumer::accept, function);
    }

    private static <V> int find(Object[] btree, V from, Comparator<V> comparator) {
        Preconditions.checkNotNull(comparator);
        int keyEnd = BTree.getKeyEnd(btree);
        return Arrays.binarySearch(btree, 0, keyEnd, from, comparator);
    }

    private static boolean isStopSentinel(long v) {
        return v == Long.MAX_VALUE;
    }

    private static <V, A> long accumulateLeaf(Object[] btree, BiLongAccumulator<A, V> accumulator, A arg, Comparator<V> comparator, V from, long initialValue) {
        int i;
        Preconditions.checkArgument((boolean)BTree.isLeaf(btree));
        long value = initialValue;
        int limit = BTree.getLeafKeyEnd(btree);
        int startIdx = 0;
        if (from != null) {
            i = BTree.find(btree, from, comparator);
            boolean isExact = i >= 0;
            startIdx = isExact ? i : -1 - i;
        }
        for (i = startIdx; i < limit && !BTree.isStopSentinel(value = accumulator.apply(arg, btree[i], value)); ++i) {
        }
        return value;
    }

    public static <V, A> long accumulate(Object[] btree, BiLongAccumulator<A, V> accumulator, A arg, Comparator<V> comparator, V from, long initialValue) {
        if (BTree.isLeaf(btree)) {
            return BTree.accumulateLeaf(btree, accumulator, arg, comparator, from, initialValue);
        }
        long value = initialValue;
        int childOffset = BTree.getChildStart(btree);
        int startChild = 0;
        if (from != null) {
            int i = BTree.find(btree, from, comparator);
            boolean isExact = i >= 0;
            int n = startChild = isExact ? i + 1 : -1 - i;
            if (isExact) {
                if (BTree.isStopSentinel(value = accumulator.apply(arg, btree[i], value))) {
                    return value;
                }
                from = null;
            }
        }
        int limit = btree.length - 1 - childOffset;
        for (int i = startChild; !(i >= limit || BTree.isStopSentinel(value = BTree.accumulate((Object[])btree[childOffset + i], accumulator, arg, comparator, from, value)) || i < childOffset && BTree.isStopSentinel(value = accumulator.apply(arg, btree[i], value))); ++i) {
            if (from == null) continue;
            from = null;
        }
        return value;
    }

    public static <V> long accumulate(Object[] btree, LongAccumulator<V> accumulator, Comparator<V> comparator, V from, long initialValue) {
        return BTree.accumulate(btree, LongAccumulator::apply, accumulator, comparator, from, initialValue);
    }

    public static <V> long accumulate(Object[] btree, LongAccumulator<V> accumulator, long initialValue) {
        return BTree.accumulate(btree, accumulator, null, null, initialValue);
    }

    public static <V, A> long accumulate(Object[] btree, BiLongAccumulator<A, V> accumulator, A arg, long initialValue) {
        return BTree.accumulate(btree, accumulator, arg, null, null, initialValue);
    }

    private static int minHeight(int size) {
        return BTree.heightAtSize2n(size, BRANCH_SHIFT);
    }

    private static int heightAtSize2n(int size, int branchShift) {
        int lengthInBinary = 64 - Long.numberOfLeadingZeros(size);
        return (branchShift - 1 + lengthInBinary) / branchShift;
    }

    private static int[][] buildBalancedSizeMaps(int branchShift) {
        int count = 32 / branchShift - 1;
        int childCount = 1 << branchShift;
        int[][] sizeMaps = new int[count][childCount];
        for (int height = 0; height < count; ++height) {
            int childSize = BTree.treeSize2n(height + 1, branchShift);
            int size = 0;
            int[] sizeMap = sizeMaps[height];
            for (int i = 0; i < childCount; ++i) {
                size += childSize;
                sizeMap[i] = size++;
            }
        }
        return sizeMaps;
    }

    private static void reverse(Object[] array, int from, int to) {
        int mid = (from + to) / 2;
        for (int i = from; i < mid; ++i) {
            int j = to - (1 + i - from);
            Object tmp = array[i];
            array[i] = array[j];
            array[j] = tmp;
        }
    }

    private static void reverse(int[] array, int from, int to) {
        int mid = (from + to) / 2;
        for (int i = from; i < mid; ++i) {
            int j = to - (1 + i - from);
            int tmp = array[i];
            array[i] = array[j];
            array[j] = tmp;
        }
    }

    private static int sizesToSizeMap(int[] sizeMap) {
        int total = sizeMap[0];
        for (int i = 1; i < sizeMap.length; ++i) {
            sizeMap[i] = total += 1 + sizeMap[i];
        }
        return total;
    }

    private static int sizesToSizeMap(int[] sizes, int count) {
        int total = sizes[0];
        for (int i = 1; i < count; ++i) {
            sizes[i] = total += 1 + sizes[i];
        }
        return total;
    }

    private static void sizeMapToSizes(int[] sizeMap) {
        for (int i = sizeMap.length; i > 1; --i) {
            int n = i;
            sizeMap[n] = sizeMap[n] - (1 + sizeMap[i - 1]);
        }
    }

    private static <Compare> int compareWithMaybeInfinity(Comparator<? super Compare> comparator, Compare key, Compare ub) {
        if (ub == null) {
            return -1;
        }
        return comparator.compare(key, ub);
    }

    static <Compare> int exponentialSearchForMaybeInfinity(Comparator<? super Compare> comparator, Object[] in, int from, int to, Compare find) {
        if (find == null) {
            return -(1 + to);
        }
        return BTree.exponentialSearch(comparator, in, from, to, find);
    }

    private static <Compare> int exponentialSearch(Comparator<? super Compare> comparator, Object[] in, int from, int to, Compare find) {
        int step = 0;
        while (from + step < to) {
            int i = from + step;
            int c = comparator.compare(find, in[i]);
            if (c < 0) {
                to = i;
                break;
            }
            if (c == 0) {
                return i;
            }
            from = i + 1;
            step = step * 2 + 1;
        }
        return Arrays.binarySearch(in, from, to, find, comparator);
    }

    static <Compare> int exponentialSearchWithUpperBound(Comparator<? super Compare> comparator, Object[] in, int from, int to, Compare ub, Compare find) {
        int step = 0;
        while (true) {
            int c;
            int i;
            if ((i = from + step) >= to) {
                c = BTree.compareWithMaybeInfinity(comparator, find, ub);
                if (c < 0) break;
                return -(2 + to);
            }
            c = comparator.compare(find, in[i]);
            if (c < 0) {
                to = i;
                break;
            }
            if (c == 0) {
                return i;
            }
            from = i + 1;
            step = step * 2 + 1;
        }
        return Arrays.binarySearch(in, from, to, find, comparator);
    }

    private static long[] sizeOnHeapOfPerfectTrees(int branchShift) {
        long[] result = new long[BTree.heightAtSize2n(Integer.MAX_VALUE, branchShift)];
        int branchFactor = 1 << branchShift;
        result[0] = branchFactor - 1;
        for (int i = 1; i < result.length; ++i) {
            result[i] = BTree.sizeOnHeapOfPerfectTree(i + 1, branchFactor);
        }
        return result;
    }

    private static long sizeOnHeapOfPerfectTree(int height, int branchShift) {
        int branchFactor = 1 << branchShift;
        long branchSize = ObjectSizes.sizeOfReferenceArray(branchFactor * 2);
        int branchCount = height == 2 ? 1 : 2 + BTree.treeSize2n(height - 2, branchShift);
        long leafSize = ObjectSizes.sizeOfReferenceArray(branchFactor - 1 | 1);
        int leafCount = 1 + BTree.treeSize2n(height - 1, branchShift);
        return branchSize * (long)branchCount + leafSize * (long)leafCount;
    }

    public static int height(Object[] tree) {
        if (BTree.isLeaf(tree)) {
            return 1;
        }
        int height = 1;
        while (!BTree.isLeaf(tree)) {
            ++height;
            tree = (Object[])tree[BTree.shallowSizeOfBranch(tree)];
        }
        return height;
    }

    private static int denseSize(int height) {
        return BTree.treeSize2n(height, BRANCH_SHIFT);
    }

    private static int checkedDenseSize(int height) {
        assert (height * BRANCH_SHIFT < 32);
        return BTree.denseSize(height);
    }

    private static int treeSize2n(int height, int branchShift) {
        return (1 << branchShift * height) - 1;
    }

    private static int maxRootHeight(int size) {
        if (size <= BRANCH_FACTOR) {
            return 1;
        }
        return 1 + BTree.heightAtSize2n((size - 1) / 2, BRANCH_SHIFT - 1);
    }

    private static int sizeOfBranch(Object[] branch) {
        int length = branch.length;
        return ((int[])branch[length - 1])[length / 2 - 1];
    }

    private static boolean isSimple(UpdateFunction<?, ?> updateF) {
        return updateF instanceof UpdateFunction.Simple;
    }

    static int[] sizeMap(Object[] branch) {
        return (int[])branch[branch.length - 1];
    }

    public static long sizeOnHeapOf(Object[] tree) {
        if (BTree.isEmpty(tree)) {
            return 0L;
        }
        long size = ObjectSizes.sizeOfArray(tree);
        if (BTree.isLeaf(tree)) {
            return size;
        }
        for (int i = BTree.childOffset(tree); i < BTree.childEndOffset(tree); ++i) {
            size += BTree.sizeOnHeapOf((Object[])tree[i]);
        }
        return size += ObjectSizes.sizeOfArray(BTree.sizeMap(tree));
    }

    private static long sizeOnHeapOfLeaf(Object[] tree) {
        if (BTree.isEmpty(tree)) {
            return 0L;
        }
        return ObjectSizes.sizeOfArray(tree);
    }

    private static <V> int compareWellFormed(Comparator<V> cmp, Object a, Object b) {
        if (a == b) {
            return 0;
        }
        if (a == NEGATIVE_INFINITY | b == POSITIVE_INFINITY) {
            return -1;
        }
        if (b == NEGATIVE_INFINITY | a == POSITIVE_INFINITY) {
            return 1;
        }
        return cmp.compare(a, b);
    }

    public static boolean isWellFormed(Object[] btree, Comparator<?> cmp) {
        return BTree.isWellFormedReturnHeight(cmp, btree, true, NEGATIVE_INFINITY, POSITIVE_INFINITY) >= 0;
    }

    private static int isWellFormedReturnHeight(Comparator<?> cmp, Object[] node, boolean isRoot, Object min, Object max) {
        if (BTree.isEmpty(node)) {
            return 0;
        }
        if (cmp != null && !BTree.isNodeWellFormed(cmp, node, min, max)) {
            return -1;
        }
        int keyCount = BTree.shallowSize(node);
        if (keyCount < 1) {
            return -1;
        }
        if (!isRoot && keyCount < BRANCH_FACTOR / 2 - 1) {
            return -1;
        }
        if (keyCount >= BRANCH_FACTOR) {
            return -1;
        }
        if (BTree.isLeaf(node)) {
            return 0;
        }
        int[] sizeMap = BTree.sizeMap(node);
        int size = 0;
        int childHeight = -1;
        for (int i = BTree.childOffset(node); i < BTree.childEndOffset(node); ++i) {
            Object[] child = (Object[])node[i];
            Object localmax = i < node.length - 2 ? node[i - BTree.childOffset(node)] : max;
            int height = BTree.isWellFormedReturnHeight(cmp, child, false, min, localmax);
            if (height == -1) {
                return -1;
            }
            if (childHeight == -1) {
                childHeight = height;
            }
            if (childHeight != height) {
                return -1;
            }
            min = localmax;
            if (sizeMap[i - BTree.childOffset(node)] != (size += BTree.size(child))) {
                return -1;
            }
            ++size;
        }
        return childHeight + 1;
    }

    private static boolean isNodeWellFormed(Comparator<?> cmp, Object[] node, Object min, Object max) {
        Object previous = min;
        int end = BTree.shallowSize(node);
        for (int i = 0; i < end; ++i) {
            Object current = node[i];
            if (BTree.compareWellFormed(cmp, previous, current) >= 0) {
                return false;
            }
            previous = current;
        }
        return BTree.compareWellFormed(cmp, previous, max) < 0;
    }

    public static <V> FastBuilder<V> fastBuilder() {
        TinyThreadLocalPool.TinyPool pool = (TinyThreadLocalPool.TinyPool)FastBuilder.POOL.get();
        FastBuilder builder = (FastBuilder)pool.poll();
        if (builder == null) {
            builder = new FastBuilder();
        }
        builder.pool = pool;
        return builder;
    }

    static int searchResultToComparison(int searchResult) {
        return searchResult >> 31;
    }

    private static class SimpleTreeKeysIterator<Compare, Insert extends Compare> {
        int leafSize;
        int leafPos;
        Object[] leaf;
        Object[][] nodes;
        int[] positions;
        int depth;

        private SimpleTreeKeysIterator() {
        }

        void init(Object[] tree) {
            int maxHeight = BTree.maxRootHeight(BTree.size(tree));
            if (this.positions == null || maxHeight >= this.positions.length) {
                this.positions = new int[maxHeight + 1];
                this.nodes = new Object[maxHeight + 1][];
            }
            this.depth = 0;
            this.descendToLeaf(tree);
        }

        void reset() {
            this.leaf = null;
            Arrays.fill((Object[])this.nodes, 0, this.nodes.length, null);
        }

        Insert next() {
            if (this.leafPos < this.leafSize) {
                return (Insert)this.leaf[this.leafPos++];
            }
            if (this.depth == 0) {
                return null;
            }
            Object[] node = this.nodes[this.depth - 1];
            int position = this.positions[this.depth - 1];
            Object result = node[position];
            this.advanceBranch(node, position + 1);
            return (Insert)result;
        }

        private void advanceBranch(Object[] node, int position) {
            int count = BTree.shallowSizeOfBranch(node);
            if (position < count) {
                this.positions[this.depth - 1] = position;
            } else {
                --this.depth;
            }
            this.descendToLeaf((Object[])node[count + position]);
        }

        void descendToLeaf(Object[] node) {
            while (!BTree.isLeaf(node)) {
                this.nodes[this.depth] = node;
                this.positions[this.depth] = 0;
                node = (Object[])node[BTree.shallowSizeOfBranch(node)];
                ++this.depth;
            }
            this.leaf = node;
            this.leafPos = 0;
            this.leafSize = BTree.sizeOfLeaf(node);
        }

        <Update> int copyKeysSmallerThan(Compare bound, Comparator<? super Compare> comparator, LeafBuilder builder, UpdateFunction<Insert, Update> transformer) {
            while (true) {
                int lim;
                int end;
                int n = end = (lim = BTree.exponentialSearchForMaybeInfinity(comparator, this.leaf, this.leafPos, this.leafSize, bound)) >= 0 ? lim : -1 - lim;
                if (end > this.leafPos) {
                    builder.copy(this.leaf, this.leafPos, end - this.leafPos, transformer);
                    this.leafPos = end;
                }
                if (end < this.leafSize) {
                    return BTree.searchResultToComparison(lim);
                }
                if (this.depth == 0) {
                    return -1;
                }
                Object[] node = this.nodes[this.depth - 1];
                int position = this.positions[this.depth - 1];
                Object branchKey = node[position];
                int cmp = BTree.compareWithMaybeInfinity(comparator, branchKey, bound);
                if (cmp >= 0) {
                    return -cmp;
                }
                builder.addKey(BTree.isSimple(transformer) ? branchKey : transformer.insert(branchKey));
                this.advanceBranch(node, position + 1);
            }
        }
    }

    private static class SimpleTreeIterator
    extends SimpleTreeStack {
        private SimpleTreeIterator() {
        }

        int init(Object[] tree) {
            int maxHeight = BTree.maxRootHeight(BTree.size(tree));
            if (this.positions == null || maxHeight >= this.positions.length) {
                this.positions = new int[maxHeight + 1];
                this.nodes = new Object[maxHeight + 1][];
            }
            this.nodes[0] = tree;
            if (BTree.isEmpty(tree)) {
                this.leafDepth = 0;
                this.depth = -1;
            } else {
                this.depth = 0;
                this.positions[0] = 0;
                while (!BTree.isLeaf(tree)) {
                    tree = (Object[])tree[BTree.shallowSizeOfBranch(tree)];
                    this.nodes[++this.depth] = tree;
                    this.positions[this.depth] = 0;
                }
                this.leafDepth = this.depth;
            }
            return this.leafDepth + 1;
        }

        void descendIntoNextLeaf(Object[] node, int pos, int sz) {
            this.positions[this.depth] = ++pos;
            ++this.depth;
            node = (Object[])node[sz + pos];
            this.nodes[this.depth] = node;
            this.positions[this.depth] = 0;
            while (this.depth < this.leafDepth) {
                ++this.depth;
                node = (Object[])node[BTree.shallowSizeOfBranch(node)];
                this.nodes[this.depth] = node;
                this.positions[this.depth] = 0;
            }
        }

        boolean ascendToParent() {
            if (this.depth < 0) {
                return false;
            }
            return --this.depth >= 0;
        }
    }

    private static abstract class SimpleTreeStack {
        Object[][] nodes;
        int[] positions;
        int depth;
        int leafDepth;

        private SimpleTreeStack() {
        }

        void reset() {
            Arrays.fill((Object[])this.nodes, 0, this.leafDepth + 1, null);
        }

        Object[] node() {
            return this.nodes[this.depth];
        }

        int position() {
            return this.positions[this.depth];
        }
    }

    private static class BiTransformer<I, I2, O>
    extends AbstractTransformer<I, O> {
        static final TinyThreadLocalPool<BiTransformer> POOL = new TinyThreadLocalPool();
        BiFunction<? super I, ? super I2, ? extends O> apply;
        I2 i2;
        TinyThreadLocalPool.TinyPool<BiTransformer> pool;

        private BiTransformer() {
        }

        @Override
        O apply(I i1) {
            return this.apply.apply(i1, this.i2);
        }

        static <I, I2, O> BiTransformer<I, I2, O> get(BiFunction<? super I, ? super I2, ? extends O> apply, I2 i2) {
            TinyThreadLocalPool.TinyPool pool = (TinyThreadLocalPool.TinyPool)POOL.get();
            BiTransformer<I, I2, O> transformer = (BiTransformer<I, I2, O>)pool.poll();
            if (transformer == null) {
                transformer = new BiTransformer<I, I2, O>();
            }
            transformer.pool = pool;
            transformer.apply = apply;
            transformer.i2 = i2;
            return transformer;
        }

        @Override
        public void close() {
            this.apply = null;
            this.i2 = null;
            this.reset();
            this.pool.offer(this);
            this.pool = null;
        }
    }

    private static class Transformer<I, O>
    extends AbstractTransformer<I, O> {
        static final TinyThreadLocalPool<Transformer> POOL = new TinyThreadLocalPool();
        TinyThreadLocalPool.TinyPool<Transformer> pool;
        Function<? super I, ? extends O> apply;

        private Transformer() {
        }

        @Override
        O apply(I v) {
            return this.apply.apply(v);
        }

        static <I, O> Transformer<I, O> get(Function<? super I, ? extends O> apply) {
            TinyThreadLocalPool.TinyPool pool = (TinyThreadLocalPool.TinyPool)POOL.get();
            Transformer<I, O> transformer = (Transformer<I, O>)pool.poll();
            if (transformer == null) {
                transformer = new Transformer<I, O>();
            }
            transformer.pool = pool;
            transformer.apply = apply;
            return transformer;
        }

        @Override
        public void close() {
            this.apply = null;
            this.reset();
            this.pool.offer(this);
            this.pool = null;
        }
    }

    private static abstract class AbstractTransformer<I, O>
    extends AbstractUpdater
    implements AutoCloseable {
        final SimpleTreeIterator update = new SimpleTreeIterator();
        Object[][] queuedToFinish = new Object[1][];

        AbstractTransformer() {
            this.allocated = -1L;
            this.ensureParent();
            this.parent.inUse = false;
        }

        abstract O apply(I var1);

        Object[] apply(Object[] update) {
            int height = this.update.init(update);
            if (this.queuedToFinish.length < height - 1) {
                this.queuedToFinish = new Object[height - 1][];
            }
            return this.apply();
        }

        private Object[] apply() {
            Object[] unode = this.update.node();
            int upos = this.update.position();
            int usz = BTree.sizeOfLeaf(unode);
            while (true) {
                Object[] originalNode;
                Object nextKey;
                boolean propagatedOriginalLeaf = false;
                if (this.leaf().count == 0) {
                    if (upos == 0) {
                        O out;
                        Object in;
                        while ((in = unode[upos]) == (out = this.apply(in)) && ++upos < usz) {
                        }
                        propagatedOriginalLeaf = upos == usz;
                        if (propagatedOriginalLeaf) {
                            AbstractTransformer.markUsed(this.parent).addChild(unode, usz);
                        } else {
                            this.leaf().copyNoOverflow(unode, 0, upos++);
                            if (out != null) {
                                this.leaf().addKeyNoOverflow(out);
                            }
                        }
                    }
                    if (!propagatedOriginalLeaf) {
                        this.transformLeafNoOverflow(unode, upos, usz);
                    }
                } else {
                    this.transformLeaf(unode, upos, usz);
                }
                int finishToHeight = 0;
                do {
                    this.update.ascendToParent();
                    BranchBuilder level = this.parent;
                    unode = this.update.node();
                    upos = this.update.position();
                    usz = BTree.shallowSizeOfBranch(unode);
                    while (upos == usz) {
                        this.queuedToFinish[level.height - 2] = unode;
                        finishToHeight = Math.max(finishToHeight, level.height);
                        if (!this.update.ascendToParent()) {
                            return this.finishAndDrain(propagatedOriginalLeaf);
                        }
                        level = level.ensureParent();
                        unode = this.update.node();
                        upos = this.update.position();
                        usz = BTree.shallowSizeOfBranch(unode);
                    }
                    nextKey = this.apply(unode[upos]);
                    if (nextKey == null && this.leaf().count > MIN_KEYS) {
                        nextKey = this.leaf().buffer[--this.leaf().count];
                    }
                    this.update.descendIntoNextLeaf(unode, upos, usz);
                    unode = this.update.node();
                    upos = this.update.position();
                    usz = BTree.sizeOfLeaf(unode);
                    while (nextKey == null && upos < usz) {
                        nextKey = this.apply(unode[upos++]);
                    }
                } while (nextKey == null);
                if (!propagatedOriginalLeaf && !this.finish(this.leaf(), null)) {
                    this.leaf().addKeyNoOverflow(nextKey);
                    continue;
                }
                BranchBuilder finish = this.parent;
                while (finish.height <= finishToHeight && this.finish(finish, originalNode = this.queuedToFinish[finish.height - 2])) {
                    finish = finish.parent;
                }
                finish.addKey(nextKey);
            }
        }

        private void transformLeafNoOverflow(Object[] unode, int upos, int usz) {
            while (upos < usz) {
                O v = this.apply(unode[upos++]);
                this.leaf().maybeAddKeyNoOverflow(v);
            }
        }

        private void transformLeaf(Object[] unode, int upos, int usz) {
            while (upos < usz) {
                O v = this.apply(unode[upos++]);
                this.leaf().maybeAddKey(v);
            }
        }

        private boolean finish(LeafOrBranchBuilder level, Object[] unode) {
            if (!level.isSufficient()) {
                return false;
            }
            level.drainAndPropagate(unode, level.ensureParent());
            return true;
        }

        private Object[] finishAndDrain(boolean skipLeaf) {
            LeafOrBranchBuilder level = this.leaf();
            if (skipLeaf && (level = this.nonEmptyParentMaybeSteal(level)) == null) {
                return (Object[])this.leaf().parent.buffer[MAX_KEYS];
            }
            while (true) {
                BranchBuilder parent;
                if ((parent = this.nonEmptyParentMaybeSteal(level)) != null && !level.isSufficient()) {
                    Object[] result = this.stealAndMaybeRepropagate(level, parent);
                    if (result != null) {
                        return result;
                    }
                } else {
                    Object[] originalNode = level == this.leaf() ? null : this.queuedToFinish[level.height - 2];
                    Object[] result = level.drainAndPropagate(originalNode, parent);
                    if (parent == null) {
                        return result;
                    }
                }
                level = parent;
            }
        }

        BranchBuilder nonEmptyParentMaybeSteal(LeafOrBranchBuilder level) {
            if (level.hasOverflow()) {
                return level.ensureParent();
            }
            BranchBuilder parent = level.parent;
            if (parent == null || !parent.inUse || parent.isEmpty() && !this.tryPrependFromParent(parent)) {
                return null;
            }
            return parent;
        }

        private Object[] stealAndMaybeRepropagate(LeafOrBranchBuilder fill, BranchBuilder parent) {
            boolean exhausted;
            this.prependFromParent(fill, parent);
            boolean bl = exhausted = !fill.hasOverflow() && parent.isEmpty() && !this.tryPrependFromParent(parent);
            if (exhausted) {
                return fill.drain();
            }
            fill.drainAndPropagate(null, parent);
            return null;
        }

        private boolean tryPrependFromParent(BranchBuilder parent) {
            BranchBuilder grandparent = this.nonEmptyParentMaybeSteal(parent);
            if (grandparent == null) {
                return false;
            }
            this.prependFromParent(parent, grandparent);
            return true;
        }

        private void prependFromParent(LeafOrBranchBuilder fill, BranchBuilder parent) {
            Object[] predecessor;
            Object predecessorNextKey;
            assert (!parent.isEmpty());
            if (parent.count == 0 && parent.hasOverflow()) {
                predecessorNextKey = parent.savedNextKey;
                predecessor = (Object[])parent.savedBuffer[2 * MAX_KEYS];
                Object[] tmpBuffer = parent.savedBuffer;
                int[] tmpSizes = parent.savedSizes;
                parent.savedBuffer = parent.buffer;
                parent.savedSizes = parent.sizes;
                parent.buffer = tmpBuffer;
                parent.sizes = tmpSizes;
                parent.savedNextKey = null;
                parent.count = MAX_KEYS;
            } else {
                --parent.count;
                predecessor = (Object[])parent.buffer[MAX_KEYS + parent.count];
                predecessorNextKey = parent.buffer[parent.count];
            }
            fill.prepend(predecessor, predecessorNextKey);
        }

        @Override
        void reset() {
            Arrays.fill((Object[])this.queuedToFinish, 0, this.update.leafDepth, null);
            this.update.reset();
            super.reset();
        }
    }

    private static class Updater<Compare, Existing extends Compare, Insert extends Compare>
    extends AbstractUpdater
    implements AutoCloseable {
        static final TinyThreadLocalPool<Updater> POOL = new TinyThreadLocalPool();
        TinyThreadLocalPool.TinyPool<Updater> pool;
        final SimpleTreeKeysIterator<Compare, Insert> insert = new SimpleTreeKeysIterator();
        Comparator<? super Compare> comparator;
        UpdateFunction<Insert, Existing> updateF;

        private Updater() {
        }

        static <Compare, Existing extends Compare, Insert extends Compare> Updater<Compare, Existing, Insert> get() {
            TinyThreadLocalPool.TinyPool pool = (TinyThreadLocalPool.TinyPool)POOL.get();
            Updater<Compare, Existing, Insert> updater = (Updater<Compare, Existing, Insert>)pool.poll();
            if (updater == null) {
                updater = new Updater<Compare, Existing, Insert>();
            }
            updater.pool = pool;
            return updater;
        }

        Object[] update(Object[] update, Object[] insert, Comparator<? super Compare> comparator, UpdateFunction<Insert, Existing> updateF) {
            this.insert.init(insert);
            this.updateF = updateF;
            this.comparator = comparator;
            this.allocated = BTree.isSimple(updateF) ? -1L : 0L;
            int leafDepth = BTree.depth(update) - 1;
            LeafOrBranchBuilder builder = this.leaf();
            for (int i = 0; i < leafDepth; ++i) {
                builder = builder.ensureParent();
            }
            Insert ik = this.insert.next();
            ik = this.updateRecursive(ik, update, null, builder);
            assert (ik == null);
            Object[] result = builder.completeBuild();
            if (this.allocated > 0L) {
                updateF.onAllocatedOnHeap(this.allocated);
            }
            return result;
        }

        private Insert updateRecursive(Insert ik, Object[] unode, Existing uub, LeafOrBranchBuilder builder) {
            return builder == this.leaf() ? this.updateRecursive(ik, unode, uub, (LeafBuilder)builder) : this.updateRecursive(ik, unode, uub, (BranchBuilder)builder);
        }

        private Insert updateRecursive(Insert ik, Object[] unode, Existing uub, BranchBuilder builder) {
            int upos = 0;
            int usz = BTree.shallowSizeOfBranch(unode);
            while (ik != null) {
                int find = BTree.exponentialSearchWithUpperBound(this.comparator, unode, upos, usz, uub, ik);
                int c = BTree.searchResultToComparison(find);
                if (find < 0) {
                    find = -1 - find;
                }
                if (find > usz) break;
                if (find > upos) {
                    builder.copyPreceding(unode, usz, upos, find - upos);
                }
                Existing nextUKey = find < usz ? unode[find] : uub;
                Object[] childUNode = (Object[])unode[find + usz];
                if (c < 0) {
                    LeafOrBranchBuilder childBuilder = builder.child;
                    ik = this.updateRecursive(ik, childUNode, nextUKey, childBuilder);
                    childBuilder.drainAndPropagate(childUNode, builder);
                    if (find == usz) {
                        return ik;
                    }
                    c = ik != null ? this.comparator.compare(nextUKey, ik) : -1;
                } else {
                    builder.addChild(childUNode);
                }
                if (c == 0) {
                    builder.addKey(this.updateF.merge(nextUKey, ik));
                    ik = this.insert.next();
                } else {
                    builder.addKey(nextUKey);
                }
                upos = find + 1;
            }
            if (upos <= usz) {
                builder.copyPreceding(unode, usz, upos, usz - upos);
                builder.addChild((Object[])unode[usz + usz]);
            }
            return ik;
        }

        private Insert updateRecursive(Insert ik, Object[] unode, Existing uub, LeafBuilder builder) {
            block9: {
                int upos = 0;
                int usz = BTree.sizeOfLeaf(unode);
                Object uk = unode[upos];
                int c = this.comparator.compare(uk, ik);
                while (true) {
                    if (c == 0) {
                        this.leaf().addKey(this.updateF.merge(uk, ik));
                        if (++upos < usz) {
                            uk = unode[upos];
                        }
                        if ((ik = this.insert.next()) == null) {
                            builder.copy(unode, upos, usz - upos);
                            return null;
                        }
                        if (upos != usz) {
                            c = this.comparator.compare(uk, ik);
                            continue;
                        }
                        break block9;
                    }
                    if (c < 0) {
                        int ulim = BTree.exponentialSearch(this.comparator, unode, upos + 1, usz, ik);
                        c = -BTree.searchResultToComparison(ulim);
                        if (ulim < 0) {
                            ulim = -(1 + ulim);
                        }
                        builder.copy(unode, upos, ulim - upos);
                        upos = ulim;
                        if (upos != usz) {
                            uk = unode[upos];
                            continue;
                        }
                        break block9;
                    }
                    builder.addKey(BTree.isSimple(this.updateF) ? ik : this.updateF.insert(ik));
                    c = this.insert.copyKeysSmallerThan((Compare)uk, this.comparator, builder, this.updateF);
                    ik = this.insert.next();
                    if (ik == null) break;
                }
                builder.copy(unode, upos, usz - upos);
                return null;
            }
            if (uub == null || this.comparator.compare(ik, uub) < 0) {
                builder.addKey(BTree.isSimple(this.updateF) ? ik : this.updateF.insert(ik));
                this.insert.copyKeysSmallerThan((Compare)uub, this.comparator, builder, this.updateF);
                ik = this.insert.next();
            }
            return ik;
        }

        @Override
        public void close() {
            this.reset();
            this.pool.offer(this);
            this.pool = null;
        }

        @Override
        void reset() {
            super.reset();
            this.insert.reset();
        }
    }

    private static abstract class AbstractUpdater
    extends AbstractFastBuilder
    implements AutoCloseable {
        private AbstractUpdater() {
        }

        @Override
        void reset() {
            assert (this.leaf().count == 0);
            this.clearLeafBuffer(this.leaf().buffer);
            if (this.leaf().savedBuffer != null) {
                Arrays.fill(this.leaf().savedBuffer, null);
            }
            BranchBuilder branch = this.leaf().parent;
            while (branch != null && branch.inUse) {
                assert (branch.count == 0);
                this.clearBranchBuffer(branch.buffer);
                if (branch.savedBuffer != null && branch.savedBuffer[0] != null) {
                    Arrays.fill(branch.savedBuffer, null);
                }
                branch.inUse = false;
                branch = branch.parent;
            }
        }

        private void clearLeafBuffer(Object[] array) {
            int i;
            if (array[0] == null) {
                return;
            }
            for (i = 1; i < array.length && array[i] != null; ++i) {
            }
            Arrays.fill(array, 0, i, null);
        }

        private void clearBranchBuffer(Object[] array) {
            int i;
            if (array[0] == null) {
                return;
            }
            for (i = 1; i < MAX_KEYS && array[i] != null; ++i) {
            }
            Arrays.fill(array, 0, i, null);
            Arrays.fill(array, MAX_KEYS, MAX_KEYS + i + 1, null);
        }
    }

    public static class FastBuilder<V>
    extends AbstractFastBuilder
    implements AutoCloseable {
        private static final TinyThreadLocalPool<FastBuilder<?>> POOL = new TinyThreadLocalPool();
        private TinyThreadLocalPool.TinyPool<FastBuilder<?>> pool;

        FastBuilder() {
            this.allocated = -1L;
        }

        public void add(V value) {
            this.leaf().addKey(value);
        }

        public void add(Object[] from, int offset, int count) {
            this.leaf().copy(from, offset, count);
        }

        public Object[] build() {
            return this.leaf().completeBuild();
        }

        public Object[] buildReverse() {
            Object[] result = this.build();
            BTree.reverseInSitu(result, BTree.height(result), false);
            return result;
        }

        @Override
        public void close() {
            this.reset();
            this.pool.offer(this);
            this.pool = null;
        }

        @Override
        void reset() {
            Arrays.fill(this.leaf().buffer, 0, this.leaf().count, null);
            this.leaf().count = 0;
            BranchBuilder branch = this.leaf().parent;
            while (branch != null && branch.inUse) {
                Arrays.fill(branch.buffer, 0, branch.count, null);
                Arrays.fill(branch.buffer, MAX_KEYS, MAX_KEYS + 1 + branch.count, null);
                branch.count = 0;
                branch.inUse = false;
                branch = branch.parent;
            }
        }
    }

    private static abstract class AbstractFastBuilder
    extends LeafBuilder {
        private AbstractFastBuilder() {
        }

        @Override
        final boolean producesOnlyDense() {
            return this.getClass() == FastBuilder.class;
        }

        final LeafBuilder leaf() {
            return this;
        }

        abstract void reset();
    }

    static class BranchBuilder
    extends LeafOrBranchBuilder {
        final LeafBuilder leaf;
        int[] sizes;
        int[] savedSizes;
        boolean inUse;

        BranchBuilder(LeafOrBranchBuilder child) {
            super(child);
            this.buffer = new Object[2 * (MAX_KEYS + 1)];
            if (!child.producesOnlyDense()) {
                this.sizes = new int[MAX_KEYS + 1];
            }
            this.leaf = child instanceof LeafBuilder ? (LeafBuilder)child : ((BranchBuilder)child).leaf;
        }

        void addKey(Object key) {
            if (this.count == MAX_KEYS) {
                this.overflow(key);
            } else {
                this.buffer[this.count++] = key;
            }
        }

        void addChild(Object[] child, int sizeOfChild) {
            this.buffer[BTree.MAX_KEYS + this.count] = child;
            this.recordSizeOfChild(sizeOfChild);
        }

        void recordSizeOfChild(int sizeOfChild) {
            if (this.sizes != null) {
                this.sizes[this.count] = sizeOfChild;
            }
        }

        void addChild(Object[] child) {
            this.addChild(child, this.sizes == null ? 0 : BTree.size(child));
        }

        void addChildAndNextKey(Object[] newChild, int newChildSize, Object nextKey) {
            this.buffer[BTree.MAX_KEYS + this.count] = newChild;
            this.recordSizeOfChild(newChildSize);
            this.addKey(nextKey);
        }

        void propagateOverflow() {
            if (this.leaf.allocated >= 0L) {
                this.leaf.allocated += ObjectSizes.sizeOfReferenceArray(2 * (1 + MAX_KEYS));
            }
            int size = this.setOverflowSizeMap(this.savedBuffer, MAX_KEYS);
            this.ensureParent().addChildAndNextKey(this.savedBuffer, size, this.savedNextKey);
            this.savedBuffer = null;
            this.savedNextKey = null;
        }

        void overflow(Object nextKey) {
            if (this.hasOverflow()) {
                this.propagateOverflow();
            }
            Object[] restoreBuffer = this.savedBuffer;
            int[] restoreSizes = this.savedSizes;
            this.savedBuffer = this.buffer;
            this.savedSizes = this.sizes;
            this.savedNextKey = nextKey;
            this.sizes = restoreSizes == null && this.savedSizes != null ? new int[MAX_KEYS + 1] : restoreSizes;
            this.buffer = restoreBuffer == null ? new Object[2 * (MAX_KEYS + 1)] : restoreBuffer;
            this.count = 0;
        }

        Object[] redistributeOverflowAndDrain() {
            int steal = MIN_KEYS - this.count;
            Object[] newBranch = new Object[2 * (MIN_KEYS + 1)];
            System.arraycopy(this.savedBuffer, MAX_KEYS - (steal - 1), newBranch, 0, steal - 1);
            newBranch[steal - 1] = this.savedNextKey;
            System.arraycopy(this.buffer, 0, newBranch, steal, this.count);
            System.arraycopy(this.savedBuffer, 2 * MAX_KEYS + 1 - steal, newBranch, MIN_KEYS, steal);
            System.arraycopy(this.buffer, MAX_KEYS, newBranch, MIN_KEYS + steal, this.count + 1);
            this.setRedistributedSizeMap(newBranch, steal);
            int savedBranchCount = MAX_KEYS - steal;
            Object[] savedBranch = new Object[2 * (savedBranchCount + 1)];
            System.arraycopy(this.savedBuffer, 0, savedBranch, 0, savedBranchCount);
            System.arraycopy(this.savedBuffer, MAX_KEYS, savedBranch, savedBranchCount, savedBranchCount + 1);
            int savedBranchSize = this.setOverflowSizeMap(savedBranch, savedBranchCount);
            if (this.leaf.allocated >= 0L) {
                this.leaf.allocated += ObjectSizes.sizeOfReferenceArray(2 * (1 + savedBranchCount));
            }
            this.ensureParent().addChildAndNextKey(savedBranch, savedBranchSize, this.savedBuffer[savedBranchCount]);
            this.savedNextKey = null;
            return newBranch;
        }

        @Override
        void prepend(Object[] pred, Object predNextKey) {
            assert (!this.hasOverflow());
            int predKeys = BTree.shallowSizeOfBranch(pred);
            int[] sizeMap = (int[])pred[2 * predKeys + 1];
            int newKeys = 1 + predKeys;
            if (newKeys + this.count <= MAX_KEYS) {
                System.arraycopy(this.buffer, 0, this.buffer, newKeys, this.count);
                System.arraycopy(this.sizes, 0, this.sizes, newKeys, this.count + 1);
                System.arraycopy(this.buffer, MAX_KEYS, this.buffer, MAX_KEYS + newKeys, this.count + 1);
                System.arraycopy(pred, 0, this.buffer, 0, predKeys);
                this.buffer[predKeys] = predNextKey;
                System.arraycopy(pred, predKeys, this.buffer, MAX_KEYS, predKeys + 1);
                BranchBuilder.copySizeMapToSizes(sizeMap, 0, this.sizes, 0, predKeys + 1);
                this.count += newKeys;
            } else {
                if (this.savedBuffer == null) {
                    this.savedBuffer = new Object[2 * (1 + MAX_KEYS)];
                    this.savedSizes = new int[1 + MAX_KEYS];
                }
                System.arraycopy(pred, 0, this.savedBuffer, 0, predKeys);
                System.arraycopy(pred, predKeys, this.savedBuffer, MAX_KEYS, predKeys + 1);
                BranchBuilder.copySizeMapToSizes(sizeMap, 0, this.savedSizes, 0, predKeys + 1);
                if (newKeys == MAX_KEYS + 1) {
                    this.savedNextKey = predNextKey;
                } else {
                    int removeKeys = 1 + MAX_KEYS - newKeys;
                    int remainingKeys = this.count - removeKeys;
                    this.savedBuffer[predKeys] = predNextKey;
                    System.arraycopy(this.buffer, 0, this.savedBuffer, newKeys, MAX_KEYS - newKeys);
                    this.savedNextKey = this.buffer[MAX_KEYS - newKeys];
                    System.arraycopy(this.sizes, 0, this.savedSizes, newKeys, MAX_KEYS + 1 - newKeys);
                    System.arraycopy(this.buffer, MAX_KEYS, this.savedBuffer, MAX_KEYS + newKeys, MAX_KEYS + 1 - newKeys);
                    System.arraycopy(this.buffer, removeKeys, this.buffer, 0, remainingKeys);
                    System.arraycopy(this.buffer, MAX_KEYS + removeKeys, this.buffer, MAX_KEYS, remainingKeys + 1);
                    System.arraycopy(this.sizes, removeKeys, this.sizes, 0, remainingKeys + 1);
                    this.count = remainingKeys;
                }
            }
        }

        @Override
        boolean producesOnlyDense() {
            return this.sizes == null;
        }

        @Override
        Object[] drainAndPropagate(Object[] unode, BranchBuilder propagateTo) {
            int sizeOfBranch;
            Object[] branch;
            if (this.mustRedistribute()) {
                branch = this.redistributeOverflowAndDrain();
                sizeOfBranch = BTree.sizeOfBranch(branch);
            } else {
                int usz;
                int n = usz = unode != null ? BTree.shallowSizeOfBranch(unode) : -1;
                if (!this.hasOverflow() && usz == this.count && BranchBuilder.areIdentical(this.buffer, 0, unode, 0, usz) && BranchBuilder.areIdentical(this.buffer, MAX_KEYS, unode, usz, usz + 1)) {
                    branch = unode;
                    sizeOfBranch = BTree.sizeOfBranch(branch);
                } else {
                    if (this.hasOverflow()) {
                        this.propagateOverflow();
                    }
                    assert (this.count > 0);
                    branch = new Object[2 * (this.count + 1)];
                    System.arraycopy(this.buffer, 0, branch, 0, this.count);
                    System.arraycopy(this.buffer, MAX_KEYS, branch, this.count, this.count + 1);
                    sizeOfBranch = this.setDrainSizeMap(unode, usz, branch, this.count);
                }
            }
            this.count = 0;
            if (propagateTo != null) {
                propagateTo.addChild(branch, sizeOfBranch);
            }
            return branch;
        }

        @Override
        Object[] drain() {
            assert (!this.hasOverflow());
            int keys = this.count;
            this.count = 0;
            Object[] branch = new Object[2 * (keys + 1)];
            if (keys == MAX_KEYS) {
                Object[] tmp = this.buffer;
                this.buffer = branch;
                branch = tmp;
            } else {
                System.arraycopy(this.buffer, 0, branch, 0, keys);
                System.arraycopy(this.buffer, MAX_KEYS, branch, keys, keys + 1);
            }
            this.setDrainSizeMap(null, -1, branch, keys);
            return branch;
        }

        int setDrainSizeMap(Object[] original, int keysInOriginal, Object[] branch, int keysInBranch) {
            int[] sizeMap;
            if (this.producesOnlyDense()) {
                return this.setImperfectSizeMap(branch, keysInBranch);
            }
            int size = BTree.sizesToSizeMap(this.sizes, keysInBranch + 1);
            if (keysInOriginal != keysInBranch || !BranchBuilder.areIdentical(sizeMap = BTree.sizeMap(original), 0, this.sizes, 0, keysInBranch + 1)) {
                sizeMap = this.sizes;
                if (keysInBranch < MAX_KEYS) {
                    sizeMap = Arrays.copyOf(sizeMap, keysInBranch + 1);
                } else {
                    this.sizes = new int[MAX_KEYS + 1];
                }
            }
            branch[2 * keysInBranch + 1] = sizeMap;
            return size;
        }

        int setOverflowSizeMap(Object[] branch, int keys) {
            if (this.producesOnlyDense()) {
                int[] sizeMap = DENSE_SIZE_MAPS[this.height - 2];
                if (keys < MAX_KEYS) {
                    sizeMap = Arrays.copyOf(sizeMap, keys + 1);
                }
                branch[2 * keys + 1] = sizeMap;
                return keys < MAX_KEYS ? sizeMap[keys] : BTree.checkedDenseSize(this.height + 1);
            }
            int[] sizes = this.savedSizes;
            if (keys < MAX_KEYS) {
                sizes = Arrays.copyOf(sizes, keys + 1);
            } else {
                this.savedSizes = null;
            }
            branch[2 * keys + 1] = sizes;
            return BTree.sizesToSizeMap(sizes);
        }

        void setRedistributedSizeMap(Object[] branch, int steal) {
            if (this.producesOnlyDense()) {
                this.setImperfectSizeMap(branch, MIN_KEYS);
            } else {
                int[] sizeMap = new int[MIN_KEYS + 1];
                System.arraycopy(this.sizes, 0, sizeMap, steal, this.count + 1);
                System.arraycopy(this.savedSizes, MAX_KEYS + 1 - steal, sizeMap, 0, steal);
                branch[2 * BTree.MIN_KEYS + 1] = sizeMap;
                BTree.sizesToSizeMap(sizeMap);
            }
        }

        private int setImperfectSizeMap(Object[] branch, int keys) {
            int[] sizeMap = Arrays.copyOf(DENSE_SIZE_MAPS[this.height - 2], keys + 1);
            int size = keys == 1 ? 0 : 1 + sizeMap[keys - 2];
            sizeMap[keys - 1] = size += BTree.size((Object[])branch[2 * keys - 1]);
            sizeMap[keys] = size += 1 + BTree.size((Object[])branch[2 * keys]);
            branch[2 * keys + 1] = sizeMap;
            return size;
        }

        void copyPreceding(Object[] unode, int usz, int offset, int length) {
            int[] uszmap = BTree.sizeMap(unode);
            if (this.count + length > MAX_KEYS) {
                int copy = MAX_KEYS - this.count;
                this.copyPrecedingNoOverflow(unode, usz, uszmap, offset, copy);
                this.buffer[BTree.MAX_KEYS + BTree.MAX_KEYS] = unode[usz + (offset += copy)];
                this.sizes[BTree.MAX_KEYS] = uszmap[offset] - (offset > 0 ? 1 + uszmap[offset - 1] : 0);
                this.overflow(unode[offset]);
                length -= 1 + copy;
                ++offset;
            }
            this.copyPrecedingNoOverflow(unode, usz, uszmap, offset, length);
        }

        private void copyPrecedingNoOverflow(Object[] unode, int usz, int[] uszmap, int offset, int length) {
            if (length <= 1) {
                if (length == 0) {
                    return;
                }
                this.buffer[this.count] = unode[offset];
                this.buffer[BTree.MAX_KEYS + this.count] = unode[usz + offset];
                this.sizes[this.count] = uszmap[offset] - (offset > 0 ? 1 + uszmap[offset - 1] : 0);
                ++this.count;
            } else {
                System.arraycopy(unode, offset, this.buffer, this.count, length);
                System.arraycopy(unode, usz + offset, this.buffer, MAX_KEYS + this.count, length);
                BranchBuilder.copySizeMapToSizes(uszmap, offset, this.sizes, this.count, length);
                this.count += length;
            }
        }

        static void copySizeMapToSizes(int[] in, int inOffset, int[] out, int outOffset, int count) {
            assert (count > 0);
            if (inOffset == 0) {
                out[outOffset++] = in[inOffset++];
                --count;
            }
            for (int i = 0; i < count; ++i) {
                out[outOffset + i] = in[inOffset + i] - (1 + in[inOffset + i - 1]);
            }
        }
    }

    private static abstract class LeafBuilder
    extends LeafOrBranchBuilder {
        long allocated;

        LeafBuilder() {
            super(null);
            this.buffer = new Object[MAX_KEYS];
        }

        public void addKey(Object nextKey) {
            if (this.count == MAX_KEYS) {
                this.overflow(nextKey);
            } else {
                this.buffer[this.count++] = nextKey;
            }
        }

        public void addKeyNoOverflow(Object nextKey) {
            this.buffer[this.count++] = nextKey;
        }

        public void maybeAddKeyNoOverflow(Object nextKey) {
            this.buffer[this.count] = nextKey;
            this.count += nextKey != null ? 1 : 0;
        }

        public void maybeAddKey(Object nextKey) {
            if (this.count == MAX_KEYS) {
                if (nextKey != null) {
                    this.overflow(nextKey);
                }
            } else {
                this.buffer[this.count] = nextKey;
                this.count += nextKey != null ? 1 : 0;
            }
        }

        void copy(Object[] source, int offset, int length) {
            if (this.count + length > MAX_KEYS) {
                int copy = MAX_KEYS - this.count;
                System.arraycopy(source, offset, this.buffer, this.count, copy);
                offset += copy;
                this.overflow(source[offset++]);
                length -= 1 + copy;
            }
            System.arraycopy(source, offset, this.buffer, this.count, length);
            this.count += length;
        }

        void copyNoOverflow(Object[] source, int offset, int length) {
            System.arraycopy(source, offset, this.buffer, this.count, length);
            this.count += length;
        }

        <Insert, Existing> void copy(Object[] source, int offset, int length, UpdateFunction<Insert, Existing> apply) {
            if (BTree.isSimple(apply)) {
                this.copy(source, offset, length);
                return;
            }
            if (this.count + length > MAX_KEYS) {
                int copy = MAX_KEYS - this.count;
                for (int i = 0; i < copy; ++i) {
                    this.buffer[this.count + i] = apply.insert(source[offset + i]);
                }
                offset += copy;
                this.overflow(apply.insert(source[offset++]));
                length -= 1 + copy;
            }
            for (int i = 0; i < length; ++i) {
                this.buffer[this.count + i] = apply.insert(source[offset + i]);
            }
            this.count += length;
        }

        void overflow(Object nextKey) {
            Object[] newBuffer;
            if (this.hasOverflow()) {
                this.propagateOverflow();
            }
            if ((newBuffer = this.savedBuffer) == null) {
                newBuffer = new Object[MAX_KEYS];
            }
            this.savedBuffer = this.buffer;
            this.savedNextKey = nextKey;
            this.buffer = newBuffer;
            this.count = 0;
        }

        Object[] redistributeOverflowAndDrain() {
            Object[] newLeaf = this.redistributeAndDrain(this.savedBuffer, MAX_KEYS, this.savedNextKey);
            this.savedNextKey = null;
            return newLeaf;
        }

        Object[] redistributeAndDrain(Object[] pred, int predSize, Object predNextKey) {
            int steal = MIN_KEYS - this.count;
            Object[] newLeaf = new Object[MIN_KEYS];
            System.arraycopy(pred, predSize - (steal - 1), newLeaf, 0, steal - 1);
            newLeaf[steal - 1] = predNextKey;
            System.arraycopy(this.buffer, 0, newLeaf, steal, this.count);
            int newPredecessorCount = predSize - steal;
            Object[] newPredecessor = new Object[newPredecessorCount | 1];
            System.arraycopy(pred, 0, newPredecessor, 0, newPredecessorCount);
            if (this.allocated >= 0L) {
                this.allocated += ObjectSizes.sizeOfReferenceArray(newPredecessorCount | 1);
            }
            this.ensureParent().addChildAndNextKey(newPredecessor, newPredecessorCount, pred[newPredecessorCount]);
            return newLeaf;
        }

        @Override
        void prepend(Object[] pred, Object predNextKey) {
            assert (!this.hasOverflow());
            int predSize = BTree.sizeOfLeaf(pred);
            int newKeys = 1 + predSize;
            if (newKeys + this.count <= MAX_KEYS) {
                System.arraycopy(this.buffer, 0, this.buffer, newKeys, this.count);
                System.arraycopy(pred, 0, this.buffer, 0, predSize);
                this.buffer[predSize] = predNextKey;
                this.count += newKeys;
            } else {
                if (this.savedBuffer == null) {
                    this.savedBuffer = new Object[MAX_KEYS];
                }
                System.arraycopy(pred, 0, this.savedBuffer, 0, predSize);
                if (predSize == MAX_KEYS) {
                    this.savedNextKey = predNextKey;
                } else {
                    int removeKeys = MAX_KEYS - predSize;
                    this.count -= removeKeys;
                    this.savedBuffer[predSize] = predNextKey;
                    System.arraycopy(this.buffer, 0, this.savedBuffer, predSize + 1, MAX_KEYS - newKeys);
                    this.savedNextKey = this.buffer[MAX_KEYS - newKeys];
                    System.arraycopy(this.buffer, removeKeys, this.buffer, 0, this.count);
                }
            }
        }

        void propagateOverflow() {
            if (this.allocated >= 0L) {
                this.allocated += ObjectSizes.sizeOfReferenceArray(MAX_KEYS);
            }
            this.ensureParent().addChildAndNextKey(this.savedBuffer, MAX_KEYS, this.savedNextKey);
            this.savedBuffer = null;
            this.savedNextKey = null;
        }

        @Override
        Object[] drainAndPropagate(Object[] unode, BranchBuilder propagateTo) {
            int sizeOfLeaf;
            Object[] leaf;
            if (this.mustRedistribute()) {
                leaf = this.redistributeOverflowAndDrain();
                sizeOfLeaf = MIN_KEYS;
            } else if (!this.hasOverflow() && unode != null && this.count == BTree.sizeOfLeaf(unode) && LeafBuilder.areIdentical(this.buffer, 0, unode, 0, this.count)) {
                leaf = unode;
                sizeOfLeaf = this.count;
            } else {
                if (this.hasOverflow()) {
                    this.propagateOverflow();
                }
                sizeOfLeaf = this.count;
                leaf = this.drain();
                if (this.allocated >= 0L && sizeOfLeaf > 0) {
                    this.allocated += ObjectSizes.sizeOfReferenceArray(sizeOfLeaf | 1) - (unode == null ? 0L : BTree.sizeOnHeapOfLeaf(unode));
                }
            }
            this.count = 0;
            if (propagateTo != null) {
                propagateTo.addChild(leaf, sizeOfLeaf);
            }
            return leaf;
        }

        @Override
        Object[] drain() {
            assert (!this.hasOverflow());
            if (this.count == 0) {
                return BTree.empty();
            }
            Object[] newLeaf = new Object[this.count | 1];
            System.arraycopy(this.buffer, 0, newLeaf, 0, this.count);
            this.count = 0;
            return newLeaf;
        }
    }

    private static abstract class LeafOrBranchBuilder {
        final int height;
        final LeafOrBranchBuilder child;
        BranchBuilder parent;
        Object[] buffer;
        int count;
        Object[] savedBuffer;
        Object savedNextKey;

        LeafOrBranchBuilder(LeafOrBranchBuilder child) {
            this.height = child == null ? 1 : 1 + child.height;
            this.child = child;
        }

        final boolean isSufficient() {
            return this.hasOverflow() || this.count >= MIN_KEYS;
        }

        final boolean hasOverflow() {
            return this.savedNextKey != null;
        }

        final boolean mustRedistribute() {
            return this.hasOverflow() && this.count < MIN_KEYS;
        }

        final boolean isEmpty() {
            return this.count == 0 && this.savedNextKey == null;
        }

        abstract Object[] drainAndPropagate(Object[] var1, BranchBuilder var2);

        abstract Object[] drain();

        public Object[] completeBuild() {
            LeafOrBranchBuilder level = this;
            while (level.hasOverflow()) {
                BranchBuilder parent = level.ensureParent();
                level.drainAndPropagate(null, parent);
                if (level.savedBuffer != null) {
                    Arrays.fill(level.savedBuffer, null);
                }
                level = parent;
            }
            return level.drain();
        }

        abstract void prepend(Object[] var1, Object var2);

        abstract boolean producesOnlyDense();

        final BranchBuilder ensureParent() {
            if (this.parent == null) {
                this.parent = new BranchBuilder(this);
            }
            this.parent.inUse = true;
            return this.parent;
        }

        static BranchBuilder markUsed(BranchBuilder branch) {
            branch.inUse = true;
            return branch;
        }

        static boolean areIdentical(Object[] a, int aOffset, Object[] b, int bOffset, int count) {
            for (int i = 0; i < count; ++i) {
                if (a[i + aOffset] == b[i + bOffset]) continue;
                return false;
            }
            return true;
        }

        static boolean areIdentical(int[] a, int aOffset, int[] b, int bOffset, int count) {
            for (int i = 0; i < count; ++i) {
                if (a[i + aOffset] == b[i + bOffset]) continue;
                return false;
            }
            return true;
        }
    }

    public static class Builder<V> {
        Comparator<? super V> comparator;
        Object[] values;
        int count;
        boolean detected = true;
        boolean auto = true;
        QuickResolver<V> quickResolver;

        protected Builder(Comparator<? super V> comparator) {
            this(comparator, 16);
        }

        protected Builder(Comparator<? super V> comparator, int initialCapacity) {
            if (initialCapacity == 0) {
                initialCapacity = 16;
            }
            this.comparator = comparator;
            this.values = new Object[initialCapacity];
        }

        @VisibleForTesting
        public Builder() {
            this.values = new Object[16];
        }

        private Builder(Builder<V> builder) {
            this.comparator = builder.comparator;
            this.values = Arrays.copyOf(builder.values, builder.values.length);
            this.count = builder.count;
            this.detected = builder.detected;
            this.auto = builder.auto;
            this.quickResolver = builder.quickResolver;
        }

        public Builder<V> copy() {
            return new Builder<V>(this);
        }

        public Builder<V> setQuickResolver(QuickResolver<V> quickResolver) {
            this.quickResolver = quickResolver;
            return this;
        }

        public void reuse() {
            this.reuse(this.comparator);
        }

        public void reuse(Comparator<? super V> comparator) {
            this.comparator = comparator;
            Arrays.fill(this.values, null);
            this.count = 0;
            this.detected = true;
        }

        public Builder<V> auto(boolean auto) {
            this.auto = auto;
            return this;
        }

        public Builder<V> add(V v) {
            if (this.count == this.values.length) {
                this.values = Arrays.copyOf(this.values, this.count * 2);
            }
            Object[] values = this.values;
            int prevCount = this.count++;
            values[prevCount] = v;
            if (this.auto && this.detected && prevCount > 0) {
                Object prev = values[prevCount - 1];
                int c = this.comparator.compare(prev, v);
                if (c == 0 && this.auto) {
                    this.count = prevCount;
                    if (this.quickResolver != null) {
                        values[prevCount - 1] = this.quickResolver.resolve(prev, v);
                    }
                } else if (c > 0) {
                    this.detected = false;
                }
            }
            return this;
        }

        public Builder<V> addAll(Collection<V> add) {
            if (this.auto && add instanceof SortedSet && Builder.equalComparators(this.comparator, ((SortedSet)add).comparator())) {
                return this.mergeAll(add, add.size());
            }
            this.detected = false;
            if (this.values.length < this.count + add.size()) {
                this.values = Arrays.copyOf(this.values, Math.max(this.count + add.size(), this.count * 2));
            }
            for (V v : add) {
                this.values[this.count++] = v;
            }
            return this;
        }

        private static boolean equalComparators(Comparator<?> a, Comparator<?> b) {
            return a == b || Builder.isNaturalComparator(a) && Builder.isNaturalComparator(b);
        }

        private static boolean isNaturalComparator(Comparator<?> a) {
            return a == null || a == Comparator.naturalOrder() || a == Ordering.natural();
        }

        private Builder<V> mergeAll(Iterable<V> add, int addCount) {
            assert (this.auto);
            this.autoEnforce();
            int curCount = this.count;
            if (this.values.length < curCount * 2 + addCount) {
                this.values = Arrays.copyOf(this.values, Math.max(curCount * 2 + addCount, curCount * 3));
            }
            if (add instanceof BTreeSet) {
                ((BTreeSet)add).toArray(this.values, curCount);
            } else {
                int i = curCount;
                for (V v : add) {
                    this.values[i++] = v;
                }
            }
            return this.mergeAll(addCount);
        }

        private Builder<V> mergeAll(int addCount) {
            int i;
            Object[] a = this.values;
            int addOffset = this.count;
            int j = addOffset;
            int curEnd = addOffset;
            int addEnd = addOffset + addCount;
            for (i = 0; i < curEnd && j < addEnd; ++i) {
                int c;
                Object ai = a[i];
                Object aj = a[j];
                int n = c = ai == aj ? 0 : this.comparator.compare(ai, aj);
                if (c > 0) break;
                if (c != 0) continue;
                if (this.quickResolver != null) {
                    a[i] = this.quickResolver.resolve(ai, aj);
                }
                ++j;
            }
            if (j == addEnd) {
                return this;
            }
            int newCount = i;
            System.arraycopy(a, i, a, addEnd, this.count - i);
            curEnd = addEnd + (this.count - i);
            i = addEnd;
            while (i < curEnd && j < addEnd) {
                Object ai = a[i];
                Object aj = a[j];
                int c = this.comparator.compare(ai, aj);
                if (c == 0) {
                    Object newValue = this.quickResolver == null ? ai : this.quickResolver.resolve(ai, aj);
                    a[newCount++] = newValue;
                    ++i;
                    ++j;
                    continue;
                }
                a[newCount++] = c < 0 ? a[i++] : a[j++];
            }
            if (i < curEnd) {
                System.arraycopy(a, i, a, newCount, curEnd - i);
                newCount += curEnd - i;
            } else if (j < addEnd) {
                if (j != newCount) {
                    System.arraycopy(a, j, a, newCount, addEnd - j);
                }
                newCount += addEnd - j;
            }
            this.count = newCount;
            return this;
        }

        public boolean isEmpty() {
            return this.count == 0;
        }

        public Builder<V> reverse() {
            assert (!this.auto);
            int mid = this.count / 2;
            for (int i = 0; i < mid; ++i) {
                Object t = this.values[i];
                this.values[i] = this.values[this.count - (1 + i)];
                this.values[this.count - (1 + i)] = t;
            }
            return this;
        }

        public Builder<V> sort() {
            Arrays.sort(this.values, 0, this.count, this.comparator);
            return this;
        }

        private void autoEnforce() {
            if (!this.detected && this.count > 1) {
                this.sort();
                int prevIdx = 0;
                Object prev = this.values[0];
                for (int i = 1; i < this.count; ++i) {
                    Object next = this.values[i];
                    if (this.comparator.compare(prev, next) != 0) {
                        this.values[++prevIdx] = prev = next;
                        continue;
                    }
                    if (this.quickResolver == null) continue;
                    this.values[prevIdx] = prev = this.quickResolver.resolve(prev, next);
                }
                this.count = prevIdx + 1;
            }
            this.detected = true;
        }

        public Builder<V> resolve(Resolver resolver) {
            if (this.count > 0) {
                int c = 0;
                int prev = 0;
                for (int i = 1; i < this.count; ++i) {
                    if (this.comparator.compare(this.values[i], this.values[prev]) == 0) continue;
                    this.values[c++] = resolver.resolve(this.values, prev, i);
                    prev = i;
                }
                this.values[c++] = resolver.resolve(this.values, prev, this.count);
                this.count = c;
            }
            return this;
        }

        public Object[] build() {
            if (this.auto) {
                this.autoEnforce();
            }
            try (BulkIterator.FromArray iterator = BulkIterator.of(this.values, 0);){
                Object[] objectArray = BTree.build(iterator, this.count, UpdateFunction.noOp());
                return objectArray;
            }
        }

        public static interface QuickResolver<V> {
            public V resolve(V var1, V var2);
        }

        public static interface Resolver {
            public Object resolve(Object[] var1, int var2, int var3);
        }
    }

    public static enum Dir {
        ASC,
        DESC;


        public Dir invert() {
            return this == ASC ? DESC : ASC;
        }

        public static Dir desc(boolean desc) {
            return desc ? DESC : ASC;
        }
    }
}

