/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *   * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */

package org.apache.commons.vfs.provider.truezip;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.vfs.*;
import org.apache.commons.vfs.impl.DefaultFileSystemManager;
import org.apache.commons.vfs.provider.AbstractFileObject;
import org.apache.commons.vfs.provider.UriParser;
import org.apache.commons.vfs.provider.LayeredFileName;
import org.apache.commons.vfs.provider.local.LocalFile;
import org.apache.commons.vfs.util.RandomAccessMode;

import de.schlichtherle.io.ArchiveDetector;
import de.schlichtherle.io.File;
import de.schlichtherle.io.FileInputStream;
import de.schlichtherle.io.FileOutputStream;

/**
 * A file object implementation which uses Truezip (https://truezip.dev.java.net/)
 */
public class TzFileObject extends AbstractFileObject implements FileObject {

    private File tzFile;
    private FileObject vfsFile;

    /**
     * Creates a non-root file.
     */
    protected TzFileObject(final TzFileSystem fileSystem, final FileName name) throws FileSystemException {
        super(name, fileSystem);
    }

    /**
     * Returns the local file that this file object represents.
     */
    protected File getLocalFile() {
        return tzFile;
    }

    /**
     * Attaches this file object to its file resource.
     */
    protected void doAttach() throws Exception {
        if (tzFile == null) {

//            original------------------
//            LayeredFileName layeredFileName = (LayeredFileName) getName();
//            String fileName = layeredFileName.getOuterName().getPathDecoded()
//                + getName().getPathDecoded();
//            tzFile = new File(fileName, ArchiveDetector.ALL);

            LayeredFileName layeredFileName = (LayeredFileName) getName();
            String fileName = layeredFileName.getOuterName().getRootURI() +
                layeredFileName.getOuterName().getPathDecoded();

            FileObject outer = getFileSystem().resolveFile(fileName);
            if (outer instanceof TzFileObject) {
                fileName = layeredFileName.getOuterName().getPathDecoded()
                    + getName().getPathDecoded();
                tzFile = new File(fileName, ArchiveDetector.ALL);

            } else {
                vfsFile = outer;
                DefaultFileSystemManager dfsMgr = (DefaultFileSystemManager) VFS.getManager();
                tzFile = new File(
                    dfsMgr.getTemporaryFileStore().allocateFile(getName().getBaseName()));
            }

//            LayeredFileName layeredFileName = (LayeredFileName) getName();
//
//            String fileName = layeredFileName.getOuterName().getRootURI() +
//                layeredFileName.getOuterName().getPathDecoded();
//
//            FileObject outer = getFileSystem().getFileSystemManager().resolveFile(fileName);
//
//            if (outer.exists()) {
//                file = new File(getFileSystem().replicateFile(outer, Selectors.SELECT_SELF).getPath());
//            } else {
//                DefaultFileSystemManager dfsMgr = (DefaultFileSystemManager) VFS.getManager();
//                file = new File(
//                    dfsMgr.getTemporaryFileStore().allocateFile(
//                        /*layeredFileName.getOuterName().getBaseName() +**/ getName().getBaseName()));
//                setFileType(FileType.IMAGINARY);
//            }
        }
    }

    /**
     * Returns the file's type.
     */
    protected FileType doGetType() throws Exception {

        if (vfsFile != null) {
            return vfsFile.getType();
        }

        if (!tzFile.exists() && tzFile.length() < 1) {
            return FileType.IMAGINARY;
        }
        if (tzFile.isDirectory()) {
            return FileType.FOLDER;
        }
        return FileType.FILE;
    }

    /**
     * Returns the children of the file.
     */
    protected String[] doListChildren()
        throws Exception {
        return UriParser.encode(tzFile.list());
    }

    /**
     * Deletes this file, and all children.
     */
    protected void doDelete() throws Exception {
        if (!tzFile.deleteAll()) {
            throw new FileSystemException("vfs.provider.truezip/delete-file.error", tzFile);
        }
    }

    /**
     * rename this file
     */
    protected void doRename(FileObject newfile) throws Exception {
        if (!tzFile.renameTo(((TzFileObject) newfile).getLocalFile())) {
            throw new FileSystemException("vfs.provider.truezip/rename-file.error",
                new String[]{tzFile.toString(), newfile.toString()});
        }
    }

    /**
     * Creates this folder.
     */
    protected void doCreateFolder() throws Exception {
        if (!tzFile.mkdirs()) {
            throw new FileSystemException("vfs.provider.truezip/create-folder.error", tzFile);
        }
    }

    /**
     * Determines if this file can be written to.
     */
    protected boolean doIsWriteable() throws FileSystemException {
        return tzFile.canWrite();
    }

    /**
     * Determines if this file is hidden.
     */
    protected boolean doIsHidden() {
        return tzFile.isHidden();
    }

    /**
     * Determines if this file can be read.
     */
    protected boolean doIsReadable() throws FileSystemException {
        return tzFile.canRead();
    }

    /**
     * Gets the last modified time of this file.
     */
    protected long doGetLastModifiedTime() throws FileSystemException {
        return tzFile.lastModified();
    }

    /**
     * Sets the last modified time of this file.
     */
    protected void doSetLastModifiedTime(final long modtime)
        throws FileSystemException {
        tzFile.setLastModified(modtime);
    }

    /**
     * Creates an input stream to read the content from.
     */
    protected InputStream doGetInputStream()
        throws Exception {
        return new FileInputStream(tzFile);
    }

    /**
     * Creates an output stream to write the file content to.
     */
    protected OutputStream doGetOutputStream(boolean bAppend) throws Exception {

        return new FileOutputStream(tzFile, bAppend);
    }

    /**
     * Returns the size of the file content (in bytes).
     */
    protected long doGetContentSize()
        throws Exception {
        return tzFile.length();
    }

    protected RandomAccessContent doGetRandomAccessContent(final RandomAccessMode mode) throws Exception {
        throw new IOException("Not implemented"); //return new LocalFileRandomAccessContent(file, mode);
    }
}