/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *   * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */

package org.apache.commons.vfs.provider.truezip;

import org.apache.commons.vfs.*;
import org.apache.commons.vfs.provider.AbstractLayeredFileProvider;
import org.apache.commons.vfs.provider.LayeredFileName;
import org.apache.commons.vfs.provider.local.LocalFileName;
import org.apache.commons.vfs.provider.local.LocalFileNameParser;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;

/**
 * A file system provider, which uses Truezip (https://truezip.dev.java.net/)
 */
public class TzFileProvider extends AbstractLayeredFileProvider {

    public final static Collection capabilities =
        Collections.unmodifiableCollection(Arrays.asList(new Capability[]{
            Capability.CREATE,
            Capability.DELETE,
            Capability.RENAME,
            Capability.GET_TYPE,
            Capability.GET_LAST_MODIFIED,
            Capability.SET_LAST_MODIFIED_FILE,
            Capability.SET_LAST_MODIFIED_FOLDER,
            Capability.LIST_CHILDREN,
            Capability.READ_CONTENT,
            Capability.URI,
            Capability.WRITE_CONTENT,
            Capability.APPEND_CONTENT,
            //Capability.RANDOM_ACCESS_READ,
            //Capability.RANDOM_ACCESS_WRITE
        }));

    public TzFileProvider() {
        super();
    }

    /**
     * Creates a layered file system.  This method is called if the file system
     * is not cached.
     *
     * @param scheme The URI scheme.
     * @param file   The file to create the file system on top of.
     * @return The file system.
     */
    protected FileSystem doCreateFileSystem(final String scheme,
        final FileObject file,
        final FileSystemOptions fileSystemOptions)
        throws FileSystemException {

        final FileName name =
            new LayeredFileName(scheme, file.getName(), FileName.ROOT_PATH, FileType.FOLDER);
        return new TzFileSystem(name, file, fileSystemOptions);
    }

    /**
     * Determines if a name is an absolute file name.
     */
    public boolean isAbsoluteLocalName(final String name) {
        return ((LocalFileNameParser) getFileNameParser()).isAbsoluteName(name);
    }

    /**
     * Creates the filesystem.
     */
    protected FileSystem doCreateFileSystem(final FileName name,
        final FileSystemOptions fileSystemOptions)
        throws FileSystemException {

        // Create the file system
        final LocalFileName rootName = (LocalFileName) name;
        return new TzFileSystem(rootName, rootName.getRootFile(), fileSystemOptions);
    }

    public Collection getCapabilities() {
        return capabilities;
    }
}
