/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.query;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableMap;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.Arrays;
import java.util.Map;
import javax.annotation.Nullable;
import org.apache.druid.client.ResultLevelCacheUtil;
import org.apache.druid.client.cache.Cache;
import org.apache.druid.client.cache.CacheConfig;
import org.apache.druid.java.util.common.RE;
import org.apache.druid.java.util.common.StringUtils;
import org.apache.druid.java.util.common.guava.Sequence;
import org.apache.druid.java.util.common.guava.SequenceWrapper;
import org.apache.druid.java.util.common.guava.Sequences;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.query.CacheStrategy;
import org.apache.druid.query.Query;
import org.apache.druid.query.QueryPlus;
import org.apache.druid.query.QueryRunner;
import org.apache.druid.query.QueryToolChest;

public class ResultLevelCachingQueryRunner<T>
implements QueryRunner<T> {
    private static final Logger log = new Logger(ResultLevelCachingQueryRunner.class);
    private final QueryRunner baseRunner;
    private ObjectMapper objectMapper;
    private final Cache cache;
    private final CacheConfig cacheConfig;
    private final boolean useResultCache;
    private final boolean populateResultCache;
    private Query<T> query;
    private final CacheStrategy<T, Object, Query<T>> strategy;

    public ResultLevelCachingQueryRunner(QueryRunner baseRunner, QueryToolChest queryToolChest, Query<T> query, ObjectMapper objectMapper, Cache cache, CacheConfig cacheConfig) {
        this.baseRunner = baseRunner;
        this.objectMapper = objectMapper;
        this.cache = cache;
        this.cacheConfig = cacheConfig;
        this.query = query;
        this.strategy = queryToolChest.getCacheStrategy(query);
        this.populateResultCache = ResultLevelCacheUtil.populateResultLevelCacheOnBrokers(query, this.strategy, cacheConfig);
        this.useResultCache = ResultLevelCacheUtil.useResultLevelCacheOnBrokers(query, this.strategy, cacheConfig);
    }

    public Sequence<T> run(QueryPlus queryPlus, Map responseContext) {
        if (this.useResultCache || this.populateResultCache) {
            String cacheKeyStr = StringUtils.fromUtf8((byte[])this.strategy.computeResultLevelCacheKey(this.query));
            byte[] cachedResultSet = this.fetchResultsFromResultLevelCache(cacheKeyStr);
            String existingResultSetId = this.extractEtagFromResults(cachedResultSet);
            existingResultSetId = existingResultSetId == null ? "" : existingResultSetId;
            this.query = this.query.withOverriddenContext((Map)ImmutableMap.of((Object)"If-None-Match", (Object)existingResultSetId));
            Sequence resultFromClient = this.baseRunner.run(QueryPlus.wrap(this.query), responseContext);
            String newResultSetId = (String)responseContext.get("ETag");
            if (this.useResultCache && newResultSetId != null && newResultSetId.equals(existingResultSetId)) {
                log.debug("Return cached result set as there is no change in identifiers for query %s ", new Object[]{this.query.getId()});
                return this.deserializeResults(cachedResultSet, this.strategy, existingResultSetId);
            }
            final ResultLevelCachePopulator resultLevelCachePopulator = this.createResultLevelCachePopulator(cacheKeyStr, newResultSetId);
            if (resultLevelCachePopulator == null) {
                return resultFromClient;
            }
            final Function cacheFn = this.strategy.prepareForCache(true);
            return Sequences.wrap((Sequence)Sequences.map((Sequence)resultFromClient, (Function)new Function<T, T>(){

                public T apply(T input) {
                    if (resultLevelCachePopulator.isShouldPopulate()) {
                        resultLevelCachePopulator.cacheResultEntry(input, cacheFn);
                    }
                    return input;
                }
            }), (SequenceWrapper)new SequenceWrapper(){

                public void after(boolean isDone, Throwable thrown) {
                    Preconditions.checkNotNull((Object)resultLevelCachePopulator, (Object)"ResultLevelCachePopulator cannot be null during cache population");
                    if (thrown != null) {
                        log.error(thrown, "Error while preparing for result level caching for query %s with error %s ", new Object[]{ResultLevelCachingQueryRunner.this.query.getId(), thrown.getMessage()});
                    } else if (resultLevelCachePopulator.isShouldPopulate()) {
                        resultLevelCachePopulator.populateResults();
                        log.debug("Cache population complete for query %s", new Object[]{ResultLevelCachingQueryRunner.this.query.getId()});
                    }
                    resultLevelCachePopulator.stopPopulating();
                }
            });
        }
        return this.baseRunner.run(queryPlus, responseContext);
    }

    private byte[] fetchResultsFromResultLevelCache(String queryCacheKey) {
        if (this.useResultCache && queryCacheKey != null) {
            return this.cache.get(ResultLevelCacheUtil.computeResultLevelCacheKey(queryCacheKey));
        }
        return null;
    }

    private String extractEtagFromResults(byte[] cachedResult) {
        if (cachedResult == null) {
            return null;
        }
        log.debug("Fetching result level cache identifier for query: %s", new Object[]{this.query.getId()});
        int etagLength = ByteBuffer.wrap(cachedResult, 0, 4).getInt();
        return StringUtils.fromUtf8((byte[])Arrays.copyOfRange(cachedResult, 4, etagLength + 4));
    }

    private Sequence<T> deserializeResults(byte[] cachedResult, CacheStrategy strategy, String resultSetId) {
        if (cachedResult == null) {
            log.error("Cached result set is null", new Object[0]);
        }
        Function pullFromCacheFunction = strategy.pullFromCache(true);
        TypeReference cacheObjectClazz = strategy.getCacheObjectClazz();
        Sequence cachedSequence = Sequences.simple(() -> {
            try {
                int resultOffset = 4 + resultSetId.length();
                return this.objectMapper.readValues(this.objectMapper.getFactory().createParser(cachedResult, resultOffset, cachedResult.length - resultOffset), cacheObjectClazz);
            }
            catch (IOException e) {
                throw new RE((Throwable)e, "Failed to retrieve results from cache for query ID [%s]", new Object[]{this.query.getId()});
            }
        });
        return Sequences.map((Sequence)cachedSequence, (Function)pullFromCacheFunction);
    }

    private ResultLevelCachePopulator createResultLevelCachePopulator(String cacheKeyStr, String resultSetId) {
        if (resultSetId != null && this.populateResultCache) {
            ResultLevelCachePopulator resultLevelCachePopulator = new ResultLevelCachePopulator(this.cache, this.objectMapper, ResultLevelCacheUtil.computeResultLevelCacheKey(cacheKeyStr), this.cacheConfig, true);
            try {
                resultLevelCachePopulator.cacheObjectStream.write(ByteBuffer.allocate(4).putInt(resultSetId.length()).array());
                resultLevelCachePopulator.cacheObjectStream.write(StringUtils.toUtf8((String)resultSetId));
            }
            catch (IOException ioe) {
                log.error((Throwable)ioe, "Failed to write cached values for query %s", new Object[]{this.query.getId()});
                return null;
            }
            return resultLevelCachePopulator;
        }
        return null;
    }

    private class ResultLevelCachePopulator {
        private final Cache cache;
        private final ObjectMapper mapper;
        private final Cache.NamedKey key;
        private final CacheConfig cacheConfig;
        @Nullable
        private ByteArrayOutputStream cacheObjectStream;

        private ResultLevelCachePopulator(Cache cache, ObjectMapper mapper, Cache.NamedKey key, CacheConfig cacheConfig, boolean shouldPopulate) {
            this.cache = cache;
            this.mapper = mapper;
            this.key = key;
            this.cacheConfig = cacheConfig;
            this.cacheObjectStream = shouldPopulate ? new ByteArrayOutputStream() : null;
        }

        boolean isShouldPopulate() {
            return this.cacheObjectStream != null;
        }

        void stopPopulating() {
            this.cacheObjectStream = null;
        }

        private void cacheResultEntry(T resultEntry, Function<T, Object> cacheFn) {
            Preconditions.checkNotNull((Object)this.cacheObjectStream, (Object)"cacheObjectStream");
            int cacheLimit = this.cacheConfig.getResultLevelCacheLimit();
            try (JsonGenerator gen = this.mapper.getFactory().createGenerator((OutputStream)this.cacheObjectStream);){
                gen.writeObject(cacheFn.apply(resultEntry));
                if (cacheLimit > 0 && this.cacheObjectStream.size() > cacheLimit) {
                    this.stopPopulating();
                }
            }
            catch (IOException ex) {
                log.error((Throwable)ex, "Failed to retrieve entry to be cached. Result Level caching will not be performed!", new Object[0]);
                this.stopPopulating();
            }
        }

        public void populateResults() {
            ResultLevelCacheUtil.populate(this.cache, this.key, ((ByteArrayOutputStream)Preconditions.checkNotNull((Object)this.cacheObjectStream, (Object)"cacheObjectStream")).toByteArray());
        }
    }
}

