/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.util;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import javax.annotation.Nullable;
import javax.annotation.concurrent.GuardedBy;
import javax.annotation.concurrent.ThreadSafe;
import org.apache.flink.util.Preconditions;

@ThreadSafe
public class KeyedBudgetManager<K> {
    private final Map<K, Long> maxBudgetByKey;
    private final long defaultPageSize;
    private final long totalNumberOfPages;
    @GuardedBy(value="lock")
    private final Map<K, Long> availableBudgetByKey;
    private final Object lock = new Object();

    public KeyedBudgetManager(Map<K, Long> maxBudgetByKey, long defaultPageSize) {
        Preconditions.checkNotNull(maxBudgetByKey);
        Preconditions.checkArgument((defaultPageSize > 0L ? 1 : 0) != 0, (Object)"The default page size has to be greater than zero");
        this.maxBudgetByKey = new HashMap<K, Long>(maxBudgetByKey);
        this.availableBudgetByKey = new HashMap<K, Long>(maxBudgetByKey);
        this.defaultPageSize = defaultPageSize;
        this.totalNumberOfPages = KeyedBudgetManager.calculateTotalNumberOfPages(maxBudgetByKey, defaultPageSize);
    }

    public long getDefaultPageSize() {
        return this.defaultPageSize;
    }

    public long acquireBudgetForKey(K key, long size) {
        Preconditions.checkNotNull(key);
        AcquisitionResult<K> result = this.acquirePagedBudgetForKeys(Collections.singletonList(key), size, 1L);
        return result.isSuccess() ? result.getAcquiredPerKey().get(key).longValue() : result.getTotalAvailableForAllQueriedKeys();
    }

    public AcquisitionResult<K> acquirePagedBudget(Iterable<K> keys, long numberOfPages) {
        return this.acquirePagedBudgetForKeys(keys, numberOfPages, this.defaultPageSize);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    AcquisitionResult<K> acquirePagedBudgetForKeys(Iterable<K> keys, long numberOfPages, long pageSize) {
        Preconditions.checkNotNull(keys);
        Preconditions.checkArgument((numberOfPages >= 0L ? 1 : 0) != 0, (Object)"The requested number of pages has to be positive");
        Preconditions.checkArgument((pageSize > 0L ? 1 : 0) != 0, (Object)"The page size has to be greater than zero");
        Object object = this.lock;
        synchronized (object) {
            boolean possibleToAcquire;
            long leftPagesToReserve = numberOfPages;
            HashMap<K, Long> pagesToReserveByKey = new HashMap<K, Long>();
            for (K key : keys) {
                long availableBudgetOfCurrentKey = this.availableBudgetByKey.getOrDefault(key, 0L);
                long availablePagesOfCurrentKey = availableBudgetOfCurrentKey / pageSize;
                if (leftPagesToReserve <= availablePagesOfCurrentKey) {
                    pagesToReserveByKey.put(key, leftPagesToReserve);
                    leftPagesToReserve = 0L;
                    break;
                }
                if (availablePagesOfCurrentKey <= 0L) continue;
                pagesToReserveByKey.put(key, availablePagesOfCurrentKey);
                leftPagesToReserve -= availablePagesOfCurrentKey;
            }
            boolean bl = possibleToAcquire = leftPagesToReserve == 0L;
            if (possibleToAcquire) {
                for (Map.Entry pagesToReserveForKey : pagesToReserveByKey.entrySet()) {
                    this.availableBudgetByKey.compute(pagesToReserveForKey.getKey(), (k, v) -> v - (Long)pagesToReserveForKey.getValue() * pageSize);
                }
            }
            return possibleToAcquire ? AcquisitionResult.success(pagesToReserveByKey) : AcquisitionResult.failure(numberOfPages - leftPagesToReserve);
        }
    }

    public void releasePageForKey(K key) {
        this.releaseBudgetForKey(key, this.defaultPageSize);
    }

    public void releaseBudgetForKey(K key, long size) {
        Preconditions.checkNotNull(key);
        Preconditions.checkArgument((size >= 0L ? 1 : 0) != 0, (Object)"The budget to release has to be positive");
        this.releaseBudgetForKeys(Collections.singletonMap(key, size));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseBudgetForKeys(Map<K, Long> sizeByKey) {
        Preconditions.checkNotNull(sizeByKey);
        Object object = this.lock;
        synchronized (object) {
            for (Map.Entry<K, Long> toReleaseForKey : sizeByKey.entrySet()) {
                long toRelease = toReleaseForKey.getValue();
                Preconditions.checkArgument((toRelease >= 0L ? 1 : 0) != 0, (String)"The budget to release for key %s has to be positive", (Object[])new Object[]{toReleaseForKey.getKey()});
                if (toRelease == 0L) continue;
                Object keyToReleaseFor = toReleaseForKey.getKey();
                long maxBudgetForKey = this.maxBudgetByKey.get(keyToReleaseFor);
                this.availableBudgetByKey.compute(keyToReleaseFor, (k, currentBudget) -> {
                    if (currentBudget == null) {
                        throw new IllegalArgumentException("The budget key is not supported: " + keyToReleaseFor);
                    }
                    if (currentBudget + toRelease > maxBudgetForKey) {
                        throw new IllegalStateException(String.format("The budget to release %d exceeds the limit %d for key %s", toRelease, maxBudgetForKey, keyToReleaseFor));
                    }
                    return currentBudget + toRelease;
                });
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void releaseAll() {
        Object object = this.lock;
        synchronized (object) {
            this.availableBudgetByKey.putAll(this.maxBudgetByKey);
        }
    }

    public long maxTotalBudget() {
        return this.maxBudgetByKey.values().stream().mapToLong(b -> b).sum();
    }

    public long maxTotalNumberOfPages() {
        return this.totalNumberOfPages;
    }

    public long maxTotalBudgetForKey(K key) {
        Preconditions.checkNotNull(key);
        return this.maxBudgetByKey.get(key);
    }

    public long totalAvailableBudget() {
        return this.availableBudgetForKeys(this.maxBudgetByKey.keySet());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long availableBudgetForKeys(Iterable<K> keys) {
        Preconditions.checkNotNull(keys);
        Object object = this.lock;
        synchronized (object) {
            long totalSize = 0L;
            for (K key : keys) {
                totalSize += this.availableBudgetForKey(key);
            }
            return totalSize;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long availableBudgetForKey(K key) {
        Preconditions.checkNotNull(key);
        Object object = this.lock;
        synchronized (object) {
            return this.availableBudgetByKey.getOrDefault(key, 0L);
        }
    }

    private static <K> long calculateTotalNumberOfPages(Map<K, Long> budgetByType, long pageSize) {
        long numPages = 0L;
        for (long sizeForType : budgetByType.values()) {
            numPages += sizeForType / pageSize;
        }
        return numPages;
    }

    public static class AcquisitionResult<K> {
        @Nullable
        private final Map<K, Long> acquiredBudgetPerKey;
        @Nullable
        private final Long totalAvailableBudgetForAllQueriedKeys;

        private AcquisitionResult(@Nullable Map<K, Long> acquiredBudgetPerKey, @Nullable Long totalAvailableBudgetForAllQueriedKeys) {
            this.acquiredBudgetPerKey = acquiredBudgetPerKey;
            this.totalAvailableBudgetForAllQueriedKeys = totalAvailableBudgetForAllQueriedKeys;
        }

        public static <K> AcquisitionResult<K> success(Map<K, Long> acquiredBudgetPerKey) {
            return new AcquisitionResult<K>(acquiredBudgetPerKey, null);
        }

        public static <K> AcquisitionResult<K> failure(long totalAvailableBudgetForAllQueriedKeys) {
            return new AcquisitionResult<K>(null, totalAvailableBudgetForAllQueriedKeys);
        }

        public boolean isSuccess() {
            return this.acquiredBudgetPerKey != null;
        }

        public boolean isFailure() {
            return this.totalAvailableBudgetForAllQueriedKeys != null;
        }

        public Map<K, Long> getAcquiredPerKey() {
            if (this.acquiredBudgetPerKey == null) {
                throw new IllegalStateException("The acquisition failed. Nothing was acquired.");
            }
            return Collections.unmodifiableMap(this.acquiredBudgetPerKey);
        }

        public long getTotalAvailableForAllQueriedKeys() {
            if (this.totalAvailableBudgetForAllQueriedKeys == null) {
                throw new IllegalStateException("The acquisition succeeded. All requested pages were acquired.");
            }
            return this.totalAvailableBudgetForAllQueriedKeys;
        }
    }
}

