/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.checkpoint;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;
import org.apache.flink.api.common.JobStatus;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.runtime.checkpoint.CheckpointStoreUtil;
import org.apache.flink.runtime.checkpoint.CheckpointsCleaner;
import org.apache.flink.runtime.checkpoint.CompletedCheckpoint;
import org.apache.flink.runtime.checkpoint.CompletedCheckpointStore;
import org.apache.flink.runtime.persistence.ResourceVersion;
import org.apache.flink.runtime.persistence.StateHandleStore;
import org.apache.flink.runtime.state.RetrievableStateHandle;
import org.apache.flink.util.FlinkException;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultCompletedCheckpointStore<R extends ResourceVersion<R>>
implements CompletedCheckpointStore {
    private static final Logger LOG = LoggerFactory.getLogger(DefaultCompletedCheckpointStore.class);
    private static final Comparator<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> STRING_COMPARATOR = Comparator.comparing(o -> (String)o.f1);
    private final StateHandleStore<CompletedCheckpoint, R> checkpointStateHandleStore;
    private final int maxNumberOfCheckpointsToRetain;
    private final ArrayDeque<CompletedCheckpoint> completedCheckpoints;
    private final Executor ioExecutor;
    private final CheckpointStoreUtil completedCheckpointStoreUtil;

    public DefaultCompletedCheckpointStore(int maxNumberOfCheckpointsToRetain, StateHandleStore<CompletedCheckpoint, R> stateHandleStore, CheckpointStoreUtil completedCheckpointStoreUtil, Executor executor) {
        Preconditions.checkArgument((maxNumberOfCheckpointsToRetain >= 1 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint.");
        this.maxNumberOfCheckpointsToRetain = maxNumberOfCheckpointsToRetain;
        this.checkpointStateHandleStore = (StateHandleStore)Preconditions.checkNotNull(stateHandleStore);
        this.completedCheckpoints = new ArrayDeque(maxNumberOfCheckpointsToRetain + 1);
        this.ioExecutor = (Executor)Preconditions.checkNotNull((Object)executor);
        this.completedCheckpointStoreUtil = (CheckpointStoreUtil)Preconditions.checkNotNull((Object)completedCheckpointStoreUtil);
    }

    @Override
    public boolean requiresExternalizedCheckpoints() {
        return true;
    }

    @Override
    public void recover() throws Exception {
        LOG.info("Recovering checkpoints from {}.", this.checkpointStateHandleStore);
        List<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> initialCheckpoints = this.checkpointStateHandleStore.getAllAndLock();
        initialCheckpoints.sort(STRING_COMPARATOR);
        int numberOfInitialCheckpoints = initialCheckpoints.size();
        LOG.info("Found {} checkpoints in {}.", (Object)numberOfInitialCheckpoints, this.checkpointStateHandleStore);
        if (this.haveAllDownloaded(initialCheckpoints)) {
            LOG.info("All {} checkpoints found are already downloaded.", (Object)numberOfInitialCheckpoints);
            return;
        }
        ArrayList<CompletedCheckpoint> lastTryRetrievedCheckpoints = new ArrayList<CompletedCheckpoint>(numberOfInitialCheckpoints);
        ArrayList<CompletedCheckpoint> retrievedCheckpoints = new ArrayList<CompletedCheckpoint>(numberOfInitialCheckpoints);
        Exception retrieveException = null;
        do {
            LOG.info("Trying to fetch {} checkpoints from storage.", (Object)numberOfInitialCheckpoints);
            lastTryRetrievedCheckpoints.clear();
            lastTryRetrievedCheckpoints.addAll(retrievedCheckpoints);
            retrievedCheckpoints.clear();
            for (Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> checkpointStateHandle : initialCheckpoints) {
                try {
                    CompletedCheckpoint completedCheckpoint = this.retrieveCompletedCheckpoint(checkpointStateHandle);
                    if (completedCheckpoint == null) continue;
                    retrievedCheckpoints.add(completedCheckpoint);
                }
                catch (Exception e) {
                    LOG.warn("Could not retrieve checkpoint, not adding to list of recovered checkpoints.", (Throwable)e);
                    retrieveException = e;
                }
            }
        } while (retrievedCheckpoints.size() != numberOfInitialCheckpoints && !CompletedCheckpoint.checkpointsMatch(lastTryRetrievedCheckpoints, retrievedCheckpoints));
        this.completedCheckpoints.clear();
        this.completedCheckpoints.addAll(retrievedCheckpoints);
        if (this.completedCheckpoints.isEmpty() && numberOfInitialCheckpoints > 0) {
            throw new FlinkException("Could not read any of the " + numberOfInitialCheckpoints + " checkpoints from storage.", (Throwable)retrieveException);
        }
        if (this.completedCheckpoints.size() != numberOfInitialCheckpoints) {
            LOG.warn("Could only fetch {} of {} checkpoints from storage.", (Object)this.completedCheckpoints.size(), (Object)numberOfInitialCheckpoints);
        }
    }

    @Override
    public void addCheckpoint(CompletedCheckpoint checkpoint, CheckpointsCleaner checkpointsCleaner, Runnable postCleanup) throws Exception {
        Preconditions.checkNotNull((Object)checkpoint, (String)"Checkpoint");
        String path = this.completedCheckpointStoreUtil.checkpointIDToName(checkpoint.getCheckpointID());
        this.checkpointStateHandleStore.addAndLock(path, checkpoint);
        this.completedCheckpoints.addLast(checkpoint);
        while (this.completedCheckpoints.size() > this.maxNumberOfCheckpointsToRetain) {
            CompletedCheckpoint completedCheckpoint = this.completedCheckpoints.removeFirst();
            this.tryRemoveCompletedCheckpoint(completedCheckpoint, completedCheckpoint.shouldBeDiscardedOnSubsume(), checkpointsCleaner, postCleanup);
        }
        LOG.debug("Added {} to {}.", (Object)checkpoint, (Object)path);
    }

    @Override
    public List<CompletedCheckpoint> getAllCheckpoints() {
        return new ArrayList<CompletedCheckpoint>(this.completedCheckpoints);
    }

    @Override
    public int getNumberOfRetainedCheckpoints() {
        return this.completedCheckpoints.size();
    }

    @Override
    public int getMaxNumberOfRetainedCheckpoints() {
        return this.maxNumberOfCheckpointsToRetain;
    }

    @Override
    public void shutdown(JobStatus jobStatus, CheckpointsCleaner checkpointsCleaner, Runnable postCleanup) throws Exception {
        if (jobStatus.isGloballyTerminalState()) {
            LOG.info("Shutting down");
            for (CompletedCheckpoint checkpoint : this.completedCheckpoints) {
                this.tryRemoveCompletedCheckpoint(checkpoint, checkpoint.shouldBeDiscardedOnShutdown(jobStatus), checkpointsCleaner, postCleanup);
            }
            this.completedCheckpoints.clear();
            this.checkpointStateHandleStore.clearEntries();
        } else {
            LOG.info("Suspending");
            this.completedCheckpoints.clear();
            this.checkpointStateHandleStore.releaseAll();
        }
    }

    private void tryRemoveCompletedCheckpoint(CompletedCheckpoint completedCheckpoint, boolean shouldDiscard, CheckpointsCleaner checkpointsCleaner, Runnable postCleanup) {
        try {
            if (this.tryRemove(completedCheckpoint.getCheckpointID())) {
                checkpointsCleaner.cleanCheckpoint(completedCheckpoint, shouldDiscard, postCleanup, this.ioExecutor);
            }
        }
        catch (Exception e) {
            LOG.warn("Failed to subsume the old checkpoint", (Throwable)e);
        }
    }

    private boolean haveAllDownloaded(List<Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String>> checkpointPointers) {
        if (this.completedCheckpoints.size() != checkpointPointers.size()) {
            return false;
        }
        Set localIds = this.completedCheckpoints.stream().map(CompletedCheckpoint::getCheckpointID).collect(Collectors.toSet());
        for (Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> initialCheckpoint : checkpointPointers) {
            if (localIds.contains(this.completedCheckpointStoreUtil.nameToCheckpointID((String)initialCheckpoint.f1))) continue;
            return false;
        }
        return true;
    }

    private boolean tryRemove(long checkpointId) throws Exception {
        return this.checkpointStateHandleStore.releaseAndTryRemove(this.completedCheckpointStoreUtil.checkpointIDToName(checkpointId));
    }

    private CompletedCheckpoint retrieveCompletedCheckpoint(Tuple2<RetrievableStateHandle<CompletedCheckpoint>, String> stateHandle) throws FlinkException {
        long checkpointId = this.completedCheckpointStoreUtil.nameToCheckpointID((String)stateHandle.f1);
        LOG.info("Trying to retrieve checkpoint {}.", (Object)checkpointId);
        try {
            return (CompletedCheckpoint)((RetrievableStateHandle)stateHandle.f0).retrieveState();
        }
        catch (ClassNotFoundException cnfe) {
            throw new FlinkException("Could not retrieve checkpoint " + checkpointId + " from state handle under " + (String)stateHandle.f1 + ". This indicates that you are trying to recover from state written by an older Flink version which is not compatible. Try cleaning the state handle store.", (Throwable)cnfe);
        }
        catch (IOException ioe) {
            throw new FlinkException("Could not retrieve checkpoint " + checkpointId + " from state handle under " + (String)stateHandle.f1 + ". This indicates that the retrieved state handle is broken. Try cleaning the state handle store.", (Throwable)ioe);
        }
    }
}

