/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.runtime.resourcemanager.slotmanager;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import org.apache.flink.annotation.VisibleForTesting;
import org.apache.flink.api.common.JobID;
import org.apache.flink.api.common.time.Time;
import org.apache.flink.runtime.clusterframework.types.AllocationID;
import org.apache.flink.runtime.clusterframework.types.ResourceProfile;
import org.apache.flink.runtime.clusterframework.types.SlotID;
import org.apache.flink.runtime.clusterframework.types.TaskManagerSlot;
import org.apache.flink.runtime.concurrent.ScheduledExecutor;
import org.apache.flink.runtime.instance.InstanceID;
import org.apache.flink.runtime.messages.Acknowledge;
import org.apache.flink.runtime.resourcemanager.ResourceManagerId;
import org.apache.flink.runtime.resourcemanager.SlotRequest;
import org.apache.flink.runtime.resourcemanager.exceptions.ResourceManagerException;
import org.apache.flink.runtime.resourcemanager.registration.TaskExecutorConnection;
import org.apache.flink.runtime.resourcemanager.slotmanager.PendingSlotRequest;
import org.apache.flink.runtime.resourcemanager.slotmanager.PendingTaskManagerSlot;
import org.apache.flink.runtime.resourcemanager.slotmanager.ResourceActions;
import org.apache.flink.runtime.resourcemanager.slotmanager.SlotManagerException;
import org.apache.flink.runtime.resourcemanager.slotmanager.TaskManagerRegistration;
import org.apache.flink.runtime.resourcemanager.slotmanager.TaskManagerSlotId;
import org.apache.flink.runtime.taskexecutor.SlotReport;
import org.apache.flink.runtime.taskexecutor.SlotStatus;
import org.apache.flink.runtime.taskexecutor.TaskExecutorGateway;
import org.apache.flink.runtime.taskexecutor.exceptions.SlotAllocationException;
import org.apache.flink.runtime.taskexecutor.exceptions.SlotOccupiedException;
import org.apache.flink.util.FlinkException;
import org.apache.flink.util.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SlotManager
implements AutoCloseable {
    private static final Logger LOG = LoggerFactory.getLogger(SlotManager.class);
    private final ScheduledExecutor scheduledExecutor;
    private final Time taskManagerRequestTimeout;
    private final Time slotRequestTimeout;
    private final Time taskManagerTimeout;
    private final HashMap<SlotID, TaskManagerSlot> slots;
    private final LinkedHashMap<SlotID, TaskManagerSlot> freeSlots;
    private final HashMap<InstanceID, TaskManagerRegistration> taskManagerRegistrations;
    private final HashMap<AllocationID, SlotID> fulfilledSlotRequests;
    private final HashMap<AllocationID, PendingSlotRequest> pendingSlotRequests;
    private final HashMap<TaskManagerSlotId, PendingTaskManagerSlot> pendingSlots;
    private ResourceManagerId resourceManagerId;
    private Executor mainThreadExecutor;
    private ResourceActions resourceActions;
    private ScheduledFuture<?> taskManagerTimeoutCheck;
    private ScheduledFuture<?> slotRequestTimeoutCheck;
    private boolean started;
    private final boolean waitResultConsumedBeforeRelease;

    public SlotManager(ScheduledExecutor scheduledExecutor, Time taskManagerRequestTimeout, Time slotRequestTimeout, Time taskManagerTimeout, boolean waitResultConsumedBeforeRelease) {
        this.scheduledExecutor = (ScheduledExecutor)Preconditions.checkNotNull((Object)scheduledExecutor);
        this.taskManagerRequestTimeout = (Time)Preconditions.checkNotNull((Object)taskManagerRequestTimeout);
        this.slotRequestTimeout = (Time)Preconditions.checkNotNull((Object)slotRequestTimeout);
        this.taskManagerTimeout = (Time)Preconditions.checkNotNull((Object)taskManagerTimeout);
        this.waitResultConsumedBeforeRelease = waitResultConsumedBeforeRelease;
        this.slots = new HashMap(16);
        this.freeSlots = new LinkedHashMap(16);
        this.taskManagerRegistrations = new HashMap(4);
        this.fulfilledSlotRequests = new HashMap(16);
        this.pendingSlotRequests = new HashMap(16);
        this.pendingSlots = new HashMap(16);
        this.resourceManagerId = null;
        this.resourceActions = null;
        this.mainThreadExecutor = null;
        this.taskManagerTimeoutCheck = null;
        this.slotRequestTimeoutCheck = null;
        this.started = false;
    }

    public int getNumberRegisteredSlots() {
        return this.slots.size();
    }

    public int getNumberRegisteredSlotsOf(InstanceID instanceId) {
        TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)instanceId);
        if (taskManagerRegistration != null) {
            return taskManagerRegistration.getNumberRegisteredSlots();
        }
        return 0;
    }

    public int getNumberFreeSlots() {
        return this.freeSlots.size();
    }

    public int getNumberFreeSlotsOf(InstanceID instanceId) {
        TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)instanceId);
        if (taskManagerRegistration != null) {
            return taskManagerRegistration.getNumberFreeSlots();
        }
        return 0;
    }

    public int getNumberPendingTaskManagerSlots() {
        return this.pendingSlots.size();
    }

    @VisibleForTesting
    int getNumberAssignedPendingTaskManagerSlots() {
        return (int)this.pendingSlots.values().stream().filter(slot -> slot.getAssignedPendingSlotRequest() != null).count();
    }

    public void start(ResourceManagerId newResourceManagerId, Executor newMainThreadExecutor, ResourceActions newResourceActions) {
        LOG.info("Starting the SlotManager.");
        this.resourceManagerId = (ResourceManagerId)((Object)Preconditions.checkNotNull((Object)((Object)newResourceManagerId)));
        this.mainThreadExecutor = (Executor)Preconditions.checkNotNull((Object)newMainThreadExecutor);
        this.resourceActions = (ResourceActions)Preconditions.checkNotNull((Object)newResourceActions);
        this.started = true;
        this.taskManagerTimeoutCheck = this.scheduledExecutor.scheduleWithFixedDelay(() -> this.mainThreadExecutor.execute(() -> this.checkTaskManagerTimeouts()), 0L, this.taskManagerTimeout.toMilliseconds(), TimeUnit.MILLISECONDS);
        this.slotRequestTimeoutCheck = this.scheduledExecutor.scheduleWithFixedDelay(() -> this.mainThreadExecutor.execute(() -> this.checkSlotRequestTimeouts()), 0L, this.slotRequestTimeout.toMilliseconds(), TimeUnit.MILLISECONDS);
    }

    public void suspend() {
        LOG.info("Suspending the SlotManager.");
        if (this.taskManagerTimeoutCheck != null) {
            this.taskManagerTimeoutCheck.cancel(false);
            this.taskManagerTimeoutCheck = null;
        }
        if (this.slotRequestTimeoutCheck != null) {
            this.slotRequestTimeoutCheck.cancel(false);
            this.slotRequestTimeoutCheck = null;
        }
        for (PendingSlotRequest pendingSlotRequest : this.pendingSlotRequests.values()) {
            this.cancelPendingSlotRequest(pendingSlotRequest);
        }
        this.pendingSlotRequests.clear();
        ArrayList<InstanceID> registeredTaskManagers = new ArrayList<InstanceID>(this.taskManagerRegistrations.keySet());
        for (InstanceID registeredTaskManager : registeredTaskManagers) {
            this.unregisterTaskManager(registeredTaskManager);
        }
        this.resourceManagerId = null;
        this.resourceActions = null;
        this.started = false;
    }

    @Override
    public void close() throws Exception {
        LOG.info("Closing the SlotManager.");
        this.suspend();
    }

    public boolean registerSlotRequest(SlotRequest slotRequest) throws SlotManagerException {
        this.checkInit();
        if (this.checkDuplicateRequest(slotRequest.getAllocationId())) {
            LOG.debug("Ignoring a duplicate slot request with allocation id {}.", (Object)slotRequest.getAllocationId());
            return false;
        }
        PendingSlotRequest pendingSlotRequest = new PendingSlotRequest(slotRequest);
        this.pendingSlotRequests.put(slotRequest.getAllocationId(), pendingSlotRequest);
        try {
            this.internalRequestSlot(pendingSlotRequest);
        }
        catch (ResourceManagerException e) {
            this.pendingSlotRequests.remove((Object)slotRequest.getAllocationId());
            throw new SlotManagerException("Could not fulfill slot request " + (Object)((Object)slotRequest.getAllocationId()) + '.', e);
        }
        return true;
    }

    public boolean unregisterSlotRequest(AllocationID allocationId) {
        this.checkInit();
        PendingSlotRequest pendingSlotRequest = this.pendingSlotRequests.remove((Object)allocationId);
        if (null != pendingSlotRequest) {
            LOG.debug("Cancel slot request {}.", (Object)allocationId);
            this.cancelPendingSlotRequest(pendingSlotRequest);
            return true;
        }
        LOG.debug("No pending slot request with allocation id {} found. Ignoring unregistration request.", (Object)allocationId);
        return false;
    }

    public void registerTaskManager(TaskExecutorConnection taskExecutorConnection, SlotReport initialSlotReport) {
        this.checkInit();
        LOG.debug("Registering TaskManager {} under {} at the SlotManager.", (Object)taskExecutorConnection.getResourceID(), (Object)taskExecutorConnection.getInstanceID());
        if (this.taskManagerRegistrations.containsKey((Object)taskExecutorConnection.getInstanceID())) {
            this.reportSlotStatus(taskExecutorConnection.getInstanceID(), initialSlotReport);
        } else {
            ArrayList<SlotID> reportedSlots = new ArrayList<SlotID>();
            for (SlotStatus slotStatus : initialSlotReport) {
                reportedSlots.add(slotStatus.getSlotID());
            }
            TaskManagerRegistration taskManagerRegistration = new TaskManagerRegistration(taskExecutorConnection, reportedSlots);
            this.taskManagerRegistrations.put(taskExecutorConnection.getInstanceID(), taskManagerRegistration);
            for (SlotStatus slotStatus : initialSlotReport) {
                this.registerSlot(slotStatus.getSlotID(), slotStatus.getAllocationID(), slotStatus.getJobID(), slotStatus.getResourceProfile(), taskExecutorConnection);
            }
        }
    }

    public boolean unregisterTaskManager(InstanceID instanceId) {
        this.checkInit();
        LOG.debug("Unregister TaskManager {} from the SlotManager.", (Object)instanceId);
        TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.remove((Object)instanceId);
        if (null != taskManagerRegistration) {
            this.internalUnregisterTaskManager(taskManagerRegistration);
            return true;
        }
        LOG.debug("There is no task manager registered with instance ID {}. Ignoring this message.", (Object)instanceId);
        return false;
    }

    public boolean reportSlotStatus(InstanceID instanceId, SlotReport slotReport) {
        this.checkInit();
        LOG.debug("Received slot report from instance {}: {}.", (Object)instanceId, (Object)slotReport);
        TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)instanceId);
        if (null != taskManagerRegistration) {
            for (SlotStatus slotStatus : slotReport) {
                this.updateSlot(slotStatus.getSlotID(), slotStatus.getAllocationID(), slotStatus.getJobID());
            }
            return true;
        }
        LOG.debug("Received slot report for unknown task manager with instance id {}. Ignoring this report.", (Object)instanceId);
        return false;
    }

    public void freeSlot(SlotID slotId, AllocationID allocationId) {
        this.checkInit();
        TaskManagerSlot slot = this.slots.get(slotId);
        if (null != slot) {
            if (slot.getState() == TaskManagerSlot.State.ALLOCATED) {
                if (Objects.equals((Object)allocationId, (Object)slot.getAllocationId())) {
                    TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)slot.getInstanceId());
                    if (taskManagerRegistration == null) {
                        throw new IllegalStateException("Trying to free a slot from a TaskManager " + (Object)((Object)slot.getInstanceId()) + " which has not been registered.");
                    }
                    this.updateSlotState(slot, taskManagerRegistration, null, null);
                } else {
                    LOG.debug("Received request to free slot {} with expected allocation id {}, but actual allocation id {} differs. Ignoring the request.", new Object[]{slotId, allocationId, slot.getAllocationId()});
                }
            } else {
                LOG.debug("Slot {} has not been allocated.", (Object)allocationId);
            }
        } else {
            LOG.debug("Trying to free a slot {} which has not been registered. Ignoring this message.", (Object)slotId);
        }
    }

    protected PendingSlotRequest findMatchingRequest(ResourceProfile slotResourceProfile) {
        for (PendingSlotRequest pendingSlotRequest : this.pendingSlotRequests.values()) {
            if (pendingSlotRequest.isAssigned() || !slotResourceProfile.isMatching(pendingSlotRequest.getResourceProfile())) continue;
            return pendingSlotRequest;
        }
        return null;
    }

    protected TaskManagerSlot findMatchingSlot(ResourceProfile requestResourceProfile) {
        Iterator<Map.Entry<SlotID, TaskManagerSlot>> iterator = this.freeSlots.entrySet().iterator();
        while (iterator.hasNext()) {
            TaskManagerSlot taskManagerSlot = iterator.next().getValue();
            Preconditions.checkState((taskManagerSlot.getState() == TaskManagerSlot.State.FREE ? 1 : 0) != 0, (String)"TaskManagerSlot %s is not in state FREE but %s.", (Object[])new Object[]{taskManagerSlot.getSlotId(), taskManagerSlot.getState()});
            if (!taskManagerSlot.getResourceProfile().isMatching(requestResourceProfile)) continue;
            iterator.remove();
            return taskManagerSlot;
        }
        return null;
    }

    private void registerSlot(SlotID slotId, AllocationID allocationId, JobID jobId, ResourceProfile resourceProfile, TaskExecutorConnection taskManagerConnection) {
        if (this.slots.containsKey(slotId)) {
            this.removeSlot(slotId);
        }
        TaskManagerSlot slot = this.createAndRegisterTaskManagerSlot(slotId, resourceProfile, taskManagerConnection);
        PendingTaskManagerSlot pendingTaskManagerSlot = allocationId == null ? this.findExactlyMatchingPendingTaskManagerSlot(resourceProfile) : null;
        if (pendingTaskManagerSlot == null) {
            this.updateSlot(slotId, allocationId, jobId);
        } else {
            this.pendingSlots.remove((Object)pendingTaskManagerSlot.getTaskManagerSlotId());
            PendingSlotRequest assignedPendingSlotRequest = pendingTaskManagerSlot.getAssignedPendingSlotRequest();
            if (assignedPendingSlotRequest == null) {
                this.handleFreeSlot(slot);
            } else {
                assignedPendingSlotRequest.unassignPendingTaskManagerSlot();
                this.allocateSlot(slot, assignedPendingSlotRequest);
            }
        }
    }

    @Nonnull
    private TaskManagerSlot createAndRegisterTaskManagerSlot(SlotID slotId, ResourceProfile resourceProfile, TaskExecutorConnection taskManagerConnection) {
        TaskManagerSlot slot = new TaskManagerSlot(slotId, resourceProfile, taskManagerConnection);
        this.slots.put(slotId, slot);
        return slot;
    }

    @Nullable
    private PendingTaskManagerSlot findExactlyMatchingPendingTaskManagerSlot(ResourceProfile resourceProfile) {
        for (PendingTaskManagerSlot pendingTaskManagerSlot : this.pendingSlots.values()) {
            if (!pendingTaskManagerSlot.getResourceProfile().equals(resourceProfile)) continue;
            return pendingTaskManagerSlot;
        }
        return null;
    }

    private boolean updateSlot(SlotID slotId, AllocationID allocationId, JobID jobId) {
        TaskManagerSlot slot = this.slots.get(slotId);
        if (slot != null) {
            TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)slot.getInstanceId());
            if (taskManagerRegistration != null) {
                this.updateSlotState(slot, taskManagerRegistration, allocationId, jobId);
                return true;
            }
            throw new IllegalStateException("Trying to update a slot from a TaskManager " + (Object)((Object)slot.getInstanceId()) + " which has not been registered.");
        }
        LOG.debug("Trying to update unknown slot with slot id {}.", (Object)slotId);
        return false;
    }

    private void updateSlotState(TaskManagerSlot slot, TaskManagerRegistration taskManagerRegistration, @Nullable AllocationID allocationId, @Nullable JobID jobId) {
        if (null != allocationId) {
            switch (slot.getState()) {
                case PENDING: {
                    PendingSlotRequest pendingSlotRequest = slot.getAssignedSlotRequest();
                    if (Objects.equals((Object)pendingSlotRequest.getAllocationId(), (Object)allocationId)) {
                        this.cancelPendingSlotRequest(pendingSlotRequest);
                        this.pendingSlotRequests.remove((Object)pendingSlotRequest.getAllocationId());
                        slot.completeAllocation(allocationId, jobId);
                    } else {
                        slot.clearPendingSlotRequest();
                        slot.updateAllocation(allocationId, jobId);
                        PendingSlotRequest actualPendingSlotRequest = this.pendingSlotRequests.remove((Object)allocationId);
                        if (actualPendingSlotRequest != null) {
                            this.cancelPendingSlotRequest(actualPendingSlotRequest);
                        }
                        this.rejectPendingSlotRequest(pendingSlotRequest, new Exception("Task manager reported slot " + slot.getSlotId() + " being already allocated."));
                    }
                    taskManagerRegistration.occupySlot();
                    break;
                }
                case ALLOCATED: {
                    if (Objects.equals((Object)allocationId, (Object)slot.getAllocationId())) break;
                    slot.freeSlot();
                    slot.updateAllocation(allocationId, jobId);
                    break;
                }
                case FREE: {
                    this.freeSlots.remove(slot.getSlotId());
                    slot.updateAllocation(allocationId, jobId);
                    taskManagerRegistration.occupySlot();
                }
            }
            this.fulfilledSlotRequests.put(allocationId, slot.getSlotId());
        } else {
            switch (slot.getState()) {
                case FREE: {
                    this.handleFreeSlot(slot);
                    break;
                }
                case PENDING: {
                    break;
                }
                case ALLOCATED: {
                    AllocationID oldAllocation = slot.getAllocationId();
                    slot.freeSlot();
                    this.fulfilledSlotRequests.remove((Object)oldAllocation);
                    taskManagerRegistration.freeSlot();
                    this.handleFreeSlot(slot);
                }
            }
        }
    }

    private void internalRequestSlot(PendingSlotRequest pendingSlotRequest) throws ResourceManagerException {
        ResourceProfile resourceProfile = pendingSlotRequest.getResourceProfile();
        TaskManagerSlot taskManagerSlot = this.findMatchingSlot(resourceProfile);
        if (taskManagerSlot != null) {
            this.allocateSlot(taskManagerSlot, pendingSlotRequest);
        } else {
            Optional<PendingTaskManagerSlot> pendingTaskManagerSlotOptional = this.findFreeMatchingPendingTaskManagerSlot(resourceProfile);
            if (!pendingTaskManagerSlotOptional.isPresent()) {
                pendingTaskManagerSlotOptional = this.allocateResource(resourceProfile);
            }
            pendingTaskManagerSlotOptional.ifPresent(pendingTaskManagerSlot -> this.assignPendingTaskManagerSlot(pendingSlotRequest, (PendingTaskManagerSlot)pendingTaskManagerSlot));
        }
    }

    private Optional<PendingTaskManagerSlot> findFreeMatchingPendingTaskManagerSlot(ResourceProfile requiredResourceProfile) {
        for (PendingTaskManagerSlot pendingTaskManagerSlot : this.pendingSlots.values()) {
            if (pendingTaskManagerSlot.getAssignedPendingSlotRequest() != null || !pendingTaskManagerSlot.getResourceProfile().isMatching(requiredResourceProfile)) continue;
            return Optional.of(pendingTaskManagerSlot);
        }
        return Optional.empty();
    }

    private Optional<PendingTaskManagerSlot> allocateResource(ResourceProfile resourceProfile) throws ResourceManagerException {
        Collection<ResourceProfile> requestedSlots = this.resourceActions.allocateResource(resourceProfile);
        if (requestedSlots.isEmpty()) {
            return Optional.empty();
        }
        Iterator<ResourceProfile> slotIterator = requestedSlots.iterator();
        PendingTaskManagerSlot pendingTaskManagerSlot = new PendingTaskManagerSlot(slotIterator.next());
        this.pendingSlots.put(pendingTaskManagerSlot.getTaskManagerSlotId(), pendingTaskManagerSlot);
        while (slotIterator.hasNext()) {
            PendingTaskManagerSlot additionalPendingTaskManagerSlot = new PendingTaskManagerSlot(slotIterator.next());
            this.pendingSlots.put(additionalPendingTaskManagerSlot.getTaskManagerSlotId(), additionalPendingTaskManagerSlot);
        }
        return Optional.of(pendingTaskManagerSlot);
    }

    private void assignPendingTaskManagerSlot(PendingSlotRequest pendingSlotRequest, PendingTaskManagerSlot pendingTaskManagerSlot) {
        pendingTaskManagerSlot.assignPendingSlotRequest(pendingSlotRequest);
        pendingSlotRequest.assignPendingTaskManagerSlot(pendingTaskManagerSlot);
    }

    private void allocateSlot(TaskManagerSlot taskManagerSlot, PendingSlotRequest pendingSlotRequest) {
        Preconditions.checkState((taskManagerSlot.getState() == TaskManagerSlot.State.FREE ? 1 : 0) != 0);
        TaskExecutorConnection taskExecutorConnection = taskManagerSlot.getTaskManagerConnection();
        TaskExecutorGateway gateway = taskExecutorConnection.getTaskExecutorGateway();
        CompletableFuture<Acknowledge> completableFuture = new CompletableFuture<Acknowledge>();
        AllocationID allocationId = pendingSlotRequest.getAllocationId();
        SlotID slotId = taskManagerSlot.getSlotId();
        InstanceID instanceID = taskManagerSlot.getInstanceId();
        taskManagerSlot.assignPendingSlotRequest(pendingSlotRequest);
        pendingSlotRequest.setRequestFuture(completableFuture);
        this.returnPendingTaskManagerSlotIfAssigned(pendingSlotRequest);
        TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)instanceID);
        if (taskManagerRegistration == null) {
            throw new IllegalStateException("Could not find a registered task manager for instance id " + (Object)((Object)instanceID) + '.');
        }
        taskManagerRegistration.markUsed();
        CompletableFuture<Acknowledge> requestFuture = gateway.requestSlot(slotId, pendingSlotRequest.getJobId(), allocationId, pendingSlotRequest.getTargetAddress(), this.resourceManagerId, this.taskManagerRequestTimeout);
        requestFuture.whenComplete((acknowledge, throwable) -> {
            if (acknowledge != null) {
                completableFuture.complete((Acknowledge)acknowledge);
            } else {
                completableFuture.completeExceptionally((Throwable)throwable);
            }
        });
        completableFuture.whenCompleteAsync((acknowledge, throwable) -> {
            try {
                if (acknowledge != null) {
                    this.updateSlot(slotId, allocationId, pendingSlotRequest.getJobId());
                } else {
                    if (throwable instanceof SlotOccupiedException) {
                        SlotOccupiedException exception = (SlotOccupiedException)throwable;
                        this.updateSlot(slotId, exception.getAllocationId(), exception.getJobId());
                    } else {
                        this.removeSlotRequestFromSlot(slotId, allocationId);
                    }
                    if (!(throwable instanceof CancellationException)) {
                        this.handleFailedSlotRequest(slotId, allocationId, (Throwable)throwable);
                    } else {
                        LOG.debug("Slot allocation request {} has been cancelled.", (Object)allocationId, throwable);
                    }
                }
            }
            catch (Exception e) {
                LOG.error("Error while completing the slot allocation.", (Throwable)e);
            }
        }, this.mainThreadExecutor);
    }

    private void returnPendingTaskManagerSlotIfAssigned(PendingSlotRequest pendingSlotRequest) {
        PendingTaskManagerSlot pendingTaskManagerSlot = pendingSlotRequest.getAssignedPendingTaskManagerSlot();
        if (pendingTaskManagerSlot != null) {
            pendingTaskManagerSlot.unassignPendingSlotRequest();
            pendingSlotRequest.unassignPendingTaskManagerSlot();
        }
    }

    private void handleFreeSlot(TaskManagerSlot freeSlot) {
        Preconditions.checkState((freeSlot.getState() == TaskManagerSlot.State.FREE ? 1 : 0) != 0);
        PendingSlotRequest pendingSlotRequest = this.findMatchingRequest(freeSlot.getResourceProfile());
        if (null != pendingSlotRequest) {
            this.allocateSlot(freeSlot, pendingSlotRequest);
        } else {
            this.freeSlots.put(freeSlot.getSlotId(), freeSlot);
        }
    }

    private void removeSlots(Iterable<SlotID> slotsToRemove) {
        for (SlotID slotId : slotsToRemove) {
            this.removeSlot(slotId);
        }
    }

    private void removeSlot(SlotID slotId) {
        TaskManagerSlot slot = this.slots.remove(slotId);
        if (null != slot) {
            AllocationID oldAllocationId;
            this.freeSlots.remove(slotId);
            if (slot.getState() == TaskManagerSlot.State.PENDING) {
                this.rejectPendingSlotRequest(slot.getAssignedSlotRequest(), new Exception("The assigned slot " + slot.getSlotId() + " was removed."));
            }
            if ((oldAllocationId = slot.getAllocationId()) != null) {
                this.fulfilledSlotRequests.remove((Object)oldAllocationId);
                this.resourceActions.notifyAllocationFailure(slot.getJobId(), oldAllocationId, (Exception)((Object)new FlinkException("The assigned slot " + slot.getSlotId() + " was removed.")));
            }
        } else {
            LOG.debug("There was no slot registered with slot id {}.", (Object)slotId);
        }
    }

    private void removeSlotRequestFromSlot(SlotID slotId, AllocationID allocationId) {
        TaskManagerSlot taskManagerSlot = this.slots.get(slotId);
        if (null != taskManagerSlot) {
            if (taskManagerSlot.getState() == TaskManagerSlot.State.PENDING && Objects.equals((Object)allocationId, (Object)taskManagerSlot.getAssignedSlotRequest().getAllocationId())) {
                TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)taskManagerSlot.getInstanceId());
                if (taskManagerRegistration == null) {
                    throw new IllegalStateException("Trying to remove slot request from slot for which there is no TaskManager " + (Object)((Object)taskManagerSlot.getInstanceId()) + " is registered.");
                }
                taskManagerSlot.clearPendingSlotRequest();
                this.updateSlotState(taskManagerSlot, taskManagerRegistration, null, null);
            } else {
                LOG.debug("Ignore slot request removal for slot {}.", (Object)slotId);
            }
        } else {
            LOG.debug("There was no slot with {} registered. Probably this slot has been already freed.", (Object)slotId);
        }
    }

    private void handleFailedSlotRequest(SlotID slotId, AllocationID allocationId, Throwable cause) {
        PendingSlotRequest pendingSlotRequest = this.pendingSlotRequests.get((Object)allocationId);
        LOG.debug("Slot request with allocation id {} failed for slot {}.", new Object[]{allocationId, slotId, cause});
        if (null != pendingSlotRequest) {
            pendingSlotRequest.setRequestFuture(null);
            try {
                this.internalRequestSlot(pendingSlotRequest);
            }
            catch (ResourceManagerException e) {
                this.pendingSlotRequests.remove((Object)allocationId);
                this.resourceActions.notifyAllocationFailure(pendingSlotRequest.getJobId(), allocationId, e);
            }
        } else {
            LOG.debug("There was not pending slot request with allocation id {}. Probably the request has been fulfilled or cancelled.", (Object)allocationId);
        }
    }

    private void rejectPendingSlotRequest(PendingSlotRequest pendingSlotRequest, Exception cause) {
        CompletableFuture<Acknowledge> request = pendingSlotRequest.getRequestFuture();
        if (null != request) {
            request.completeExceptionally(new SlotAllocationException(cause));
        } else {
            LOG.debug("Cannot reject pending slot request {}, since no request has been sent.", (Object)pendingSlotRequest.getAllocationId());
        }
    }

    private void cancelPendingSlotRequest(PendingSlotRequest pendingSlotRequest) {
        CompletableFuture<Acknowledge> request = pendingSlotRequest.getRequestFuture();
        this.returnPendingTaskManagerSlotIfAssigned(pendingSlotRequest);
        if (null != request) {
            request.cancel(false);
        }
    }

    @VisibleForTesting
    void checkTaskManagerTimeouts() {
        if (!this.taskManagerRegistrations.isEmpty()) {
            long currentTime = System.currentTimeMillis();
            ArrayList<TaskManagerRegistration> timedOutTaskManagers = new ArrayList<TaskManagerRegistration>(this.taskManagerRegistrations.size());
            for (TaskManagerRegistration taskManagerRegistration : this.taskManagerRegistrations.values()) {
                if (currentTime - taskManagerRegistration.getIdleSince() < this.taskManagerTimeout.toMilliseconds()) continue;
                timedOutTaskManagers.add(taskManagerRegistration);
            }
            for (TaskManagerRegistration taskManagerRegistration : timedOutTaskManagers) {
                if (this.waitResultConsumedBeforeRelease) {
                    this.releaseTaskExecutorIfPossible(taskManagerRegistration);
                    continue;
                }
                this.releaseTaskExecutor(taskManagerRegistration.getInstanceId());
            }
        }
    }

    private void releaseTaskExecutorIfPossible(TaskManagerRegistration taskManagerRegistration) {
        long idleSince = taskManagerRegistration.getIdleSince();
        taskManagerRegistration.getTaskManagerConnection().getTaskExecutorGateway().canBeReleased().thenAcceptAsync(canBeReleased -> {
            boolean stillIdle;
            InstanceID timedOutTaskManagerId = taskManagerRegistration.getInstanceId();
            boolean bl = stillIdle = idleSince == taskManagerRegistration.getIdleSince();
            if (stillIdle && canBeReleased.booleanValue()) {
                this.releaseTaskExecutor(timedOutTaskManagerId);
            }
        }, this.mainThreadExecutor);
    }

    private void releaseTaskExecutor(InstanceID timedOutTaskManagerId) {
        FlinkException cause = new FlinkException("TaskExecutor exceeded the idle timeout.");
        LOG.debug("Release TaskExecutor {} because it exceeded the idle timeout.", (Object)timedOutTaskManagerId);
        this.resourceActions.releaseResource(timedOutTaskManagerId, (Exception)((Object)cause));
    }

    private void checkSlotRequestTimeouts() {
        if (!this.pendingSlotRequests.isEmpty()) {
            long currentTime = System.currentTimeMillis();
            Iterator<Map.Entry<AllocationID, PendingSlotRequest>> slotRequestIterator = this.pendingSlotRequests.entrySet().iterator();
            while (slotRequestIterator.hasNext()) {
                PendingSlotRequest slotRequest = slotRequestIterator.next().getValue();
                if (currentTime - slotRequest.getCreationTimestamp() < this.slotRequestTimeout.toMilliseconds()) continue;
                slotRequestIterator.remove();
                if (slotRequest.isAssigned()) {
                    this.cancelPendingSlotRequest(slotRequest);
                }
                this.resourceActions.notifyAllocationFailure(slotRequest.getJobId(), slotRequest.getAllocationId(), new TimeoutException("The allocation could not be fulfilled in time."));
            }
        }
    }

    private void internalUnregisterTaskManager(TaskManagerRegistration taskManagerRegistration) {
        Preconditions.checkNotNull((Object)taskManagerRegistration);
        this.removeSlots(taskManagerRegistration.getSlots());
    }

    private boolean checkDuplicateRequest(AllocationID allocationId) {
        return this.pendingSlotRequests.containsKey((Object)allocationId) || this.fulfilledSlotRequests.containsKey((Object)allocationId);
    }

    private void checkInit() {
        Preconditions.checkState((boolean)this.started, (Object)"The slot manager has not been started.");
    }

    @VisibleForTesting
    TaskManagerSlot getSlot(SlotID slotId) {
        return this.slots.get(slotId);
    }

    @VisibleForTesting
    PendingSlotRequest getSlotRequest(AllocationID allocationId) {
        return this.pendingSlotRequests.get((Object)allocationId);
    }

    @VisibleForTesting
    boolean isTaskManagerIdle(InstanceID instanceId) {
        TaskManagerRegistration taskManagerRegistration = this.taskManagerRegistrations.get((Object)instanceId);
        if (null != taskManagerRegistration) {
            return taskManagerRegistration.isIdle();
        }
        return false;
    }

    @VisibleForTesting
    public void unregisterTaskManagersAndReleaseResources() {
        Iterator<Map.Entry<InstanceID, TaskManagerRegistration>> taskManagerRegistrationIterator = this.taskManagerRegistrations.entrySet().iterator();
        while (taskManagerRegistrationIterator.hasNext()) {
            TaskManagerRegistration taskManagerRegistration = taskManagerRegistrationIterator.next().getValue();
            taskManagerRegistrationIterator.remove();
            this.internalUnregisterTaskManager(taskManagerRegistration);
            this.resourceActions.releaseResource(taskManagerRegistration.getInstanceId(), (Exception)((Object)new FlinkException("Triggering of SlotManager#unregisterTaskManagersAndReleaseResources.")));
        }
    }
}

