/*
 * Decompiled with CFR 0.152.
 */
package org.apache.heron.api.utils;

import java.io.File;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.PathMatcher;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Logger;
import org.apache.heron.api.Config;
import org.apache.heron.api.exception.InvalidTopologyException;
import org.apache.heron.api.generated.TopologyAPI;
import org.apache.heron.common.basics.ByteAmount;

public final class TopologyUtils {
    private static final Logger LOG = Logger.getLogger(TopologyUtils.class.getName());

    private TopologyUtils() {
    }

    public static TopologyAPI.Topology getTopology(String topologyDefnFile) throws InvalidTopologyException {
        try {
            byte[] topologyDefn = Files.readAllBytes(Paths.get(topologyDefnFile, new String[0]));
            TopologyAPI.Topology topology = TopologyAPI.Topology.parseFrom(topologyDefn);
            TopologyUtils.validateTopology(topology);
            return topology;
        }
        catch (IOException e) {
            throw new RuntimeException("Failed to read/parse content of " + topologyDefnFile, e);
        }
    }

    public static String getConfigWithDefault(List<TopologyAPI.Config.KeyValue> config, String key, String defaultValue) {
        for (TopologyAPI.Config.KeyValue kv : config) {
            if (!kv.getKey().equals(key)) continue;
            return kv.getValue();
        }
        return defaultValue;
    }

    public static Long getConfigWithDefault(List<TopologyAPI.Config.KeyValue> config, String key, Long defaultValue) {
        return Long.parseLong(TopologyUtils.getConfigWithDefault(config, key, Long.toString(defaultValue)));
    }

    public static Integer getConfigWithDefault(List<TopologyAPI.Config.KeyValue> config, String key, Integer defaultValue) {
        return Integer.parseInt(TopologyUtils.getConfigWithDefault(config, key, Integer.toString(defaultValue)));
    }

    public static Double getConfigWithDefault(List<TopologyAPI.Config.KeyValue> config, String key, Double defaultValue) {
        return Double.parseDouble(TopologyUtils.getConfigWithDefault(config, key, Double.toString(defaultValue)));
    }

    public static ByteAmount getConfigWithDefault(List<TopologyAPI.Config.KeyValue> config, String key, ByteAmount defaultValue) {
        long defaultBytes = defaultValue.asBytes();
        return ByteAmount.fromBytes(TopologyUtils.getConfigWithDefault(config, key, defaultBytes));
    }

    public static Boolean getConfigWithDefault(List<TopologyAPI.Config.KeyValue> config, String key, boolean defaultValue) {
        return Boolean.parseBoolean(TopologyUtils.getConfigWithDefault(config, key, Boolean.toString(defaultValue)));
    }

    public static String getConfigWithException(List<TopologyAPI.Config.KeyValue> config, String key) {
        for (TopologyAPI.Config.KeyValue kv : config) {
            if (!kv.getKey().equals(key)) continue;
            return kv.getValue();
        }
        throw new RuntimeException("Missing config for required key " + key);
    }

    public static Map<String, Integer> getComponentParallelism(TopologyAPI.Topology topology) {
        String parallelism;
        String componentName;
        HashMap<String, Integer> parallelismMap = new HashMap<String, Integer>();
        for (TopologyAPI.Spout spout : topology.getSpoutsList()) {
            componentName = spout.getComp().getName();
            parallelism = TopologyUtils.getConfigWithException(spout.getComp().getConfig().getKvsList(), "topology.component.parallelism").trim();
            parallelismMap.put(componentName, Integer.parseInt(parallelism));
        }
        for (TopologyAPI.Bolt bolt : topology.getBoltsList()) {
            componentName = bolt.getComp().getName();
            parallelism = TopologyUtils.getConfigWithException(bolt.getComp().getConfig().getKvsList(), "topology.component.parallelism").trim();
            parallelismMap.put(componentName, Integer.parseInt(parallelism));
        }
        return parallelismMap;
    }

    public static String getInstanceJvmOptions(TopologyAPI.Topology topology) {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        return TopologyUtils.getConfigWithDefault(topologyConfig, "topology.worker.childopts", "");
    }

    public static String getComponentJvmOptions(TopologyAPI.Topology topology) {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        return TopologyUtils.getConfigWithDefault(topologyConfig, "topology.component.jvmopts", "");
    }

    public static int getTotalInstance(Map<String, Integer> parallelismMap) {
        int numInstances = 0;
        for (int parallelism : parallelismMap.values()) {
            numInstances += parallelism;
        }
        return numInstances;
    }

    public static int getTotalInstance(TopologyAPI.Topology topology) {
        Map<String, Integer> parallelismMap = TopologyUtils.getComponentParallelism(topology);
        return TopologyUtils.getTotalInstance(parallelismMap);
    }

    public static boolean shouldStartCkptMgr(TopologyAPI.Topology topology) {
        String mode = TopologyUtils.getConfigWithDefault(topology.getTopologyConfig().getKvsList(), "topology.reliability.mode", "");
        if (mode.isEmpty()) {
            return false;
        }
        return Config.TopologyReliabilityMode.valueOf(mode) == Config.TopologyReliabilityMode.EFFECTIVELY_ONCE;
    }

    public static ByteAmount getCheckpointManagerRam(TopologyAPI.Topology topology) {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        return TopologyUtils.getConfigWithDefault(topologyConfig, "topology.stateful.checkpointmanager.ram", ByteAmount.fromGigabytes(1L));
    }

    public static void validateTopology(TopologyAPI.Topology topology) throws InvalidTopologyException {
        if (!TopologyUtils.verifyTopology(topology)) {
            throw new InvalidTopologyException();
        }
    }

    public static boolean verifyTopology(TopologyAPI.Topology topology) {
        if (!topology.hasName() || topology.getName().isEmpty()) {
            LOG.severe("Missing topology name");
            return false;
        }
        if (topology.getName().contains(".") || topology.getName().contains("/")) {
            LOG.severe("Invalid topology name. Topology name shouldn't have . or /");
            return false;
        }
        TopologyUtils.getComponentRamMapConfig(topology);
        HashSet<String> outputStreams = new HashSet<String>();
        for (TopologyAPI.Spout spout : topology.getSpoutsList()) {
            for (TopologyAPI.OutputStream outputStream : spout.getOutputsList()) {
                outputStreams.add(outputStream.getStream().getComponentName() + "/" + outputStream.getStream().getId());
            }
        }
        for (TopologyAPI.Bolt bolt : topology.getBoltsList()) {
            for (TopologyAPI.OutputStream outputStream : bolt.getOutputsList()) {
                outputStreams.add(outputStream.getStream().getComponentName() + "/" + outputStream.getStream().getId());
            }
        }
        for (TopologyAPI.Bolt bolt : topology.getBoltsList()) {
            for (TopologyAPI.InputStream inputStream : bolt.getInputsList()) {
                String key = inputStream.getStream().getComponentName() + "/" + inputStream.getStream().getId();
                if (outputStreams.contains(key)) continue;
                LOG.severe("Invalid input stream " + key + " existing streams are " + outputStreams);
                return false;
            }
        }
        return true;
    }

    public static String getAdditionalClassPath(TopologyAPI.Topology topology) {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        return TopologyUtils.getConfigWithDefault(topologyConfig, "topology.additional.classpath", "");
    }

    public static Map<String, Double> getComponentCpuMapConfig(TopologyAPI.Topology topology) throws RuntimeException {
        Map<String, String> configMap = TopologyUtils.getComponentConfigMap(topology, "topology.component.cpumap");
        HashMap<String, Double> cpuMap = new HashMap<String, Double>();
        for (Map.Entry<String, String> entry : configMap.entrySet()) {
            Double requiredCpu = Double.parseDouble(entry.getValue());
            cpuMap.put(entry.getKey(), requiredCpu);
        }
        return cpuMap;
    }

    public static Map<String, ByteAmount> getComponentRamMapConfig(TopologyAPI.Topology topology) throws RuntimeException {
        Map<String, String> configMap = TopologyUtils.getComponentConfigMap(topology, "topology.component.rammap");
        HashMap<String, ByteAmount> ramMap = new HashMap<String, ByteAmount>();
        for (Map.Entry<String, String> entry : configMap.entrySet()) {
            long requiredRam = Long.parseLong(entry.getValue());
            ramMap.put(entry.getKey(), ByteAmount.fromBytes(requiredRam));
        }
        return ramMap;
    }

    public static Map<String, ByteAmount> getComponentDiskMapConfig(TopologyAPI.Topology topology) throws RuntimeException {
        Map<String, String> configMap = TopologyUtils.getComponentConfigMap(topology, "topology.component.diskmap");
        HashMap<String, ByteAmount> diskMap = new HashMap<String, ByteAmount>();
        for (Map.Entry<String, String> entry : configMap.entrySet()) {
            long requiredDisk = Long.parseLong(entry.getValue());
            diskMap.put(entry.getKey(), ByteAmount.fromBytes(requiredDisk));
        }
        return diskMap;
    }

    protected static Map<String, String> getComponentConfigMap(TopologyAPI.Topology topology, String key) throws RuntimeException {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        HashMap<String, String> configMap = new HashMap<String, String>();
        Set<String> componentNames = TopologyUtils.getComponentParallelism(topology).keySet();
        String mapStr = TopologyUtils.getConfigWithDefault(topologyConfig, key, (String)null);
        if (mapStr != null) {
            String[] mapTokens;
            for (String token : mapTokens = mapStr.split(",")) {
                if (token.trim().isEmpty()) continue;
                String[] componentAndValue = token.split(":");
                if (componentAndValue.length != 2) {
                    throw new RuntimeException("Malformed component config " + key);
                }
                if (!componentNames.contains(componentAndValue[0])) {
                    throw new RuntimeException("Invalid component. " + componentAndValue[0] + " not found");
                }
                configMap.put(componentAndValue[0], componentAndValue[1]);
            }
        }
        return configMap;
    }

    public static int getNumContainers(TopologyAPI.Topology topology) {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        return Integer.parseInt(TopologyUtils.getConfigWithDefault(topologyConfig, "topology.stmgrs", "1").trim());
    }

    public static String makeClassPath(TopologyAPI.Topology topology, String originalPackageFile) {
        String originalPackage = new File(originalPackageFile).getName();
        StringBuilder classPathBuilder = new StringBuilder();
        if (originalPackage.endsWith(".jar")) {
            classPathBuilder.append(originalPackage);
        } else {
            String topologyJar = originalPackage.replace(".tar.gz", "").replace(".tar", "") + ".jar";
            classPathBuilder.append(String.format("libs/*:%s", topologyJar));
        }
        String additionalClasspath = TopologyUtils.getAdditionalClassPath(topology);
        if (!additionalClasspath.isEmpty()) {
            classPathBuilder.append(":");
            classPathBuilder.append(TopologyUtils.getAdditionalClassPath(topology));
        }
        return classPathBuilder.toString();
    }

    public static String lookUpTopologyDefnFile(String dir, String filename) {
        String pattern = String.format("glob:%s/%s.defn", dir, filename);
        PathMatcher matcher = FileSystems.getDefault().getPathMatcher(pattern);
        for (File file : new File(dir).listFiles()) {
            if (!matcher.matches(file.toPath())) continue;
            return file.getPath();
        }
        throw new IllegalStateException("Failed to find topology defn file");
    }

    public static boolean getTopologyRemoteDebuggingEnabled(TopologyAPI.Topology topology) {
        List<TopologyAPI.Config.KeyValue> topologyConfig = topology.getTopologyConfig().getKvsList();
        return Boolean.parseBoolean(TopologyUtils.getConfigWithDefault(topologyConfig, "topology.remote.debugging.enable", "false"));
    }
}

