/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.processors.kafka;

import java.io.Closeable;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import kafka.producer.Partitioner;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.Producer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.apache.kafka.common.serialization.ByteArraySerializer;
import org.apache.nifi.logging.ProcessorLog;
import org.apache.nifi.processors.kafka.PublishingContext;
import org.apache.nifi.stream.io.util.StreamDemarcator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class KafkaPublisher
implements Closeable {
    private static final Logger logger = LoggerFactory.getLogger(KafkaPublisher.class);
    private final Producer<byte[], byte[]> kafkaProducer;
    private long ackWaitTime = 30000L;
    private ProcessorLog processLog;
    private final Partitioner partitioner;
    private final int ackCheckSize;

    KafkaPublisher(Properties kafkaProperties) {
        this(kafkaProperties, 100);
    }

    KafkaPublisher(Properties kafkaProperties, int ackCheckSize) {
        kafkaProperties.put("value.serializer", ByteArraySerializer.class.getName());
        kafkaProperties.put("key.serializer", ByteArraySerializer.class.getName());
        this.kafkaProducer = new KafkaProducer(kafkaProperties);
        this.ackCheckSize = ackCheckSize;
        try {
            this.partitioner = kafkaProperties.containsKey("partitioner.class") ? (Partitioner)Class.forName(kafkaProperties.getProperty("partitioner.class")).newInstance() : null;
        }
        catch (Exception e) {
            throw new IllegalStateException("Failed to create partitioner", e);
        }
    }

    KafkaPublisherResult publish(PublishingContext publishingContext) {
        byte[] messageBytes;
        StreamDemarcator streamTokenizer = new StreamDemarcator(publishingContext.getContentStream(), publishingContext.getDelimiterBytes(), publishingContext.getMaxRequestSize());
        int prevLastAckedMessageIndex = publishingContext.getLastAckedMessageIndex();
        ArrayList<Future<RecordMetadata>> resultFutures = new ArrayList<Future<RecordMetadata>>();
        int tokenCounter = 0;
        boolean continueSending = true;
        KafkaPublisherResult result = null;
        while (continueSending && (messageBytes = streamTokenizer.nextToken()) != null) {
            if (prevLastAckedMessageIndex < tokenCounter) {
                Integer partitionId = publishingContext.getPartitionId();
                if (partitionId == null && publishingContext.getKeyBytes() != null) {
                    partitionId = this.getPartition(publishingContext.getKeyBytes(), publishingContext.getTopic());
                }
                ProducerRecord message = new ProducerRecord(publishingContext.getTopic(), publishingContext.getPartitionId(), (Object)publishingContext.getKeyBytes(), (Object)messageBytes);
                resultFutures.add(this.kafkaProducer.send(message));
                if (tokenCounter % this.ackCheckSize == 0) {
                    int lastAckedMessageIndex = this.processAcks(resultFutures, prevLastAckedMessageIndex);
                    resultFutures.clear();
                    if (lastAckedMessageIndex % this.ackCheckSize != 0) {
                        continueSending = false;
                        result = new KafkaPublisherResult(tokenCounter, lastAckedMessageIndex);
                    }
                    prevLastAckedMessageIndex = lastAckedMessageIndex;
                }
            }
            ++tokenCounter;
        }
        if (result == null) {
            int lastAckedMessageIndex = this.processAcks(resultFutures, prevLastAckedMessageIndex);
            resultFutures.clear();
            result = new KafkaPublisherResult(tokenCounter, lastAckedMessageIndex);
        }
        return result;
    }

    void setAckWaitTime(long ackWaitTime) {
        this.ackWaitTime = ackWaitTime;
    }

    private int processAcks(List<Future<RecordMetadata>> sendFutures, int lastAckMessageIndex) {
        boolean exceptionThrown = false;
        for (int segmentCounter = 0; segmentCounter < sendFutures.size() && !exceptionThrown; ++segmentCounter) {
            Future<RecordMetadata> future = sendFutures.get(segmentCounter);
            try {
                future.get(this.ackWaitTime, TimeUnit.MILLISECONDS);
                ++lastAckMessageIndex;
                continue;
            }
            catch (InterruptedException e) {
                exceptionThrown = true;
                Thread.currentThread().interrupt();
                this.warnOrError("Interrupted while waiting for acks from Kafka", null);
                continue;
            }
            catch (ExecutionException e) {
                exceptionThrown = true;
                this.warnOrError("Failed while waiting for acks from Kafka", e);
                continue;
            }
            catch (TimeoutException e) {
                exceptionThrown = true;
                this.warnOrError("Timed out while waiting for acks from Kafka", null);
            }
        }
        return lastAckMessageIndex;
    }

    @Override
    public void close() {
        this.kafkaProducer.close();
    }

    void setProcessLog(ProcessorLog processLog) {
        this.processLog = processLog;
    }

    private void warnOrError(String message, Exception e) {
        if (e == null) {
            logger.warn(message);
            if (this.processLog != null) {
                this.processLog.warn(message);
            }
        } else {
            logger.error(message, (Throwable)e);
            if (this.processLog != null) {
                this.processLog.error(message, (Throwable)e);
            }
        }
    }

    private int getPartition(Object key, String topicName) {
        if (this.partitioner != null) {
            int partSize = this.kafkaProducer.partitionsFor(topicName).size();
            return this.partitioner.partition(key, partSize);
        }
        return 0;
    }

    static class KafkaPublisherResult {
        private final int messagesSent;
        private final int lastMessageAcked;

        KafkaPublisherResult(int messagesSent, int lastMessageAcked) {
            this.messagesSent = messagesSent;
            this.lastMessageAcked = lastMessageAcked;
        }

        public int getMessagesSent() {
            return this.messagesSent;
        }

        public int getLastMessageAcked() {
            return this.lastMessageAcked;
        }

        public boolean isAllAcked() {
            return this.messagesSent - 1 == this.lastMessageAcked;
        }

        public String toString() {
            return "Sent:" + this.messagesSent + "; Last ACK:" + this.lastMessageAcked;
        }
    }
}

