/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.schema;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.hash.HashFunction;
import com.google.common.hash.Hashing;
import com.google.protobuf.ByteString;
import com.google.protobuf.InvalidProtocolBufferException;
import java.nio.charset.StandardCharsets;
import java.time.Clock;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.stream.Collectors;
import javax.validation.constraints.NotNull;
import org.apache.avro.Schema;
import org.apache.bookkeeper.common.concurrent.FutureUtils;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.mutable.MutableLong;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.service.schema.SchemaCompatibilityCheck;
import org.apache.pulsar.broker.service.schema.SchemaRegistry;
import org.apache.pulsar.broker.service.schema.SchemaRegistryService;
import org.apache.pulsar.broker.service.schema.SchemaRegistryStats;
import org.apache.pulsar.broker.service.schema.exceptions.IncompatibleSchemaException;
import org.apache.pulsar.broker.service.schema.exceptions.NotExistSchemaException;
import org.apache.pulsar.broker.service.schema.exceptions.SchemaException;
import org.apache.pulsar.broker.service.schema.proto.SchemaRegistryFormat;
import org.apache.pulsar.common.policies.data.SchemaCompatibilityStrategy;
import org.apache.pulsar.common.protocol.schema.SchemaData;
import org.apache.pulsar.common.protocol.schema.SchemaHash;
import org.apache.pulsar.common.protocol.schema.SchemaStorage;
import org.apache.pulsar.common.protocol.schema.SchemaVersion;
import org.apache.pulsar.common.protocol.schema.StoredSchema;
import org.apache.pulsar.common.schema.LongSchemaVersion;
import org.apache.pulsar.common.schema.SchemaType;
import org.apache.pulsar.common.util.FutureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SchemaRegistryServiceImpl
implements SchemaRegistryService {
    private static final Logger log = LoggerFactory.getLogger(SchemaRegistryServiceImpl.class);
    private static HashFunction hashFunction = Hashing.sha256();
    private final Map<SchemaType, SchemaCompatibilityCheck> compatibilityChecks;
    private final SchemaStorage schemaStorage;
    private final Clock clock;
    private final SchemaRegistryStats stats;

    @VisibleForTesting
    SchemaRegistryServiceImpl(SchemaStorage schemaStorage, Map<SchemaType, SchemaCompatibilityCheck> compatibilityChecks, Clock clock, PulsarService pulsarService) {
        this.schemaStorage = schemaStorage;
        this.compatibilityChecks = compatibilityChecks;
        this.clock = clock;
        this.stats = new SchemaRegistryStats(pulsarService);
    }

    SchemaRegistryServiceImpl(SchemaStorage schemaStorage, Map<SchemaType, SchemaCompatibilityCheck> compatibilityChecks, PulsarService pulsarService) {
        this(schemaStorage, compatibilityChecks, Clock.systemUTC(), pulsarService);
    }

    @Override
    @NotNull
    public CompletableFuture<SchemaRegistry.SchemaAndMetadata> getSchema(String schemaId) {
        return this.getSchema(schemaId, SchemaVersion.Latest).thenApply(schema -> {
            if (schema != null && schema.schema.isDeleted()) {
                return null;
            }
            return schema;
        });
    }

    @Override
    @NotNull
    public CompletableFuture<SchemaRegistry.SchemaAndMetadata> getSchema(String schemaId, SchemaVersion version) {
        CompletionStage completableFuture;
        long start = this.clock.millis();
        if (version == SchemaVersion.Latest) {
            completableFuture = this.schemaStorage.get(schemaId, version);
        } else {
            long longVersion = ((LongSchemaVersion)version).getVersion();
            completableFuture = ((CompletableFuture)this.trimDeletedSchemaAndGetList(schemaId).thenApply(metadataList -> metadataList.stream().filter(schemaAndMetadata -> ((LongSchemaVersion)schemaAndMetadata.version).getVersion() == longVersion).collect(Collectors.toList()))).thenCompose(metadataList -> {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Meta data list size {}", (Object)schemaId, (Object)(CollectionUtils.isEmpty((Collection)metadataList) ? 0 : metadataList.size()));
                }
                if (CollectionUtils.isNotEmpty((Collection)metadataList)) {
                    return this.schemaStorage.get(schemaId, version);
                }
                return CompletableFuture.completedFuture(null);
            });
        }
        return ((CompletableFuture)completableFuture.thenCompose(stored -> {
            if (Objects.isNull(stored)) {
                return CompletableFuture.completedFuture(null);
            }
            return ((CompletableFuture)Functions.bytesToSchemaInfo(stored.data).thenApply(Functions::schemaInfoToSchema)).thenApply(schema -> new SchemaRegistry.SchemaAndMetadata(schemaId, (SchemaData)schema, stored.version));
        })).whenComplete((v, t) -> {
            long latencyMs = this.clock.millis() - start;
            if (t != null) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Get schema failed", (Object)schemaId);
                }
                this.stats.recordGetFailed(schemaId, latencyMs);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug(null == v ? "[{}] Schema not found" : "[{}] Schema is present", (Object)schemaId);
                }
                this.stats.recordGetLatency(schemaId, latencyMs);
            }
        });
    }

    @Override
    public CompletableFuture<List<CompletableFuture<SchemaRegistry.SchemaAndMetadata>>> getAllSchemas(String schemaId) {
        long start = this.clock.millis();
        return ((CompletableFuture)this.schemaStorage.getAll(schemaId).thenCompose(schemas -> this.convertToSchemaAndMetadata(schemaId, (List<CompletableFuture<StoredSchema>>)schemas))).whenComplete((v, t) -> {
            long latencyMs = this.clock.millis() - start;
            if (t != null) {
                this.stats.recordListFailed(schemaId, latencyMs);
            } else {
                this.stats.recordListLatency(schemaId, latencyMs);
            }
        });
    }

    private CompletableFuture<List<CompletableFuture<SchemaRegistry.SchemaAndMetadata>>> convertToSchemaAndMetadata(String schemaId, List<CompletableFuture<StoredSchema>> schemas) {
        List list = schemas.stream().map(future -> future.thenCompose(stored -> ((CompletableFuture)Functions.bytesToSchemaInfo(stored.data).thenApply(Functions::schemaInfoToSchema)).thenApply(schema -> new SchemaRegistry.SchemaAndMetadata(schemaId, (SchemaData)schema, stored.version)))).collect(Collectors.toList());
        if (log.isDebugEnabled()) {
            log.debug("[{}] {} schemas is found", (Object)schemaId, (Object)list.size());
        }
        return CompletableFuture.completedFuture(list);
    }

    @Override
    @NotNull
    public CompletableFuture<SchemaVersion> putSchemaIfAbsent(String schemaId, SchemaData schema, SchemaCompatibilityStrategy strategy) {
        MutableLong start = new MutableLong(0L);
        CompletableFuture<SchemaVersion> promise = new CompletableFuture<SchemaVersion>();
        this.schemaStorage.put(schemaId, schemasFuture -> ((CompletableFuture)schemasFuture.thenCompose(schemaFutureList -> this.trimDeletedSchemaAndGetList(schemaId, this.convertToSchemaAndMetadata(schemaId, (List<CompletableFuture<StoredSchema>>)schemaFutureList)))).thenCompose(schemaAndMetadataList -> this.getSchemaVersionBySchemaData((List<SchemaRegistry.SchemaAndMetadata>)schemaAndMetadataList, schema).thenCompose(schemaVersion -> {
            if (schemaVersion != null) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Schema is already exists", (Object)schemaId);
                }
                promise.complete((SchemaVersion)schemaVersion);
                return CompletableFuture.completedFuture(null);
            }
            CompletableFuture<Object> checkCompatibilityFuture = new CompletableFuture();
            if (schemaAndMetadataList.size() != 0) {
                checkCompatibilityFuture = SchemaRegistryServiceImpl.isTransitiveStrategy(strategy) ? this.checkCompatibilityWithAll(schemaId, schema, strategy, (List<SchemaRegistry.SchemaAndMetadata>)schemaAndMetadataList) : this.checkCompatibilityWithLatest(schemaId, schema, strategy);
            } else {
                checkCompatibilityFuture.complete(null);
            }
            return checkCompatibilityFuture.thenCompose(v -> {
                byte[] context = hashFunction.hashBytes(schema.getData()).asBytes();
                SchemaRegistryFormat.SchemaInfo info = SchemaRegistryFormat.SchemaInfo.newBuilder().setType(Functions.convertFromDomainType(schema.getType())).setSchema(ByteString.copyFrom((byte[])schema.getData())).setSchemaId(schemaId).setUser(schema.getUser()).setDeleted(false).setTimestamp(this.clock.millis()).addAllProps(Functions.toPairs(schema.getProps())).build();
                start.setValue(this.clock.millis());
                return CompletableFuture.completedFuture(Pair.of((Object)info.toByteArray(), (Object)context));
            });
        }))).whenComplete((v, ex) -> {
            long latencyMs = this.clock.millis() - start.getValue();
            if (ex != null) {
                log.error("[{}] Put schema failed", (Object)schemaId, ex);
                if (start.getValue() != 0L) {
                    this.stats.recordPutFailed(schemaId, latencyMs);
                }
                promise.completeExceptionally((Throwable)ex);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Put schema finished", (Object)schemaId);
                }
                if (v != null) {
                    promise.complete((SchemaVersion)v);
                    if (start.getValue() != 0L) {
                        this.stats.recordPutLatency(schemaId, this.clock.millis() - start.getValue());
                    }
                }
            }
        });
        return promise;
    }

    @Override
    public CompletableFuture<SchemaVersion> deleteSchema(String schemaId, String user, boolean force) {
        long start = this.clock.millis();
        if (force) {
            return this.deleteSchemaStorage(schemaId, true);
        }
        byte[] deletedEntry = this.deleted(schemaId, user).toByteArray();
        return this.schemaStorage.put(schemaId, deletedEntry, new byte[0]).whenComplete((v, t) -> {
            long latencyMs = this.clock.millis() - start;
            if (t != null) {
                log.error("[{}] User {} delete schema failed", (Object)schemaId, (Object)user);
                this.stats.recordDelFailed(schemaId, latencyMs);
            } else {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] User {} delete schema finished", (Object)schemaId, (Object)user);
                }
                this.stats.recordDelLatency(schemaId, latencyMs);
            }
        });
    }

    @Override
    public CompletableFuture<SchemaVersion> deleteSchemaStorage(String schemaId) {
        return this.deleteSchemaStorage(schemaId, false);
    }

    @Override
    public CompletableFuture<SchemaVersion> deleteSchemaStorage(String schemaId, boolean forcefully) {
        long start = this.clock.millis();
        return this.schemaStorage.delete(schemaId, forcefully).whenComplete((v, t) -> {
            long latencyMs = this.clock.millis() - start;
            if (t != null) {
                this.stats.recordDelFailed(schemaId, latencyMs);
                log.error("[{}] Delete schema storage failed", (Object)schemaId);
            } else {
                this.stats.recordDelLatency(schemaId, latencyMs);
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Delete schema storage finished", (Object)schemaId);
                }
            }
        });
    }

    @Override
    public CompletableFuture<Boolean> isCompatible(String schemaId, SchemaData schema, SchemaCompatibilityStrategy strategy) {
        return this.checkCompatible(schemaId, schema, strategy).thenApply(v -> true);
    }

    private static boolean isTransitiveStrategy(SchemaCompatibilityStrategy strategy) {
        return SchemaCompatibilityStrategy.FORWARD_TRANSITIVE.equals((Object)strategy) || SchemaCompatibilityStrategy.BACKWARD_TRANSITIVE.equals((Object)strategy) || SchemaCompatibilityStrategy.FULL_TRANSITIVE.equals((Object)strategy);
    }

    @Override
    public CompletableFuture<Void> checkCompatible(String schemaId, SchemaData schema, SchemaCompatibilityStrategy strategy) {
        switch (strategy) {
            case FORWARD_TRANSITIVE: 
            case BACKWARD_TRANSITIVE: 
            case FULL_TRANSITIVE: {
                return this.checkCompatibilityWithAll(schemaId, schema, strategy);
            }
        }
        return this.checkCompatibilityWithLatest(schemaId, schema, strategy);
    }

    @Override
    public SchemaVersion versionFromBytes(byte[] version) {
        return this.schemaStorage.versionFromBytes(version);
    }

    @Override
    public void close() throws Exception {
        this.schemaStorage.close();
        this.stats.close();
    }

    private SchemaRegistryFormat.SchemaInfo deleted(String schemaId, String user) {
        return SchemaRegistryFormat.SchemaInfo.newBuilder().setSchemaId(schemaId).setType(SchemaRegistryFormat.SchemaInfo.SchemaType.NONE).setSchema(ByteString.EMPTY).setUser(user).setDeleted(true).setTimestamp(this.clock.millis()).build();
    }

    private void checkCompatible(SchemaRegistry.SchemaAndMetadata existingSchema, SchemaData newSchema, SchemaCompatibilityStrategy strategy) throws IncompatibleSchemaException {
        SchemaData existingSchemaData = existingSchema.schema;
        if (newSchema.getType() != existingSchemaData.getType()) {
            throw new IncompatibleSchemaException(String.format("Incompatible schema: exists schema type %s, new schema type %s", existingSchemaData.getType(), newSchema.getType()));
        }
        SchemaHash existingHash = SchemaHash.of((SchemaData)existingSchemaData);
        SchemaHash newHash = SchemaHash.of((SchemaData)newSchema);
        if (!newHash.equals((Object)existingHash)) {
            this.compatibilityChecks.getOrDefault(newSchema.getType(), SchemaCompatibilityCheck.DEFAULT).checkCompatible(existingSchemaData, newSchema, strategy);
        }
    }

    @Override
    public CompletableFuture<Long> findSchemaVersion(String schemaId, SchemaData schemaData) {
        return this.trimDeletedSchemaAndGetList(schemaId).thenCompose(schemaAndMetadataList -> {
            SchemaHash newHash = SchemaHash.of((SchemaData)schemaData);
            for (SchemaRegistry.SchemaAndMetadata schemaAndMetadata : schemaAndMetadataList) {
                if (!newHash.equals((Object)SchemaHash.of((SchemaData)schemaAndMetadata.schema))) continue;
                return CompletableFuture.completedFuture(((LongSchemaVersion)this.schemaStorage.versionFromBytes(schemaAndMetadata.version.bytes())).getVersion());
            }
            return CompletableFuture.completedFuture(-1L);
        });
    }

    @Override
    public CompletableFuture<Void> checkConsumerCompatibility(String schemaId, SchemaData schemaData, SchemaCompatibilityStrategy strategy) {
        if (SchemaCompatibilityStrategy.ALWAYS_COMPATIBLE == strategy) {
            return CompletableFuture.completedFuture(null);
        }
        return this.getSchema(schemaId).thenCompose(existingSchema -> {
            if (existingSchema != null && !existingSchema.schema.isDeleted()) {
                if (strategy == SchemaCompatibilityStrategy.BACKWARD || strategy == SchemaCompatibilityStrategy.FORWARD || strategy == SchemaCompatibilityStrategy.FORWARD_TRANSITIVE || strategy == SchemaCompatibilityStrategy.FULL) {
                    return this.checkCompatibilityWithLatest(schemaId, schemaData, SchemaCompatibilityStrategy.BACKWARD);
                }
                return this.checkCompatibilityWithAll(schemaId, schemaData, strategy);
            }
            return FutureUtil.failedFuture((Throwable)new NotExistSchemaException("Topic does not have schema to check"));
        });
    }

    @Override
    public CompletableFuture<SchemaVersion> getSchemaVersionBySchemaData(List<SchemaRegistry.SchemaAndMetadata> schemaAndMetadataList, SchemaData schemaData) {
        if (schemaAndMetadataList == null || schemaAndMetadataList.size() == 0) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<SchemaVersion> completableFuture = new CompletableFuture<SchemaVersion>();
        if (SchemaRegistryServiceImpl.isUsingAvroSchemaParser(schemaData.getType())) {
            Schema.Parser parser = new Schema.Parser();
            Schema newSchema = parser.parse(new String(schemaData.getData(), StandardCharsets.UTF_8));
            for (SchemaRegistry.SchemaAndMetadata schemaAndMetadata : schemaAndMetadataList) {
                if (SchemaRegistryServiceImpl.isUsingAvroSchemaParser(schemaAndMetadata.schema.getType())) {
                    Schema.Parser existParser = new Schema.Parser();
                    Schema existSchema = existParser.parse(new String(schemaAndMetadata.schema.getData(), StandardCharsets.UTF_8));
                    if (!newSchema.equals((Object)existSchema) || schemaAndMetadata.schema.getType() != schemaData.getType()) continue;
                    SchemaVersion schemaVersion = schemaAndMetadata.version;
                    completableFuture.complete(schemaVersion);
                    return completableFuture;
                }
                if (!Arrays.equals(hashFunction.hashBytes(schemaAndMetadata.schema.getData()).asBytes(), hashFunction.hashBytes(schemaData.getData()).asBytes()) || schemaAndMetadata.schema.getType() != schemaData.getType()) continue;
                SchemaVersion schemaVersion = schemaAndMetadata.version;
                completableFuture.complete(schemaVersion);
                return completableFuture;
            }
        } else {
            for (SchemaRegistry.SchemaAndMetadata schemaAndMetadata : schemaAndMetadataList) {
                if (!Arrays.equals(hashFunction.hashBytes(schemaAndMetadata.schema.getData()).asBytes(), hashFunction.hashBytes(schemaData.getData()).asBytes()) || schemaAndMetadata.schema.getType() != schemaData.getType()) continue;
                SchemaVersion schemaVersion = schemaAndMetadata.version;
                completableFuture.complete(schemaVersion);
                return completableFuture;
            }
        }
        completableFuture.complete(null);
        return completableFuture;
    }

    private CompletableFuture<Void> checkCompatibilityWithLatest(String schemaId, SchemaData schema, SchemaCompatibilityStrategy strategy) {
        if (SchemaCompatibilityStrategy.ALWAYS_COMPATIBLE == strategy) {
            return CompletableFuture.completedFuture(null);
        }
        return this.getSchema(schemaId).thenCompose(existingSchema -> {
            if (existingSchema != null && !existingSchema.schema.isDeleted()) {
                CompletableFuture result = new CompletableFuture();
                result.whenComplete((__, t) -> {
                    if (t != null) {
                        log.error("[{}] Schema is incompatible", (Object)schemaId);
                        this.stats.recordSchemaIncompatible(schemaId);
                    } else {
                        if (log.isDebugEnabled()) {
                            log.debug("[{}] Schema is compatible", (Object)schemaId);
                        }
                        this.stats.recordSchemaCompatible(schemaId);
                    }
                });
                try {
                    this.checkCompatible((SchemaRegistry.SchemaAndMetadata)existingSchema, schema, strategy);
                    result.complete(null);
                }
                catch (IncompatibleSchemaException e) {
                    result.completeExceptionally(e);
                }
                return result;
            }
            return CompletableFuture.completedFuture(null);
        });
    }

    private CompletableFuture<Void> checkCompatibilityWithAll(String schemaId, SchemaData schema, SchemaCompatibilityStrategy strategy) {
        return this.trimDeletedSchemaAndGetList(schemaId).thenCompose(schemaAndMetadataList -> this.checkCompatibilityWithAll(schemaId, schema, strategy, (List<SchemaRegistry.SchemaAndMetadata>)schemaAndMetadataList));
    }

    private CompletableFuture<Void> checkCompatibilityWithAll(String schemaId, SchemaData schema, SchemaCompatibilityStrategy strategy, List<SchemaRegistry.SchemaAndMetadata> schemaAndMetadataList) {
        CompletableFuture<Void> result;
        block8: {
            result = new CompletableFuture<Void>();
            result.whenComplete((v, t) -> {
                if (t != null) {
                    this.stats.recordSchemaIncompatible(schemaId);
                    log.error("[{}] Schema is incompatible, schema type {}", (Object)schemaId, (Object)schema.getType());
                } else {
                    this.stats.recordSchemaCompatible(schemaId);
                    if (log.isDebugEnabled()) {
                        log.debug("[{}] Schema is compatible, schema type {}", (Object)schemaId, (Object)schema.getType());
                    }
                }
            });
            if (strategy == SchemaCompatibilityStrategy.ALWAYS_COMPATIBLE) {
                result.complete(null);
            } else {
                SchemaRegistry.SchemaAndMetadata breakSchema = null;
                for (SchemaRegistry.SchemaAndMetadata schemaAndMetadata2 : schemaAndMetadataList) {
                    if (schemaAndMetadata2.schema.getType() == schema.getType()) continue;
                    breakSchema = schemaAndMetadata2;
                    break;
                }
                if (breakSchema == null) {
                    try {
                        this.compatibilityChecks.getOrDefault(schema.getType(), SchemaCompatibilityCheck.DEFAULT).checkCompatible(schemaAndMetadataList.stream().map(schemaAndMetadata -> schemaAndMetadata.schema).collect(Collectors.toList()), schema, strategy);
                        result.complete(null);
                    }
                    catch (Exception e) {
                        if (e instanceof IncompatibleSchemaException) {
                            result.completeExceptionally(e);
                            break block8;
                        }
                        result.completeExceptionally(new IncompatibleSchemaException(e));
                    }
                } else {
                    result.completeExceptionally(new IncompatibleSchemaException(String.format("Incompatible schema: exists schema type %s, new schema type %s", breakSchema.schema.getType(), schema.getType())));
                }
            }
        }
        return result;
    }

    @Override
    public CompletableFuture<List<SchemaRegistry.SchemaAndMetadata>> trimDeletedSchemaAndGetList(String schemaId) {
        CompletableFuture<List<CompletableFuture<SchemaRegistry.SchemaAndMetadata>>> schemaFutureList = this.getAllSchemas(schemaId);
        return this.trimDeletedSchemaAndGetList(schemaId, schemaFutureList);
    }

    private CompletableFuture<List<SchemaRegistry.SchemaAndMetadata>> trimDeletedSchemaAndGetList(String schemaId, CompletableFuture<List<CompletableFuture<SchemaRegistry.SchemaAndMetadata>>> schemaFutureList) {
        CompletableFuture<List<SchemaRegistry.SchemaAndMetadata>> schemaResult = new CompletableFuture<List<SchemaRegistry.SchemaAndMetadata>>();
        ((CompletableFuture)schemaFutureList.thenCompose(FutureUtils::collect)).handle((schemaList, ex) -> {
            List list;
            List list2 = list = ex != null ? new ArrayList() : schemaList;
            if (ex != null) {
                boolean recoverable;
                Throwable rc = FutureUtil.unwrapCompletionException((Throwable)ex);
                boolean bl = recoverable = !(rc instanceof SchemaException) || ((SchemaException)rc).isRecoverable();
                if (recoverable) {
                    schemaResult.completeExceptionally(rc);
                    return null;
                }
                schemaFutureList.getNow(Collections.emptyList()).forEach(schemaFuture -> {
                    if (!schemaFuture.isCompletedExceptionally()) {
                        list.add(schemaFuture.getNow(null));
                        return;
                    }
                });
                this.trimDeletedSchemaAndGetList(list);
                this.deleteSchemaStorage(schemaId, true).handle((sv, th) -> {
                    log.info("Clean up non-recoverable schema {}. Deletion of schema {} {}", new Object[]{rc.getMessage(), schemaId, th == null ? "successful" : "failed, " + th.getCause().getMessage()});
                    schemaResult.complete(list);
                    return null;
                });
                return null;
            }
            List<SchemaRegistry.SchemaAndMetadata> trimmed = this.trimDeletedSchemaAndGetList(list);
            schemaResult.complete(trimmed);
            return null;
        });
        return schemaResult;
    }

    private List<SchemaRegistry.SchemaAndMetadata> trimDeletedSchemaAndGetList(List<SchemaRegistry.SchemaAndMetadata> list) {
        int lastIndex;
        for (int i = lastIndex = list.size() - 1; i >= 0; --i) {
            if (!list.get((int)i).schema.isDeleted()) continue;
            if (i == lastIndex) {
                return Collections.emptyList();
            }
            return list.subList(i + 1, list.size());
        }
        return list;
    }

    public static boolean isUsingAvroSchemaParser(SchemaType type) {
        switch (type) {
            case AVRO: 
            case JSON: 
            case PROTOBUF: {
                return true;
            }
        }
        return false;
    }

    static interface Functions {
        public static SchemaType convertToDomainType(SchemaRegistryFormat.SchemaInfo.SchemaType type) {
            if (type.getNumber() < 0) {
                return SchemaType.NONE;
            }
            return SchemaType.valueOf((int)(type.getNumber() - 1));
        }

        public static SchemaRegistryFormat.SchemaInfo.SchemaType convertFromDomainType(SchemaType type) {
            if (type.getValue() < 0) {
                return SchemaRegistryFormat.SchemaInfo.SchemaType.NONE;
            }
            return SchemaRegistryFormat.SchemaInfo.SchemaType.valueOf(type.getValue() + 1);
        }

        public static Map<String, String> toMap(List<SchemaRegistryFormat.SchemaInfo.KeyValuePair> pairs) {
            HashMap<String, String> map = new HashMap<String, String>();
            for (SchemaRegistryFormat.SchemaInfo.KeyValuePair pair : pairs) {
                map.put(pair.getKey(), pair.getValue());
            }
            return map;
        }

        public static List<SchemaRegistryFormat.SchemaInfo.KeyValuePair> toPairs(Map<String, String> map) {
            if (Objects.isNull(map)) {
                return Collections.emptyList();
            }
            ArrayList<SchemaRegistryFormat.SchemaInfo.KeyValuePair> pairs = new ArrayList<SchemaRegistryFormat.SchemaInfo.KeyValuePair>(map.size());
            for (Map.Entry<String, String> entry : map.entrySet()) {
                SchemaRegistryFormat.SchemaInfo.KeyValuePair.Builder builder = SchemaRegistryFormat.SchemaInfo.KeyValuePair.newBuilder();
                pairs.add(builder.setKey(entry.getKey()).setValue(entry.getValue()).build());
            }
            return pairs;
        }

        public static SchemaData schemaInfoToSchema(SchemaRegistryFormat.SchemaInfo info) {
            return SchemaData.builder().user(info.getUser()).type(Functions.convertToDomainType(info.getType())).data(info.getSchema().toByteArray()).timestamp(info.getTimestamp()).isDeleted(info.getDeleted()).props(Functions.toMap(info.getPropsList())).build();
        }

        public static CompletableFuture<SchemaRegistryFormat.SchemaInfo> bytesToSchemaInfo(byte[] bytes) {
            CompletableFuture<SchemaRegistryFormat.SchemaInfo> future;
            try {
                future = CompletableFuture.completedFuture(SchemaRegistryFormat.SchemaInfo.parseFrom(bytes));
            }
            catch (InvalidProtocolBufferException e) {
                future = new CompletableFuture();
                future.completeExceptionally(e);
            }
            return future;
        }
    }
}

