/*
 * Decompiled with CFR 0.152.
 */
package org.ballerinalang.util.debugger;

import io.netty.channel.Channel;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.Semaphore;
import java.util.stream.Collectors;
import org.ballerinalang.bre.bvm.BVMScheduler;
import org.ballerinalang.bre.bvm.StackFrame;
import org.ballerinalang.bre.bvm.Strand;
import org.ballerinalang.model.types.BType;
import org.ballerinalang.model.values.BBoolean;
import org.ballerinalang.model.values.BByte;
import org.ballerinalang.model.values.BFloat;
import org.ballerinalang.model.values.BInteger;
import org.ballerinalang.model.values.BString;
import org.ballerinalang.util.codegen.LineNumberInfo;
import org.ballerinalang.util.codegen.LocalVariableInfo;
import org.ballerinalang.util.codegen.PackageVarInfo;
import org.ballerinalang.util.codegen.ProgramFile;
import org.ballerinalang.util.codegen.attributes.AttributeInfo;
import org.ballerinalang.util.codegen.attributes.LocalVariableAttributeInfo;
import org.ballerinalang.util.debugger.DebugClientHandler;
import org.ballerinalang.util.debugger.DebugCommand;
import org.ballerinalang.util.debugger.DebugException;
import org.ballerinalang.util.debugger.DebugInfoHolder;
import org.ballerinalang.util.debugger.DebugServer;
import org.ballerinalang.util.debugger.ExpressionEvaluator;
import org.ballerinalang.util.debugger.VMDebugClientHandler;
import org.ballerinalang.util.debugger.dto.BreakPointDTO;
import org.ballerinalang.util.debugger.dto.CommandDTO;
import org.ballerinalang.util.debugger.dto.FrameDTO;
import org.ballerinalang.util.debugger.dto.MessageDTO;
import org.ballerinalang.util.debugger.dto.VariableDTO;
import org.ballerinalang.util.debugger.util.DebugMsgUtil;

public class Debugger {
    private boolean debugEnabled = false;
    private ProgramFile programFile;
    private DebugClientHandler clientHandler;
    private DebugInfoHolder debugInfoHolder;
    private ExpressionEvaluator expressionEvaluator;
    private volatile Semaphore executionSem;
    private static final String META_DATA_VAR_PATTERN = "$";
    private static final String GLOBAL = "Global";
    private static final String LOCAL = "Local";

    public Debugger(ProgramFile programFile) {
        this.programFile = programFile;
        this.expressionEvaluator = new ExpressionEvaluator();
        String debug = System.getProperty("debug");
        if (debug != null && !debug.isEmpty()) {
            this.debugEnabled = true;
        }
    }

    public void init() {
        this.setupDebugger();
        this.setClientHandler(new VMDebugClientHandler());
        DebugServer debugServer = new DebugServer(this);
        debugServer.startServer();
        Runtime.getRuntime().addShutdownHook(new DebuggerShutDownHook(this, debugServer));
    }

    protected void setupDebugger() {
        this.executionSem = new Semaphore(0);
        this.debugInfoHolder = new DebugInfoHolder(this.programFile);
    }

    public void waitTillDebuggeeResponds() {
        try {
            this.executionSem.acquire();
        }
        catch (InterruptedException interruptedException) {
            // empty catch block
        }
    }

    public void pauseWorker(Strand ctx) {
        ctx.getDebugContext().setStrandPaused(true);
        BVMScheduler.stateChange(ctx, Strand.State.RUNNABLE, Strand.State.PAUSED);
    }

    public LineNumberInfo getLineNumber(String packagePath, int ip) {
        return this.debugInfoHolder.getLineNumber(packagePath, ip);
    }

    void processDebugCommand(String json) {
        try {
            this.processCommand(json);
        }
        catch (Exception e) {
            MessageDTO message = new MessageDTO("INVALID", e.getMessage());
            this.clientHandler.sendCustomMsg(message);
            this.stopDebugging();
        }
    }

    private void processCommand(String json) {
        CommandDTO command;
        try {
            command = DebugMsgUtil.buildCommandDTO(json);
        }
        catch (Exception e) {
            throw new DebugException("Invalid Command");
        }
        switch (command.getCommand()) {
            case "RESUME": {
                this.resume(command.getThreadId());
                break;
            }
            case "STEP_OVER": {
                this.stepOver(command.getThreadId());
                break;
            }
            case "STEP_IN": {
                this.stepIn(command.getThreadId());
                break;
            }
            case "STEP_OUT": {
                this.stepOut(command.getThreadId());
                break;
            }
            case "STOP": {
                this.stopDebugging();
                break;
            }
            case "SET_POINTS": {
                this.addDebugPoints(command.getPoints());
                this.sendAcknowledge("Debug points updated");
                break;
            }
            case "EVALUATE_EXPRESSION": {
                String results = this.evaluateVariable(command.getThreadId(), command.getVariableName());
                this.sendResults(results);
                break;
            }
            case "START": {
                this.sendAcknowledge("Debug started.");
                this.startDebug();
                break;
            }
            default: {
                throw new DebugException("Invalid Command");
            }
        }
    }

    public void addDebugPoints(List<BreakPointDTO> breakPointDTOS) {
        ArrayList<BreakPointDTO> undeployedList = new ArrayList<BreakPointDTO>(breakPointDTOS);
        List<BreakPointDTO> deployedList = this.debugInfoHolder.addDebugPoints(breakPointDTOS);
        undeployedList.removeAll(deployedList);
        if (!undeployedList.isEmpty()) {
            String bPoints = undeployedList.stream().map(b -> "{" + b.toString() + "}").collect(Collectors.joining(", "));
            MessageDTO message = new MessageDTO("INVALID", "Invalid Breakpoint : [" + bPoints + "]");
            this.clientHandler.sendCustomMsg(message);
        }
    }

    public void startDebug() {
        this.executionSem.release();
    }

    public void resume(String strandId) {
        Strand ctx = this.getWorkerContext(strandId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.RESUME);
        BVMScheduler.stateChange(ctx, Strand.State.PAUSED, Strand.State.RUNNABLE);
        BVMScheduler.schedule(ctx);
    }

    public void stepIn(String strandId) {
        Strand ctx = this.getWorkerContext(strandId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.STEP_IN);
        BVMScheduler.stateChange(ctx, Strand.State.PAUSED, Strand.State.RUNNABLE);
        BVMScheduler.schedule(ctx);
    }

    public void stepOver(String strandId) {
        Strand ctx = this.getWorkerContext(strandId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.STEP_OVER);
        BVMScheduler.stateChange(ctx, Strand.State.PAUSED, Strand.State.RUNNABLE);
        BVMScheduler.schedule(ctx);
    }

    public void stepOut(String strandId) {
        Strand ctx = this.getWorkerContext(strandId);
        ctx.getDebugContext().setCurrentCommand(DebugCommand.STEP_OUT);
        BVMScheduler.stateChange(ctx, Strand.State.PAUSED, Strand.State.RUNNABLE);
        BVMScheduler.schedule(ctx);
    }

    public void stopDebugging() {
        this.debugInfoHolder.clearDebugLocations();
        this.clientHandler.getAllStrands().forEach((k, v) -> {
            v.getDebugContext().setCurrentCommand(DebugCommand.RESUME);
            BVMScheduler.stateChange(v, Arrays.asList(Strand.State.PAUSED, Strand.State.RUNNABLE), Strand.State.RUNNABLE);
            BVMScheduler.schedule(v);
        });
        this.clientHandler.clearChannel();
    }

    public String evaluateVariable(String workerId, String variableName) {
        Strand ctx = this.getWorkerContext(workerId);
        LineNumberInfo currentExecLine = this.getLineNumber(ctx.currentFrame.callableUnitInfo.getPackageInfo().getPkgPath(), ctx.currentFrame.ip);
        return this.expressionEvaluator.evaluateVariable(ctx, currentExecLine, variableName);
    }

    private Strand getWorkerContext(String workerId) {
        Strand ctx = this.clientHandler.getStrand(workerId);
        if (ctx == null) {
            throw new DebugException("Invalid Worker ID : " + workerId);
        }
        return ctx;
    }

    void addDebugSession(Channel channel) throws DebugException {
        this.clientHandler.setChannel(channel);
        this.sendAcknowledge("Channel registered.");
    }

    public void addStrand(Strand strand) {
        this.clientHandler.addStrand(strand);
    }

    public boolean isClientSessionActive() {
        return this.clientHandler.isChannelActive();
    }

    public void notifyDebugHit(Strand ctx, LineNumberInfo currentExecLine, String workerId) {
        MessageDTO message = this.generateDebugHitMessage(ctx, currentExecLine, workerId);
        this.clientHandler.notifyHalt(message);
    }

    public void notifyExit() {
        if (!this.isClientSessionActive()) {
            return;
        }
        this.clientHandler.notifyExit();
    }

    private void sendAcknowledge(String messageText) {
        MessageDTO message = new MessageDTO("ACK", messageText);
        this.clientHandler.sendCustomMsg(message);
    }

    private void sendResults(String results) {
        MessageDTO message = new MessageDTO("EXPRESSION_RESULTS", results);
        this.clientHandler.sendExpressionResults(message);
    }

    private MessageDTO generateDebugHitMessage(Strand ctx, LineNumberInfo currentExecLine, String workerId) {
        PackageVarInfo[] packageVarInfoEntries;
        MessageDTO message = new MessageDTO("DEBUG_HIT", "Debug point hit.");
        message.setThreadId(workerId);
        BreakPointDTO breakPointDTO = new BreakPointDTO(currentExecLine.getPackageInfo().getPkgPath(), currentExecLine.getFileName(), currentExecLine.getLineNumber());
        message.setLocation(breakPointDTO);
        int callingIp = currentExecLine.getIp();
        String pck = ctx.currentFrame.callableUnitInfo.getPackageInfo().getPkgPath();
        String functionName = ctx.currentFrame.callableUnitInfo.getName();
        LineNumberInfo callingLine = this.getLineNumber(ctx.currentFrame.callableUnitInfo.getPackageInfo().getPkgPath(), callingIp);
        FrameDTO frameDTO = new FrameDTO(pck, functionName, callingLine.getFileName(), callingLine.getLineNumber());
        message.addFrame(frameDTO);
        for (PackageVarInfo packVarInfo : packageVarInfoEntries = ctx.programFile.getPackageInfo(ctx.programFile.getEntryPkgName()).getPackageInfoEntries()) {
            int pkgIndex;
            VariableDTO variableDTO;
            if (!packVarInfo.isIdentifierLiteral() && packVarInfo.getName().contains(META_DATA_VAR_PATTERN) || (variableDTO = Debugger.constructGlobalVariable(ctx, packVarInfo, pkgIndex = ctx.currentFrame.callableUnitInfo.getPackageInfo().pkgIndex)).getValue() == null && packVarInfo.getType().getTag() != 7) continue;
            frameDTO.addVariable(variableDTO);
        }
        LocalVariableAttributeInfo localVarAttrInfo = (LocalVariableAttributeInfo)ctx.currentFrame.callableUnitInfo.getDefaultWorkerInfo().getAttributeInfo(AttributeInfo.Kind.LOCAL_VARIABLES_ATTRIBUTE);
        localVarAttrInfo.getLocalVariables().forEach(variableInfo -> {
            if (variableInfo.isIdentifierLiteral() || !variableInfo.getVariableName().contains(META_DATA_VAR_PATTERN)) {
                VariableDTO variableDTO = Debugger.constructLocalVariable(ctx.currentFrame, variableInfo);
                if (variableInfo.getScopeStartLineNumber() < callingLine.getLineNumber() && callingLine.getLineNumber() <= variableInfo.getScopeEndLineNumber()) {
                    frameDTO.addVariable(variableDTO);
                }
            }
        });
        return message;
    }

    public static VariableDTO constructGlobalVariable(Strand ctx, PackageVarInfo packVarInfo, int pkgIndex) {
        VariableDTO variableDTO = new VariableDTO(packVarInfo.getName(), GLOBAL);
        BType varType = packVarInfo.getType();
        switch (varType.getTag()) {
            case 1: {
                variableDTO.setBValue(new BInteger(ctx.programFile.globalMemArea.getIntField(pkgIndex, packVarInfo.getGlobalMemIndex())));
                break;
            }
            case 2: {
                variableDTO.setBValue(new BByte(ctx.programFile.globalMemArea.getIntField(pkgIndex, packVarInfo.getGlobalMemIndex())));
                break;
            }
            case 3: {
                variableDTO.setBValue(new BFloat(ctx.programFile.globalMemArea.getFloatField(pkgIndex, packVarInfo.getGlobalMemIndex())));
                break;
            }
            case 5: {
                variableDTO.setBValue(new BString(ctx.programFile.globalMemArea.getStringField(pkgIndex, packVarInfo.getGlobalMemIndex())));
                break;
            }
            case 6: {
                variableDTO.setBValue(new BBoolean(ctx.programFile.globalMemArea.getBooleanField(pkgIndex, packVarInfo.getGlobalMemIndex()) == 1));
                break;
            }
            default: {
                variableDTO.setBValue(ctx.programFile.globalMemArea.getRefField(pkgIndex, packVarInfo.getGlobalMemIndex()), varType);
            }
        }
        return variableDTO;
    }

    public static VariableDTO constructLocalVariable(StackFrame sf, LocalVariableInfo variableInfo) {
        VariableDTO variableDTO = new VariableDTO(variableInfo.getVariableName(), LOCAL);
        BType varType = variableInfo.getVariableType();
        switch (varType.getTag()) {
            case 1: {
                variableDTO.setBValue(new BInteger(sf.longRegs[variableInfo.getVariableIndex()]));
                break;
            }
            case 2: {
                variableDTO.setBValue(new BByte(sf.longRegs[variableInfo.getVariableIndex()]));
                break;
            }
            case 3: {
                variableDTO.setBValue(new BFloat(sf.doubleRegs[variableInfo.getVariableIndex()]));
                break;
            }
            case 5: {
                variableDTO.setBValue(new BString(sf.stringRegs[variableInfo.getVariableIndex()]));
                break;
            }
            case 6: {
                variableDTO.setBValue(new BBoolean(sf.intRegs[variableInfo.getVariableIndex()] == 1));
                break;
            }
            default: {
                variableDTO.setBValue(sf.refRegs[variableInfo.getVariableIndex()], varType);
            }
        }
        return variableDTO;
    }

    public boolean isDebugEnabled() {
        return this.debugEnabled;
    }

    public void setDebugEnabled() {
        this.debugEnabled = true;
    }

    protected void setClientHandler(DebugClientHandler clientHandler) {
        this.clientHandler = clientHandler;
    }

    static class DebuggerShutDownHook
    extends Thread {
        private Debugger debugger;
        private DebugServer debugServer;

        DebuggerShutDownHook(Debugger debugger, DebugServer debugServer) {
            this.debugger = debugger;
            this.debugServer = debugServer;
        }

        @Override
        public void run() {
            this.debugger.notifyExit();
            this.debugServer.closeServerChannel();
        }
    }
}

