/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer.ranges;

import com.google.common.base.Stopwatch;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.inject.assistedinject.Assisted;
import com.google.inject.assistedinject.AssistedInject;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import org.elasticsearch.search.SearchHit;
import org.graylog2.indexer.Deflector;
import org.graylog2.indexer.EmptyIndexException;
import org.graylog2.indexer.ranges.IndexRange;
import org.graylog2.indexer.ranges.IndexRangeService;
import org.graylog2.indexer.searches.Searches;
import org.graylog2.plugin.ServerStatus;
import org.graylog2.plugin.Tools;
import org.graylog2.shared.system.activities.Activity;
import org.graylog2.shared.system.activities.ActivityWriter;
import org.graylog2.system.jobs.SystemJob;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RebuildIndexRangesJob
extends SystemJob {
    private static final Logger LOG = LoggerFactory.getLogger(RebuildIndexRangesJob.class);
    public static final int MAX_CONCURRENCY = 1;
    private boolean cancelRequested = false;
    private int indicesToCalculate = 0;
    private int indicesCalculated = 0;
    protected final Deflector deflector;
    private final Searches searches;
    private final ActivityWriter activityWriter;
    protected final IndexRangeService indexRangeService;

    @AssistedInject
    public RebuildIndexRangesJob(@Assisted Deflector deflector, ServerStatus serverStatus, Searches searches, ActivityWriter activityWriter, IndexRangeService indexRangeService) {
        super(serverStatus);
        this.deflector = deflector;
        this.searches = searches;
        this.activityWriter = activityWriter;
        this.indexRangeService = indexRangeService;
    }

    @Override
    public void requestCancel() {
        this.cancelRequested = true;
    }

    @Override
    public int getProgress() {
        if (this.indicesToCalculate <= 0) {
            return 0;
        }
        return (int)Math.floor((float)this.indicesCalculated / (float)this.indicesToCalculate * 100.0f);
    }

    @Override
    public String getDescription() {
        return "Rebuilds index range information.";
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void execute() {
        ArrayList ranges = Lists.newArrayList();
        this.info("Re-calculating index ranges.");
        String[] indices = this.deflector.getAllDeflectorIndexNames();
        if (indices == null || indices.length == 0) {
            this.info("No indices, nothing to calculate.");
            return;
        }
        this.indicesToCalculate = indices.length;
        Stopwatch sw = Stopwatch.createStarted();
        for (String index : indices) {
            if (this.cancelRequested) {
                this.info("Stop requested. Not calculating next index range, not updating ranges.");
                sw.stop();
                return;
            }
            try {
                ranges.add(this.calculateRange(index));
            }
            catch (EmptyIndexException e) {
                LOG.info("Index [{}] is empty, inserting dummy index range.", (Object)index);
                Map<String, Object> emptyIndexRange = this.getDeflectorIndexRange(index);
                if (this.deflector.getCurrentActualTargetIndex().equals(index)) {
                    LOG.info("Index [{}] is empty but it is the current deflector target. Inserting dummy index range.", (Object)index);
                } else {
                    emptyIndexRange.put("start", 0);
                    emptyIndexRange.put("calculated_at", Tools.getUTCTimestamp());
                }
                ranges.add(emptyIndexRange);
            }
            catch (Exception e) {
                LOG.info("Could not calculate range of index [" + index + "]. Skipping.", (Throwable)e);
            }
            finally {
                ++this.indicesCalculated;
            }
        }
        this.updateCollection(ranges);
        this.info("Done calculating index ranges for " + indices.length + " indices. Took " + sw.stop().elapsed(TimeUnit.MILLISECONDS) + "ms.");
    }

    protected Map<String, Object> getDeflectorIndexRange(String index) {
        HashMap deflectorIndexRange = Maps.newHashMap();
        deflectorIndexRange.put("index", index);
        deflectorIndexRange.put("start", Tools.getUTCTimestamp());
        return deflectorIndexRange;
    }

    private static int getTimestampOfMessage(SearchHit msg) {
        Object field = msg.getSource().get("timestamp");
        if (field == null) {
            throw new RuntimeException("Document has no field timestamp.");
        }
        DateTimeFormatter formatter = DateTimeFormat.forPattern((String)"yyyy-MM-dd HH:mm:ss.SSS").withZoneUTC();
        DateTime dt = formatter.parseDateTime(field.toString());
        return (int)(dt.getMillis() / 1000L);
    }

    protected Map<String, Object> calculateRange(String index) throws EmptyIndexException {
        HashMap range = Maps.newHashMap();
        Stopwatch x = Stopwatch.createStarted();
        SearchHit doc = this.searches.firstOfIndex(index);
        if (doc == null || doc.isSourceEmpty()) {
            x.stop();
            throw new EmptyIndexException();
        }
        int rangeStart = RebuildIndexRangesJob.getTimestampOfMessage(doc);
        int took = (int)x.stop().elapsed(TimeUnit.MILLISECONDS);
        range.put("index", index);
        range.put("start", rangeStart);
        range.put("calculated_at", Tools.getUTCTimestamp());
        range.put("took_ms", took);
        LOG.info("Calculated range of [{}] in [{}ms].", (Object)index, (Object)took);
        return range;
    }

    private void updateCollection(List<Map<String, Object>> ranges) {
        this.indexRangeService.destroyAll();
        for (Map<String, Object> range : ranges) {
            IndexRange indexRange = this.indexRangeService.create(range);
            this.indexRangeService.saveWithoutValidation(indexRange);
        }
    }

    protected void info(String what) {
        LOG.info(what);
        this.activityWriter.write(new Activity(what, RebuildIndexRangesJob.class));
    }

    @Override
    public boolean providesProgress() {
        return true;
    }

    @Override
    public boolean isCancelable() {
        return true;
    }

    @Override
    public int maxConcurrency() {
        return 1;
    }

    @Override
    public String getClassName() {
        return this.getClass().getCanonicalName();
    }

    public static interface Factory {
        public RebuildIndexRangesJob create(Deflector var1);
    }
}

