/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.security.ldap;

import com.google.common.util.concurrent.SimpleTimeLimiter;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import com.google.common.util.concurrent.UncheckedTimeoutException;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Iterator;
import java.util.concurrent.Callable;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import org.apache.directory.api.ldap.model.cursor.CursorException;
import org.apache.directory.api.ldap.model.cursor.EntryCursor;
import org.apache.directory.api.ldap.model.entry.Attribute;
import org.apache.directory.api.ldap.model.entry.Entry;
import org.apache.directory.api.ldap.model.exception.LdapException;
import org.apache.directory.api.ldap.model.message.BindRequest;
import org.apache.directory.api.ldap.model.message.BindRequestImpl;
import org.apache.directory.api.ldap.model.message.BindResponse;
import org.apache.directory.api.ldap.model.message.ResultCodeEnum;
import org.apache.directory.api.ldap.model.message.SearchScope;
import org.apache.directory.ldap.client.api.LdapConnectionConfig;
import org.apache.directory.ldap.client.api.LdapNetworkConnection;
import org.graylog2.shared.security.ldap.LdapEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LdapConnector {
    private static final Logger LOG = LoggerFactory.getLogger(LdapConnector.class);
    private final int connectionTimeout;

    public LdapConnector(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    public LdapNetworkConnection connect(LdapConnectionConfig config) throws LdapException {
        final LdapNetworkConnection connection = new LdapNetworkConnection(config);
        connection.setTimeOut((long)this.connectionTimeout);
        if (LOG.isTraceEnabled()) {
            LOG.trace("Connecting to LDAP server {}:{}, binding with user {}", new Object[]{config.getLdapHost(), config.getLdapPort(), config.getName()});
        }
        ThreadFactory threadFactory = new ThreadFactoryBuilder().setNameFormat("ldap-connector-%d").build();
        SimpleTimeLimiter timeLimiter = new SimpleTimeLimiter(Executors.newSingleThreadExecutor(threadFactory));
        Callable timeLimitedConnection = (Callable)timeLimiter.newProxy((Object)new Callable<Boolean>(){

            @Override
            public Boolean call() throws Exception {
                return connection.connect();
            }
        }, Callable.class, (long)this.connectionTimeout, TimeUnit.MILLISECONDS);
        try {
            Boolean connected = (Boolean)timeLimitedConnection.call();
            if (!connected.booleanValue()) {
                return null;
            }
        }
        catch (UncheckedTimeoutException e) {
            LOG.error("Timed out connecting to LDAP server", (Throwable)e);
            throw new LdapException("Could not connect to LDAP server", e.getCause());
        }
        catch (LdapException e) {
            throw e;
        }
        catch (Exception e) {
            throw new LdapException("Unexpected error connecting to LDAP", (Throwable)e);
        }
        connection.bind();
        return connection;
    }

    public LdapEntry search(LdapNetworkConnection connection, String searchBase, String searchPattern, String principal, boolean activeDirectory) throws LdapException, CursorException {
        EntryCursor entryCursor;
        Iterator it;
        LdapEntry ldapEntry = new LdapEntry();
        String filter = MessageFormat.format(searchPattern, this.sanitizePrincipal(principal));
        if (LOG.isTraceEnabled()) {
            LOG.trace("Search {} for {}, starting at {}", new Object[]{activeDirectory ? "ActiveDirectory" : "LDAP", filter, searchBase});
        }
        if ((it = (entryCursor = connection.search(searchBase, filter, SearchScope.SUBTREE, new String[]{"*"})).iterator()).hasNext()) {
            Entry e = (Entry)it.next();
            if (!activeDirectory) {
                ldapEntry.setDn(e.getDn().getName());
            }
            for (Attribute attribute : e.getAttributes()) {
                if (activeDirectory && attribute.getId().equalsIgnoreCase("userPrincipalName")) {
                    ldapEntry.setDn(attribute.getString());
                }
                if (!attribute.isHumanReadable()) continue;
                ldapEntry.put(attribute.getId(), attribute.getString());
            }
        } else {
            LOG.trace("No LDAP entry found for filter {}", (Object)filter);
            return null;
        }
        LOG.trace("LDAP search found entry for DN {} with search filter {}", (Object)ldapEntry.getDn(), (Object)filter);
        return ldapEntry;
    }

    private String sanitizePrincipal(String input) {
        String s = "";
        for (int i = 0; i < input.length(); ++i) {
            byte[] utf8bytes;
            char c = input.charAt(i);
            if (c == '*') {
                s = s + "\\2a";
                continue;
            }
            if (c == '(') {
                s = s + "\\28";
                continue;
            }
            if (c == ')') {
                s = s + "\\29";
                continue;
            }
            if (c == '\\') {
                s = s + "\\5c";
                continue;
            }
            if (c == '\u0000') {
                s = s + "\\00";
                continue;
            }
            if (c <= '\u007f') {
                s = s + String.valueOf(c);
                continue;
            }
            if (c < '\u0080') continue;
            for (byte b : utf8bytes = String.valueOf(c).getBytes(StandardCharsets.UTF_8)) {
                s = s + String.format("\\%02x", b);
            }
        }
        return s;
    }

    public boolean authenticate(LdapNetworkConnection connection, String principal, String credentials) throws LdapException {
        BindRequestImpl bindRequest = new BindRequestImpl();
        bindRequest.setName(principal);
        bindRequest.setCredentials(credentials);
        LOG.trace("Re-binding with DN {} using password", (Object)principal);
        BindResponse bind = connection.bind((BindRequest)bindRequest);
        if (!bind.getLdapResult().getResultCode().equals((Object)ResultCodeEnum.SUCCESS)) {
            LOG.trace("Re-binding DN {} failed", (Object)principal);
            throw new RuntimeException(bind.toString());
        }
        LOG.trace("Binding DN {} did not throw, connection authenticated: {}", (Object)principal, (Object)connection.isAuthenticated());
        return connection.isAuthenticated();
    }
}

