/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.indexer;

import com.google.common.base.Strings;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSortedSet;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import org.elasticsearch.cluster.health.ClusterHealthStatus;
import org.elasticsearch.indices.InvalidAliasNameException;
import org.graylog2.audit.AuditActor;
import org.graylog2.audit.AuditEventSender;
import org.graylog2.indexer.NoTargetIndexException;
import org.graylog2.indexer.indices.Indices;
import org.graylog2.indexer.indices.TooManyAliasesException;
import org.graylog2.indexer.indices.jobs.SetIndexReadOnlyAndCalculateRangeJob;
import org.graylog2.indexer.ranges.IndexRange;
import org.graylog2.indexer.ranges.IndexRangeService;
import org.graylog2.plugin.system.NodeId;
import org.graylog2.shared.system.activities.Activity;
import org.graylog2.shared.system.activities.ActivityWriter;
import org.graylog2.system.jobs.SystemJobConcurrencyException;
import org.graylog2.system.jobs.SystemJobManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Deflector {
    private static final Logger LOG = LoggerFactory.getLogger(Deflector.class);
    public static final String DEFLECTOR_SUFFIX = "deflector";
    public static final String SEPARATOR = "_";
    public static final String RESTORED_ARCHIVE_SUFFIX = "_restored_archive";
    private final SystemJobManager systemJobManager;
    private final ActivityWriter activityWriter;
    private final IndexRangeService indexRangeService;
    private final String indexPrefix;
    private final String deflectorName;
    private final Indices indices;
    private final Pattern deflectorIndexPattern;
    private final Pattern indexPattern;
    private final AuditEventSender auditEventSender;
    private final NodeId nodeId;
    private final SetIndexReadOnlyAndCalculateRangeJob.Factory setIndexReadOnlyAndCalculateRangeJobFactory;

    @Inject
    public Deflector(SystemJobManager systemJobManager, @Named(value="elasticsearch_index_prefix") String indexPrefix, ActivityWriter activityWriter, Indices indices, IndexRangeService indexRangeService, AuditEventSender auditEventSender, NodeId nodeId, SetIndexReadOnlyAndCalculateRangeJob.Factory setIndexReadOnlyAndCalculateRangeJobFactory) {
        this.indexPrefix = indexPrefix;
        this.systemJobManager = systemJobManager;
        this.activityWriter = activityWriter;
        this.indexRangeService = indexRangeService;
        this.auditEventSender = auditEventSender;
        this.nodeId = nodeId;
        this.setIndexReadOnlyAndCalculateRangeJobFactory = setIndexReadOnlyAndCalculateRangeJobFactory;
        this.deflectorName = Deflector.buildName(indexPrefix);
        this.indices = indices;
        this.deflectorIndexPattern = Pattern.compile("^" + indexPrefix + SEPARATOR + "\\d+");
        this.indexPattern = Pattern.compile("^" + indexPrefix + SEPARATOR + "\\d+(?:" + RESTORED_ARCHIVE_SUFFIX + ")?");
    }

    public boolean isUp() {
        return this.indices.aliasExists(this.getName());
    }

    public void setUp() {
        if (this.isUp()) {
            LOG.info("Found deflector alias <{}>. Using it.", (Object)this.getName());
        } else {
            LOG.info("Did not find an deflector alias. Setting one up now.");
            try {
                String currentTarget = this.getNewestTargetName();
                LOG.info("Pointing to already existing index target <{}>", (Object)currentTarget);
                this.pointTo(currentTarget);
            }
            catch (NoTargetIndexException ex) {
                String msg = "There is no index target to point to. Creating one now.";
                LOG.info("There is no index target to point to. Creating one now.");
                this.activityWriter.write(new Activity("There is no index target to point to. Creating one now.", Deflector.class));
                this.cycle();
            }
            catch (InvalidAliasNameException e) {
                LOG.error("Seems like there already is an index called [{}]", (Object)this.getName());
            }
        }
    }

    public void cycle() {
        int oldTargetNumber;
        LOG.info("Cycling deflector to next index now.");
        try {
            oldTargetNumber = this.getNewestTargetNumber();
        }
        catch (NoTargetIndexException ex) {
            oldTargetNumber = -1;
        }
        int newTargetNumber = oldTargetNumber + 1;
        String newTarget = Deflector.buildIndexName(this.indexPrefix, newTargetNumber);
        String oldTarget = Deflector.buildIndexName(this.indexPrefix, oldTargetNumber);
        if (oldTargetNumber == -1) {
            LOG.info("Cycling from <none> to <{}>", (Object)newTarget);
        } else {
            LOG.info("Cycling from <{}> to <{}>", (Object)oldTarget, (Object)newTarget);
        }
        LOG.info("Creating index target <{}>...", (Object)newTarget);
        if (!this.indices.create(newTarget)) {
            LOG.error("Could not properly create new target <{}>", (Object)newTarget);
        }
        LOG.info("Waiting for index allocation of <{}>", (Object)newTarget);
        ClusterHealthStatus healthStatus = this.indices.waitForRecovery(newTarget);
        LOG.debug("Health status of index <{}>: {}", (Object)newTarget, (Object)healthStatus);
        this.addDeflectorIndexRange(newTarget);
        LOG.info("Done!");
        LOG.info("Pointing deflector to new target index....");
        Activity activity = new Activity(Deflector.class);
        if (oldTargetNumber == -1) {
            this.pointTo(newTarget);
            activity.setMessage("Cycled deflector from <none> to <" + newTarget + ">");
        } else {
            LOG.debug("Now switching over deflector alias.");
            this.pointTo(newTarget, oldTarget);
            SetIndexReadOnlyAndCalculateRangeJob setIndexReadOnlyAndCalculateRangeJob = this.setIndexReadOnlyAndCalculateRangeJobFactory.create(oldTarget);
            try {
                this.systemJobManager.submitWithDelay(setIndexReadOnlyAndCalculateRangeJob, 30L, TimeUnit.SECONDS);
            }
            catch (SystemJobConcurrencyException e) {
                LOG.error("Cannot set index <" + oldTarget + "> to read only and calculate its range. It won't be optimized.", (Throwable)e);
            }
            activity.setMessage("Cycled deflector from <" + oldTarget + "> to <" + newTarget + ">");
        }
        LOG.info("Done!");
        this.activityWriter.write(activity);
        this.auditEventSender.success(AuditActor.system(this.nodeId), "server:es_write_index:update", (Map<String, Object>)ImmutableMap.of((Object)"indexName", (Object)newTarget));
    }

    private void addDeflectorIndexRange(String newTarget) {
        IndexRange deflectorRange = this.indexRangeService.createUnknownRange(newTarget);
        this.indexRangeService.save(deflectorRange);
    }

    public int getNewestTargetNumber() throws NoTargetIndexException {
        Set<String> indexNames = this.indices.getIndexNamesAndAliases(this.getDeflectorWildcard()).keySet();
        if (indexNames.isEmpty()) {
            throw new NoTargetIndexException();
        }
        int highestIndexNumber = -1;
        for (String indexName : indexNames) {
            if (!this.isGraylogDeflectorIndex(indexName)) continue;
            try {
                int indexNumber = Deflector.extractIndexNumber(indexName);
                highestIndexNumber = Math.max(indexNumber, highestIndexNumber);
            }
            catch (NumberFormatException ex) {
                LOG.warn("Couldn't extract index number from index name " + indexName, (Throwable)ex);
            }
        }
        if (highestIndexNumber == -1) {
            throw new NoTargetIndexException();
        }
        return highestIndexNumber;
    }

    public String[] getAllGraylogIndexNames() {
        Set<String> indexNames = this.indices.getIndexNamesAndAliases(this.getDeflectorWildcard()).keySet();
        List<String> result = indexNames.stream().filter(this::isGraylogIndex).collect(Collectors.toList());
        return result.toArray(new String[result.size()]);
    }

    public Map<String, Set<String>> getAllGraylogDeflectorIndices() {
        Map<String, Set<String>> indexNamesAndAliases = this.indices.getIndexNamesAndAliases(this.getDeflectorWildcard());
        return indexNamesAndAliases.entrySet().stream().filter(e -> this.isGraylogDeflectorIndex((String)e.getKey())).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    }

    public String getNewestTargetName() throws NoTargetIndexException {
        return Deflector.buildIndexName(this.indexPrefix, this.getNewestTargetNumber());
    }

    public static String buildIndexName(String prefix, int number) {
        return prefix + SEPARATOR + number;
    }

    public static String buildName(String prefix) {
        return prefix + SEPARATOR + DEFLECTOR_SUFFIX;
    }

    public static int extractIndexNumber(String indexName) throws NumberFormatException {
        String[] parts = indexName.split(SEPARATOR);
        try {
            return Integer.parseInt(parts[parts.length - 1]);
        }
        catch (Exception e) {
            String msg = "Could not extract index number from index <" + indexName + ">.";
            LOG.debug(msg, (Throwable)e);
            throw new NumberFormatException(msg);
        }
    }

    public void pointTo(String newIndex, String oldIndex) {
        this.indices.cycleAlias(this.getName(), newIndex, oldIndex);
    }

    public void pointTo(String newIndex) {
        this.indices.cycleAlias(this.getName(), newIndex);
    }

    @Nullable
    public String getCurrentActualTargetIndex() throws TooManyAliasesException {
        return this.indices.aliasTarget(this.getName());
    }

    public String getName() {
        return this.deflectorName;
    }

    public String getDeflectorWildcard() {
        return this.indexPrefix + SEPARATOR + "*";
    }

    public boolean isDeflectorAlias(String indexName) {
        return this.getName().equals(indexName);
    }

    public boolean isGraylogDeflectorIndex(String indexName) {
        return !Strings.isNullOrEmpty((String)indexName) && !this.isDeflectorAlias(indexName) && this.deflectorIndexPattern.matcher(indexName).matches();
    }

    public boolean isGraylogIndex(String indexName) {
        return !Strings.isNullOrEmpty((String)indexName) && !this.isDeflectorAlias(indexName) && this.indexPattern.matcher(indexName).matches();
    }

    public void cleanupAliases(Set<String> indexNames) {
        ImmutableSortedSet sortedSet = ImmutableSortedSet.orderedBy((Comparator)new IndexNameComparator()).addAll(indexNames).build();
        this.indices.removeAliases(this.getName(), sortedSet.headSet((String)sortedSet.last()));
    }

    private static class IndexNameComparator
    implements Comparator<String> {
        private IndexNameComparator() {
        }

        @Override
        public int compare(String o1, String o2) {
            int indexNumber1 = this.silentlyExtractIndexNumber(o1);
            int indexNumber2 = this.silentlyExtractIndexNumber(o2);
            return ComparisonChain.start().compare(indexNumber1, indexNumber2).result();
        }

        private int silentlyExtractIndexNumber(String indexName) {
            try {
                return Deflector.extractIndexNumber(indexName);
            }
            catch (NumberFormatException e) {
                return -1;
            }
        }
    }
}

