/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.contentpacks;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Sets;
import com.google.common.graph.ElementOrder;
import com.google.common.graph.Graph;
import com.google.common.graph.GraphBuilder;
import com.google.common.graph.ImmutableGraph;
import com.google.common.graph.MutableGraph;
import com.google.common.graph.Traverser;
import java.time.Instant;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.graylog2.contentpacks.ContentPackInstallationPersistenceService;
import org.graylog2.contentpacks.EntityDescriptorIds;
import org.graylog2.contentpacks.constraints.ConstraintChecker;
import org.graylog2.contentpacks.exceptions.ContentPackException;
import org.graylog2.contentpacks.exceptions.EmptyDefaultValueException;
import org.graylog2.contentpacks.exceptions.FailedConstraintsException;
import org.graylog2.contentpacks.exceptions.InvalidParameterTypeException;
import org.graylog2.contentpacks.exceptions.InvalidParametersException;
import org.graylog2.contentpacks.exceptions.MissingParametersException;
import org.graylog2.contentpacks.exceptions.UnexpectedEntitiesException;
import org.graylog2.contentpacks.facades.EntityFacade;
import org.graylog2.contentpacks.facades.UnsupportedEntityFacade;
import org.graylog2.contentpacks.model.ContentPack;
import org.graylog2.contentpacks.model.ContentPackInstallation;
import org.graylog2.contentpacks.model.ContentPackUninstallDetails;
import org.graylog2.contentpacks.model.ContentPackUninstallation;
import org.graylog2.contentpacks.model.ContentPackV1;
import org.graylog2.contentpacks.model.LegacyContentPack;
import org.graylog2.contentpacks.model.ModelId;
import org.graylog2.contentpacks.model.ModelType;
import org.graylog2.contentpacks.model.constraints.Constraint;
import org.graylog2.contentpacks.model.constraints.ConstraintCheckResult;
import org.graylog2.contentpacks.model.entities.Entity;
import org.graylog2.contentpacks.model.entities.EntityDescriptor;
import org.graylog2.contentpacks.model.entities.EntityExcerpt;
import org.graylog2.contentpacks.model.entities.EntityV1;
import org.graylog2.contentpacks.model.entities.NativeEntity;
import org.graylog2.contentpacks.model.entities.NativeEntityDescriptor;
import org.graylog2.contentpacks.model.entities.references.ValueReference;
import org.graylog2.contentpacks.model.entities.references.ValueType;
import org.graylog2.contentpacks.model.parameters.Parameter;
import org.graylog2.utilities.Graphs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class ContentPackService {
    private static final Logger LOG = LoggerFactory.getLogger(ContentPackService.class);
    private final ContentPackInstallationPersistenceService contentPackInstallationPersistenceService;
    private final Set<ConstraintChecker> constraintCheckers;
    private final Map<ModelType, EntityFacade<?>> entityFacades;

    @Inject
    public ContentPackService(ContentPackInstallationPersistenceService contentPackInstallationPersistenceService, Set<ConstraintChecker> constraintCheckers, Map<ModelType, EntityFacade<?>> entityFacades) {
        this.contentPackInstallationPersistenceService = contentPackInstallationPersistenceService;
        this.constraintCheckers = constraintCheckers;
        this.entityFacades = entityFacades;
    }

    public ContentPackInstallation installContentPack(ContentPack contentPack, Map<String, ValueReference> parameters, String comment, String user) {
        if (contentPack instanceof ContentPackV1) {
            return this.installContentPack((ContentPackV1)contentPack, parameters, comment, user);
        }
        throw new IllegalArgumentException("Unsupported content pack version: " + contentPack.version());
    }

    private ContentPackInstallation installContentPack(ContentPackV1 contentPack, Map<String, ValueReference> parameters, String comment, String user) {
        this.ensureConstraints(contentPack.constraints());
        Entity rootEntity = EntityV1.createRoot(contentPack);
        ImmutableMap<String, ValueReference> validatedParameters = this.validateParameters(parameters, (Set<Parameter>)contentPack.parameters());
        ImmutableGraph<Entity> dependencyGraph = this.buildEntityGraph(rootEntity, (Set<Entity>)contentPack.entities(), (Map<String, ValueReference>)validatedParameters);
        Traverser entityTraverser = Traverser.forGraph(dependencyGraph);
        Iterable entitiesInOrder = entityTraverser.depthFirstPostOrder((Object)rootEntity);
        LinkedHashMap<EntityDescriptor, Object> createdEntities = new LinkedHashMap<EntityDescriptor, Object>();
        HashMap<EntityDescriptor, Object> allEntities = new HashMap<EntityDescriptor, Object>();
        ImmutableSet.Builder allEntityDescriptors = ImmutableSet.builder();
        try {
            for (Entity entity : entitiesInOrder) {
                if (entity.equals(rootEntity)) continue;
                EntityDescriptor entityDescriptor = entity.toEntityDescriptor();
                EntityFacade facade = this.entityFacades.getOrDefault(entity.type(), UnsupportedEntityFacade.INSTANCE);
                Optional existingEntity = facade.findExisting(entity, parameters);
                if (existingEntity.isPresent()) {
                    LOG.trace("Found existing entity for {}", (Object)entityDescriptor);
                    NativeEntity nativeEntity = existingEntity.get();
                    NativeEntityDescriptor nativeEntityDescriptor = nativeEntity.descriptor();
                    if (this.contentPackInstallationPersistenceService.countInstallationOfEntityById(nativeEntityDescriptor.id()) <= 0L || this.contentPackInstallationPersistenceService.countInstallationOfEntityByIdAndFoundOnSystem(nativeEntityDescriptor.id()) > 0L) {
                        NativeEntityDescriptor serverDescriptor = nativeEntityDescriptor.toBuilder().foundOnSystem(true).build();
                        allEntityDescriptors.add((Object)serverDescriptor);
                    } else {
                        allEntityDescriptors.add((Object)nativeEntity.descriptor());
                    }
                    allEntities.put(entityDescriptor, nativeEntity.entity());
                    continue;
                }
                LOG.trace("Creating new entity for {}", (Object)entityDescriptor);
                NativeEntity createdEntity = facade.createNativeEntity(entity, (Map<String, ValueReference>)validatedParameters, (Map<EntityDescriptor, Object>)allEntities, user);
                allEntityDescriptors.add((Object)createdEntity.descriptor());
                createdEntities.put(entityDescriptor, createdEntity.entity());
                allEntities.put(entityDescriptor, createdEntity.entity());
            }
        }
        catch (Exception e) {
            this.rollback(createdEntities);
            throw new ContentPackException("Failed to install content pack <" + contentPack.id() + "/" + contentPack.revision() + ">", e);
        }
        ContentPackInstallation installation = ContentPackInstallation.builder().contentPackId(contentPack.id()).contentPackRevision(contentPack.revision()).parameters(validatedParameters).comment(comment).entities((ImmutableSet<NativeEntityDescriptor>)allEntityDescriptors.build()).createdAt(Instant.now()).createdBy(user).build();
        return this.contentPackInstallationPersistenceService.insert(installation);
    }

    private void rollback(Map<EntityDescriptor, Object> entities) {
        ImmutableList entries = ImmutableList.copyOf(entities.entrySet());
        for (Map.Entry entry : entries.reverse()) {
            EntityDescriptor entityDescriptor = (EntityDescriptor)entry.getKey();
            Object entity = entry.getValue();
            EntityFacade facade = this.entityFacades.getOrDefault(entityDescriptor.type(), UnsupportedEntityFacade.INSTANCE);
            LOG.debug("Removing entity {}", (Object)entityDescriptor);
            facade.delete(entity);
        }
    }

    public ContentPackUninstallDetails getUninstallDetails(ContentPack contentPack, ContentPackInstallation installation) {
        if (contentPack instanceof ContentPackV1) {
            return this.getUninstallDetails((ContentPackV1)contentPack, installation);
        }
        throw new IllegalArgumentException("Unsupported content pack version: " + contentPack.version());
    }

    private ContentPackUninstallDetails getUninstallDetails(ContentPackV1 contentPack, ContentPackInstallation installation) {
        Entity rootEntity = EntityV1.createRoot(contentPack);
        ImmutableMap<String, ValueReference> parameters = installation.parameters();
        ImmutableGraph<Entity> dependencyGraph = this.buildEntityGraph(rootEntity, (Set<Entity>)contentPack.entities(), (Map<String, ValueReference>)parameters);
        Traverser entityTraverser = Traverser.forGraph(dependencyGraph);
        Iterable entitiesInOrder = entityTraverser.breadthFirst((Object)rootEntity);
        HashSet<NativeEntityDescriptor> nativeEntityDescriptors = new HashSet<NativeEntityDescriptor>();
        entitiesInOrder.forEach(entity -> {
            NativeEntityDescriptor nativeEntityDescriptor;
            if (entity.equals(rootEntity)) {
                return;
            }
            Optional<NativeEntityDescriptor> nativeEntityDescriptorOptional = installation.entities().stream().filter(descriptor -> entity.id().equals(descriptor.contentPackEntityId())).findFirst();
            if (nativeEntityDescriptorOptional.isPresent() && this.contentPackInstallationPersistenceService.countInstallationOfEntityById((nativeEntityDescriptor = nativeEntityDescriptorOptional.get()).id()) <= 1L) {
                nativeEntityDescriptors.add(nativeEntityDescriptor);
            }
        });
        return ContentPackUninstallDetails.create(nativeEntityDescriptors);
    }

    public ContentPackUninstallation uninstallContentPack(ContentPack contentPack, ContentPackInstallation installation) {
        if (contentPack instanceof ContentPackV1) {
            return this.uninstallContentPack(installation, (ContentPackV1)contentPack);
        }
        throw new IllegalArgumentException("Unsupported content pack version: " + contentPack.version());
    }

    private ContentPackUninstallation uninstallContentPack(ContentPackInstallation installation, ContentPackV1 contentPack) {
        Entity rootEntity = EntityV1.createRoot(contentPack);
        ImmutableMap<String, ValueReference> parameters = installation.parameters();
        ImmutableGraph<Entity> dependencyGraph = this.buildEntityGraph(rootEntity, (Set<Entity>)contentPack.entities(), (Map<String, ValueReference>)parameters);
        Traverser entityTraverser = Traverser.forGraph(dependencyGraph);
        Iterable entitiesInOrder = entityTraverser.breadthFirst((Object)rootEntity);
        HashSet<NativeEntityDescriptor> removedEntities = new HashSet<NativeEntityDescriptor>();
        HashSet<NativeEntityDescriptor> failedEntities = new HashSet<NativeEntityDescriptor>();
        HashSet<NativeEntityDescriptor> skippedEntities = new HashSet<NativeEntityDescriptor>();
        try {
            for (Entity entity : entitiesInOrder) {
                if (entity.equals(rootEntity)) continue;
                Optional<NativeEntityDescriptor> nativeEntityDescriptorOptional = installation.entities().stream().filter(descriptor -> entity.id().equals(descriptor.contentPackEntityId())).findFirst();
                EntityFacade facade = this.entityFacades.getOrDefault(entity.type(), UnsupportedEntityFacade.INSTANCE);
                if (!nativeEntityDescriptorOptional.isPresent()) continue;
                NativeEntityDescriptor nativeEntityDescriptor = nativeEntityDescriptorOptional.get();
                Optional nativeEntityOptional = facade.loadNativeEntity(nativeEntityDescriptor);
                ModelId entityId = nativeEntityDescriptor.id();
                long installCount = this.contentPackInstallationPersistenceService.countInstallationOfEntityById(entityId);
                long systemFoundCount = this.contentPackInstallationPersistenceService.countInstallationOfEntityByIdAndFoundOnSystem(entityId);
                if (installCount > 1L || installCount == 1L && systemFoundCount >= 1L) {
                    skippedEntities.add(nativeEntityDescriptor);
                    LOG.debug("Did not remove entity since other content pack installations still use them: {}", (Object)nativeEntityDescriptor);
                    continue;
                }
                if (nativeEntityOptional.isPresent()) {
                    NativeEntity nativeEntity = nativeEntityOptional.get();
                    LOG.trace("Removing existing native entity for {} ({})", (Object)nativeEntityDescriptor);
                    try {
                        facade.delete(nativeEntity.entity());
                        removedEntities.add(nativeEntityDescriptor);
                    }
                    catch (Exception e) {
                        LOG.warn("Couldn't remove native entity {}", nativeEntity);
                        failedEntities.add(nativeEntityDescriptor);
                    }
                    continue;
                }
                LOG.trace("Couldn't find existing native entity for {} ({})", (Object)nativeEntityDescriptor);
            }
        }
        catch (Exception e) {
            throw new ContentPackException("Failed to remove content pack <" + contentPack.id() + "/" + contentPack.revision() + ">", e);
        }
        int deletedInstallations = this.contentPackInstallationPersistenceService.deleteById(installation.id());
        LOG.debug("Deleted {} installation(s) of content pack {}", (Object)deletedInstallations, (Object)contentPack.id());
        return ContentPackUninstallation.builder().entities((ImmutableSet<NativeEntityDescriptor>)ImmutableSet.copyOf(removedEntities)).skippedEntities((ImmutableSet<NativeEntityDescriptor>)ImmutableSet.copyOf(skippedEntities)).failedEntities((ImmutableSet<NativeEntityDescriptor>)ImmutableSet.copyOf(failedEntities)).build();
    }

    public Set<EntityExcerpt> listAllEntityExcerpts() {
        ImmutableSet.Builder entityIndexBuilder = ImmutableSet.builder();
        this.entityFacades.values().forEach(facade -> entityIndexBuilder.addAll(facade.listEntityExcerpts()));
        return entityIndexBuilder.build();
    }

    public Set<EntityDescriptor> resolveEntities(Collection<EntityDescriptor> unresolvedEntities) {
        MutableGraph dependencyGraph = GraphBuilder.directed().allowsSelfLoops(false).nodeOrder(ElementOrder.insertion()).build();
        unresolvedEntities.forEach(arg_0 -> ((MutableGraph)dependencyGraph).addNode(arg_0));
        HashSet<EntityDescriptor> resolvedEntities = new HashSet<EntityDescriptor>();
        MutableGraph<EntityDescriptor> finalDependencyGraph = this.resolveDependencyGraph((Graph<EntityDescriptor>)dependencyGraph, resolvedEntities);
        LOG.debug("Final dependency graph: {}", finalDependencyGraph);
        return finalDependencyGraph.nodes();
    }

    private MutableGraph<EntityDescriptor> resolveDependencyGraph(Graph<EntityDescriptor> dependencyGraph, Set<EntityDescriptor> resolvedEntities) {
        MutableGraph mutableGraph = GraphBuilder.from(dependencyGraph).build();
        Graphs.merge(mutableGraph, dependencyGraph);
        for (EntityDescriptor entityDescriptor : dependencyGraph.nodes()) {
            LOG.debug("Resolving entity {}", (Object)entityDescriptor);
            if (resolvedEntities.contains(entityDescriptor)) {
                LOG.debug("Entity {} already resolved, skipping.", (Object)entityDescriptor);
                continue;
            }
            EntityFacade facade = this.entityFacades.getOrDefault(entityDescriptor.type(), UnsupportedEntityFacade.INSTANCE);
            Graph<EntityDescriptor> graph = facade.resolveNativeEntity(entityDescriptor);
            LOG.trace("Dependencies of entity {}: {}", (Object)entityDescriptor, graph);
            Graphs.merge(mutableGraph, graph);
            LOG.trace("New dependency graph: {}", (Object)mutableGraph);
            resolvedEntities.add(entityDescriptor);
            MutableGraph<EntityDescriptor> result = this.resolveDependencyGraph((Graph<EntityDescriptor>)mutableGraph, resolvedEntities);
            Graphs.merge(mutableGraph, result);
        }
        return mutableGraph;
    }

    public ImmutableSet<Entity> collectEntities(Collection<EntityDescriptor> resolvedEntities) {
        EntityDescriptorIds entityDescriptorIds = EntityDescriptorIds.of(resolvedEntities);
        ImmutableSet.Builder entities = ImmutableSet.builder();
        for (EntityDescriptor entityDescriptor : resolvedEntities) {
            if (EntityDescriptorIds.isSystemStreamDescriptor(entityDescriptor)) continue;
            EntityFacade facade = this.entityFacades.getOrDefault(entityDescriptor.type(), UnsupportedEntityFacade.INSTANCE);
            facade.exportEntity(entityDescriptor, entityDescriptorIds).ifPresent(arg_0 -> ((ImmutableSet.Builder)entities).add(arg_0));
        }
        return entities.build();
    }

    private ImmutableGraph<Entity> buildEntityGraph(Entity rootEntity, Set<Entity> entities, Map<String, ValueReference> parameters) {
        Map<EntityDescriptor, Entity> entityDescriptorMap = entities.stream().collect(Collectors.toMap(Entity::toEntityDescriptor, Function.identity()));
        MutableGraph dependencyGraph = GraphBuilder.directed().allowsSelfLoops(false).expectedNodeCount(entities.size()).build();
        for (Map.Entry entry : entityDescriptorMap.entrySet()) {
            EntityDescriptor entityDescriptor = entry.getKey();
            Entity entity2 = (Entity)entry.getValue();
            EntityFacade facade = this.entityFacades.getOrDefault(entity2.type(), UnsupportedEntityFacade.INSTANCE);
            Graph<Entity> entityGraph = facade.resolveForInstallation(entity2, parameters, entityDescriptorMap);
            LOG.trace("Dependencies of entity {}: {}", (Object)entityDescriptor, entityGraph);
            dependencyGraph.putEdge((Object)rootEntity, (Object)entity2);
            Graphs.merge(dependencyGraph, entityGraph);
            LOG.trace("New dependency graph: {}", (Object)dependencyGraph);
        }
        Set<Entity> unexpectedEntities = dependencyGraph.nodes().stream().filter(entity -> !rootEntity.equals(entity)).filter(entity -> !entities.contains(entity)).collect(Collectors.toSet());
        if (!unexpectedEntities.isEmpty()) {
            throw new UnexpectedEntitiesException(unexpectedEntities);
        }
        return ImmutableGraph.copyOf((Graph)dependencyGraph);
    }

    private void ensureConstraints(Set<Constraint> requiredConstraints) {
        HashSet<Constraint> fulfilledConstraints = new HashSet<Constraint>();
        for (ConstraintChecker constraintChecker : this.constraintCheckers) {
            fulfilledConstraints.addAll(constraintChecker.ensureConstraints(requiredConstraints));
        }
        if (!fulfilledConstraints.equals(requiredConstraints)) {
            Sets.SetView failedConstraints = Sets.difference(requiredConstraints, fulfilledConstraints);
            throw new FailedConstraintsException((Collection<Constraint>)failedConstraints);
        }
    }

    public Set<ConstraintCheckResult> checkConstraints(ContentPack contentPack) {
        if (contentPack instanceof ContentPackV1) {
            return this.checkConstraintsV1((ContentPackV1)contentPack);
        }
        if (contentPack instanceof LegacyContentPack) {
            return Collections.emptySet();
        }
        throw new IllegalArgumentException("Unsupported content pack version: " + contentPack.version());
    }

    private Set<ConstraintCheckResult> checkConstraintsV1(ContentPackV1 contentPackV1) {
        Set<Constraint> requiredConstraints = contentPackV1.constraints();
        HashSet<ConstraintCheckResult> fulfilledConstraints = new HashSet<ConstraintCheckResult>();
        for (ConstraintChecker constraintChecker : this.constraintCheckers) {
            fulfilledConstraints.addAll(constraintChecker.checkConstraints(requiredConstraints));
        }
        return fulfilledConstraints;
    }

    private ImmutableMap<String, ValueReference> validateParameters(Map<String, ValueReference> parameters, Set<Parameter> contentPackParameters) {
        Set<String> contentPackParameterNames = contentPackParameters.stream().map(Parameter::name).collect(Collectors.toSet());
        this.checkUnknownParameters(parameters, contentPackParameterNames);
        this.checkMissingParameters(parameters, contentPackParameterNames);
        Map<String, ValueType> contentPackParameterValueTypes = contentPackParameters.stream().collect(Collectors.toMap(Parameter::name, Parameter::valueType));
        Set<String> invalidParameters = parameters.entrySet().stream().filter(entry -> ((ValueReference)entry.getValue()).valueType() != contentPackParameterValueTypes.get(entry.getKey())).map(Map.Entry::getKey).collect(Collectors.toSet());
        if (!invalidParameters.isEmpty()) {
            throw new InvalidParametersException(invalidParameters);
        }
        ImmutableMap.Builder validatedParameters = ImmutableMap.builder();
        for (Parameter contentPackParameter : contentPackParameters) {
            String name = contentPackParameter.name();
            ValueReference providedParameter = parameters.get(name);
            if (providedParameter == null) {
                Optional defaultValue = contentPackParameter.defaultValue();
                Object value = defaultValue.orElseThrow(() -> new EmptyDefaultValueException(name));
                ValueReference valueReference = ValueReference.builder().valueType(contentPackParameter.valueType()).value(value).build();
                validatedParameters.put((Object)name, (Object)valueReference);
                continue;
            }
            if (providedParameter.valueType() != contentPackParameter.valueType()) {
                throw new InvalidParameterTypeException(contentPackParameter.valueType(), providedParameter.valueType());
            }
            validatedParameters.put((Object)name, (Object)providedParameter);
        }
        return validatedParameters.build();
    }

    private void checkUnknownParameters(Map<String, ValueReference> parameters, Set<String> contentPackParameterNames) {
        Predicate<String> containsContentPackParameter = contentPackParameterNames::contains;
        Set unknownParameters = parameters.keySet().stream().filter(containsContentPackParameter.negate()).collect(Collectors.toSet());
        if (!unknownParameters.isEmpty()) {
            LOG.debug("Unknown parameters: {}", unknownParameters);
        }
    }

    private void checkMissingParameters(Map<String, ValueReference> parameters, Set<String> contentPackParameterNames) {
        Predicate<String> containsParameter = parameters::containsKey;
        Set<String> missingParameters = contentPackParameterNames.stream().filter(containsParameter.negate()).collect(Collectors.toSet());
        if (!missingParameters.isEmpty()) {
            throw new MissingParametersException(missingParameters);
        }
    }
}

