/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.cluster.lock;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.mongodb.MongoCommandException;
import com.mongodb.client.MongoCollection;
import com.mongodb.client.model.Filters;
import com.mongodb.client.model.FindOneAndUpdateOptions;
import com.mongodb.client.model.IndexOptions;
import com.mongodb.client.model.Indexes;
import com.mongodb.client.model.ReturnDocument;
import com.mongodb.client.model.Updates;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.Objects;
import java.util.Optional;
import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.inject.Inject;
import javax.inject.Named;
import javax.inject.Singleton;
import org.bson.Document;
import org.bson.conversions.Bson;
import org.graylog2.cluster.lock.Lock;
import org.graylog2.cluster.lock.LockService;
import org.graylog2.database.MongoConnection;
import org.graylog2.database.indices.MongoDbIndexTools;
import org.graylog2.plugin.system.NodeId;

@Singleton
public class MongoLockService
implements LockService {
    public static final String COLLECTION_NAME = "cluster_locks";
    public static final Duration MIN_LOCK_TTL = Duration.ofSeconds(60L);
    private final NodeId nodeId;
    private final MongoCollection<Document> collection;

    @Inject
    public MongoLockService(NodeId nodeId, MongoConnection mongoConnection, @Named(value="lock_service_lock_ttl") Duration lockTTL) {
        this.nodeId = nodeId;
        this.collection = mongoConnection.getMongoDatabase().getCollection(COLLECTION_NAME);
        this.collection.createIndex(Indexes.ascending((String[])new String[]{"resource"}), new IndexOptions().unique(true));
        MongoDbIndexTools.ensureTTLIndex(this.collection, lockTTL, "updated_at");
    }

    @Override
    public Optional<Lock> lock(@Nonnull String resource, @Nullable String lockContext) {
        return this.doLock(resource, this.getLockedByString(lockContext));
    }

    @Override
    public Optional<Lock> lock(@Nonnull String resource) {
        return this.lock(resource, null);
    }

    @Override
    public Optional<Lock> extendLock(@Nonnull Lock lock) {
        if (lock != null) {
            return this.doLock(lock.resource(), lock.lockedBy());
        }
        return Optional.empty();
    }

    private Optional<Lock> doLock(@Nonnull String resource, @Nonnull String lockedBy) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)resource) ? 1 : 0) != 0);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)lockedBy) ? 1 : 0) != 0);
        try {
            Document doc = (Document)this.collection.findOneAndUpdate(Filters.and((Bson[])new Bson[]{Filters.eq((String)"resource", (Object)resource), Filters.eq((String)"locked_by", (Object)lockedBy)}), Updates.combine((Bson[])new Bson[]{Updates.currentDate((String)"updated_at"), Updates.setOnInsert((String)"resource", (Object)resource), Updates.setOnInsert((String)"locked_by", (Object)lockedBy)}), new FindOneAndUpdateOptions().upsert(true).returnDocument(ReturnDocument.AFTER));
            return Optional.of(this.toLock(Objects.requireNonNull(doc)));
        }
        catch (MongoCommandException e) {
            if (e.getCode() == 11000) {
                return Optional.empty();
            }
            throw e;
        }
    }

    @Override
    public Optional<Lock> unlock(@Nonnull String resource, @Nullable String lockContext) {
        return this.doUnlock(resource, this.getLockedByString(lockContext));
    }

    @Override
    public Optional<Lock> unlock(@Nonnull Lock lock) {
        if (lock != null) {
            return this.doUnlock(lock.resource(), lock.lockedBy());
        }
        return Optional.empty();
    }

    private Optional<Lock> doUnlock(@Nonnull String resource, @Nonnull String lockedBy) {
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)resource) ? 1 : 0) != 0);
        Preconditions.checkArgument((!Strings.isNullOrEmpty((String)lockedBy) ? 1 : 0) != 0);
        Document deletedDocument = (Document)this.collection.findOneAndDelete(Filters.and((Bson[])new Bson[]{Filters.eq((String)"resource", (Object)resource), Filters.eq((String)"locked_by", (Object)lockedBy)}));
        if (deletedDocument != null) {
            return Optional.of(this.toLock(deletedDocument));
        }
        return Optional.empty();
    }

    private Lock toLock(Document doc) {
        ZonedDateTime createdAt = Instant.ofEpochSecond(doc.getObjectId((Object)"_id").getTimestamp()).atZone(ZoneOffset.UTC);
        ZonedDateTime updatedAt = doc.getDate((Object)"updated_at").toInstant().atZone(ZoneOffset.UTC);
        return Lock.builder().resource(doc.getString((Object)"resource")).createdAt(createdAt).updatedAt(updatedAt).lockedBy(doc.getString((Object)"locked_by")).build();
    }

    private String getLockedByString(@Nullable String lockContext) {
        if (lockContext == null) {
            return this.nodeId.getNodeId();
        }
        return this.nodeId.getNodeId() + "-" + lockContext;
    }
}

