/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.lookup.adapters.dnslookup;

import com.google.common.base.Joiner;
import com.google.common.base.Splitter;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.net.HostAndPort;
import com.google.common.net.InetAddresses;
import com.google.common.net.InternetDomainName;
import io.netty.buffer.ByteBuf;
import io.netty.channel.AddressedEnvelope;
import io.netty.handler.codec.dns.DefaultDnsPtrRecord;
import io.netty.handler.codec.dns.DefaultDnsQuestion;
import io.netty.handler.codec.dns.DefaultDnsRawRecord;
import io.netty.handler.codec.dns.DefaultDnsRecordDecoder;
import io.netty.handler.codec.dns.DnsQuestion;
import io.netty.handler.codec.dns.DnsRecord;
import io.netty.handler.codec.dns.DnsRecordType;
import io.netty.handler.codec.dns.DnsResponse;
import io.netty.handler.codec.dns.DnsSection;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;
import java.util.Objects;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.StringUtils;
import org.graylog2.lookup.adapters.dnslookup.ADnsAnswer;
import org.graylog2.lookup.adapters.dnslookup.DnsClientNotRunningException;
import org.graylog2.lookup.adapters.dnslookup.DnsResolverPool;
import org.graylog2.lookup.adapters.dnslookup.PtrDnsAnswer;
import org.graylog2.lookup.adapters.dnslookup.TxtDnsAnswer;
import org.graylog2.shared.utilities.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DnsClient {
    private static final Logger LOG = LoggerFactory.getLogger(DnsClient.class);
    private static final int DEFAULT_DNS_PORT = 53;
    private static final int DEFAULT_REQUEST_TIMEOUT_INCREMENT = 100;
    private static final Pattern VALID_HOSTNAME_PATTERN = Pattern.compile("^(([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\\-]*[a-zA-Z0-9])\\.)*([A-Za-z0-9]|[A-Za-z0-9][A-Za-z0-9\\-]*[A-Za-z0-9])$");
    private static final String IP_4_REVERSE_SUFFIX = ".in-addr.arpa.";
    private static final String IP_6_REVERSE_SUFFIX = ".ip6.arpa.";
    private static final String IP_4_VERSION = "IPv4";
    private static final String IP_6_VERSION = "IPv6";
    private static final char[] HEX_CHARS_ARRAY = "0123456789ABCDEF".toCharArray();
    private final long queryTimeout;
    private final long requestTimeout;
    private final long resolverPoolSize;
    private final long resolverPoolRefreshSeconds;
    private DnsResolverPool resolverPool;

    public DnsClient(long queryTimeout) {
        this(queryTimeout, queryTimeout + 100L);
    }

    public DnsClient(long queryTimeout, long requestTimeout) {
        this(queryTimeout, requestTimeout, 10, 300L);
    }

    public DnsClient(long queryTimeout, int resolverPoolSize, long resolverPoolRefreshSeconds) {
        this(queryTimeout, queryTimeout + 100L, resolverPoolSize, resolverPoolRefreshSeconds);
    }

    private DnsClient(long queryTimeout, long requestTimeout, int resolverPoolSize, long resolverPoolRefreshSeconds) {
        this.queryTimeout = queryTimeout;
        this.requestTimeout = requestTimeout;
        this.resolverPoolSize = resolverPoolSize;
        this.resolverPoolRefreshSeconds = resolverPoolRefreshSeconds;
    }

    public void start(String dnsServerIps) {
        LOG.debug("Attempting to start DNS client");
        this.resolverPool = new DnsResolverPool(dnsServerIps, this.queryTimeout, this.resolverPoolSize, this.resolverPoolRefreshSeconds);
        this.resolverPool.initialize();
    }

    public void stop() {
        LOG.debug("Attempting to stop DNS client");
        if (this.resolverPool == null) {
            LOG.error("DNS resolution pool is not initialized.");
            return;
        }
        this.resolverPool.stop();
    }

    public List<ADnsAnswer> resolveIPv4AddressForHostname(String hostName, boolean includeIpVersion) throws InterruptedException, ExecutionException, UnknownHostException {
        return this.resolveIpAddresses(hostName, DnsRecordType.A, includeIpVersion);
    }

    public List<ADnsAnswer> resolveIPv6AddressForHostname(String hostName, boolean includeIpVersion) throws InterruptedException, ExecutionException, UnknownHostException {
        return this.resolveIpAddresses(hostName, DnsRecordType.AAAA, includeIpVersion);
    }

    private List<ADnsAnswer> resolveIpAddresses(String hostName, DnsRecordType dnsRecordType, boolean includeIpVersion) throws InterruptedException, ExecutionException {
        LOG.debug("Attempting to resolve [{}] records for [{}]", (Object)dnsRecordType, (Object)hostName);
        if (this.resolverPool.isStopped()) {
            throw new DnsClientNotRunningException();
        }
        this.validateHostName(hostName);
        DefaultDnsQuestion aRecordDnsQuestion = new DefaultDnsQuestion(hostName, dnsRecordType);
        DnsResolverPool.ResolverLease resolverLease = this.resolverPool.takeLease();
        try {
            List<ADnsAnswer> list = ((List)resolverLease.getResolver().resolveAll((DnsQuestion)aRecordDnsQuestion).get(this.requestTimeout, TimeUnit.MILLISECONDS)).stream().map(dnsRecord -> DnsClient.decodeDnsRecord(dnsRecord, includeIpVersion)).filter(Objects::nonNull).collect(Collectors.toList());
            return list;
        }
        catch (TimeoutException e) {
            throw new ExecutionException("Resolver future didn't return a result in " + this.requestTimeout + " ms", e);
        }
        finally {
            this.resolverPool.returnLease(resolverLease);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static ADnsAnswer decodeDnsRecord(DnsRecord dnsRecord, boolean includeIpVersion) {
        InetAddress ipAddress;
        byte[] ipAddressBytes;
        if (dnsRecord == null) {
            return null;
        }
        LOG.trace("Attempting to decode DNS record [{}]", (Object)dnsRecord);
        DefaultDnsRawRecord dnsRawRecord = (DefaultDnsRawRecord)dnsRecord;
        try {
            ByteBuf byteBuf = dnsRawRecord.content();
            ipAddressBytes = new byte[byteBuf.readableBytes()];
            int readerIndex = byteBuf.readerIndex();
            byteBuf.getBytes(readerIndex, ipAddressBytes);
        }
        finally {
            dnsRawRecord.release();
        }
        LOG.trace("The IP address has [{}] bytes", (Object)ipAddressBytes.length);
        try {
            ipAddress = InetAddress.getByAddress(ipAddressBytes);
        }
        catch (UnknownHostException e) {
            LOG.error("Could not extract IP address from DNS entry [{}]. Cause [{}]", (Object)dnsRecord.toString(), (Object)ExceptionUtils.getRootCauseMessage(e));
            return null;
        }
        LOG.trace("The resulting IP address is [{}]", (Object)ipAddress.getHostAddress());
        ADnsAnswer.Builder builder = ADnsAnswer.builder().ipAddress(ipAddress.getHostAddress()).dnsTTL(dnsRecord.timeToLive());
        if (includeIpVersion) {
            builder.ipVersion(ipAddress instanceof Inet4Address ? IP_4_VERSION : IP_6_VERSION);
        }
        return builder.build();
    }

    public PtrDnsAnswer reverseLookup(String ipAddress) throws InterruptedException, ExecutionException {
        LOG.debug("Attempting to perform reverse lookup for IP address [{}]", (Object)ipAddress);
        if (this.resolverPool.isStopped()) {
            throw new DnsClientNotRunningException();
        }
        DnsClient.validateIpAddress(ipAddress);
        String inverseAddressFormat = this.getInverseAddressFormat(ipAddress);
        DnsResponse content = null;
        DnsResolverPool.ResolverLease resolverLease = this.resolverPool.takeLease();
        try {
            content = (DnsResponse)((AddressedEnvelope)resolverLease.getResolver().query((DnsQuestion)new DefaultDnsQuestion(inverseAddressFormat, DnsRecordType.PTR)).get(this.requestTimeout, TimeUnit.MILLISECONDS)).content();
            for (int i = 0; i < content.count(DnsSection.ANSWER); ++i) {
                DnsRecord dnsRecord = content.recordAt(DnsSection.ANSWER, i);
                if (!(dnsRecord instanceof DefaultDnsPtrRecord)) continue;
                DefaultDnsPtrRecord ptrRecord = (DefaultDnsPtrRecord)dnsRecord;
                PtrDnsAnswer.Builder dnsAnswerBuilder = PtrDnsAnswer.builder();
                String hostname = ptrRecord.hostname();
                LOG.trace("PTR record retrieved with hostname [{}]", (Object)hostname);
                try {
                    DnsClient.parseReverseLookupDomain(dnsAnswerBuilder, hostname);
                }
                catch (IllegalArgumentException e) {
                    LOG.debug("Reverse lookup of [{}] was partially successful. The DNS server returned [{}], which is an invalid host name. The \"domain\" field will be left blank.", (Object)ipAddress, (Object)hostname);
                    dnsAnswerBuilder.domain("");
                }
                PtrDnsAnswer ptrDnsAnswer = dnsAnswerBuilder.dnsTTL(ptrRecord.timeToLive()).build();
                return ptrDnsAnswer;
            }
        }
        catch (TimeoutException e) {
            throw new ExecutionException("Resolver future didn't return a result in " + this.requestTimeout + " ms", e);
        }
        finally {
            if (content != null) {
                content.release();
            }
            this.resolverPool.returnLease(resolverLease);
        }
        return null;
    }

    public static void parseReverseLookupDomain(PtrDnsAnswer.Builder dnsAnswerBuilder, String hostname) {
        dnsAnswerBuilder.fullDomain(hostname);
        InternetDomainName internetDomainName = InternetDomainName.from((String)hostname);
        if (internetDomainName.hasPublicSuffix()) {
            InternetDomainName topDomainName = internetDomainName.topDomainUnderRegistrySuffix();
            dnsAnswerBuilder.domain(topDomainName.toString());
        } else {
            String[] split = hostname.split("\\.");
            if (split.length > 1) {
                dnsAnswerBuilder.domain(split[split.length - 2] + "." + split[split.length - 1]);
            } else if (split.length == 1) {
                dnsAnswerBuilder.domain(hostname);
            } else {
                dnsAnswerBuilder.domain("");
            }
        }
    }

    public List<TxtDnsAnswer> txtLookup(String hostName) throws InterruptedException, ExecutionException {
        if (this.resolverPool.isStopped()) {
            throw new DnsClientNotRunningException();
        }
        LOG.debug("Attempting to perform TXT lookup for hostname [{}]", (Object)hostName);
        this.validateHostName(hostName);
        DnsResponse content = null;
        DnsResolverPool.ResolverLease resolverLease = this.resolverPool.takeLease();
        try {
            content = (DnsResponse)((AddressedEnvelope)resolverLease.getResolver().query((DnsQuestion)new DefaultDnsQuestion(hostName, DnsRecordType.TXT)).get(this.requestTimeout, TimeUnit.MILLISECONDS)).content();
            int count = content.count(DnsSection.ANSWER);
            ArrayList<TxtDnsAnswer> txtRecords = new ArrayList<TxtDnsAnswer>(count);
            for (int i = 0; i < count; ++i) {
                DnsRecord dnsRecord = content.recordAt(DnsSection.ANSWER, i);
                LOG.trace("TXT record [{}] retrieved with content [{}].", (Object)i, (Object)dnsRecord);
                if (!(dnsRecord instanceof DefaultDnsRawRecord)) continue;
                DefaultDnsRawRecord txtRecord = (DefaultDnsRawRecord)dnsRecord;
                TxtDnsAnswer.Builder dnsAnswerBuilder = TxtDnsAnswer.builder();
                String decodeTxtRecord = DnsClient.decodeTxtRecord(txtRecord);
                LOG.trace("The decoded TXT record is [{}]", (Object)decodeTxtRecord);
                dnsAnswerBuilder.value(decodeTxtRecord).dnsTTL(txtRecord.timeToLive()).build();
                txtRecords.add(dnsAnswerBuilder.build());
            }
            ArrayList<TxtDnsAnswer> arrayList = txtRecords;
            return arrayList;
        }
        catch (TimeoutException e) {
            throw new ExecutionException("Resolver future didn't return a result in " + this.requestTimeout + " ms", e);
        }
        finally {
            if (content != null) {
                content.release();
            }
            this.resolverPool.returnLease(resolverLease);
        }
    }

    private static String decodeTxtRecord(DefaultDnsRawRecord record) {
        LOG.debug("Attempting to read TXT value from DNS record [{}]", (Object)record);
        return DefaultDnsRecordDecoder.decodeName((ByteBuf)record.content());
    }

    public String getInverseAddressFormat(String ipAddress) {
        ipAddress = StringUtils.trim((String)ipAddress);
        DnsClient.validateIpAddress(ipAddress);
        LOG.debug("Preparing inverse format for IP address [{}]", (Object)ipAddress);
        if (DnsClient.isIp6Address(ipAddress)) {
            LOG.debug("[{}] is an IPv6 address", (Object)ipAddress);
            byte[] addressBytes = InetAddresses.forString((String)ipAddress).getAddress();
            if (addressBytes.length > 16) {
                throw new IllegalArgumentException(String.format(Locale.ENGLISH, "[%s] is an invalid IPv6 address", ipAddress));
            }
            char[] resolvedHex = new char[addressBytes.length * 2];
            for (int i = 0; i < addressBytes.length; ++i) {
                int v = addressBytes[i] & 0xFF;
                resolvedHex[i * 2] = HEX_CHARS_ARRAY[v >>> 4];
                resolvedHex[i * 2 + 1] = HEX_CHARS_ARRAY[v & 0xF];
            }
            String fullHexAddress = new String(resolvedHex).toLowerCase(Locale.ENGLISH);
            Object[] reversedAndSplit = new StringBuilder(fullHexAddress).reverse().toString().split("");
            String invertedAddress = Joiner.on((String)".").join(reversedAndSplit);
            LOG.debug("Inverted address [{}] built for [{}]", (Object)invertedAddress, (Object)ipAddress);
            return invertedAddress + IP_6_REVERSE_SUFFIX;
        }
        LOG.debug("[{}] is an IPv4 address", (Object)ipAddress);
        String[] octets = ipAddress.split("\\.");
        String invertedAddress = octets[3] + "." + octets[2] + "." + octets[1] + "." + octets[0] + IP_4_REVERSE_SUFFIX;
        LOG.debug("Inverted address [{}] built for [{}]", (Object)invertedAddress, (Object)ipAddress);
        return invertedAddress;
    }

    private void validateHostName(String hostName) {
        if (!DnsClient.isHostName(hostName)) {
            throw new IllegalArgumentException(String.format(Locale.ENGLISH, "[%s] is an invalid hostname. Please supply a pure hostname (eg. api.graylog.com)", hostName));
        }
    }

    public static boolean isHostName(String hostName) {
        return VALID_HOSTNAME_PATTERN.matcher(hostName).matches();
    }

    private static void validateIpAddress(String ipAddress) {
        if (!DnsClient.isValidIpAddress(ipAddress)) {
            throw new IllegalArgumentException(String.format(Locale.ENGLISH, "[%s] is an invalid IP address.", ipAddress));
        }
    }

    private static boolean isValidIpAddress(String ipAddress) {
        return DnsClient.isIp4Address(ipAddress) || DnsClient.isIp6Address(ipAddress);
    }

    public static boolean isIp4Address(String ipAddress) {
        try {
            InetAddress address = InetAddresses.forString((String)ipAddress);
            if (address instanceof Inet4Address) {
                return true;
            }
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        return false;
    }

    public static boolean isIp6Address(String ipAddress) {
        try {
            InetAddress address = InetAddresses.forString((String)ipAddress);
            if (address instanceof Inet6Address) {
                return true;
            }
        }
        catch (IllegalArgumentException illegalArgumentException) {
            // empty catch block
        }
        return false;
    }

    public static boolean allIpAddressesValid(String ipAddresses) {
        if (!Strings.isNullOrEmpty((String)ipAddresses)) {
            return Lists.newArrayList((Iterable)Splitter.on((String)",").trimResults().omitEmptyStrings().split((CharSequence)ipAddresses)).stream().map(hostAndPort -> HostAndPort.fromString((String)hostAndPort).withDefaultPort(53)).allMatch(hostAndPort -> DnsClient.isValidIpAddress(hostAndPort.getHost()));
        }
        return false;
    }
}

