/*
 * Decompiled with CFR 0.152.
 */
package org.graylog2.utilities;

import com.google.common.net.InetAddresses;
import java.math.BigInteger;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.Objects;
import java.util.Optional;
import org.graylog2.shared.utilities.StringUtils;

public class IpSubnet {
    private static final byte[] MASK_IPV4 = new byte[]{-1, -1, -1, -1};
    private static final byte[] MASK_IPV6 = new byte[]{-1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1};
    private final InetAddress inetAddress;
    private InetAddress startAddress;
    private InetAddress endAddress;
    private final int prefixLength;

    public IpSubnet(String cidr) throws UnknownHostException {
        Objects.requireNonNull(cidr, "CIDR must not be null");
        if (!cidr.contains("/")) {
            throw new UnknownHostException("Invalid subnet: " + cidr);
        }
        int index = cidr.indexOf("/");
        String addressPart = cidr.substring(0, index);
        String networkPart = cidr.substring(index + 1);
        this.inetAddress = InetAddress.getByName(addressPart);
        this.prefixLength = Integer.parseInt(networkPart);
        this.calculate();
    }

    private void calculate() throws UnknownHostException {
        BigInteger mask;
        int targetSize;
        if (this.inetAddress.getAddress().length == 4) {
            targetSize = 4;
            mask = new BigInteger(1, MASK_IPV4).not().shiftRight(this.prefixLength);
        } else {
            targetSize = 16;
            mask = new BigInteger(1, MASK_IPV6).not().shiftRight(this.prefixLength);
        }
        BigInteger ipVal = new BigInteger(1, this.inetAddress.getAddress());
        BigInteger startIp = ipVal.and(mask);
        BigInteger endIp = startIp.add(mask.not());
        byte[] startIpArr = this.toBytes(startIp.toByteArray(), targetSize);
        byte[] endIpArr = this.toBytes(endIp.toByteArray(), targetSize);
        this.startAddress = InetAddress.getByAddress(startIpArr);
        this.endAddress = InetAddress.getByAddress(endIpArr);
    }

    private byte[] toBytes(byte[] array, int targetSize) {
        byte[] b = new byte[targetSize];
        int length = targetSize > array.length ? array.length : targetSize;
        int srcPos = array.length > targetSize ? array.length - targetSize : 0;
        int destPos = targetSize > array.length ? targetSize - array.length : 0;
        System.arraycopy(array, srcPos, b, destPos, length);
        return b;
    }

    public String getNetworkAddress() {
        return this.startAddress.getHostAddress();
    }

    public String getBroadcastAddress() {
        return this.endAddress.getHostAddress();
    }

    public boolean contains(String ipAddress) throws UnknownHostException {
        return this.contains(InetAddress.getByName(ipAddress));
    }

    public boolean contains(InetAddress address) {
        BigInteger start = new BigInteger(1, this.startAddress.getAddress());
        BigInteger end = new BigInteger(1, this.endAddress.getAddress());
        BigInteger target = new BigInteger(1, address.getAddress());
        int st = start.compareTo(target);
        int te = target.compareTo(end);
        return !(st != -1 && st != 0 || te != -1 && te != 0);
    }

    public int getPrefixLength() {
        return this.prefixLength;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        IpSubnet that = (IpSubnet)o;
        return Objects.equals(this.startAddress, that.startAddress) && Objects.equals(this.endAddress, that.endAddress) && Objects.equals(this.prefixLength, that.prefixLength);
    }

    public int hashCode() {
        return Objects.hash(this.startAddress, this.endAddress, this.prefixLength);
    }

    public String toString() {
        return this.inetAddress.getHostAddress() + "/" + this.prefixLength;
    }

    public static Optional<String> formatCIDR(String value) {
        Optional<String> cidr = Optional.empty();
        try {
            new IpSubnet(value);
            cidr = Optional.of(value);
        }
        catch (UnknownHostException e) {
            try {
                InetAddress address = InetAddresses.forString((String)value);
                if (address instanceof Inet4Address) {
                    cidr = Optional.of(StringUtils.f("%s/32", value));
                } else if (address instanceof Inet6Address) {
                    cidr = Optional.of(StringUtils.f("%s/128", value));
                }
            }
            catch (IllegalArgumentException illegalArgumentException) {
                // empty catch block
            }
        }
        return cidr;
    }
}

