/*
 * Decompiled with CFR 0.152.
 */
package org.graylog.integrations.aws.service;

import com.github.rholder.retry.Attempt;
import com.github.rholder.retry.RetryException;
import com.github.rholder.retry.RetryListener;
import com.github.rholder.retry.Retryer;
import com.github.rholder.retry.RetryerBuilder;
import com.github.rholder.retry.StopStrategies;
import com.github.rholder.retry.WaitStrategies;
import jakarta.inject.Inject;
import jakarta.ws.rs.BadRequestException;
import java.util.ArrayList;
import java.util.Locale;
import java.util.concurrent.TimeUnit;
import org.graylog.integrations.aws.AWSClientBuilderUtil;
import org.graylog.integrations.aws.resources.requests.AWSRequest;
import org.graylog.integrations.aws.resources.requests.CreateLogSubscriptionRequest;
import org.graylog.integrations.aws.resources.responses.CreateLogSubscriptionResponse;
import org.graylog.integrations.aws.resources.responses.LogGroupsResponse;
import org.graylog2.shared.utilities.ExceptionUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.services.cloudwatchlogs.CloudWatchLogsClient;
import software.amazon.awssdk.services.cloudwatchlogs.CloudWatchLogsClientBuilder;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsRequest;
import software.amazon.awssdk.services.cloudwatchlogs.model.DescribeLogGroupsResponse;
import software.amazon.awssdk.services.cloudwatchlogs.model.Distribution;
import software.amazon.awssdk.services.cloudwatchlogs.model.InvalidParameterException;
import software.amazon.awssdk.services.cloudwatchlogs.model.LogGroup;
import software.amazon.awssdk.services.cloudwatchlogs.model.PutSubscriptionFilterRequest;
import software.amazon.awssdk.services.cloudwatchlogs.paginators.DescribeLogGroupsIterable;

public class CloudWatchService {
    private static final Logger LOG = LoggerFactory.getLogger(CloudWatchService.class);
    private static final int SUBSCRIPTION_RETRY_DELAY = 1000;
    private static final int SUBSCRIPTION_RETRY_MAX_ATTEMPTS = 120;
    private static final TimeUnit SUBSCRIPTION_RETRY_DELAY_UNIT = TimeUnit.MILLISECONDS;
    private final CloudWatchLogsClientBuilder logsClientBuilder;
    private final AWSClientBuilderUtil awsClientBuilderUtil;

    @Inject
    public CloudWatchService(CloudWatchLogsClientBuilder logsClientBuilder, AWSClientBuilderUtil awsClientBuilderUtil) {
        this.logsClientBuilder = logsClientBuilder;
        this.awsClientBuilderUtil = awsClientBuilderUtil;
    }

    public LogGroupsResponse getLogGroupNames(AWSRequest request) {
        CloudWatchLogsClient cloudWatchLogsClient = this.awsClientBuilderUtil.buildClient(this.logsClientBuilder, request);
        DescribeLogGroupsRequest describeLogGroupsRequest = (DescribeLogGroupsRequest)DescribeLogGroupsRequest.builder().build();
        DescribeLogGroupsIterable responses = cloudWatchLogsClient.describeLogGroupsPaginator(describeLogGroupsRequest);
        ArrayList<String> groupNameList = new ArrayList<String>();
        for (DescribeLogGroupsResponse response : responses) {
            for (int c = 0; c < response.logGroups().size(); ++c) {
                groupNameList.add(((LogGroup)response.logGroups().get(c)).logGroupName());
            }
        }
        LOG.debug("Log groups queried: [{}]", groupNameList);
        if (groupNameList.isEmpty()) {
            throw new BadRequestException(String.format(Locale.ROOT, "No CloudWatch log groups were found in the [%s] region.", request.region()));
        }
        return LogGroupsResponse.create(groupNameList, groupNameList.size());
    }

    public CreateLogSubscriptionResponse addSubscriptionFilter(final CreateLogSubscriptionRequest request) {
        CloudWatchLogsClient cloudWatch = this.awsClientBuilderUtil.buildClient(this.logsClientBuilder, (AWSRequest)request);
        PutSubscriptionFilterRequest putSubscriptionFilterRequest = (PutSubscriptionFilterRequest)PutSubscriptionFilterRequest.builder().logGroupName(request.logGroupName()).filterName(request.filterName()).filterPattern(request.filterPattern()).destinationArn(request.destinationStreamArn()).roleArn(request.roleArn()).distribution(Distribution.BY_LOG_STREAM).build();
        try {
            Retryer retryer = RetryerBuilder.newBuilder().retryIfExceptionOfType(InvalidParameterException.class).withWaitStrategy(WaitStrategies.fixedWait((long)1000L, (TimeUnit)SUBSCRIPTION_RETRY_DELAY_UNIT)).withStopStrategy(StopStrategies.stopAfterAttempt((int)120)).withRetryListener(new RetryListener(){

                public <V> void onRetry(Attempt<V> attempt) {
                    if (attempt.hasException()) {
                        LOG.info("Failed to create log group subscription on retry attempt [{}]. This is probably normal and indicates that the specified IAM role is not ready yet due to IAM eventual consistency.Retrying now. Exception [{}]", (Object)attempt.getAttemptNumber(), (Object)attempt.getExceptionCause().getMessage());
                    } else if (attempt.hasException() && attempt.getAttemptNumber() == 120L) {
                        LOG.error("Failed to put subscription after [{}] attempts. Giving up. Exception [{}]", (Object)attempt.getAttemptNumber(), (Object)attempt.getExceptionCause());
                    } else if (attempt.getAttemptNumber() > 1L) {
                        LOG.info("Retry of CloudWatch log group [{}] subscription was finally successful on attempt [{}].", (Object)request.logGroupName(), (Object)attempt.getAttemptNumber());
                    }
                }
            }).build();
            try {
                retryer.call(() -> {
                    cloudWatch.putSubscriptionFilter(putSubscriptionFilterRequest);
                    return null;
                });
                String explanation = String.format(Locale.ROOT, "Success. The subscription filter [%s] was added for the CloudWatch log group [%s].", request.filterName(), request.logGroupName());
                return CreateLogSubscriptionResponse.create(explanation);
            }
            catch (RetryException e) {
                throw new RuntimeException(String.format(Locale.ROOT, "Failed to create the CloudWatch subscription after [%d] attempts. Exception [%s]", e.getNumberOfFailedAttempts(), e.getCause()), e.getCause());
            }
        }
        catch (Exception e) {
            String specificError = ExceptionUtils.formatMessageCause(e);
            String responseMessage = String.format(Locale.ROOT, "Attempt to add subscription [%s] to Cloudwatch log group [%s] failed due to the following exception: [%s]", request.filterName(), request.logGroupName(), specificError);
            LOG.error(responseMessage);
            throw new BadRequestException(responseMessage, (Throwable)e);
        }
    }
}

