package org.jbehave.scenario.reporters;

import static java.util.Arrays.asList;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;

import freemarker.template.Configuration;
import freemarker.template.ObjectWrapper;
import freemarker.template.Template;
import freemarker.template.TemplateException;

/**
 * <p>Freemarker-based report renderer, using the file outputs generated by the
 * reporters for the given formats. The FTL templates for the index and single
 * views are injectable via the {@link FreemarkerReportRender(Properties)}
 * constructor, but defaults are provided. To override, specify the the path the
 * new template under keys "index" and "single".</p>
 * <p>The report renderer provids the following template resources:
 * <pre>
 * resources.setProperty("index", "ftl/jbehave-reports-index.ftl");
 * resources.setProperty("single", "ftl/jbehave-reports-single.ftl");
 * resources.setProperty("renderedDirectory", "rendered");
 * resources.setProperty("defaultFormats", "stats");
 * </pre>  
 * </p>
 * @author Mauro Talevi
 */
public class FreemarkerReportRenderer implements ReportRenderer {

    private final Configuration configuration;
    private final Properties templateResources;

    public FreemarkerReportRenderer() {
        this(defaultTemplateResources());
    }

    private static Properties defaultTemplateResources() {
        Properties resources = new Properties();
        resources.setProperty("index", "ftl/jbehave-reports-index.ftl");
        resources.setProperty("single", "ftl/jbehave-reports-single.ftl");
        resources.setProperty("renderedDirectory", "rendered");
        resources.setProperty("defaultFormats", "stats");
        return resources;
    }

    public FreemarkerReportRenderer(Properties templateResources) {
        this.configuration = configure();
        this.templateResources = mergeWithDefault(templateResources);
    }

    private Properties mergeWithDefault(Properties templateResources) {
        Properties resources = defaultTemplateResources();
        resources.putAll(templateResources);
        return resources;
    }

    public void render(File outputDirectory, List<String> formats) {
        createIndex(outputDirectory, formats);
    }

    private void createIndex(File outputDirectory, List<String> formats) {
        String outputName = templateResource("renderedDirectory")+"/index.html";
        String resource = templateResource("index");
        List<String> mergedFormats = mergeWithDefaults(formats);
        Map<String, List<File>> reportFiles = generatedReportFiles(outputDirectory, outputName, mergedFormats);
        List<Report> renderedReports = renderedReports(reportFiles);
        Map<String, Object> dataModel = newDataModel();
        dataModel.put("reports", renderedReports);
        dataModel.put("date", new Date());
        write(outputDirectory, outputName, resource, dataModel);
    }

    private List<String> mergeWithDefaults(List<String> formats) {
        List<String> merged = new ArrayList<String>();
        merged.addAll(asList(templateResource("defaultFormats").split(",")));        
        merged.addAll(formats);
        return merged;
    }

    private Map<String, List<File>> generatedReportFiles(File outputDirectory, final String outputName,
            final List<String> formats) {
        Map<String, List<File>> reports = new HashMap<String, List<File>>();
        if (outputDirectory == null || !outputDirectory.exists()) {
            return reports;
        }
        String[] fileNames = outputDirectory.list(new FilenameFilter() {
            public boolean accept(File dir, String name) {
                return !name.equals(outputName) && hasFormats(name, formats);
            }

            private boolean hasFormats(String name, List<String> formats) {
                for (String format : formats) {
                    if (name.endsWith(format)) {
                        return true;
                    }
                }
                return false;
            }
        });
        for (String fileName : fileNames) {
            String name = FilenameUtils.getBaseName(fileName);
            List<File> filesByName = reports.get(name);
            if (filesByName == null) {
                filesByName = new ArrayList<File>();
                reports.put(name, filesByName);
            }
            filesByName.add(new File(outputDirectory, fileName));
        }
        return reports;
    }

    private List<Report> renderedReports(Map<String, List<File>> reportFiles) {
        try {
            String resource = templateResource("single");
            String renderedDirectory = templateResource("renderedDirectory");
            List<Report> reports = new ArrayList<Report>();
            for (String name : reportFiles.keySet()) {
                Map<String, File> filesByFormat = new HashMap<String, File>();
                for (File file : reportFiles.get(name)) {
                    String fileName = file.getName();
                    String format = FilenameUtils.getExtension(fileName);
                    Map<String, Object> dataModel = newDataModel();
                    dataModel.put("name", name);
                    dataModel.put("body", IOUtils.toString(new FileReader(file)));
                    dataModel.put("format", format);
                    File outputDirectory = file.getParentFile();
                    String outputName = renderedDirectory+ "/" + fileName;
                    if (!format.equals("html")) {
                        outputName = outputName + ".html";
                    }
                    File written = write(outputDirectory, outputName, resource, dataModel);
                    filesByFormat.put(format, written);
                }
                reports.add(new Report(name, filesByFormat));
            }
            return reports;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private File write(File outputDirectory, String outputName, String resource, Map<String, Object> dataModel) {
        try {
            File file = new File(outputDirectory, outputName);
            file.getParentFile().mkdirs();
            Writer writer = new FileWriter(file);
            process(resource, dataModel, writer);
            return file;
        } catch (Exception e) {
            throw new RenderingFailedException(resource, e);
        }
    }

    private Configuration configure() {
        Configuration configuration = new Configuration();
        configuration.setClassForTemplateLoading(FreemarkerReportRenderer.class, "/");
        configuration.setObjectWrapper(ObjectWrapper.BEANS_WRAPPER);
        return configuration;
    }

    private void process(String resource, Map<String, Object> dataModel, Writer writer) throws TemplateException,
            IOException {
        Template template = configuration.getTemplate(resource);
        template.process(dataModel, writer);
    }

    private String templateResource(String format) {
        String resource = templateResources.getProperty(format);
        if (resource == null) {
            throw new RendererTemplateNotFoundException(format);
        }
        return resource;
    }

    private Map<String, Object> newDataModel() {
        return new HashMap<String, Object>();
    }

    @SuppressWarnings("serial")
    public static class RenderingFailedException extends RuntimeException {

        public RenderingFailedException(String resource, Exception cause) {
            super("Failed rendering report with resource " + resource, cause);
        }

    }

    @SuppressWarnings("serial")
    public static class RendererTemplateNotFoundException extends RuntimeException {

        public RendererTemplateNotFoundException(String format) {
            super("Renderer template not found for format " + format);
        }

    }

    public static class Report {

        private final String name;
        private final Map<String, File> filesByFormat;

        public Report(String name, Map<String, File> filesByFormat) {
            this.name = name;
            this.filesByFormat = filesByFormat;
        }

        public String getName() {
            return name;
        }

        public Map<String, File> getFilesByFormat() {
            return filesByFormat;
        }
        
        public Properties asProperties(String format){
            Properties p = new Properties();
            File stats = filesByFormat.get(format);
            if ( stats == null ){
                return p;
            }
            try {
                p.load(new FileInputStream(stats));
            } catch (IOException e) {
                // return empty map
            }
            return p;
        }

    }
}
