package org.mapstruct.ap.internal.gem;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.lang.model.element.AnnotationMirror;
import javax.lang.model.element.AnnotationValue;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.element.VariableElement;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.util.AbstractAnnotationValueVisitor8;
import javax.lang.model.util.ElementFilter;
import org.mapstruct.tools.gem.Gem;
import org.mapstruct.tools.gem.GemValue;

import javax.lang.model.type.TypeMirror;

public class AnnotateWithGem implements Gem {

    private final GemValue<TypeMirror> value;
    private final GemValue<List<ElementGem>> elements;
    private final boolean isValid;
    private final AnnotationMirror mirror;

    private AnnotateWithGem( BuilderImpl builder ) {
        this.value = builder.value;
        this.elements = builder.elements;
        isValid = ( this.value != null ? this.value.isValid() : false )
               && ( this.elements != null ? this.elements.isValid() : false );
        mirror = builder.mirror;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link AnnotateWithGem#value}
    */
    public GemValue<TypeMirror> value( ) {
        return value;
    }

    /**
    * accessor
    *
    * @return the {@link GemValue} for {@link AnnotateWithGem#elements}
    */
    public GemValue<List<ElementGem>> elements( ) {
        return elements;
    }

    @Override
    public AnnotationMirror mirror( ) {
        return mirror;
    }

    @Override
    public boolean isValid( ) {
        return isValid;
    }

    public static AnnotateWithGem  instanceOn(Element element) {
        return build( element, new BuilderImpl() );
    }

    public static AnnotateWithGem instanceOn(AnnotationMirror mirror ) {
        return build( mirror, new BuilderImpl() );
    }

    public static  <T> T  build(Element element, Builder<T> builder) {
        AnnotationMirror mirror = element.getAnnotationMirrors().stream()
            .filter( a ->  "org.mapstruct.AnnotateWith".contentEquals( ( ( TypeElement )a.getAnnotationType().asElement() ).getQualifiedName() ) )
            .findAny()
            .orElse( null );
        return build( mirror, builder );
    }

    public static <T> T build(AnnotationMirror mirror, Builder<T> builder ) {

        // return fast
        if ( mirror == null || builder == null ) {
            return null;
        }

        // fetch defaults from all defined values in the annotation type
        List<ExecutableElement> enclosed = ElementFilter.methodsIn( mirror.getAnnotationType().asElement().getEnclosedElements() );
        Map<String, AnnotationValue> defaultValues = new HashMap<>( enclosed.size() );
        enclosed.forEach( e -> defaultValues.put( e.getSimpleName().toString(), e.getDefaultValue() ) );

        // fetch all explicitely set annotation values in the annotation instance
        Map<String, AnnotationValue> values = new HashMap<>( enclosed.size() );
        mirror.getElementValues().entrySet().forEach( e -> values.put( e.getKey().getSimpleName().toString(), e.getValue() ) );

        // iterate and populate builder
        for ( String methodName : defaultValues.keySet() ) {

            if ( "value".equals( methodName ) ) {
                builder.setValue( GemValue.create( values.get( methodName ), defaultValues.get( methodName ), TypeMirror.class ) );
            }
            else if ( "elements".equals( methodName ) ) {
                builder.setElements( GemValue.createArray( values.get( methodName ), defaultValues.get( methodName ), ElementGem::instanceOn ) );
            }
        }
        builder.setMirror( mirror );
        return builder.build();
    }

    /**
     * A builder that can be implemented by the user to define custom logic e.g. in the
     * build method, prior to creating the annotation gem.
     */
    public interface Builder<T> {

       /**
        * Sets the {@link GemValue} for {@link AnnotateWithGem#value}
        *
        * @return the {@link Builder} for this gem, representing {@link AnnotateWithGem}
        */
        Builder setValue(GemValue<TypeMirror> methodName );

       /**
        * Sets the {@link GemValue} for {@link AnnotateWithGem#elements}
        *
        * @return the {@link Builder} for this gem, representing {@link AnnotateWithGem}
        */
        Builder setElements(GemValue<List<ElementGem>> methodName );

        /**
         * Sets the annotation mirror
         *
         * @param mirror the mirror which this gem represents
         *
         * @return the {@link Builder} for this gem, representing {@link AnnotateWithGem}
         */
          Builder setMirror( AnnotationMirror mirror );

        /**
         * The build method can be overriden in a custom custom implementation, which allows
         * the user to define his own custom validation on the annotation.
         *
         * @return the representation of the annotation
         */
        T build();
    }

    private static class BuilderImpl implements Builder<AnnotateWithGem> {

        private GemValue<TypeMirror> value;
        private GemValue<List<ElementGem>> elements;
        private AnnotationMirror mirror;

        public Builder setValue(GemValue<TypeMirror> value ) {
            this.value = value;
            return this;
        }

        public Builder setElements(GemValue<List<ElementGem>> elements ) {
            this.elements = elements;
            return this;
        }

        public Builder  setMirror( AnnotationMirror mirror ) {
            this.mirror = mirror;
            return this;
        }

        public AnnotateWithGem build() {
            return new AnnotateWithGem( this );
        }
    }

}
