/*
	Milyn - Copyright (C) 2006 - 2010

	This library is free software; you can redistribute it and/or
	modify it under the terms of the GNU Lesser General Public
	License (version 2.1) as published by the Free Software
	Foundation.

	This library is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.

	See the GNU Lesser General Public License for more details:
	http://www.gnu.org/licenses/lgpl.txt
*/

package org.milyn.templating.stringtemplate.v4;

import java.io.IOException;
import java.util.Map;

import org.hsqldb.lib.Iterator;
import org.milyn.cdr.SmooksConfigurationException;
import org.milyn.cdr.SmooksResourceConfiguration;
import org.milyn.cdr.annotation.AppContext;
import org.milyn.cdr.annotation.Configurator;
import org.milyn.container.ApplicationContext;
import org.milyn.container.ExecutionContext;
import org.milyn.delivery.ContentHandler;
import org.milyn.delivery.ContentHandlerFactory;
import org.milyn.delivery.annotation.Resource;
import org.milyn.delivery.dom.serialize.TextSerializationUnit;
import org.milyn.delivery.ordering.Consumer;
import org.milyn.event.report.annotation.VisitAfterReport;
import org.milyn.event.report.annotation.VisitBeforeReport;
import org.milyn.javabean.context.BeanContext;
import org.milyn.templating.AbstractTemplateProcessor;
import org.stringtemplate.v4.ST;
import org.stringtemplate.v4.STRawGroupDir;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * StringTemplate {@link org.milyn.delivery.dom.DOMElementVisitor} Creator class.
 * <p/>
 * Creates {@link org.milyn.delivery.dom.DOMElementVisitor} instances for applying
 * <a href="http://www.stringtemplate.org/">StringTemplate</a> transformations (i.e. ".st" files).
 * <p/>
 * This templating solution relies on the <a href="http://milyn.codehaus.org/downloads">Smooks JavaBean Cartridge</a>
 * to perform the JavaBean population that's required by <a href="http://www.stringtemplate.org/">StringTemplate</a>.
 *
 * <h2>Targeting ".st" Files for Transformation</h2>
 * <pre>
 * &lt;resource-config selector="<i>target-element</i>"&gt;
 *     &lt;!-- See {@link org.milyn.resource.URIResourceLocator} --&gt;
 *     &lt;resource&gt;<b>/com/acme/AcmeStringTemplate.st</b>&lt;/resource&gt;
 *
 *     &lt;!-- (Optional) The action to be applied on the template content. Should the content
 *          generated by the template:
 *          1. replace ("replace") the target element, or
 *          2. be added to ("addto") the target element, or
 *          3. be inserted before ("insertbefore") the target element, or
 *          4. be inserted after ("insertafter") the target element.
 *          5. be bound to ("bindto") a {@link BeanContext} variable named by the "bindId" param.
 *          Default "replace".--&gt;
 *     &lt;param name="<b>action</b>"&gt;<i>replace/addto/insertbefore/insertafter</i>&lt;/param&gt;
 *
 *     &lt;!-- (Optional) Should the template be applied before (true) or
 *             after (false) Smooks visits the child elements of the target element.
 *             Default "false".--&gt;
 *     &lt;param name="<b>applyTemplateBefore</b>"&gt;<i>true/false</i>&lt;/param&gt;
 *
 *     &lt;!-- (Optional) The name of the {@link org.milyn.io.AbstractOutputStreamResource OutputStreamResource}
 *             to which the result should be written. If set, the "action" param is ignored. --&gt;
 *     &lt;param name="<b>outputStreamResource</b>"&gt;<i>xyzResource</i>&lt;/param&gt;
 *
 *     &lt;!-- (Optional) Template encoding.
 *          Default "UTF-8".--&gt;
 *     &lt;param name="<b>encoding</b>"&gt;<i>encoding</i>&lt;/param&gt;
 *
 *     &lt;!-- (Optional) bindId when "action" is "bindto".
 *     &lt;param name="<b>bindId</b>"&gt;<i>xxxx</i>&lt;/param&gt;
 *
 * &lt;/resource-config&gt;
 * </pre>
 *
 * @author tfennelly
 */
@Resource(type="stv4")
public class StringTemplateContentHandlerFactory implements ContentHandlerFactory {

	@AppContext
	private ApplicationContext applicationContext;

	/**
	 * Create a StringTemplate based ContentHandler.
     * @param resourceConfig The SmooksResourceConfiguration for the StringTemplate.
     * @return The StringTemplate {@link org.milyn.delivery.ContentHandler} instance.
	 */
	public synchronized ContentHandler create(SmooksResourceConfiguration resourceConfig) throws SmooksConfigurationException, InstantiationException {
        try {
            return Configurator.configure(new StringTemplateTemplateProcessor(), resourceConfig, applicationContext);
        } catch (SmooksConfigurationException e) {
            throw e;
        } catch (Exception e) {
			InstantiationException instanceException = new InstantiationException("StringTemplate ProcessingUnit resource [" + resourceConfig.getResource() + "] not loadable.  StringTemplate resource invalid.");
			instanceException.initCause(e);
			throw instanceException;
		}
	}

	/**
	 * StringTemplate template application ProcessingUnit.
	 * @author tfennelly
	 */
    @VisitBeforeReport(condition = "false")
    @VisitAfterReport(summary = "Applied StringTemplate Template.", detailTemplate = "reporting/StringTemplateTemplateProcessor_After.html")
	private static class StringTemplateTemplateProcessor extends AbstractTemplateProcessor implements Consumer {

    	private ST template = null;
    	private String templateName;
    	STRawGroupDir templateGroupDir;

        @Override
	protected void loadTemplate(SmooksResourceConfiguration config) throws IOException {
            String path = config.getResource();
            
            if(path.charAt(0) == '/') {
            	path = path.substring(1);
            }

            String dir = path.substring(0, path.lastIndexOf('/'));
            
            templateName = path.substring(path.lastIndexOf('/'), path.indexOf(".st"));
            templateGroupDir = new STRawGroupDir(dir, getEncoding().displayName(),'$','$');
            template = templateGroupDir.getInstanceOf(templateName);
        }

        @Override
	protected void visit(Element element, ExecutionContext executionContext) {
            // First thing we do is clone the template for this transformation...
       	    // Commented out as due to https://github.com/antlr/stringtemplate4/issues/100
            // ST transform = new ST(template);
            ST transform = templateGroupDir.getInstanceOf(templateName);
        	        	
            Map<String, Object> beans = executionContext.getBeanContext().getBeanMap();
            
            // Set the document data beans on the template and apply it...
            for (Map.Entry<String, Object> entry : beans.entrySet()) {
           		transform.add(entry.getKey(), entry.getValue());
            }
    
            String templatingResult = transform.render().trim();

            Node resultNode = TextSerializationUnit.createTextElement(element, templatingResult);

            // Process the templating action, supplying the templating result...
            processTemplateAction(element, resultNode, executionContext);
        }

        public boolean consumes(Object object) {
        	
        	if (template.impl.getTemplateSource().indexOf(object.toString()) != -1) {
        		return true;
        	}
            
            return false;
        }
    }
}
