/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.elasticsearch;

import java.io.IOException;
import java.io.InputStream;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;
import java.util.EnumMap;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.apache.commons.lang.exception.ExceptionUtils;
import org.elasticsearch.action.admin.indices.delete.DeleteIndexRequest;
import org.elasticsearch.action.support.master.AcknowledgedResponse;
import org.elasticsearch.client.RequestOptions;
import org.elasticsearch.client.RestHighLevelClient;
import org.elasticsearch.client.indices.CreateIndexRequest;
import org.elasticsearch.client.indices.CreateIndexResponse;
import org.elasticsearch.client.indices.GetIndexRequest;
import org.elasticsearch.client.indices.PutMappingRequest;
import org.elasticsearch.common.xcontent.XContentType;
import org.json.JSONObject;
import org.openmetadata.schema.service.configuration.elasticsearch.ElasticSearchConfiguration;
import org.openmetadata.schema.system.EventPublisherJob;
import org.openmetadata.schema.system.Failure;
import org.openmetadata.schema.system.FailureDetails;
import org.openmetadata.schema.type.IndexMappingLanguage;
import org.openmetadata.service.Entity;
import org.openmetadata.service.jdbi3.CollectionDAO;
import org.openmetadata.service.util.JsonUtils;
import org.openmetadata.service.workflows.searchIndex.ReindexingUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ElasticSearchIndexDefinition {
    private static final Logger LOG = LoggerFactory.getLogger(ElasticSearchIndexDefinition.class);
    public static final String ELASTIC_SEARCH_EXTENSION = "service.eventPublisher";
    public static final String ELASTIC_SEARCH_ENTITY_FQN_STREAM = "eventPublisher:ElasticSearch:STREAM";
    private static final String MAPPINGS_KEY = "mappings";
    private static final String PROPERTIES_KEY = "properties";
    private static final String REASON_TRACE = "Reason: [%s] , Trace : [%s]";
    public static final String ENTITY_REPORT_DATA = "entityReportData";
    public static final String WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA = "webAnalyticEntityViewReportData";
    public static final String WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA = "webAnalyticUserActivityReportData";
    private final CollectionDAO dao;
    final EnumMap<ElasticSearchIndexType, ElasticSearchIndexStatus> elasticSearchIndexes = new EnumMap(ElasticSearchIndexType.class);
    public static final HashMap<String, String> ENTITY_TYPE_TO_INDEX_MAP;
    private static final Map<ElasticSearchIndexType, Set<String>> INDEX_TO_MAPPING_FIELDS_MAP;
    private final RestHighLevelClient client;

    public ElasticSearchIndexDefinition(RestHighLevelClient client, CollectionDAO dao) {
        this.dao = dao;
        this.client = client;
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.elasticSearchIndexes.put(elasticSearchIndexType, ElasticSearchIndexStatus.NOT_CREATED);
        }
    }

    public void createIndexes(ElasticSearchConfiguration esConfig) {
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.createIndex(elasticSearchIndexType, esConfig.getSearchIndexMappingLanguage().value());
        }
    }

    public void updateIndexes(ElasticSearchConfiguration esConfig) {
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.updateIndex(elasticSearchIndexType, esConfig.getSearchIndexMappingLanguage().value());
        }
    }

    public void dropIndexes() {
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            this.deleteIndex(elasticSearchIndexType);
        }
    }

    public boolean createIndex(ElasticSearchIndexType elasticSearchIndexType, String lang) {
        try {
            GetIndexRequest gRequest = new GetIndexRequest(new String[]{elasticSearchIndexType.indexName});
            gRequest.local(false);
            boolean exists = this.client.indices().exists(gRequest, RequestOptions.DEFAULT);
            if (!exists) {
                String elasticSearchIndexMapping = ElasticSearchIndexDefinition.getIndexMapping(elasticSearchIndexType, lang);
                CreateIndexRequest request = new CreateIndexRequest(elasticSearchIndexType.indexName);
                request.source(elasticSearchIndexMapping, XContentType.JSON);
                CreateIndexResponse createIndexResponse = this.client.indices().create(request, RequestOptions.DEFAULT);
                LOG.info("{} Created {}", (Object)elasticSearchIndexType.indexName, (Object)createIndexResponse.isAcknowledged());
            }
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.CREATED);
        }
        catch (Exception e) {
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.FAILED);
            this.updateElasticSearchFailureStatus(this.getContext("Creating Index", elasticSearchIndexType.indexName), String.format(REASON_TRACE, e.getMessage(), ExceptionUtils.getStackTrace((Throwable)e)));
            LOG.error("Failed to create Elastic Search indexes due to", (Throwable)e);
            return false;
        }
        return true;
    }

    private String getContext(String type, String info) {
        return String.format("Failed While : %s \n Additional Info:  %s ", type, info);
    }

    private void updateIndex(ElasticSearchIndexType elasticSearchIndexType, String lang) {
        try {
            GetIndexRequest gRequest = new GetIndexRequest(new String[]{elasticSearchIndexType.indexName});
            gRequest.local(false);
            boolean exists = this.client.indices().exists(gRequest, RequestOptions.DEFAULT);
            String elasticSearchIndexMapping = ElasticSearchIndexDefinition.getIndexMapping(elasticSearchIndexType, lang);
            if (exists) {
                PutMappingRequest request = new PutMappingRequest(new String[]{elasticSearchIndexType.indexName});
                request.source(elasticSearchIndexMapping, XContentType.JSON);
                AcknowledgedResponse putMappingResponse = this.client.indices().putMapping(request, RequestOptions.DEFAULT);
                LOG.info("{} Updated {}", (Object)elasticSearchIndexType.indexName, (Object)putMappingResponse.isAcknowledged());
            } else {
                CreateIndexRequest request = new CreateIndexRequest(elasticSearchIndexType.indexName);
                request.source(elasticSearchIndexMapping, XContentType.JSON);
                CreateIndexResponse createIndexResponse = this.client.indices().create(request, RequestOptions.DEFAULT);
                LOG.info("{} Created {}", (Object)elasticSearchIndexType.indexName, (Object)createIndexResponse.isAcknowledged());
            }
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.CREATED);
        }
        catch (Exception e) {
            this.setIndexStatus(elasticSearchIndexType, ElasticSearchIndexStatus.FAILED);
            this.updateElasticSearchFailureStatus(this.getContext("Updating Index", elasticSearchIndexType.indexName), String.format(REASON_TRACE, e.getMessage(), ExceptionUtils.getStackTrace((Throwable)e)));
            LOG.error("Failed to update Elastic Search indexes due to", (Throwable)e);
        }
    }

    public void deleteIndex(ElasticSearchIndexType elasticSearchIndexType) {
        try {
            GetIndexRequest gRequest = new GetIndexRequest(new String[]{elasticSearchIndexType.indexName});
            gRequest.local(false);
            boolean exists = this.client.indices().exists(gRequest, RequestOptions.DEFAULT);
            if (exists) {
                DeleteIndexRequest request = new DeleteIndexRequest(elasticSearchIndexType.indexName);
                AcknowledgedResponse deleteIndexResponse = this.client.indices().delete(request, RequestOptions.DEFAULT);
                LOG.info("{} Deleted {}", (Object)elasticSearchIndexType.indexName, (Object)deleteIndexResponse.isAcknowledged());
            }
        }
        catch (IOException e) {
            this.updateElasticSearchFailureStatus(this.getContext("Deleting Index", elasticSearchIndexType.indexName), String.format(REASON_TRACE, e.getMessage(), ExceptionUtils.getStackTrace((Throwable)e)));
            LOG.error("Failed to delete Elastic Search indexes due to", (Throwable)e);
        }
    }

    private void setIndexStatus(ElasticSearchIndexType indexType, ElasticSearchIndexStatus elasticSearchIndexStatus) {
        this.elasticSearchIndexes.put(indexType, elasticSearchIndexStatus);
    }

    public static String getIndexMapping(ElasticSearchIndexType elasticSearchIndexType, String lang) throws IOException {
        InputStream in = ElasticSearchIndexDefinition.class.getResourceAsStream(String.format(elasticSearchIndexType.indexMappingFile, lang.toLowerCase()));
        assert (in != null);
        return new String(in.readAllBytes());
    }

    private static void populateEsFieldsForIndexes(ElasticSearchIndexType elasticSearchIndexType, IndexMappingLanguage lang) {
        if (!ReindexingUtil.isDataInsightIndex(elasticSearchIndexType.entityType)) {
            String indexData = ElasticSearchIndexDefinition.getIndexMapping(elasticSearchIndexType, lang.value());
            JSONObject object = new JSONObject(indexData).getJSONObject(MAPPINGS_KEY).getJSONObject(PROPERTIES_KEY);
            Set<String> keySet = Entity.getEntityRepository(elasticSearchIndexType.entityType).getCommonFields(object.keySet());
            INDEX_TO_MAPPING_FIELDS_MAP.put(elasticSearchIndexType, keySet);
        }
    }

    public static ElasticSearchIndexType getIndexMappingByEntityType(String type) {
        if (type.equalsIgnoreCase("table")) {
            return ElasticSearchIndexType.TABLE_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("dashboard")) {
            return ElasticSearchIndexType.DASHBOARD_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("pipeline")) {
            return ElasticSearchIndexType.PIPELINE_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("topic")) {
            return ElasticSearchIndexType.TOPIC_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("user")) {
            return ElasticSearchIndexType.USER_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("team")) {
            return ElasticSearchIndexType.TEAM_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("glossary")) {
            return ElasticSearchIndexType.GLOSSARY_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("mlmodel")) {
            return ElasticSearchIndexType.MLMODEL_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("glossaryTerm")) {
            return ElasticSearchIndexType.GLOSSARY_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("tag")) {
            return ElasticSearchIndexType.TAG_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase(ENTITY_REPORT_DATA)) {
            return ElasticSearchIndexType.ENTITY_REPORT_DATA_INDEX;
        }
        if (type.equalsIgnoreCase(WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA)) {
            return ElasticSearchIndexType.WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA_INDEX;
        }
        if (type.equalsIgnoreCase(WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA)) {
            return ElasticSearchIndexType.WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA_INDEX;
        }
        if (type.equalsIgnoreCase("container")) {
            return ElasticSearchIndexType.CONTAINER_SEARCH_INDEX;
        }
        if (type.equalsIgnoreCase("query")) {
            return ElasticSearchIndexType.QUERY_SEARCH_INDEX;
        }
        throw new RuntimeException("Failed to find index doc for type " + type);
    }

    public static Set<String> getIndexFields(String entityType, IndexMappingLanguage lang) {
        Set<String> fields = INDEX_TO_MAPPING_FIELDS_MAP.get((Object)ElasticSearchIndexDefinition.getIndexMappingByEntityType(entityType));
        if (fields != null) {
            return fields;
        }
        ElasticSearchIndexDefinition.populateEsFieldsForIndexes(ElasticSearchIndexDefinition.getIndexMappingByEntityType(entityType), lang);
        fields = INDEX_TO_MAPPING_FIELDS_MAP.get((Object)ElasticSearchIndexDefinition.getIndexMappingByEntityType(entityType));
        return fields;
    }

    private void updateElasticSearchFailureStatus(String failedFor, String failureMessage) {
        try {
            long updateTime = Date.from(LocalDateTime.now().atZone(ZoneId.systemDefault()).toInstant()).getTime();
            String recordString = this.dao.entityExtensionTimeSeriesDao().getExtension(ELASTIC_SEARCH_ENTITY_FQN_STREAM, ELASTIC_SEARCH_EXTENSION);
            EventPublisherJob lastRecord = JsonUtils.readValue(recordString, EventPublisherJob.class);
            long originalLastUpdate = lastRecord.getTimestamp();
            lastRecord.setStatus(EventPublisherJob.Status.ACTIVE_WITH_ERROR);
            lastRecord.setTimestamp(Long.valueOf(updateTime));
            lastRecord.setFailure(new Failure().withSinkError(new FailureDetails().withContext(failedFor).withLastFailedAt(Long.valueOf(updateTime)).withLastFailedReason(failureMessage)));
            this.dao.entityExtensionTimeSeriesDao().update(ELASTIC_SEARCH_ENTITY_FQN_STREAM, ELASTIC_SEARCH_EXTENSION, JsonUtils.pojoToJson(lastRecord), originalLastUpdate);
        }
        catch (Exception e) {
            LOG.error("Failed to Update Elastic Search Job Info");
        }
    }

    static {
        INDEX_TO_MAPPING_FIELDS_MAP = new HashMap<ElasticSearchIndexType, Set<String>>();
        ENTITY_TYPE_TO_INDEX_MAP = new HashMap();
        for (ElasticSearchIndexType elasticSearchIndexType : ElasticSearchIndexType.values()) {
            ENTITY_TYPE_TO_INDEX_MAP.put(elasticSearchIndexType.entityType, elasticSearchIndexType.indexName);
        }
    }

    public static enum ElasticSearchIndexType {
        TABLE_SEARCH_INDEX("table", "table_search_index", "/elasticsearch/%s/table_index_mapping.json"),
        TOPIC_SEARCH_INDEX("topic", "topic_search_index", "/elasticsearch/%s/topic_index_mapping.json"),
        DASHBOARD_SEARCH_INDEX("dashboard", "dashboard_search_index", "/elasticsearch/%s/dashboard_index_mapping.json"),
        PIPELINE_SEARCH_INDEX("pipeline", "pipeline_search_index", "/elasticsearch/%s/pipeline_index_mapping.json"),
        USER_SEARCH_INDEX("user", "user_search_index", "/elasticsearch/%s/user_index_mapping.json"),
        TEAM_SEARCH_INDEX("team", "team_search_index", "/elasticsearch/%s/team_index_mapping.json"),
        GLOSSARY_SEARCH_INDEX("glossary", "glossary_search_index", "/elasticsearch/%s/glossary_index_mapping.json"),
        MLMODEL_SEARCH_INDEX("mlmodel", "mlmodel_search_index", "/elasticsearch/%s/mlmodel_index_mapping.json"),
        CONTAINER_SEARCH_INDEX("container", "container_search_index", "/elasticsearch/%s/container_index_mapping.json"),
        QUERY_SEARCH_INDEX("query", "query_search_index", "/elasticsearch/%s/query_index_mapping.json"),
        TAG_SEARCH_INDEX("tag", "tag_search_index", "/elasticsearch/%s/tag_index_mapping.json"),
        ENTITY_REPORT_DATA_INDEX("entityReportData", "entity_report_data_index", "/elasticsearch/entity_report_data_index.json"),
        WEB_ANALYTIC_ENTITY_VIEW_REPORT_DATA_INDEX("webAnalyticEvent", "web_analytic_entity_view_report_data_index", "/elasticsearch/web_analytic_entity_view_report_data_index.json"),
        WEB_ANALYTIC_USER_ACTIVITY_REPORT_DATA_INDEX("webAnalyticUserActivityReportData", "web_analytic_user_activity_report_data_index", "/elasticsearch/web_analytic_user_activity_report_data_index.json");

        public final String indexName;
        public final String indexMappingFile;
        public final String entityType;

        private ElasticSearchIndexType(String entityType, String indexName, String indexMappingFile) {
            this.entityType = entityType;
            this.indexName = indexName;
            this.indexMappingFile = indexMappingFile;
        }
    }

    public static enum ElasticSearchIndexStatus {
        CREATED,
        NOT_CREATED,
        FAILED;

    }
}

