/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.security;

import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.ImmutableMap;
import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.JWSAlgorithm;
import com.nimbusds.jwt.JWT;
import com.nimbusds.oauth2.sdk.auth.ClientAuthentication;
import com.nimbusds.oauth2.sdk.auth.ClientAuthenticationMethod;
import com.nimbusds.oauth2.sdk.auth.ClientSecretBasic;
import com.nimbusds.oauth2.sdk.auth.ClientSecretPost;
import com.nimbusds.oauth2.sdk.auth.PrivateKeyJWT;
import com.nimbusds.oauth2.sdk.auth.Secret;
import com.nimbusds.oauth2.sdk.id.ClientID;
import com.nimbusds.oauth2.sdk.token.AccessToken;
import com.nimbusds.oauth2.sdk.token.BearerAccessToken;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.net.HttpURLConnection;
import java.net.URL;
import java.nio.charset.StandardCharsets;
import java.security.Principal;
import java.security.PrivateKey;
import java.text.ParseException;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeMap;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.client.Invocation;
import javax.ws.rs.client.WebTarget;
import javax.ws.rs.core.SecurityContext;
import org.openmetadata.common.utils.CommonUtil;
import org.openmetadata.schema.security.client.OidcClientConfig;
import org.openmetadata.service.OpenMetadataApplicationConfig;
import org.openmetadata.service.security.AuthenticationException;
import org.openmetadata.service.util.JsonUtils;
import org.pac4j.core.exception.TechnicalException;
import org.pac4j.core.util.CommonHelper;
import org.pac4j.core.util.HttpUtils;
import org.pac4j.oidc.client.AzureAd2Client;
import org.pac4j.oidc.client.GoogleOidcClient;
import org.pac4j.oidc.client.OidcClient;
import org.pac4j.oidc.config.AzureAd2OidcConfiguration;
import org.pac4j.oidc.config.OidcConfiguration;
import org.pac4j.oidc.config.PrivateKeyJWTClientAuthnMethodConfig;
import org.pac4j.oidc.credentials.OidcCredentials;
import org.pac4j.oidc.credentials.authenticator.OidcAuthenticator;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class SecurityUtil {
    private static final Logger LOG = LoggerFactory.getLogger(SecurityUtil.class);
    public static final String DEFAULT_PRINCIPAL_DOMAIN = "openmetadata.org";
    private static final Collection<ClientAuthenticationMethod> SUPPORTED_METHODS = Arrays.asList(ClientAuthenticationMethod.CLIENT_SECRET_POST, ClientAuthenticationMethod.CLIENT_SECRET_BASIC, ClientAuthenticationMethod.PRIVATE_KEY_JWT, ClientAuthenticationMethod.NONE);

    private SecurityUtil() {
    }

    public static String getUserName(SecurityContext securityContext) {
        Principal principal = securityContext.getUserPrincipal();
        return principal == null ? null : principal.getName().split("[/@]")[0];
    }

    public static Map<String, String> authHeaders(String username) {
        ImmutableMap.Builder builder = ImmutableMap.builder();
        if (username != null) {
            builder.put((Object)"X-Auth-Params-Email", (Object)username);
        }
        return builder.build();
    }

    public static String getPrincipalName(Map<String, String> authHeaders) {
        if (authHeaders == null) {
            return null;
        }
        String principal = authHeaders.get("X-Auth-Params-Email");
        return principal == null ? null : principal.split("@")[0];
    }

    public static String getDomain(OpenMetadataApplicationConfig config) {
        String principalDomain = config.getAuthorizerConfiguration().getPrincipalDomain();
        return CommonUtil.nullOrEmpty((String)principalDomain) ? DEFAULT_PRINCIPAL_DOMAIN : principalDomain;
    }

    public static Invocation.Builder addHeaders(WebTarget target, Map<String, String> headers) {
        if (headers != null) {
            return target.request().header("X-Auth-Params-Email", (Object)headers.get("X-Auth-Params-Email"));
        }
        return target.request();
    }

    public static OidcClient tryCreateOidcClient(OidcClientConfig clientConfig) {
        String id = clientConfig.getId();
        String secret = clientConfig.getSecret();
        if (CommonHelper.isNotBlank((String)id) && CommonHelper.isNotBlank((String)secret)) {
            GoogleOidcClient oidcClient;
            String type;
            String clientAuthenticationMethod;
            String maxClockSkew;
            String jwsAlgo;
            String useNonce;
            String discoveryUri;
            String scope;
            String responseType;
            OidcConfiguration configuration = new OidcConfiguration();
            configuration.setClientId(id);
            configuration.setResponseMode("query");
            if (CommonHelper.isNotBlank((String)secret)) {
                configuration.setSecret(secret);
            }
            if (CommonHelper.isNotBlank((String)(responseType = clientConfig.getResponseType()))) {
                configuration.setResponseType(responseType);
            }
            if (CommonHelper.isNotBlank((String)(scope = clientConfig.getScope()))) {
                configuration.setScope(scope);
            }
            if (CommonHelper.isNotBlank((String)(discoveryUri = clientConfig.getDiscoveryUri()))) {
                configuration.setDiscoveryURI(discoveryUri);
            }
            if (CommonHelper.isNotBlank((String)(useNonce = clientConfig.getUseNonce()))) {
                configuration.setUseNonce(Boolean.parseBoolean(useNonce));
            }
            if (CommonHelper.isNotBlank((String)(jwsAlgo = clientConfig.getPreferredJwsAlgorithm()))) {
                configuration.setPreferredJwsAlgorithm(JWSAlgorithm.parse((String)jwsAlgo));
            }
            if (CommonHelper.isNotBlank((String)(maxClockSkew = clientConfig.getMaxClockSkew()))) {
                configuration.setMaxClockSkew(Integer.parseInt(maxClockSkew));
            }
            if (CommonHelper.isNotBlank((String)(clientAuthenticationMethod = clientConfig.getClientAuthenticationMethod().value()))) {
                configuration.setClientAuthenticationMethod(ClientAuthenticationMethod.parse((String)clientAuthenticationMethod));
            }
            configuration.setDisablePkce(clientConfig.getDisablePkce().booleanValue());
            if (clientConfig.getCustomParams() != null) {
                for (int j = 1; j <= 5; ++j) {
                    if (!clientConfig.getCustomParams().containsKey(String.format("customParamKey%d", j))) continue;
                    configuration.addCustomParam((String)clientConfig.getCustomParams().get(String.format("customParamKey%d", j)), (String)clientConfig.getCustomParams().get(String.format("customParamValue%d", j)));
                }
            }
            if ("azure".equalsIgnoreCase(type = clientConfig.getType())) {
                AzureAd2OidcConfiguration azureAdConfiguration = new AzureAd2OidcConfiguration(configuration);
                String tenant = clientConfig.getTenant();
                if (CommonHelper.isNotBlank((String)tenant)) {
                    azureAdConfiguration.setTenant(tenant);
                }
                oidcClient = new AzureAd2Client(azureAdConfiguration);
            } else if ("google".equalsIgnoreCase(type)) {
                oidcClient = new GoogleOidcClient(configuration);
                oidcClient.getConfiguration().getCustomParams().put("access_type", "offline");
            } else {
                oidcClient = new OidcClient(configuration);
            }
            oidcClient.setName(String.format("OMOidcClient%s", oidcClient.getName()));
            return oidcClient;
        }
        throw new IllegalArgumentException("Client ID and Client Secret is required to create OidcClient");
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public static ClientAuthentication getClientAuthentication(OidcConfiguration configuration) {
        ClientAuthenticationMethod chosenMethod;
        ClientID clientID = new ClientID(configuration.getClientId());
        ClientSecretPost clientAuthenticationMechanism = null;
        if (configuration.getSecret() == null) return clientAuthenticationMechanism;
        List metadataMethods = configuration.findProviderMetadata().getTokenEndpointAuthMethods();
        ClientAuthenticationMethod preferredMethod = SecurityUtil.getPreferredAuthenticationMethod(configuration);
        if (CommonHelper.isNotEmpty((Collection)metadataMethods)) {
            if (preferredMethod != null) {
                if (!metadataMethods.contains(preferredMethod)) throw new TechnicalException("Preferred authentication method (" + preferredMethod + ") not supported by provider according to provider metadata (" + metadataMethods + ").");
                chosenMethod = preferredMethod;
            } else {
                chosenMethod = SecurityUtil.firstSupportedMethod(metadataMethods);
            }
        } else {
            chosenMethod = preferredMethod != null ? preferredMethod : ClientAuthenticationMethod.getDefault();
            LOG.info("Provider metadata does not provide Token endpoint authentication methods. Using: {}", (Object)chosenMethod);
        }
        if (ClientAuthenticationMethod.CLIENT_SECRET_POST.equals((Object)chosenMethod)) {
            Secret clientSecret = new Secret(configuration.getSecret());
            return new ClientSecretPost(clientID, clientSecret);
        }
        if (ClientAuthenticationMethod.CLIENT_SECRET_BASIC.equals((Object)chosenMethod)) {
            Secret clientSecret = new Secret(configuration.getSecret());
            return new ClientSecretBasic(clientID, clientSecret);
        }
        if (!ClientAuthenticationMethod.PRIVATE_KEY_JWT.equals((Object)chosenMethod)) return clientAuthenticationMechanism;
        PrivateKeyJWTClientAuthnMethodConfig privateKetJwtConfig = configuration.getPrivateKeyJWTClientAuthnMethodConfig();
        CommonHelper.assertNotNull((String)"privateKetJwtConfig", (Object)privateKetJwtConfig);
        JWSAlgorithm jwsAlgo = privateKetJwtConfig.getJwsAlgorithm();
        CommonHelper.assertNotNull((String)"privateKetJwtConfig.getJwsAlgorithm()", (Object)jwsAlgo);
        PrivateKey privateKey = privateKetJwtConfig.getPrivateKey();
        CommonHelper.assertNotNull((String)"privateKetJwtConfig.getPrivateKey()", (Object)privateKey);
        String keyID = privateKetJwtConfig.getKeyID();
        try {
            return new PrivateKeyJWT(clientID, configuration.findProviderMetadata().getTokenEndpointURI(), jwsAlgo, privateKey, keyID, null);
        }
        catch (JOSEException e) {
            throw new TechnicalException("Cannot instantiate private key JWT client authentication method", (Throwable)e);
        }
    }

    private static ClientAuthenticationMethod getPreferredAuthenticationMethod(OidcConfiguration config) {
        ClientAuthenticationMethod configurationMethod = config.getClientAuthenticationMethod();
        if (configurationMethod == null) {
            return null;
        }
        if (!SUPPORTED_METHODS.contains(configurationMethod)) {
            throw new TechnicalException("Configured authentication method (" + configurationMethod + ") is not supported.");
        }
        return configurationMethod;
    }

    private static ClientAuthenticationMethod firstSupportedMethod(List<ClientAuthenticationMethod> metadataMethods) {
        Optional<ClientAuthenticationMethod> firstSupported = metadataMethods.stream().filter(SUPPORTED_METHODS::contains).findFirst();
        if (firstSupported.isPresent()) {
            return firstSupported.get();
        }
        throw new TechnicalException("None of the Token endpoint provider metadata authentication methods are supported: " + metadataMethods);
    }

    public static void getErrorMessage(HttpServletResponse resp, Exception e) {
        resp.setContentType("text/html; charset=UTF-8");
        LOG.error("[Auth Callback Servlet] Failed in Auth Login : {}", (Object)e.getMessage());
        resp.getOutputStream().println(String.format("<p> [Auth Callback Servlet] Failed in Auth Login : %s </p>", e.getMessage()));
    }

    public static void sendRedirectWithToken(HttpServletResponse response, OidcCredentials credentials, String serverUrl, List<String> claimsOrder, String defaultDomain) throws ParseException, IOException {
        JWT jwt = credentials.getIdToken();
        TreeMap claims = new TreeMap(String.CASE_INSENSITIVE_ORDER);
        claims.putAll(jwt.getJWTClaimsSet().getClaims());
        String preferredJwtClaim = claimsOrder.stream().filter(claims::containsKey).findFirst().map(claims::get).map(String.class::cast).orElseThrow(() -> new AuthenticationException("Invalid JWT token, none of the following claims are present " + claimsOrder));
        String userName = preferredJwtClaim.contains("@") ? preferredJwtClaim.split("@")[0] : preferredJwtClaim;
        String email = String.format("%s@%s", userName, defaultDomain);
        String url = String.format("%s/auth/callback?id_token=%s&email=%s&name=%s", serverUrl, credentials.getIdToken().getParsedString(), email, userName);
        response.sendRedirect(url);
    }

    public static boolean isCredentialsExpired(OidcCredentials credentials) throws ParseException {
        Date expiration = credentials.getIdToken().getJWTClaimsSet().getExpirationTime();
        return expiration != null && expiration.toInstant().isBefore(Instant.now().plusSeconds(30L));
    }

    public static Optional<OidcCredentials> getUserCredentialsFromSession(HttpServletRequest request, OidcClient client) throws ParseException {
        OidcCredentials credentials = (OidcCredentials)request.getSession().getAttribute("oidcCredentialProfile");
        if (credentials != null && credentials.getRefreshToken() != null) {
            SecurityUtil.removeOrRenewOidcCredentials(request, client, credentials);
            return Optional.of(credentials);
        }
        if (credentials == null) {
            LOG.error("No credentials found against session. ID: {}", (Object)request.getSession().getId());
        } else {
            LOG.error("No refresh token found against session. ID: {}", (Object)request.getSession().getId());
        }
        return Optional.empty();
    }

    private static void removeOrRenewOidcCredentials(HttpServletRequest request, OidcClient client, OidcCredentials credentials) throws ParseException {
        boolean profilesUpdated = false;
        if (SecurityUtil.isCredentialsExpired(credentials)) {
            LOG.debug("Expired credentials found, trying to renew.");
            profilesUpdated = true;
            OidcConfiguration oidcConfiguration = client.getConfiguration();
            if (oidcConfiguration instanceof AzureAd2OidcConfiguration) {
                AzureAd2OidcConfiguration azureAd2OidcConfiguration = (AzureAd2OidcConfiguration)oidcConfiguration;
                SecurityUtil.refreshAccessTokenAzureAd2Token(azureAd2OidcConfiguration, credentials);
            } else {
                OidcAuthenticator authenticator = new OidcAuthenticator(client.getConfiguration(), client);
                authenticator.refresh(credentials);
            }
        }
        if (profilesUpdated) {
            request.getSession().setAttribute("oidcCredentialProfile", (Object)credentials);
        }
    }

    private static void refreshAccessTokenAzureAd2Token(AzureAd2OidcConfiguration azureConfig, OidcCredentials azureAdProfile) {
        HttpURLConnection connection = null;
        try {
            HashMap<String, String> headers = new HashMap<String, String>();
            headers.put("Content-Type", "application/x-www-form-urlencoded");
            headers.put("Accept", "application/json");
            URL tokenEndpointURL = azureConfig.findProviderMetadata().getTokenEndpointURI().toURL();
            connection = HttpUtils.openPostConnection((URL)tokenEndpointURL, headers);
            BufferedWriter out = new BufferedWriter(new OutputStreamWriter(connection.getOutputStream(), StandardCharsets.UTF_8));
            out.write(azureConfig.makeOauth2TokenRequest(azureAdProfile.getRefreshToken().getValue()));
            out.close();
            int responseCode = connection.getResponseCode();
            if (responseCode != 200) {
                throw new TechnicalException("request for access token failed: " + HttpUtils.buildHttpErrorMessage((HttpURLConnection)connection));
            }
            String body = HttpUtils.readBody((HttpURLConnection)connection);
            Map<String, Object> res = JsonUtils.readValue(body, new TypeReference<Map<String, Object>>(){});
            azureAdProfile.setAccessToken((AccessToken)new BearerAccessToken((String)res.get("access_token")));
        }
        catch (IOException e) {
            try {
                throw new TechnicalException((Throwable)e);
            }
            catch (Throwable throwable) {
                HttpUtils.closeConnection(connection);
                throw throwable;
            }
        }
        HttpUtils.closeConnection((HttpURLConnection)connection);
    }
}

