/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.migration.api;

import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Stream;
import org.jdbi.v3.core.Handle;
import org.jdbi.v3.core.Jdbi;
import org.json.JSONObject;
import org.openmetadata.service.jdbi3.MigrationDAO;
import org.openmetadata.service.jdbi3.locator.ConnectionType;
import org.openmetadata.service.migration.QueryStatus;
import org.openmetadata.service.migration.api.MigrationProcess;
import org.openmetadata.service.migration.context.MigrationContext;
import org.openmetadata.service.migration.context.MigrationWorkflowContext;
import org.openmetadata.service.migration.utils.MigrationFile;
import org.openmetadata.service.util.OpenMetadataOperations;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MigrationWorkflow {
    private static final Logger LOG = LoggerFactory.getLogger(MigrationWorkflow.class);
    public static final String SUCCESS_MSG = "Success";
    public static final String FAILED_MSG = "Failed due to : ";
    private List<MigrationProcess> migrations;
    private final String nativeSQLScriptRootPath;
    private final ConnectionType connectionType;
    private final String extensionSQLScriptRootPath;
    private final MigrationDAO migrationDAO;
    private final Jdbi jdbi;
    private final boolean forceMigrations;
    private Optional<String> currentMaxMigrationVersion;

    public MigrationWorkflow(Jdbi jdbi, String nativeSQLScriptRootPath, ConnectionType connectionType, String extensionSQLScriptRootPath, boolean forceMigrations) {
        this.jdbi = jdbi;
        this.migrationDAO = (MigrationDAO)jdbi.onDemand(MigrationDAO.class);
        this.forceMigrations = forceMigrations;
        this.nativeSQLScriptRootPath = nativeSQLScriptRootPath;
        this.connectionType = connectionType;
        this.extensionSQLScriptRootPath = extensionSQLScriptRootPath;
    }

    public void loadMigrations() {
        List<MigrationFile> availableMigrations = this.getMigrationFiles(this.nativeSQLScriptRootPath, this.connectionType, this.extensionSQLScriptRootPath);
        this.migrations = this.filterAndGetMigrationsToRun(availableMigrations);
    }

    public void validateMigrationsForServer() {
        if (!this.migrations.isEmpty()) {
            throw new IllegalStateException("There are pending migrations to be run on the database. Please backup your data and run `./bootstrap/bootstrap_storage.sh migrate-all`. You can find more information on upgrading OpenMetadata at https://docs.open-metadata.org/deployment/upgrade ");
        }
    }

    public List<MigrationFile> getMigrationFiles(String nativeSQLScriptRootPath, ConnectionType connectionType, String extensionSQLScriptRootPath) {
        List<MigrationFile> availableOMNativeMigrations = this.getMigrationFilesFromPath(nativeSQLScriptRootPath, connectionType);
        if (extensionSQLScriptRootPath == null || extensionSQLScriptRootPath.isEmpty()) {
            return availableOMNativeMigrations;
        }
        List<MigrationFile> availableExtensionMigrations = this.getMigrationFilesFromPath(extensionSQLScriptRootPath, connectionType);
        return Stream.concat(availableOMNativeMigrations.stream(), availableExtensionMigrations.stream()).sorted().toList();
    }

    public List<MigrationFile> getMigrationFilesFromPath(String path, ConnectionType connectionType) {
        return Arrays.stream(Objects.requireNonNull(new File(path).listFiles(File::isDirectory))).map(dir -> new MigrationFile((File)dir, this.migrationDAO, connectionType)).sorted().toList();
    }

    private List<MigrationProcess> filterAndGetMigrationsToRun(List<MigrationFile> availableMigrations) {
        LOG.debug("Filtering Server Migrations");
        this.currentMaxMigrationVersion = this.migrationDAO.getMaxServerMigrationVersion();
        List<MigrationFile> applyMigrations = this.currentMaxMigrationVersion.isPresent() && !this.forceMigrations ? availableMigrations.stream().filter(migration -> migration.biggerThan(this.currentMaxMigrationVersion.get())).toList() : availableMigrations;
        ArrayList<MigrationProcess> processes = new ArrayList<MigrationProcess>();
        try {
            for (MigrationFile file : applyMigrations) {
                file.parseSQLFiles();
                String clazzName = file.getMigrationProcessClassName();
                MigrationProcess process = (MigrationProcess)Class.forName(clazzName).getConstructor(MigrationFile.class).newInstance(file);
                processes.add(process);
            }
        }
        catch (Exception e) {
            LOG.error("Failed to list and add migrations to run due to ", (Throwable)e);
        }
        return processes;
    }

    public void printMigrationInfo() {
        LOG.info("Following Migrations will be performed, with Force Migration : {}", (Object)this.forceMigrations);
        List<String> columns = Arrays.asList("Version", "ConnectionType", "MigrationsFilePath");
        ArrayList<List<String>> allRows = new ArrayList<List<String>>();
        for (MigrationProcess process : this.migrations) {
            ArrayList<String> row = new ArrayList<String>();
            row.add(process.getVersion());
            row.add(process.getDatabaseConnectionType());
            row.add(process.getMigrationsPath());
            allRows.add(row);
        }
        OpenMetadataOperations.printToAsciiTable(columns.stream().toList(), allRows, "No Server Migration To be Run");
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void runMigrationWorkflows() {
        List<String> columns = Arrays.asList("Version", "Initialization", "SchemaChanges", "DataMigration", "PostDDLScripts", "Context");
        ArrayList<List<String>> allRows = new ArrayList<List<String>>();
        try (Handle transactionHandler = this.jdbi.open();){
            MigrationWorkflowContext context = new MigrationWorkflowContext(transactionHandler);
            if (this.currentMaxMigrationVersion.isPresent()) {
                LOG.debug("Current Max version {}", (Object)this.currentMaxMigrationVersion.get());
                context.computeInitialContext(this.currentMaxMigrationVersion.get());
            } else {
                context.computeInitialContext("1.1.0");
            }
            LOG.info("[MigrationWorkflow] WorkFlow Started");
            try {
                for (MigrationProcess process : this.migrations) {
                    LOG.info("[MigrationWorkFlow] Migration Run started for Version: {}, with Force Migration : {}", (Object)process.getVersion(), (Object)this.forceMigrations);
                    ArrayList<String> row = new ArrayList<String>();
                    row.add(process.getVersion());
                    try {
                        this.runStepAndAddStatus(row, () -> process.initialize(transactionHandler));
                        this.runSchemaChanges(row, process);
                        this.runStepAndAddStatus(row, process::runDataMigration);
                        this.runPostDDLChanges(row, process);
                        context.computeMigrationContext(process);
                        row.add(context.getMigrationContext().get(process.getVersion()).getResults().toString());
                        this.updateMigrationStepInDB(process, context);
                    }
                    finally {
                        allRows.add(row);
                        LOG.info("[MigrationWorkFlow] Migration Run finished for Version: {}", (Object)process.getVersion());
                    }
                }
                OpenMetadataOperations.printToAsciiTable(columns, allRows, "Status Unavailable");
            }
            catch (Exception e) {
                LOG.error("Encountered Exception in MigrationWorkflow", (Throwable)e);
                throw e;
            }
        }
        LOG.info("[MigrationWorkflow] WorkFlow Completed");
    }

    private void runSchemaChanges(List<String> row, MigrationProcess process) {
        block2: {
            try {
                List<String> schemaChangesColumns = Arrays.asList("Query", "Query Status");
                Map<String, QueryStatus> queryStatusMap = process.runSchemaChanges(this.forceMigrations);
                ArrayList<List<String>> allSchemaChangesRows = new ArrayList<List<String>>(queryStatusMap.entrySet().stream().map(entry -> Arrays.asList((String)entry.getKey(), String.format("Status : %s , Message: %s", new Object[]{((QueryStatus)entry.getValue()).getStatus(), ((QueryStatus)entry.getValue()).getMessage()}))).toList());
                LOG.info("[MigrationWorkflow] Version : {} Run Schema Changes Query Status", (Object)process.getVersion());
                OpenMetadataOperations.printToAsciiTable(schemaChangesColumns, allSchemaChangesRows, "No New Queries");
                row.add(SUCCESS_MSG);
            }
            catch (Exception e) {
                row.add(FAILED_MSG + e.getMessage());
                if (this.forceMigrations) break block2;
                throw e;
            }
        }
    }

    private void runPostDDLChanges(List<String> row, MigrationProcess process) {
        block2: {
            try {
                List<String> schemaChangesColumns = Arrays.asList("Query", "Query Status");
                Map<String, QueryStatus> queryStatusMap = process.runPostDDLScripts(this.forceMigrations);
                ArrayList<List<String>> allSchemaChangesRows = new ArrayList<List<String>>(queryStatusMap.entrySet().stream().map(entry -> Arrays.asList((String)entry.getKey(), String.format("Status : %s , Message: %s", new Object[]{((QueryStatus)entry.getValue()).getStatus(), ((QueryStatus)entry.getValue()).getMessage()}))).toList());
                LOG.info("[MigrationWorkflow] Version : {} Run Post DDL Query Status", (Object)process.getVersion());
                OpenMetadataOperations.printToAsciiTable(schemaChangesColumns, allSchemaChangesRows, "No New Queries");
                row.add(SUCCESS_MSG);
            }
            catch (Exception e) {
                row.add(FAILED_MSG + e.getMessage());
                if (this.forceMigrations) break block2;
                throw e;
            }
        }
    }

    private void runStepAndAddStatus(List<String> row, MigrationProcess.MigrationProcessCallback process) {
        block2: {
            try {
                process.call();
                row.add(SUCCESS_MSG);
            }
            catch (Exception e) {
                row.add(FAILED_MSG + e.getMessage());
                if (this.forceMigrations) break block2;
                throw e;
            }
        }
    }

    public void updateMigrationStepInDB(MigrationProcess step, MigrationWorkflowContext workflowContext) {
        MigrationContext context = workflowContext.getMigrationContext().get(step.getVersion());
        JSONObject metrics = new JSONObject(context.getResults());
        this.migrationDAO.upsertServerMigration(step.getVersion(), step.getMigrationsPath(), UUID.randomUUID().toString(), metrics.toString());
    }
}

