/*
 * Decompiled with CFR 0.152.
 */
package org.openmetadata.service.security;

import com.auth0.jwk.Jwk;
import com.auth0.jwk.JwkProvider;
import com.auth0.jwt.JWT;
import com.auth0.jwt.algorithms.Algorithm;
import com.auth0.jwt.exceptions.JWTDecodeException;
import com.auth0.jwt.interfaces.Claim;
import com.auth0.jwt.interfaces.DecodedJWT;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.collect.ImmutableList;
import java.net.URL;
import java.security.interfaces.RSAPublicKey;
import java.util.Arrays;
import java.util.Calendar;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TimeZone;
import java.util.TreeMap;
import java.util.stream.Collectors;
import javax.ws.rs.container.ContainerRequestContext;
import javax.ws.rs.container.ContainerRequestFilter;
import javax.ws.rs.core.MultivaluedMap;
import javax.ws.rs.core.SecurityContext;
import javax.ws.rs.core.UriInfo;
import javax.ws.rs.ext.Provider;
import org.openmetadata.common.utils.CommonUtil;
import org.openmetadata.schema.api.security.AuthenticationConfiguration;
import org.openmetadata.schema.api.security.AuthorizerConfiguration;
import org.openmetadata.schema.auth.LogoutRequest;
import org.openmetadata.schema.auth.ServiceTokenType;
import org.openmetadata.schema.services.connections.metadata.AuthProvider;
import org.openmetadata.service.security.AuthenticationException;
import org.openmetadata.service.security.CatalogPrincipal;
import org.openmetadata.service.security.MultiUrlJwkProvider;
import org.openmetadata.service.security.SecurityUtil;
import org.openmetadata.service.security.auth.BotTokenCache;
import org.openmetadata.service.security.auth.CatalogSecurityContext;
import org.openmetadata.service.security.auth.UserTokenCache;
import org.openmetadata.service.security.saml.JwtTokenCacheManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Provider
public class JwtFilter
implements ContainerRequestFilter {
    private static final Logger LOG = LoggerFactory.getLogger(JwtFilter.class);
    public static final String EMAIL_CLAIM_KEY = "email";
    public static final String USERNAME_CLAIM_KEY = "username";
    public static final String AUTHORIZATION_HEADER = "Authorization";
    public static final String TOKEN_PREFIX = "Bearer";
    public static final String BOT_CLAIM = "isBot";
    private List<String> jwtPrincipalClaims;
    private Map<String, String> jwtPrincipalClaimsMapping;
    private JwkProvider jwkProvider;
    private String principalDomain;
    private boolean enforcePrincipalDomain;
    private AuthProvider providerType;
    private boolean useRolesFromProvider = false;
    private static final List<String> DEFAULT_PUBLIC_KEY_URLS = Arrays.asList("http://localhost:8585/api/v1/system/config/jwks", "http://host.docker.internal:8585/api/v1/system/config/jwks");
    public static final List<String> EXCLUDED_ENDPOINTS = List.of("v1/system/config/jwks", "v1/system/config/authorizer", "v1/system/config/customUiThemePreference", "v1/system/config/auth", "v1/users/signup", "v1/system/version", "v1/users/registrationConfirmation", "v1/users/resendRegistrationToken", "v1/users/generatePasswordResetLink", "v1/users/password/reset", "v1/users/checkEmailInUse", "v1/users/login", "v1/users/refresh");

    private JwtFilter() {
    }

    public JwtFilter(AuthenticationConfiguration authenticationConfiguration, AuthorizerConfiguration authorizerConfiguration) {
        this.providerType = authenticationConfiguration.getProvider();
        this.jwtPrincipalClaims = authenticationConfiguration.getJwtPrincipalClaims();
        this.jwtPrincipalClaimsMapping = CommonUtil.listOrEmpty((List)authenticationConfiguration.getJwtPrincipalClaimsMapping()).stream().map(s -> s.split(":")).collect(Collectors.toMap(s -> s[0], s -> s[1]));
        SecurityUtil.validatePrincipalClaimsMapping(this.jwtPrincipalClaimsMapping);
        ImmutableList.Builder publicKeyUrlsBuilder = ImmutableList.builder();
        for (String publicKeyUrlStr : authenticationConfiguration.getPublicKeyUrls()) {
            publicKeyUrlsBuilder.add((Object)new URL(publicKeyUrlStr));
        }
        for (String publicKeyUrl : DEFAULT_PUBLIC_KEY_URLS) {
            if (authenticationConfiguration.getPublicKeyUrls().contains(publicKeyUrl)) continue;
            publicKeyUrlsBuilder.add((Object)new URL(publicKeyUrl));
        }
        this.jwkProvider = new MultiUrlJwkProvider((List<URL>)publicKeyUrlsBuilder.build());
        this.principalDomain = authorizerConfiguration.getPrincipalDomain();
        this.enforcePrincipalDomain = authorizerConfiguration.getEnforcePrincipalDomain();
        this.useRolesFromProvider = authorizerConfiguration.getUseRolesFromProvider();
    }

    @VisibleForTesting
    JwtFilter(JwkProvider jwkProvider, List<String> jwtPrincipalClaims, String principalDomain, boolean enforcePrincipalDomain) {
        this.jwkProvider = jwkProvider;
        this.jwtPrincipalClaims = jwtPrincipalClaims;
        this.principalDomain = principalDomain;
        this.enforcePrincipalDomain = enforcePrincipalDomain;
    }

    public void filter(ContainerRequestContext requestContext) {
        UriInfo uriInfo = requestContext.getUriInfo();
        if (EXCLUDED_ENDPOINTS.stream().anyMatch(endpoint -> uriInfo.getPath().equalsIgnoreCase((String)endpoint))) {
            return;
        }
        String tokenFromHeader = JwtFilter.extractToken((MultivaluedMap<String, String>)requestContext.getHeaders());
        LOG.debug("Token from header:{}", (Object)tokenFromHeader);
        Map<String, Claim> claims = this.validateJwtAndGetClaims(tokenFromHeader);
        String userName = SecurityUtil.findUserNameFromClaims(this.jwtPrincipalClaimsMapping, this.jwtPrincipalClaims, claims);
        String email = SecurityUtil.findEmailFromClaims(this.jwtPrincipalClaimsMapping, this.jwtPrincipalClaims, claims, this.principalDomain);
        this.checkValidationsForToken(claims, tokenFromHeader, userName);
        CatalogPrincipal catalogPrincipal = new CatalogPrincipal(userName, email);
        String scheme = requestContext.getUriInfo().getRequestUri().getScheme();
        CatalogSecurityContext catalogSecurityContext = new CatalogSecurityContext(catalogPrincipal, scheme, "DIGEST", this.getUserRolesFromClaims(claims, SecurityUtil.isBot(claims)));
        LOG.debug("SecurityContext {}", (Object)catalogSecurityContext);
        requestContext.setSecurityContext((SecurityContext)catalogSecurityContext);
    }

    public void checkValidationsForToken(Map<String, Claim> claims, String tokenFromHeader, String userName) {
        this.validateTokenIsNotUsedAfterLogout(tokenFromHeader);
        SecurityUtil.validateDomainEnforcement(this.jwtPrincipalClaimsMapping, this.jwtPrincipalClaims, claims, this.principalDomain, this.enforcePrincipalDomain);
        if (SecurityUtil.isBot(claims)) {
            this.validateBotToken(tokenFromHeader, userName);
        }
        this.validatePersonalAccessToken(claims, tokenFromHeader, userName);
    }

    private Set<String> getUserRolesFromClaims(Map<String, Claim> claims, boolean isBot) {
        List roles;
        HashSet<String> userRoles = new HashSet<String>();
        if (this.useRolesFromProvider && !isBot && claims.containsKey("roles") && !CommonUtil.nullOrEmpty((List)(roles = claims.get("roles").asList(String.class)))) {
            userRoles = new HashSet(claims.get("roles").asList(String.class));
        }
        return userRoles;
    }

    public Map<String, Claim> validateJwtAndGetClaims(String token) {
        DecodedJWT jwt;
        try {
            jwt = JWT.decode((String)token);
        }
        catch (JWTDecodeException e) {
            throw new AuthenticationException("Invalid token", e);
        }
        if (jwt.getExpiresAt() != null && jwt.getExpiresAt().before(Calendar.getInstance(TimeZone.getTimeZone("UTC")).getTime())) {
            throw new AuthenticationException("Expired token!");
        }
        Jwk jwk = this.jwkProvider.get(jwt.getKeyId());
        Algorithm algorithm = Algorithm.RSA256((RSAPublicKey)((RSAPublicKey)jwk.getPublicKey()), null);
        try {
            algorithm.verify(jwt);
        }
        catch (RuntimeException runtimeException) {
            throw new AuthenticationException("Invalid token", runtimeException);
        }
        TreeMap<String, Claim> claims = new TreeMap<String, Claim>(String.CASE_INSENSITIVE_ORDER);
        claims.putAll(jwt.getClaims());
        return claims;
    }

    protected static String extractToken(MultivaluedMap<String, String> headers) {
        LOG.debug("Request Headers:{}", headers);
        String source = (String)headers.getFirst((Object)AUTHORIZATION_HEADER);
        if (CommonUtil.nullOrEmpty((String)source)) {
            throw AuthenticationException.getTokenNotPresentException();
        }
        if (source.startsWith(TOKEN_PREFIX)) {
            return source.substring(TOKEN_PREFIX.length() + 1);
        }
        throw AuthenticationException.getTokenNotPresentException();
    }

    public static String extractToken(String tokenFromHeader) {
        LOG.debug("Request Token:{}", (Object)tokenFromHeader);
        if (CommonUtil.nullOrEmpty((String)tokenFromHeader)) {
            throw AuthenticationException.getTokenNotPresentException();
        }
        if (tokenFromHeader.startsWith(TOKEN_PREFIX)) {
            return tokenFromHeader.substring(TOKEN_PREFIX.length() + 1);
        }
        throw AuthenticationException.getTokenNotPresentException();
    }

    private void validateBotToken(String tokenFromHeader, String userName) {
        if (tokenFromHeader.equals(BotTokenCache.getToken(userName))) {
            return;
        }
        throw AuthenticationException.getInvalidTokenException();
    }

    private void validatePersonalAccessToken(Map<String, Claim> claims, String tokenFromHeader, String userName) {
        if (claims.containsKey("tokenType") && ServiceTokenType.PERSONAL_ACCESS.value().equals(claims.get("tokenType") != null ? "" : claims.get("tokenType").asString())) {
            Set<String> userTokens = UserTokenCache.getToken(userName);
            if (userTokens != null && userTokens.contains(tokenFromHeader)) {
                return;
            }
            throw AuthenticationException.getInvalidTokenException();
        }
    }

    private void validateTokenIsNotUsedAfterLogout(String authToken) {
        LogoutRequest previouslyLoggedOutEvent;
        if ((AuthProvider.BASIC.equals((Object)this.providerType) || AuthProvider.SAML.equals((Object)this.providerType)) && (previouslyLoggedOutEvent = JwtTokenCacheManager.getInstance().getLogoutEventForToken(authToken)) != null) {
            throw new AuthenticationException("Expired token!");
        }
    }

    public List<String> getJwtPrincipalClaims() {
        return this.jwtPrincipalClaims;
    }

    public Map<String, String> getJwtPrincipalClaimsMapping() {
        return this.jwtPrincipalClaimsMapping;
    }
}

