/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.mongodb.core;

import com.mongodb.DB;
import com.mongodb.Mongo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.authentication.UserCredentials;
import org.springframework.data.mongodb.CannotGetMongoDbConnectionException;
import org.springframework.data.mongodb.core.DbHolder;
import org.springframework.data.mongodb.core.MongoSynchronization;
import org.springframework.transaction.support.ResourceHolder;
import org.springframework.transaction.support.TransactionSynchronization;
import org.springframework.transaction.support.TransactionSynchronizationManager;
import org.springframework.util.Assert;

public abstract class MongoDbUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(MongoDbUtils.class);

    private MongoDbUtils() {
    }

    public static DB getDB(Mongo mongo, String databaseName) {
        return MongoDbUtils.doGetDB(mongo, databaseName, UserCredentials.NO_CREDENTIALS, true);
    }

    public static DB getDB(Mongo mongo, String databaseName, UserCredentials credentials) {
        Assert.notNull((Object)mongo, (String)"No Mongo instance specified!");
        Assert.hasText((String)databaseName, (String)"Database name must be given!");
        Assert.notNull((Object)credentials, (String)"Credentials must not be null, use UserCredentials.NO_CREDENTIALS!");
        return MongoDbUtils.doGetDB(mongo, databaseName, credentials, true);
    }

    private static DB doGetDB(Mongo mongo, String databaseName, UserCredentials credentials, boolean allowCreate) {
        String password;
        String username;
        boolean credentialsGiven;
        DB db;
        DbHolder dbHolder = (DbHolder)((Object)TransactionSynchronizationManager.getResource((Object)mongo));
        if (dbHolder != null && !dbHolder.isEmpty()) {
            db = null;
            if (TransactionSynchronizationManager.isSynchronizationActive() && dbHolder.doesNotHoldNonDefaultDB() && (db = dbHolder.getDB(databaseName)) != null && !dbHolder.isSynchronizedWithTransaction()) {
                LOGGER.debug("Registering Spring transaction synchronization for existing MongoDB {}.", (Object)databaseName);
                TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new MongoSynchronization((ResourceHolder)dbHolder, mongo));
                dbHolder.setSynchronizedWithTransaction(true);
            }
            if (db != null) {
                return db;
            }
        }
        LOGGER.debug("Getting Mongo Database name=[{}]", (Object)databaseName);
        db = mongo.getDB(databaseName);
        boolean bl = credentialsGiven = credentials.hasUsername() && credentials.hasPassword();
        if (credentialsGiven && !db.isAuthenticated() && !db.authenticate(username = credentials.getUsername(), (password = credentials.hasPassword() ? credentials.getPassword() : null) == null ? null : password.toCharArray())) {
            throw new CannotGetMongoDbConnectionException("Failed to authenticate to database [" + databaseName + "], username = [" + username + "], password = [" + password + "]", databaseName, credentials);
        }
        if (TransactionSynchronizationManager.isSynchronizationActive()) {
            LOGGER.debug("Registering Spring transaction synchronization for MongoDB instance {}.", (Object)databaseName);
            DbHolder holderToUse = dbHolder;
            if (holderToUse == null) {
                holderToUse = new DbHolder(databaseName, db);
            } else {
                holderToUse.addDB(databaseName, db);
            }
            TransactionSynchronizationManager.registerSynchronization((TransactionSynchronization)new MongoSynchronization((ResourceHolder)holderToUse, mongo));
            holderToUse.setSynchronizedWithTransaction(true);
            if (holderToUse != dbHolder) {
                TransactionSynchronizationManager.bindResource((Object)mongo, (Object)((Object)holderToUse));
            }
        }
        if (!allowCreate && !MongoDbUtils.isDBTransactional(db, mongo)) {
            throw new IllegalStateException("No Mongo DB bound to thread, and configuration does not allow creation of non-transactional one here");
        }
        return db;
    }

    public static boolean isDBTransactional(DB db, Mongo mongo) {
        if (mongo == null) {
            return false;
        }
        DbHolder dbHolder = (DbHolder)((Object)TransactionSynchronizationManager.getResource((Object)mongo));
        return dbHolder != null && dbHolder.containsDB(db);
    }

    public static void closeDB(DB db) {
        if (db != null) {
            LOGGER.debug("Closing Mongo DB object");
            try {
                db.requestDone();
            }
            catch (Throwable ex) {
                LOGGER.debug("Unexpected exception on closing Mongo DB object", ex);
            }
        }
    }
}

