// Utilities
import { computed, getCurrentInstance, inject, onBeforeUnmount, onMounted, provide, reactive, toRef } from 'vue';
import { useProxiedModel } from "./proxiedModel.mjs";
import { consoleWarn, deepEqual, findChildren, getUid, propsFactory, wrapInArray } from "../util/index.mjs"; // Types

export const makeGroupProps = propsFactory({
  modelValue: {
    type: [Number, Boolean, String, Array, Object],
    default: undefined
  },
  multiple: Boolean,
  mandatory: [Boolean, String],
  max: Number,
  selectedClass: String,
  disabled: Boolean
}, 'group');
export const makeGroupItemProps = propsFactory({
  value: {
    type: [Number, Boolean, String, Object],
    default: undefined
  },
  disabled: Boolean,
  selectedClass: String
}, 'group-item'); // Composables

export function useGroupItem(props, injectKey) {
  const vm = getCurrentInstance();

  if (!vm) {
    throw new Error('[Vuetify] useGroupItem composable must be used inside a component setup function');
  }

  const group = inject(injectKey, null);

  if (!group) {
    throw new Error(`[Vuetify] Could not find useGroup injection with symbol ${injectKey.description}`);
  }

  const id = getUid();
  const value = toRef(props, 'value');
  const disabled = computed(() => group.disabled.value || props.disabled);
  group.register({
    id,
    value,
    disabled
  }, vm);
  onBeforeUnmount(() => {
    group.unregister(id);
  });
  const isSelected = computed(() => {
    return group.isSelected(id);
  });
  const selectedClass = computed(() => {
    var _group$selectedClass$;

    return isSelected.value && ((_group$selectedClass$ = group.selectedClass.value) != null ? _group$selectedClass$ : props.selectedClass);
  });
  return {
    id,
    isSelected,
    toggle: () => group.select(id, !isSelected.value),
    select: value => group.select(id, value),
    selectedClass,
    value,
    disabled,
    group
  };
}
export function useGroup(props, injectKey) {
  let isUnmounted = false;
  const items = reactive([]);
  const selected = useProxiedModel(props, 'modelValue', [], v => {
    if (v == null) return [];
    return getIds(items, wrapInArray(v));
  }, v => {
    const arr = getValues(items, v);
    return props.multiple ? arr : arr[0];
  });
  const groupVm = getCurrentInstance();

  function register(item, vm) {
    // Is there a better way to fix this typing?
    const unwrapped = item;
    const children = findChildren(groupVm == null ? void 0 : groupVm.vnode);
    const instances = children.slice(1) // First one is group component itself
    .filter(cmp => !!cmp.provides[injectKey]); // TODO: Fix in TS 4.4

    const index = instances.indexOf(vm);
    if (index > -1) items.splice(index, 0, unwrapped);else items.push(unwrapped);
  }

  function unregister(id) {
    if (isUnmounted) return;
    selected.value = selected.value.filter(v => v !== id);
    forceMandatoryValue();
    const index = items.findIndex(item => item.id === id);
    items.splice(index, 1);
  } // If mandatory and nothing is selected, then select first non-disabled item


  function forceMandatoryValue() {
    const item = items.find(item => !item.disabled);

    if (item && props.mandatory === 'force' && !selected.value.length) {
      selected.value = [item.id];
    }
  }

  onMounted(() => {
    forceMandatoryValue();
  });
  onBeforeUnmount(() => {
    isUnmounted = true;
  });

  function select(id, isSelected) {
    const item = items.find(item => item.id === id);
    if (isSelected && item != null && item.disabled) return;

    if (props.multiple) {
      const internalValue = selected.value.slice();
      const index = internalValue.findIndex(v => v === id); // We can't remove value if group is
      // mandatory, value already exists,
      // and it is the only value

      if (props.mandatory && index > -1 && internalValue.length <= 1) return; // We can't add value if it would
      // cause max limit to be exceeded

      if (props.max != null && index < 0 && internalValue.length + 1 > props.max) return;
      if (index < 0 && isSelected) internalValue.push(id);else if (index >= 0 && !isSelected) internalValue.splice(index, 1);
      selected.value = internalValue;
    } else {
      if (props.mandatory && selected.value.includes(id)) return;
      selected.value = isSelected ? [id] : [];
    }
  }

  function step(offset) {
    // getting an offset from selected value obviously won't work with multiple values
    if (props.multiple) consoleWarn('This method is not supported when using "multiple" prop');

    if (!selected.value.length) {
      const item = items.find(item => !item.disabled);
      item && (selected.value = [item.id]);
    } else {
      const currentId = selected.value[0];
      const currentIndex = items.findIndex(i => i.id === currentId);
      let newIndex = (currentIndex + offset) % items.length;
      let newItem = items[newIndex];

      while (newItem.disabled && newIndex !== currentIndex) {
        newIndex = (newIndex + offset) % items.length;
        newItem = items[newIndex];
      }

      if (newItem.disabled) return;
      selected.value = [items[newIndex].id];
    }
  }

  const state = {
    register,
    unregister,
    selected,
    select,
    disabled: toRef(props, 'disabled'),
    prev: () => step(items.length - 1),
    next: () => step(1),
    isSelected: id => selected.value.includes(id),
    selectedClass: computed(() => props.selectedClass),
    items: computed(() => items.map(({
      id
    }) => id))
  };
  provide(injectKey, state);
  return state;
}

function getIds(items, modelValue) {
  const ids = [];

  for (const item of items) {
    if (item.value != null) {
      if (modelValue.find(value => deepEqual(value, item.value))) {
        ids.push(item.id);
      }
    } else if (modelValue.includes(item.id)) {
      ids.push(item.id);
    }
  }

  return ids;
}

function getValues(items, ids) {
  const values = [];

  for (const item of items) {
    if (ids.includes(item.id)) {
      values.push(item.value != null ? item.value : item.id);
    }
  }

  return values;
}
//# sourceMappingURL=group.mjs.map