/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details from the malware scan that created a finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EbsVolumeScanDetails implements SdkPojo, Serializable,
        ToCopyableBuilder<EbsVolumeScanDetails.Builder, EbsVolumeScanDetails> {
    private static final SdkField<String> SCAN_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("ScanId")
            .getter(getter(EbsVolumeScanDetails::scanId)).setter(setter(Builder::scanId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanId").build()).build();

    private static final SdkField<Instant> SCAN_STARTED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ScanStartedAt").getter(getter(EbsVolumeScanDetails::scanStartedAt))
            .setter(setter(Builder::scanStartedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanStartedAt").build()).build();

    private static final SdkField<Instant> SCAN_COMPLETED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("ScanCompletedAt").getter(getter(EbsVolumeScanDetails::scanCompletedAt))
            .setter(setter(Builder::scanCompletedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanCompletedAt").build()).build();

    private static final SdkField<String> TRIGGER_FINDING_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("TriggerFindingId").getter(getter(EbsVolumeScanDetails::triggerFindingId))
            .setter(setter(Builder::triggerFindingId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("triggerFindingId").build()).build();

    private static final SdkField<List<String>> SOURCES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Sources")
            .getter(getter(EbsVolumeScanDetails::sources))
            .setter(setter(Builder::sources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ScanDetections> SCAN_DETECTIONS_FIELD = SdkField
            .<ScanDetections> builder(MarshallingType.SDK_POJO).memberName("ScanDetections")
            .getter(getter(EbsVolumeScanDetails::scanDetections)).setter(setter(Builder::scanDetections))
            .constructor(ScanDetections::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanDetections").build()).build();

    private static final SdkField<String> SCAN_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ScanType").getter(getter(EbsVolumeScanDetails::scanTypeAsString)).setter(setter(Builder::scanType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("scanType").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SCAN_ID_FIELD,
            SCAN_STARTED_AT_FIELD, SCAN_COMPLETED_AT_FIELD, TRIGGER_FINDING_ID_FIELD, SOURCES_FIELD, SCAN_DETECTIONS_FIELD,
            SCAN_TYPE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("scanId", SCAN_ID_FIELD);
                    put("scanStartedAt", SCAN_STARTED_AT_FIELD);
                    put("scanCompletedAt", SCAN_COMPLETED_AT_FIELD);
                    put("triggerFindingId", TRIGGER_FINDING_ID_FIELD);
                    put("sources", SOURCES_FIELD);
                    put("scanDetections", SCAN_DETECTIONS_FIELD);
                    put("scanType", SCAN_TYPE_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String scanId;

    private final Instant scanStartedAt;

    private final Instant scanCompletedAt;

    private final String triggerFindingId;

    private final List<String> sources;

    private final ScanDetections scanDetections;

    private final String scanType;

    private EbsVolumeScanDetails(BuilderImpl builder) {
        this.scanId = builder.scanId;
        this.scanStartedAt = builder.scanStartedAt;
        this.scanCompletedAt = builder.scanCompletedAt;
        this.triggerFindingId = builder.triggerFindingId;
        this.sources = builder.sources;
        this.scanDetections = builder.scanDetections;
        this.scanType = builder.scanType;
    }

    /**
     * <p>
     * Unique Id of the malware scan that generated the finding.
     * </p>
     * 
     * @return Unique Id of the malware scan that generated the finding.
     */
    public final String scanId() {
        return scanId;
    }

    /**
     * <p>
     * Returns the start date and time of the malware scan.
     * </p>
     * 
     * @return Returns the start date and time of the malware scan.
     */
    public final Instant scanStartedAt() {
        return scanStartedAt;
    }

    /**
     * <p>
     * Returns the completion date and time of the malware scan.
     * </p>
     * 
     * @return Returns the completion date and time of the malware scan.
     */
    public final Instant scanCompletedAt() {
        return scanCompletedAt;
    }

    /**
     * <p>
     * GuardDuty finding ID that triggered a malware scan.
     * </p>
     * 
     * @return GuardDuty finding ID that triggered a malware scan.
     */
    public final String triggerFindingId() {
        return triggerFindingId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Sources property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasSources() {
        return sources != null && !(sources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Contains list of threat intelligence sources used to detect threats.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasSources} method.
     * </p>
     * 
     * @return Contains list of threat intelligence sources used to detect threats.
     */
    public final List<String> sources() {
        return sources;
    }

    /**
     * <p>
     * Contains a complete view providing malware scan result details.
     * </p>
     * 
     * @return Contains a complete view providing malware scan result details.
     */
    public final ScanDetections scanDetections() {
        return scanDetections;
    }

    /**
     * <p>
     * Specifies the scan type that invoked the malware scan.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanType} will
     * return {@link ScanType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanTypeAsString}.
     * </p>
     * 
     * @return Specifies the scan type that invoked the malware scan.
     * @see ScanType
     */
    public final ScanType scanType() {
        return ScanType.fromValue(scanType);
    }

    /**
     * <p>
     * Specifies the scan type that invoked the malware scan.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #scanType} will
     * return {@link ScanType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #scanTypeAsString}.
     * </p>
     * 
     * @return Specifies the scan type that invoked the malware scan.
     * @see ScanType
     */
    public final String scanTypeAsString() {
        return scanType;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(scanId());
        hashCode = 31 * hashCode + Objects.hashCode(scanStartedAt());
        hashCode = 31 * hashCode + Objects.hashCode(scanCompletedAt());
        hashCode = 31 * hashCode + Objects.hashCode(triggerFindingId());
        hashCode = 31 * hashCode + Objects.hashCode(hasSources() ? sources() : null);
        hashCode = 31 * hashCode + Objects.hashCode(scanDetections());
        hashCode = 31 * hashCode + Objects.hashCode(scanTypeAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EbsVolumeScanDetails)) {
            return false;
        }
        EbsVolumeScanDetails other = (EbsVolumeScanDetails) obj;
        return Objects.equals(scanId(), other.scanId()) && Objects.equals(scanStartedAt(), other.scanStartedAt())
                && Objects.equals(scanCompletedAt(), other.scanCompletedAt())
                && Objects.equals(triggerFindingId(), other.triggerFindingId()) && hasSources() == other.hasSources()
                && Objects.equals(sources(), other.sources()) && Objects.equals(scanDetections(), other.scanDetections())
                && Objects.equals(scanTypeAsString(), other.scanTypeAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EbsVolumeScanDetails").add("ScanId", scanId()).add("ScanStartedAt", scanStartedAt())
                .add("ScanCompletedAt", scanCompletedAt()).add("TriggerFindingId", triggerFindingId())
                .add("Sources", hasSources() ? sources() : null).add("ScanDetections", scanDetections())
                .add("ScanType", scanTypeAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ScanId":
            return Optional.ofNullable(clazz.cast(scanId()));
        case "ScanStartedAt":
            return Optional.ofNullable(clazz.cast(scanStartedAt()));
        case "ScanCompletedAt":
            return Optional.ofNullable(clazz.cast(scanCompletedAt()));
        case "TriggerFindingId":
            return Optional.ofNullable(clazz.cast(triggerFindingId()));
        case "Sources":
            return Optional.ofNullable(clazz.cast(sources()));
        case "ScanDetections":
            return Optional.ofNullable(clazz.cast(scanDetections()));
        case "ScanType":
            return Optional.ofNullable(clazz.cast(scanTypeAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<EbsVolumeScanDetails, T> g) {
        return obj -> g.apply((EbsVolumeScanDetails) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EbsVolumeScanDetails> {
        /**
         * <p>
         * Unique Id of the malware scan that generated the finding.
         * </p>
         * 
         * @param scanId
         *        Unique Id of the malware scan that generated the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanId(String scanId);

        /**
         * <p>
         * Returns the start date and time of the malware scan.
         * </p>
         * 
         * @param scanStartedAt
         *        Returns the start date and time of the malware scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanStartedAt(Instant scanStartedAt);

        /**
         * <p>
         * Returns the completion date and time of the malware scan.
         * </p>
         * 
         * @param scanCompletedAt
         *        Returns the completion date and time of the malware scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanCompletedAt(Instant scanCompletedAt);

        /**
         * <p>
         * GuardDuty finding ID that triggered a malware scan.
         * </p>
         * 
         * @param triggerFindingId
         *        GuardDuty finding ID that triggered a malware scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder triggerFindingId(String triggerFindingId);

        /**
         * <p>
         * Contains list of threat intelligence sources used to detect threats.
         * </p>
         * 
         * @param sources
         *        Contains list of threat intelligence sources used to detect threats.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sources(Collection<String> sources);

        /**
         * <p>
         * Contains list of threat intelligence sources used to detect threats.
         * </p>
         * 
         * @param sources
         *        Contains list of threat intelligence sources used to detect threats.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sources(String... sources);

        /**
         * <p>
         * Contains a complete view providing malware scan result details.
         * </p>
         * 
         * @param scanDetections
         *        Contains a complete view providing malware scan result details.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scanDetections(ScanDetections scanDetections);

        /**
         * <p>
         * Contains a complete view providing malware scan result details.
         * </p>
         * This is a convenience method that creates an instance of the {@link ScanDetections.Builder} avoiding the need
         * to create one manually via {@link ScanDetections#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ScanDetections.Builder#build()} is called immediately and its
         * result is passed to {@link #scanDetections(ScanDetections)}.
         * 
         * @param scanDetections
         *        a consumer that will call methods on {@link ScanDetections.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #scanDetections(ScanDetections)
         */
        default Builder scanDetections(Consumer<ScanDetections.Builder> scanDetections) {
            return scanDetections(ScanDetections.builder().applyMutation(scanDetections).build());
        }

        /**
         * <p>
         * Specifies the scan type that invoked the malware scan.
         * </p>
         * 
         * @param scanType
         *        Specifies the scan type that invoked the malware scan.
         * @see ScanType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanType
         */
        Builder scanType(String scanType);

        /**
         * <p>
         * Specifies the scan type that invoked the malware scan.
         * </p>
         * 
         * @param scanType
         *        Specifies the scan type that invoked the malware scan.
         * @see ScanType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ScanType
         */
        Builder scanType(ScanType scanType);
    }

    static final class BuilderImpl implements Builder {
        private String scanId;

        private Instant scanStartedAt;

        private Instant scanCompletedAt;

        private String triggerFindingId;

        private List<String> sources = DefaultSdkAutoConstructList.getInstance();

        private ScanDetections scanDetections;

        private String scanType;

        private BuilderImpl() {
        }

        private BuilderImpl(EbsVolumeScanDetails model) {
            scanId(model.scanId);
            scanStartedAt(model.scanStartedAt);
            scanCompletedAt(model.scanCompletedAt);
            triggerFindingId(model.triggerFindingId);
            sources(model.sources);
            scanDetections(model.scanDetections);
            scanType(model.scanType);
        }

        public final String getScanId() {
            return scanId;
        }

        public final void setScanId(String scanId) {
            this.scanId = scanId;
        }

        @Override
        public final Builder scanId(String scanId) {
            this.scanId = scanId;
            return this;
        }

        public final Instant getScanStartedAt() {
            return scanStartedAt;
        }

        public final void setScanStartedAt(Instant scanStartedAt) {
            this.scanStartedAt = scanStartedAt;
        }

        @Override
        public final Builder scanStartedAt(Instant scanStartedAt) {
            this.scanStartedAt = scanStartedAt;
            return this;
        }

        public final Instant getScanCompletedAt() {
            return scanCompletedAt;
        }

        public final void setScanCompletedAt(Instant scanCompletedAt) {
            this.scanCompletedAt = scanCompletedAt;
        }

        @Override
        public final Builder scanCompletedAt(Instant scanCompletedAt) {
            this.scanCompletedAt = scanCompletedAt;
            return this;
        }

        public final String getTriggerFindingId() {
            return triggerFindingId;
        }

        public final void setTriggerFindingId(String triggerFindingId) {
            this.triggerFindingId = triggerFindingId;
        }

        @Override
        public final Builder triggerFindingId(String triggerFindingId) {
            this.triggerFindingId = triggerFindingId;
            return this;
        }

        public final Collection<String> getSources() {
            if (sources instanceof SdkAutoConstructList) {
                return null;
            }
            return sources;
        }

        public final void setSources(Collection<String> sources) {
            this.sources = SourcesCopier.copy(sources);
        }

        @Override
        public final Builder sources(Collection<String> sources) {
            this.sources = SourcesCopier.copy(sources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder sources(String... sources) {
            sources(Arrays.asList(sources));
            return this;
        }

        public final ScanDetections.Builder getScanDetections() {
            return scanDetections != null ? scanDetections.toBuilder() : null;
        }

        public final void setScanDetections(ScanDetections.BuilderImpl scanDetections) {
            this.scanDetections = scanDetections != null ? scanDetections.build() : null;
        }

        @Override
        public final Builder scanDetections(ScanDetections scanDetections) {
            this.scanDetections = scanDetections;
            return this;
        }

        public final String getScanType() {
            return scanType;
        }

        public final void setScanType(String scanType) {
            this.scanType = scanType;
        }

        @Override
        public final Builder scanType(String scanType) {
            this.scanType = scanType;
            return this;
        }

        @Override
        public final Builder scanType(ScanType scanType) {
            this.scanType(scanType == null ? null : scanType.toString());
            return this;
        }

        @Override
        public EbsVolumeScanDetails build() {
            return new EbsVolumeScanDetails(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
