/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.redshift.waiters;

import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Immutable;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.services.redshift.RedshiftAsyncClient;
import software.amazon.awssdk.services.redshift.model.DescribeClusterSnapshotsRequest;
import software.amazon.awssdk.services.redshift.model.DescribeClusterSnapshotsResponse;
import software.amazon.awssdk.services.redshift.model.DescribeClustersRequest;
import software.amazon.awssdk.services.redshift.model.DescribeClustersResponse;
import software.amazon.awssdk.utils.SdkAutoCloseable;

/**
 * Waiter utility class that polls a resource until a desired state is reached or until it is determined that the
 * resource will never enter into the desired state. This can be created using the static {@link #builder()} method
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
@Immutable
public interface RedshiftAsyncWaiter extends SdkAutoCloseable {
    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterAvailable} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeClustersRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterAvailable(
            DescribeClustersRequest describeClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterAvailable} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClustersRequest#builder()}
     *
     * @param describeClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterAvailable(
            Consumer<DescribeClustersRequest.Builder> describeClustersRequest) {
        return waitUntilClusterAvailable(DescribeClustersRequest.builder().applyMutation(describeClustersRequest).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterAvailable} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeClustersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterAvailable(
            DescribeClustersRequest describeClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterAvailable} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterAvailable(
            Consumer<DescribeClustersRequest.Builder> describeClustersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterAvailable(DescribeClustersRequest.builder().applyMutation(describeClustersRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeClustersRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterDeleted(
            DescribeClustersRequest describeClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClustersRequest#builder()}
     *
     * @param describeClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterDeleted(
            Consumer<DescribeClustersRequest.Builder> describeClustersRequest) {
        return waitUntilClusterDeleted(DescribeClustersRequest.builder().applyMutation(describeClustersRequest).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state
     *
     * @param describeClustersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterDeleted(
            DescribeClustersRequest describeClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterDeleted} is met,
     * or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterDeleted(
            Consumer<DescribeClustersRequest.Builder> describeClustersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterDeleted(DescribeClustersRequest.builder().applyMutation(describeClustersRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterRestored} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeClustersRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterRestored(
            DescribeClustersRequest describeClustersRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterRestored} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClustersRequest#builder()}
     *
     * @param describeClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterRestored(
            Consumer<DescribeClustersRequest.Builder> describeClustersRequest) {
        return waitUntilClusterRestored(DescribeClustersRequest.builder().applyMutation(describeClustersRequest).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterRestored} is
     * met, or until it is determined that the resource will never enter into the desired state
     *
     * @param describeClustersRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterRestored(
            DescribeClustersRequest describeClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusters} API until the desired condition {@code ClusterRestored} is
     * met, or until it is determined that the resource will never enter into the desired state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClustersRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClustersResponse>> waitUntilClusterRestored(
            Consumer<DescribeClustersRequest.Builder> describeClustersRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilClusterRestored(DescribeClustersRequest.builder().applyMutation(describeClustersRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusterSnapshots} API until the desired condition
     * {@code SnapshotAvailable} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeClusterSnapshotsRequest
     *        the request to be used for polling
     * @return CompletableFuture containing the WaiterResponse. It completes successfully when the resource enters into
     *         a desired state or exceptionally when it is determined that the resource will never enter into the
     *         desired state.
     */
    default CompletableFuture<WaiterResponse<DescribeClusterSnapshotsResponse>> waitUntilSnapshotAvailable(
            DescribeClusterSnapshotsRequest describeClusterSnapshotsRequest) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusterSnapshots} API until the desired condition
     * {@code SnapshotAvailable} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder without the need to create one manually
     * using {@link DescribeClusterSnapshotsRequest#builder()}
     *
     * @param describeClusterSnapshotsRequest
     *        The consumer that will configure the request to be used for polling
     * @return CompletableFuture of the WaiterResponse containing either a response or an exception that has matched
     *         with the waiter success condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterSnapshotsResponse>> waitUntilSnapshotAvailable(
            Consumer<DescribeClusterSnapshotsRequest.Builder> describeClusterSnapshotsRequest) {
        return waitUntilSnapshotAvailable(DescribeClusterSnapshotsRequest.builder()
                .applyMutation(describeClusterSnapshotsRequest).build());
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusterSnapshots} API until the desired condition
     * {@code SnapshotAvailable} is met, or until it is determined that the resource will never enter into the desired
     * state
     *
     * @param describeClusterSnapshotsRequest
     *        The request to be used for polling
     * @param overrideConfig
     *        Per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterSnapshotsResponse>> waitUntilSnapshotAvailable(
            DescribeClusterSnapshotsRequest describeClusterSnapshotsRequest, WaiterOverrideConfiguration overrideConfig) {
        throw new UnsupportedOperationException();
    }

    /**
     * Polls {@link RedshiftAsyncClient#describeClusterSnapshots} API until the desired condition
     * {@code SnapshotAvailable} is met, or until it is determined that the resource will never enter into the desired
     * state.
     * <p>
     * This is a convenience method to create an instance of the request builder and instance of the override config
     * builder
     *
     * @param describeClusterSnapshotsRequest
     *        The consumer that will configure the request to be used for polling
     * @param overrideConfig
     *        The consumer that will configure the per request override configuration for waiters
     * @return WaiterResponse containing either a response or an exception that has matched with the waiter success
     *         condition
     */
    default CompletableFuture<WaiterResponse<DescribeClusterSnapshotsResponse>> waitUntilSnapshotAvailable(
            Consumer<DescribeClusterSnapshotsRequest.Builder> describeClusterSnapshotsRequest,
            Consumer<WaiterOverrideConfiguration.Builder> overrideConfig) {
        return waitUntilSnapshotAvailable(DescribeClusterSnapshotsRequest.builder()
                .applyMutation(describeClusterSnapshotsRequest).build(),
                WaiterOverrideConfiguration.builder().applyMutation(overrideConfig).build());
    }

    /**
     * Create a builder that can be used to configure and create a {@link RedshiftAsyncWaiter}.
     *
     * @return a builder
     */
    static Builder builder() {
        return DefaultRedshiftAsyncWaiter.builder();
    }

    /**
     * Create an instance of {@link RedshiftAsyncWaiter} with the default configuration.
     * <p>
     * <b>A default {@link RedshiftAsyncClient} will be created to poll resources. It is recommended to share a single
     * instance of the waiter created via this method. If it is not desirable to share a waiter instance, invoke
     * {@link #close()} to release the resources once the waiter is not needed.</b>
     *
     * @return an instance of {@link RedshiftAsyncWaiter}
     */
    static RedshiftAsyncWaiter create() {
        return DefaultRedshiftAsyncWaiter.builder().build();
    }

    interface Builder {
        /**
         * Sets a custom {@link ScheduledExecutorService} that will be used to schedule async polling attempts
         * <p>
         * This executorService must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * executorService when the waiter is closed
         *
         * @param executorService
         *        the executorService to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder scheduledExecutorService(ScheduledExecutorService executorService);

        /**
         * Defines overrides to the default SDK waiter configuration that should be used for waiters created from this
         * builder
         *
         * @param overrideConfiguration
         *        the override configuration to set
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration);

        /**
         * This is a convenient method to pass the override configuration without the need to create an instance
         * manually via {@link WaiterOverrideConfiguration#builder()}
         *
         * @param overrideConfiguration
         *        The consumer that will configure the overrideConfiguration
         * @return a reference to this object so that method calls can be chained together.
         * @see #overrideConfiguration(WaiterOverrideConfiguration)
         */
        default Builder overrideConfiguration(Consumer<WaiterOverrideConfiguration.Builder> overrideConfiguration) {
            WaiterOverrideConfiguration.Builder builder = WaiterOverrideConfiguration.builder();
            overrideConfiguration.accept(builder);
            return overrideConfiguration(builder.build());
        }

        /**
         * Sets a custom {@link RedshiftAsyncClient} that will be used to poll the resource
         * <p>
         * This SDK client must be closed by the caller when it is ready to be disposed. The SDK will not close the
         * client when the waiter is closed
         *
         * @param client
         *        the client to send the request
         * @return a reference to this object so that method calls can be chained together.
         */
        Builder client(RedshiftAsyncClient client);

        /**
         * Builds an instance of {@link RedshiftAsyncWaiter} based on the configurations supplied to this builder
         *
         * @return An initialized {@link RedshiftAsyncWaiter}
         */
        RedshiftAsyncWaiter build();
    }
}
