package org.apache.axis2.deployment.util;

import org.apache.axis2.AxisFault;
import org.apache.axis2.Constants;
import org.apache.axis2.deployment.DeploymentException;
import org.apache.axis2.description.AxisMessage;
import org.apache.axis2.description.AxisOperation;
import org.apache.axis2.description.AxisOperationFactory;
import org.apache.axis2.description.AxisService;
import org.apache.axis2.description.Flow;
import org.apache.axis2.description.HandlerDescription;
import org.apache.axis2.description.Parameter;
import org.apache.axis2.description.ParameterImpl;
import org.apache.axis2.engine.AxisConfiguration;
import org.apache.axis2.engine.Handler;
import org.apache.axis2.engine.MessageReceiver;
import org.apache.axis2.wsdl.java2wsdl.SchemaGenerator;
import org.apache.axis2.wsdl.java2wsdl.TypeTable;
import org.apache.wsdl.WSDLConstants;
import org.codehaus.jam.JMethod;

import javax.xml.namespace.QName;
import java.io.File;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.ArrayList;

/*
* Copyright 2004,2005 The Apache Software Foundation.
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*
*/

public class Utils {
    public static void addFlowHandlers(Flow flow, ClassLoader clsLoader) throws AxisFault {
        int count = flow.getHandlerCount();

        for (int j = 0; j < count; j++) {
            HandlerDescription handlermd = flow.getHandler(j);
            Class handlerClass;
            Handler handler;

            handlerClass = getHandlerClass(handlermd.getClassName(), clsLoader);

            try {
                handler = (Handler) handlerClass.newInstance();
                handler.init(handlermd);
                handlermd.setHandler(handler);
            } catch (InstantiationException e) {
                throw new AxisFault(e);
            } catch (IllegalAccessException e) {
                throw new AxisFault(e);
            }
        }
    }

    public static void loadHandler(ClassLoader loader1, HandlerDescription desc)
            throws DeploymentException {
        String handlername = desc.getClassName();
        Handler handler;
        Class handlerClass;

        try {
            handlerClass = Class.forName(handlername, true, loader1);
            handler = (Handler) handlerClass.newInstance();
            handler.init(desc);
            desc.setHandler(handler);
        } catch (ClassNotFoundException e) {
            throw new DeploymentException(e);
        } catch (Exception e) {
            throw new DeploymentException(e);
        }
    }

    public static ClassLoader getClassLoader(ClassLoader parent, String path)
            throws DeploymentException {
        return getClassLoader(parent, new File(path));
    }

    public static ClassLoader getClassLoader(ClassLoader parent, File file)
            throws DeploymentException {
        URLClassLoader classLoader;

        if (file != null) {
            try {
                ArrayList urls = new ArrayList();

                urls.add(file.toURL());

                // lower case directory name
                File libfiles = new File(file, "lib");

                if (libfiles.exists()) {
                    urls.add(libfiles.toURL());

                    File jarfiles[] = libfiles.listFiles();

                    for (int i = 0; i < jarfiles.length; i++) {
                        File jarfile = jarfiles[i];

                        if (jarfile.getName().endsWith(".jar")) {
                            urls.add(jarfile.toURL());
                        }
                    }
                }

                // upper case directory name
                libfiles = new File(file, "Lib");

                if (libfiles.exists()) {
                    urls.add(libfiles.toURL());

                    File jarfiles[] = libfiles.listFiles();

                    for (int i = 0; i < jarfiles.length; i++) {
                        File jarfile = jarfiles[i];

                        if (jarfile.getName().endsWith(".jar")) {
                            urls.add(jarfile.toURL());
                        }
                    }
                }

                URL urllist[] = new URL[urls.size()];

                for (int i = 0; i < urls.size(); i++) {
                    urllist[i] = (URL) urls.get(i);
                }

                classLoader = new URLClassLoader(urllist, parent);

                return classLoader;
            } catch (MalformedURLException e) {
                throw new DeploymentException(e);
            }
        }

        return null;
    }

    private static Class getHandlerClass(String className, ClassLoader loader1) throws AxisFault {
        Class handlerClass;

        try {
            handlerClass = Class.forName(className, true, loader1);
        } catch (ClassNotFoundException e) {
            throw new AxisFault(e.getMessage());
        }

        return handlerClass;
    }

    /**
     * This guy will create a AxisService using java replection
     */
    public static void fillAxisService(AxisService axisService,
                                       AxisConfiguration axisConfig) throws Exception {
        Parameter implInfoParam = axisService.getParameter(Constants.SERVICE_CLASS);
        String serviceClass = (String) implInfoParam.getValue();
        ClassLoader serviceClassLoader = axisService.getClassLoader();
        SchemaGenerator schemaGenerator = new SchemaGenerator(serviceClassLoader,
                serviceClass, null, null);
        axisService.setSchema(schemaGenerator.generateSchema());

        JMethod [] method = schemaGenerator.getMethods();
        TypeTable table = schemaGenerator.getTypeTable();
        PhasesInfo pinfo = axisConfig.getPhasesInfo();

        for (int i = 0; i < method.length; i++) {
            JMethod jmethod = method[i];
            if (!jmethod.isPublic()) {
                // no need to expose , private and protected methods
                continue;
            }
            if (jmethod.getSimpleName().equals("init"))
                continue;
            String opName = jmethod.getSimpleName();
            AxisOperation operation = axisService.getOperation(new QName(opName));
            // if the opeartion there in services.xml then try to set it schema element name
            if (operation != null) {
                AxisMessage inMessage = operation.getMessage(
                        WSDLConstants.MESSAGE_LABEL_IN_VALUE);
                if (inMessage != null) {
                    inMessage.setElementQName(table.getComplexScheamType(jmethod.getSimpleName() +
                            SchemaGenerator.METHOD_REQUEST_WRAPPER));
                }
                if (!jmethod.getReturnType().isVoidType()) {
                    AxisMessage outMessage = operation.getMessage(
                            WSDLConstants.MESSAGE_LABEL_OUT_VALUE);
                    outMessage.setElementQName(table.getQNamefortheType(jmethod.getSimpleName() +
                            SchemaGenerator.METHOD_RESPONSE_WRAPPER));
                }
            } else {
                operation = getAxisOperationforJmethod(jmethod, table);
                MessageReceiver mr = axisService.getMessageReceiver(
                        operation.getMessageExchangePattern());
                if (mr != null) {
                    operation.setMessageReceiver(mr);
                } else {
                    mr = axisConfig.getMessageReceiver(operation.getMessageExchangePattern());
                    operation.setMessageReceiver(mr);
                }
                pinfo.setOperationPhases(operation);
                axisService.addOperation(operation);
            }

        }
    }

    /**
     * To create an AxisService using given service impl class name
     * fisrt generate schema corresponding to the given java class , next for each methods AxisOperation
     * will be created.
     * <p/>
     * Note : Inorder to work this properly RPCMessageReceiver should be availble in the class path
     * otherewise operation can not continue
     *
     * @param implClass
     * @param axisConfig
     * @return return created AxisSrevice
     */
    public static AxisService createService(String implClass,
                                            AxisConfiguration axisConfig) throws AxisFault {
        Parameter parameter = new ParameterImpl(Constants.SERVICE_CLASS, implClass);
        AxisService axisService = new AxisService();
        axisService.setUseDefaultChains(false);
        axisService.addParameter(parameter);

        int index = implClass.lastIndexOf(".");
        String serviceName;
        if (index > 0) {
            serviceName = implClass.substring(index + 1, implClass.length());
        } else {
            serviceName = implClass;
        }

        axisService.setName(serviceName);
        axisService.setClassLoader(axisConfig.getServiceClassLoader());

        ClassLoader serviceClassLoader = axisService.getClassLoader();
        SchemaGenerator schemaGenerator = new SchemaGenerator(serviceClassLoader,
                implClass, null, null);
        try {
            axisService.setSchema(schemaGenerator.generateSchema());
        } catch (Exception e) {
            throw new AxisFault(e);
        }

        JMethod [] method = schemaGenerator.getMethods();
        TypeTable table = schemaGenerator.getTypeTable();

        PhasesInfo pinfo = axisConfig.getPhasesInfo();

        for (int i = 0; i < method.length; i++) {
            JMethod jmethod = method[i];
            if (!jmethod.isPublic()) {
                // no need to expose , private and protected methods
                continue;
            }
            AxisOperation operation = getAxisOperationforJmethod(jmethod, table);

            // loading message recivers
            try {
                Class clazz = Class.forName("org.apache.axis2.rpc.receivers.RPCMessageReceiver");
                MessageReceiver messageReceiver = (MessageReceiver) clazz.newInstance();
                operation.setMessageReceiver(messageReceiver);
            } catch (ClassNotFoundException e) {
                throw new AxisFault("ClassNotFoundException occurd during message receiver loading"
                        + e.getMessage());
            } catch (IllegalAccessException e) {
                throw new AxisFault("IllegalAccessException occurd during message receiver loading"
                        + e.getMessage());
            } catch (InstantiationException e) {
                throw new AxisFault("InstantiationException occurd during message receiver loading"
                        + e.getMessage());
            }

            pinfo.setOperationPhases(operation);
            axisService.addOperation(operation);
        }
        return axisService;

    }

    private static AxisOperation getAxisOperationforJmethod(JMethod jmethod,
                                                            TypeTable table) throws AxisFault {
        AxisOperation operation;
        String opName = jmethod.getSimpleName();
        if (jmethod.getReturnType().isVoidType()) {
            operation = AxisOperationFactory.getAxisOperation(WSDLConstants.MEP_CONSTANT_IN_ONLY);
        } else {
            operation = AxisOperationFactory.getAxisOperation(WSDLConstants.MEP_CONSTANT_IN_OUT);
            AxisMessage outMessage = operation.getMessage(
                    WSDLConstants.MESSAGE_LABEL_OUT_VALUE);
            outMessage.setElementQName(table.getQNamefortheType(jmethod.getSimpleName() +
                    SchemaGenerator.METHOD_RESPONSE_WRAPPER));
        }

        operation.setName(new QName(opName));
        AxisMessage inMessage = operation.getMessage(WSDLConstants.MESSAGE_LABEL_IN_VALUE);
        if (inMessage != null) {
            inMessage.setElementQName(table.getComplexScheamType(jmethod.getSimpleName() +
                    SchemaGenerator.METHOD_REQUEST_WRAPPER));
        }
        return operation;
    }
}