package cat.inspiracio.orange;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;

import org.apache.maven.plugin.logging.Log;
import org.w3c.dom.Document;
import org.w3c.dom.DocumentType;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import cat.inspiracio.html.DocumentWriter;

/** Transforms a html document into a java class that renders it. 
 * 
 * Has state: not threadsafe. */
class Programmer extends DocumentWriter{

	// state ----------------------------------------------------
	
	private Log log;
	
	/** Package name of the class being written.
	 * Corresponds to file path. 
	 * Like "cat.inspiracio.orange.webapp.cacti" 
	 * for /cactus/cactus.html. */
	private String packageName;
	
	/** Class name of the class being written. 
	 * Corresponds to file name.
	 * Like "cactus" for /cacti/cactus.html. */
	private String className;
	
	/** Is the cursor at the start of a new line? */
	private boolean newline=true;
	
	/** How many tabs at the start of the line? */
	private int indentation=0;
	
	/** Class names for imports. Fully-qualified. */
	private Set<String>imports;
	
	// construction --------------------------------------------
	
	Programmer(Writer w){super(w);}

	void setLog(Log l){log=l;}
	void setPackage(String p){packageName=p;}
	void setClass(String c){className=c;}

	// recursers -----------------------------------------------

	@Override protected Programmer open(Document d) throws Exception {
		packageDeclaration();
		importStatements(d);
		startClass();
		constructor();
		startWrite();
		return this;
	}

	private void startClass() throws IOException{
		write("public class ").write(className).writeln(" extends Template{");
		writeln();
		indent();
	}

	private void constructor() throws IOException{
		write("public ").write(className).writeln("(){}");
		writeln();
	}
	
	private void packageDeclaration() throws IOException{
		write("package ").write(packageName).writeln(';');
		writeln();
	}

	private void startWrite() throws IOException{
		writeln("@Override public final void write() throws Exception {");
		indent();
	}
	
	private void importStatements(Document d) throws IOException{
		imports=new TreeSet<String>();
		imports.add("cat.inspiracio.orange.Template");
		imports(d.getDocumentElement());
		for(String i : imports)
			writeln("import " + i + ";");
		writeln();
	}

	/** Recurses over the element to find all class-imports. */
	private void imports(Element e){
		if(e.hasAttribute("data-import")){
			String a=e.getAttribute("data-import");
			String[]is=a.split(",");
			for(String i : is){
				i=i.trim();
				imports.add(i);
			}
			e.removeAttribute("data-import");
		}
		for(Element c : childElements(e))
			imports(c);
	}

	/** Convenience: gets the child elements of an element. */
	private Collection<Element>childElements(Element e){
		NodeList children=e.getChildNodes();
		ArrayList<Element>elements=new ArrayList<Element>();
		for(int i=0; i<children.getLength(); i++){
			Node n=children.item(i);
			if(n instanceof Element)
				elements.add((Element)n);
		}
		return elements;
	}
	
	@Override protected Programmer doctype(DocumentType type)throws IOException{return ww("<!DOCTYPE html>");}

    /** Check for data-substitute. */
    @Override protected Programmer element(Element e) throws Exception{

    	if(e.hasAttribute("data-if"))
    		return dataIf(e);
    	
    	if(e.hasAttribute("data-for"))
    		return dataFor(e);
    	
    	if(e.hasAttribute("data-substitute"))
    		return dataSubstitute(e);
    	
    	if(e.hasAttribute("data-while"))
    		return dataWhile(e);
    	
		super.element(e);
		return this;
    }

    /** Renders an element with data-if attribute.
     * Generates an if-statement. */
    private Programmer dataIf(Element e) throws Exception{
    	String v=e.getAttribute("data-if");
		e.removeAttribute("data-if");
    	log.info("data-if = " + v);
		return writeln("if(" + v + "){").indent().element(e).outdent().writeln('}');
    }
    
    /** Renders an element with data-for attribute.
     * Generates a for-loop. */    
    private Programmer dataFor(Element e) throws Exception{
    	String v=e.getAttribute("data-for");
		e.removeAttribute("data-for");
		log.info("data-for = " + v);
		return writeln("for(" + v + "){").indent().element(e).outdent().writeln('}');
    }
    
    /** Renders an element with data-while attribute.
     * Generates a while-loop. */    
    private Programmer dataWhile(Element e) throws Exception{
    	String v=e.getAttribute("data-while");
		e.removeAttribute("data-while");
		log.info("data-while = " + v);
		return writeln("while(" + v + "){").indent().element(e).outdent().writeln('}');
    }
    
    /** Renders an element with data-substitute attribute. 
     * Generates a call to another class. */
    private Programmer dataSubstitute(Element e) throws IOException{
    	String v=e.getAttribute("data-substitute");
		log.info("data-substitute = " + v);
		return write("substitute(").literal(v).writeln(");");
    }
    
    /** Writes opening tag and the attributes.
	 * If the element has no child nodes, the final ">" of the opening tag
	 * is not written, so that close(e) can write "/>" --- unless the element
	 * is one of the few elements that need a separate closing tag even if they
	 * have no children. */
	@Override protected Programmer open(Element e) throws Exception {
		String tag=e.getTagName();
		boolean b=e.hasChildNodes() || needClosingTag(tag);
		
		if(0==e.getAttributes().getLength()){
			if(b)
				ww("<" + tag + ">");
			else
				ww("<" + tag);
		}
		
		else{
			ww("<" + tag);
			attributes(e);
			if(b)
				ww(">");
		}
		return this;
	}

	/** Writes " key=\"value\". 
	 * 
	 * In value, " is escaped to "&amp;". 
	 * 
	 * If the value is null or empty, writes just the key. 
	 * 
	 * Does no support escaped expressions \${E}.
	 * */
    @Override protected Programmer attribute(String key, String value) throws Exception {
        //no value --- normal case
    	if(empty(value))
            return ww(" " + key);
        
        //value is literally "true" or "false" --- degenerate case
    	if("true".equals(value))
    		return ww(" " + key);//or: key=key
    	
    	if("false".equals(value))
    		return this;

    	//Value has no ${E} --- usual case, must be fast and neat
    	if(-1==value.indexOf("${"))
    		return ww(" " + key + "=" + quote(value));//Escapes " in the value
    	
    	//Whole value is just one ${E} which may be boolean --- a normal case
    	if(isExpression(value)){
    		String e=value.substring(2, value.length()-1);
    		return write("attribute(").literal(key).writeln(", " + e + ");");
    	}

    	//Value is partly literal and contains expressions --- unusual case
    	ww(" " + key + "=\"");	//opens "
    	List<Part>parts=Part.parse(value);
    	for(Part p : parts){
        	if(p.isLiteral())
        		ww(p.getLiteral());
        	else
        		write("write(escape(").write(p.getExpression()).writeln("));");
    	}
    	return ww("\"");		//closes "
    }
    
    /** Is whole value just one expression, like "${karte.getGenus()}"?
     * That's a frequent case. */
    private boolean isExpression(String value){
    	//XXX Too simple. Fails for "${x} and ${y}".
    	return value!=null &&
    			3<value.length() &&
    			value.startsWith("${") &&
    			value.endsWith("}") &&
    			-1==value.indexOf("${", 2);
    }
    
    /** Writes a text node. 
     * 
     * Collapses white space into a single space. 
     * */
	@Override protected Programmer text(String s) throws Exception {
		// Collapse multiple whitespace into single space would be ok.
		//https://www.dotnetperls.com/whitespace-java
		//http://stackoverflow.com/questions/2932392/java-how-to-replace-2-or-more-spaces-with-single-space-in-string-and-delete-lead
		if(whitespace(s))
			return ww(" ");
		
		List<Part>parts=Part.parse(s);
		for(Part p : parts)
			if(p.isLiteral())
				ww(escape(p.getLiteral()));
			else
				writeln("write(" + p.getExpression() + ");");//need some toString(Object)?
		return this;
	}
	
	private boolean whitespace(String s){return s.trim().isEmpty();}

	/** escape for html: < & */
	private String escape(String s){
		if(s==null)
			return null;
		return s.replace("&", "&amp;").replace("<", "&lt;");
	}
	
	/** Writes an element's closing tag.
     * If the element has no children, only writes "/>" ---
     * unless the element is one of the few elements that need a separate
     * closing tag even if they have no child elements. */
	@Override protected Programmer close(Element e) throws IOException {
		String tag=e.getTagName();
		boolean b=e.hasChildNodes() || needClosingTag(tag);
		if(b)
			return ww("</" + tag + ">");
		else
			return ww("/>");//opt: for void elements like <meta> could output ">" rather than "/>"
	}

	/** Ends the write()-methods, inserts any declaration, ends the class. */
	@Override protected Void close(Document d)throws Exception{
		endWrite();
		declarations();
		endClass();
		return null;
	}
	
	private Programmer declarations() throws IOException{
		if(declarations!=null)
			for(String d : declarations)
				writeln(d);
		return this;
	}

	private Programmer endWrite() throws IOException{
		return outdent().writeln().writeln('}').writeln();
	}

	private Programmer endClass() throws IOException{
		return outdent().writeln('}').writeln();
	}
	
	private Set<String> declarations=new HashSet<>();
	
	/** Writes a script element.
     * 
     * Script elements are special:
     * They have no child elements except for text, 
     * and the text should not be escaped. In that way,
     * the javascript program in there can have < > &.
     * 
     * @param element Must be script and must have no children
     * other than text.
     */
	@Override protected void script(Element element) throws Exception{
		if(isDeclaration(element)){
			String source=element.getTextContent();
			declarations.add(source);
			return;
		}
		
		if(isServerScript(element)){
			//Design decision:
			//We don't enclose the script in a new block: { }
			//so that it can initialise variables that are used further down.
			String source=element.getTextContent();
			if(!empty(source))
				writeln(source);//no escaping at all
			return;
		}
		
        open(element);
        String s=element.getTextContent();//correct if precondition holds
        if(!empty(s))
        	ww(s);//no escaping for html
        close(element);
    }
	
	private boolean isServerScript(Element e){
		String type=e.getAttribute("type");
		return "server/java".equals(type);
	}

	private boolean isDeclaration(Element e){return e.hasAttribute("data-declare");}

	/** No output */
    @Override protected Programmer cdata(String s) throws Exception{return this;}

	/** No output */
	@Override protected Programmer comment(String s) throws Exception{return this;}
	
    /** Quotes a value so that it can be an attribute's value.
     * Escapes " by &quot; and encloses in ". */
    protected String quote(String value){
    	//if(okUnquoted(value))return value;

        //optimise usual case
        if(contains(value, '"'))
            value=value.replaceAll("\"", "&quot;");
        return '"' + value + '"';
    }

    /** Is it ok to use this attribute value without quotes in html5?
     * 
     * https://www.w3.org/TR/html-markup/syntax.html#syntax-attributes
     * Must not contain any literal space characters.
     * Must not contain any """, "'", "=", ">", "<", or "`", characters.
     * Must not be the empty string. 
     * 
     * I'm not using it because it's a bit dangerous. Example:
     * <link type="text/css"/> becomes <link type=text/css/>. 
     * After unquoted attribute:
     * -another attribute, fine it starts with space
     * -close element, */
    @SuppressWarnings("unused")
	private boolean okUnquoted(String s){
    	if(s.isEmpty())
    		return false;
    	for(int i=0; i<s.length(); i++){
    		char c=s.charAt(i);
    		if(Character.isWhitespace(c))
    			return false;
    		switch(c){
    		case '"': 
    		case '\'':
    		case '=':
    		case '>':
    		case '<':
    		case '`':
    			return false;
    		}
    	}
    	return true;
    }

    /** Does the string contain this character? */
    private boolean contains(String value, char c){return 0<=value.indexOf(c);}

	// helpers -------------------------------------------------
	
	private Programmer indent() throws IOException{
		flush();
		indentation++;
		return this;
	}
	
	private Programmer outdent() throws IOException{
		flush();
		indentation--;
		return this;
	}
	
	/** If at start of new line, write the indentation. */
	private void dent() throws IOException{
		if(newline)
			for(int i=0; i<indentation; i++)
				super.write('\t');
		newline=false;
	}

	@Override public Programmer write(char c) throws IOException{
		flush();
		dent();
		super.write(c);
		return this;
	}

	@Override public Programmer write(String s) throws IOException{
		if(empty(s))
			return this;
		flush();
		dent();
		super.write(s);
		return this;
	}

	protected Programmer writeln() throws IOException{
		flush();
		super.writeln();
		newline=true;
		return this;
	}

	/** Writes a single character. */
	@Override public Programmer writeln(char c) throws IOException{
		flush();
		dent();
		super.writeln(c);
		return this;
	}
	
	@Override public Programmer writeln(String s) throws IOException{
		flush();
		dent();
		super.writeln(s);
		return this;
	}
	
	/** accumulated string that will got into write(s). */
	private StringBuilder builder=new StringBuilder();
	
	/** Accumulate some more string for next write() operation. */
	private Programmer accumulate(String s){
		builder.append(s);
		return this;
	}
	
	/** Flushes accumulated string to write(s). */
	private Programmer flush() throws IOException{
		if(builder.length()==0)
			return this;
		String s=builder.toString();
		builder.setLength(0);
		return write("write(").literal(s).writeln(");");
	}
	
	/** Generates code to write a string as-is.
	 * write("write("s");")
	 * @throws IOException */
	private Programmer ww(String s) throws IOException{
		if(empty(s))
			return this;
		return accumulate(s);
	}

	/** Write s as a Java literal string. 
	 * @throws IOException */
	private Programmer literal(String s) throws IOException{
		write('"');
		for(int i=0; i<s.length(); i++){
			char c=s.charAt(i);
			switch(c){
			case '\\': write("\\\\");break;
			case '\"': write("\\\"");break;
			case '\n': write("\\n");break;
			case '\r': write("\\r");break;
			default: write(c);break;
			}
		}
		return write('"');
	}

	private boolean empty(String value){return value==null || 0==value.length();}
}
