/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.git;

import com.google.common.collect.Sets;
import com.google.gerrit.common.data.GarbageCollectionResult;
import com.google.gerrit.reviewdb.client.Project;
import com.google.gerrit.server.git.GarbageCollectionQueue;
import com.google.gerrit.server.git.GitRepositoryManager;
import com.google.inject.Inject;
import java.io.PrintWriter;
import java.util.List;
import java.util.Properties;
import java.util.Set;
import org.eclipse.jgit.api.GarbageCollectCommand;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.errors.RepositoryNotFoundException;
import org.eclipse.jgit.lib.Config;
import org.eclipse.jgit.lib.NullProgressMonitor;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.lib.StoredConfig;
import org.eclipse.jgit.lib.TextProgressMonitor;
import org.eclipse.jgit.storage.pack.PackConfig;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GarbageCollection {
    private static final Logger log = LoggerFactory.getLogger(GarbageCollection.class);
    public static final String LOG_NAME = "gc_log";
    private static final Logger gcLog = LoggerFactory.getLogger("gc_log");
    private final GitRepositoryManager repoManager;
    private final GarbageCollectionQueue gcQueue;

    @Inject
    GarbageCollection(GitRepositoryManager repoManager, GarbageCollectionQueue gcQueue) {
        this.repoManager = repoManager;
        this.gcQueue = gcQueue;
    }

    public GarbageCollectionResult run(List<Project.NameKey> projectNames) {
        return this.run(projectNames, null);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public GarbageCollectionResult run(List<Project.NameKey> projectNames, PrintWriter writer) {
        GarbageCollectionResult result = new GarbageCollectionResult();
        Set<Project.NameKey> projectsToGc = this.gcQueue.addAll(projectNames);
        for (Project.NameKey projectName : Sets.difference(Sets.newHashSet(projectNames), projectsToGc)) {
            result.addError(new GarbageCollectionResult.Error(GarbageCollectionResult.Error.Type.GC_ALREADY_SCHEDULED, projectName));
        }
        for (Project.NameKey p : projectsToGc) {
            Repository repo = null;
            try {
                repo = this.repoManager.openRepository(p);
                GarbageCollection.logGcConfiguration(p, repo);
                GarbageCollection.print(writer, "collecting garbage for \"" + p + "\":\n");
                GarbageCollectCommand gc = Git.wrap(repo).gc();
                GarbageCollection.logGcInfo(p, "before:", gc.getStatistics());
                gc.setProgressMonitor(writer != null ? new TextProgressMonitor(writer) : NullProgressMonitor.INSTANCE);
                Properties statistics = gc.call();
                GarbageCollection.logGcInfo(p, "after: ", statistics);
                GarbageCollection.print(writer, "done.\n\n");
            }
            catch (RepositoryNotFoundException e) {
                GarbageCollection.logGcError(writer, p, e);
                result.addError(new GarbageCollectionResult.Error(GarbageCollectionResult.Error.Type.REPOSITORY_NOT_FOUND, p));
            }
            catch (Exception e) {
                GarbageCollection.logGcError(writer, p, e);
                result.addError(new GarbageCollectionResult.Error(GarbageCollectionResult.Error.Type.GC_FAILED, p));
            }
            finally {
                if (repo != null) {
                    repo.close();
                }
                this.gcQueue.gcFinished(p);
            }
        }
        return result;
    }

    private static void logGcInfo(Project.NameKey projectName, String msg) {
        GarbageCollection.logGcInfo(projectName, msg, null);
    }

    private static void logGcInfo(Project.NameKey projectName, String msg, Properties statistics) {
        StringBuilder b = new StringBuilder();
        b.append("[").append(projectName.get()).append("] ");
        b.append(msg);
        if (statistics != null) {
            b.append(" ");
            String s = statistics.toString();
            if (s.startsWith("{") && s.endsWith("}")) {
                s = s.substring(1, s.length() - 1);
            }
            b.append(s);
        }
        gcLog.info(b.toString());
    }

    private static void logGcConfiguration(Project.NameKey projectName, Repository repo) {
        StringBuilder b = new StringBuilder();
        StoredConfig cfg = repo.getConfig();
        b.append(GarbageCollection.formatConfigValues(cfg, "gc", null));
        for (String subsection : cfg.getSubsections("gc")) {
            b.append(GarbageCollection.formatConfigValues(cfg, "gc", subsection));
        }
        if (b.length() == 0) {
            b.append("no set");
        }
        GarbageCollection.logGcInfo(projectName, "gc config: " + b.toString());
        GarbageCollection.logGcInfo(projectName, "pack config: " + new PackConfig(repo).toString());
    }

    private static String formatConfigValues(Config config, String section, String subsection) {
        StringBuilder b = new StringBuilder();
        Set<String> names = config.getNames(section, subsection);
        for (String name : names) {
            String value = config.getString(section, subsection, name);
            b.append(section);
            if (subsection != null) {
                b.append(".").append(subsection);
            }
            b.append(".");
            b.append(name).append("=").append(value);
            b.append("; ");
        }
        return b.toString();
    }

    private static void logGcError(PrintWriter writer, Project.NameKey projectName, Exception e) {
        GarbageCollection.print(writer, "failed.\n\n");
        StringBuilder b = new StringBuilder();
        b.append("[").append(projectName.get()).append("]");
        gcLog.error(b.toString(), e);
        log.error(b.toString(), e);
    }

    private static void print(PrintWriter writer, String message) {
        if (writer != null) {
            writer.print(message);
        }
    }

    public static interface Factory {
        public GarbageCollection create();
    }
}

