/*
 * Decompiled with CFR 0.152.
 */
package com.google.gerrit.server.mail;

import com.google.common.primitives.Ints;
import com.google.gerrit.common.Version;
import com.google.gerrit.common.errors.EmailException;
import com.google.gerrit.server.config.ConfigUtil;
import com.google.gerrit.server.config.GerritServerConfig;
import com.google.gerrit.server.mail.Address;
import com.google.gerrit.server.mail.EmailHeader;
import com.google.gerrit.server.mail.EmailSender;
import com.google.gerrit.server.util.TimeUtil;
import com.google.inject.AbstractModule;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.Writer;
import java.text.SimpleDateFormat;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import org.apache.commons.net.smtp.AuthSMTPClient;
import org.apache.commons.net.smtp.SMTPClient;
import org.apache.commons.net.smtp.SMTPReply;
import org.eclipse.jgit.lib.Config;

@Singleton
public class SmtpEmailSender
implements EmailSender {
    private static final int DEFAULT_CONNECT_TIMEOUT = 0;
    private final boolean enabled;
    private final int connectTimeout;
    private String smtpHost;
    private int smtpPort;
    private String smtpUser;
    private String smtpPass;
    private Encryption smtpEncryption;
    private boolean sslVerify;
    private Set<String> allowrcpt;
    private String importance;
    private int expiryDays;

    @Inject
    SmtpEmailSender(@GerritServerConfig Config cfg) {
        int defaultPort;
        this.enabled = cfg.getBoolean("sendemail", null, "enable", true);
        this.connectTimeout = Ints.checkedCast(ConfigUtil.getTimeUnit(cfg, "sendemail", null, "connectTimeout", 0L, TimeUnit.MILLISECONDS));
        this.smtpHost = cfg.getString("sendemail", null, "smtpserver");
        if (this.smtpHost == null) {
            this.smtpHost = "127.0.0.1";
        }
        this.smtpEncryption = ConfigUtil.getEnum(cfg, "sendemail", null, "smtpencryption", Encryption.NONE);
        this.sslVerify = cfg.getBoolean("sendemail", null, "sslverify", true);
        switch (this.smtpEncryption) {
            case SSL: {
                defaultPort = 465;
                break;
            }
            default: {
                defaultPort = 25;
            }
        }
        this.smtpPort = cfg.getInt("sendemail", null, "smtpserverport", defaultPort);
        this.smtpUser = cfg.getString("sendemail", null, "smtpuser");
        this.smtpPass = cfg.getString("sendemail", null, "smtppass");
        HashSet<String> rcpt = new HashSet<String>();
        for (String addr : cfg.getStringList("sendemail", null, "allowrcpt")) {
            rcpt.add(addr);
        }
        this.allowrcpt = Collections.unmodifiableSet(rcpt);
        this.importance = cfg.getString("sendemail", null, "importance");
        this.expiryDays = cfg.getInt("sendemail", null, "expiryDays", 0);
    }

    @Override
    public boolean isEnabled() {
        return this.enabled;
    }

    @Override
    public boolean canEmail(String address) {
        if (!this.isEnabled()) {
            return false;
        }
        if (this.allowrcpt.isEmpty()) {
            return true;
        }
        if (this.allowrcpt.contains(address)) {
            return true;
        }
        String domain = address.substring(address.lastIndexOf(64) + 1);
        return this.allowrcpt.contains(domain) || this.allowrcpt.contains("@" + domain);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void send(Address from, Collection<Address> rcpt, Map<String, EmailHeader> callerHeaders, String body) throws EmailException {
        if (!this.isEnabled()) {
            throw new EmailException("Sending email is disabled");
        }
        LinkedHashMap<String, EmailHeader> hdrs = new LinkedHashMap<String, EmailHeader>(callerHeaders);
        this.setMissingHeader(hdrs, "MIME-Version", "1.0");
        this.setMissingHeader(hdrs, "Content-Type", "text/plain; charset=UTF-8");
        this.setMissingHeader(hdrs, "Content-Transfer-Encoding", "8bit");
        this.setMissingHeader(hdrs, "Content-Disposition", "inline");
        this.setMissingHeader(hdrs, "User-Agent", "Gerrit/" + Version.getVersion());
        if (this.importance != null) {
            this.setMissingHeader(hdrs, "Importance", this.importance);
        }
        if (this.expiryDays > 0) {
            Date expiry = new Date(TimeUtil.nowMs() + (long)(this.expiryDays * 24 * 60 * 60) * 1000L);
            this.setMissingHeader(hdrs, "Expiry-Date", new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss Z").format(expiry));
        }
        StringBuffer rejected = new StringBuffer();
        try {
            SMTPClient client = this.open();
            try {
                if (!client.setSender(from.email)) {
                    throw new EmailException("Server " + this.smtpHost + " rejected from address " + from.email);
                }
                for (Address addr : rcpt) {
                    if (client.addRecipient(addr.email)) continue;
                    String error = client.getReplyString();
                    rejected.append("Server ").append(this.smtpHost).append(" rejected recipient ").append(addr).append(": ").append(error);
                }
                Writer w = client.sendMessageData();
                if (w == null) {
                    throw new EmailException(rejected + "Server " + this.smtpHost + " rejected DATA command: " + client.getReplyString());
                }
                w = new BufferedWriter(w);
                for (Map.Entry h : hdrs.entrySet()) {
                    if (((EmailHeader)h.getValue()).isEmpty()) continue;
                    w.write((String)h.getKey());
                    w.write(": ");
                    ((EmailHeader)h.getValue()).write(w);
                    w.write("\r\n");
                }
                w.write("\r\n");
                w.write(body);
                w.flush();
                w.close();
                if (!client.completePendingCommand()) {
                    throw new EmailException("Server " + this.smtpHost + " rejected message body: " + client.getReplyString());
                }
                client.logout();
                if (rejected.length() > 0) {
                    throw new EmailException(rejected.toString());
                }
            }
            finally {
                client.disconnect();
            }
        }
        catch (IOException e) {
            throw new EmailException("Cannot send outgoing email", e);
        }
    }

    private void setMissingHeader(Map<String, EmailHeader> hdrs, String name, String value) {
        if (!hdrs.containsKey(name) || hdrs.get(name).isEmpty()) {
            hdrs.put(name, new EmailHeader.String(value));
        }
    }

    private SMTPClient open() throws EmailException {
        AuthSMTPClient client = new AuthSMTPClient("UTF-8");
        if (this.smtpEncryption == Encryption.SSL) {
            client.enableSSL(this.sslVerify);
        }
        try {
            client.setConnectTimeout(this.connectTimeout);
            client.connect(this.smtpHost, this.smtpPort);
            if (!SMTPReply.isPositiveCompletion(client.getReplyCode())) {
                throw new EmailException("SMTP server rejected connection");
            }
            if (!client.login()) {
                String e = client.getReplyString();
                throw new EmailException("SMTP server rejected HELO/EHLO greeting: " + e);
            }
            if (this.smtpEncryption == Encryption.TLS) {
                if (!client.startTLS(this.smtpHost, this.smtpPort, this.sslVerify)) {
                    throw new EmailException("SMTP server does not support TLS");
                }
                if (!client.login()) {
                    String e = client.getReplyString();
                    throw new EmailException("SMTP server rejected login: " + e);
                }
            }
            if (this.smtpUser != null && !client.auth(this.smtpUser, this.smtpPass)) {
                String e = client.getReplyString();
                throw new EmailException("SMTP server rejected auth: " + e);
            }
        }
        catch (IOException e) {
            if (client.isConnected()) {
                try {
                    client.disconnect();
                }
                catch (IOException e2) {
                    // empty catch block
                }
            }
            throw new EmailException(e.getMessage(), e);
        }
        catch (EmailException e) {
            if (client.isConnected()) {
                try {
                    client.disconnect();
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            throw e;
        }
        return client;
    }

    public static enum Encryption {
        NONE,
        SSL,
        TLS;

    }

    public static class Module
    extends AbstractModule {
        @Override
        protected void configure() {
            this.bind(EmailSender.class).to(SmtpEmailSender.class);
        }
    }
}

