/*
 *    GraphProcessor.java
 *
 *    Copyright 2002, Bill2, Inc. All rights reserved.
 *
 *    This program contains the confidential trade secret
 *    information of Bill2, Inc.  Use, disclosure, or
 *    copying without written consent is strictly prohibited.
 *
 *    @author topping
 *    @version $Revision$
 */
package org.dentaku.gentaku.cartridge.event.graph;

import org.omg.uml.behavioralelements.statemachines.StateVertex;
import org.omg.uml.behavioralelements.statemachines.Transition;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

public class GraphProcessor {
    private GraphResults dfsResults = null;
    private GraphResults sccResults = null;
    private ArrayList topological = null;

    public GraphResults scc(Collection verticies, GraphIterator nav) {
        dfsResults = dfs(verticies, nav);
        topological = topologicalSort(dfsResults);
        sccResults = dfs(topological, new ReverseGraphIterator(nav));
        return sccResults;
    }

    public GraphResults dfs(Collection verticies, GraphIterator nav) {
        GraphResults results = new GraphResults();
        for (Iterator it = verticies.iterator(); it.hasNext();) {
            Object vertex = (Object) it.next();
            if (!results.getDiscoveredVertex().containsKey(vertex)) {
                Collection roots = results.getRoots();
                if (roots != null) {
                    roots.add(vertex);
                }
                dfsVisit(results, vertex, nav);
            }
        }
        return results;
    }

    private ArrayList topologicalSort(GraphResults resultSet1) {
        Object[] finished = resultSet1.getFinishedVertex().toArray();
        ArrayList topological = new ArrayList(finished.length);
        for (int i = finished.length - 1; i >= 0; i--) {
            topological.add(finished[i]);
        }
        return topological;
    }

    private void dfsVisit(GraphResults results, Object vertex, GraphIterator nav) {
        results.getDiscoveredVertex().put(vertex, new Integer(results.getNextTime()));

        Collection nextEdges = nav.nextEdges(vertex);
        for (Iterator i = nextEdges.iterator(); i.hasNext();) {
            Object nextEdge = i.next();
            Object nextVertex = nav.getTarget(nextEdge);
            System.out.println("dfsVisit: thisVertex="+((StateVertex)vertex).getName()+"; nextEdge="+getEdgeType(results, nextEdge)+"; nextVertex="+((StateVertex)nextVertex).getName());
            classifyEdge(results, nav, nextEdge);
            if (!results.getDiscoveredVertex().containsKey(nextVertex)) {
                dfsVisit(results, nextVertex, nav);
            }
        }
        results.getFinishedVertex().add(vertex);
    }

    private void classifyEdge(GraphResults results, GraphIterator nav, Object edge) {
        Object source = nav.getSource(edge);
        Object target = nav.getTarget(edge);
        if (results.getFinishedVertex().contains(target)) {
            // black, forward or cross edge, distinguish by time
            int sourceTime = ((Integer) results.getDiscoveredVertex().get(source)).intValue();
            int targetTime = ((Integer) results.getDiscoveredVertex().get(target)).intValue();
            if (sourceTime < targetTime) {
                System.out.println("classifyEdge - FORWARD: source="+((StateVertex)source).getName()+"; dest="+((StateVertex)target).getName());
                results.getForwardEdges().add(edge);
            } else {
                System.out.println("classifyEdge - CROSS: source="+((StateVertex)source).getName()+"; dest="+((StateVertex)target).getName());
                results.getCrossEdges().add(edge);
            }
        } else {
            if (results.getDiscoveredVertex().containsKey(target)) {
                // gray, back edge
                System.out.println("classifyEdge - BACK: source="+((StateVertex)source).getName()+"; dest="+((StateVertex)target).getName());
                results.getBackEdges().add(edge);
            } else {
                // white, tree edge
                System.out.println("classifyEdge - TREE: source="+((StateVertex)source).getName()+"; dest="+((StateVertex)target).getName());
                results.getTreeEdges().add(edge);
            }
        }
    }

    private String getEdgeType(GraphResults results, Object edge) {
        if (results.getBackEdges().contains(edge)) {
            return "back";
        } else if (results.getForwardEdges().contains(edge)) {
            return "forward";
        } else if (results.getCrossEdges().contains(edge)) {
            return "cross";
        } else if (results.getTreeEdges().contains(edge)) {
            return "tree";
        }
        return "unknown";
    }

    public void validate(Collection verticies, GraphIterator nav) throws GraphException {
        // do the basics
        dfsResults = dfs(verticies, nav);
        topological = topologicalSort(dfsResults);
        sccResults = dfs(topological, new ReverseGraphIterator(nav));

        // TEST 1: nodes are have only one path
        StateVertex lastVertex = null;
        Iterator it = topological.iterator();
        lastVertex = (StateVertex) it.next();
        while (it.hasNext()) {
            StateVertex thisVertex = (StateVertex) it.next();
            Collection outgoing = lastVertex.getOutgoing();
            if (!(outgoing.size() == 1
                    && ((Transition)lastVertex.getOutgoing().iterator().next()).getTarget() == thisVertex)) {
                throw new GraphException("graph is not a single path");
            }
            lastVertex = thisVertex;
        }
    }

    public GraphResults getDfsResults() {
        return dfsResults;
    }

    public GraphResults getSccResults() {
        return sccResults;
    }

    public ArrayList getTopological() {
        return topological;
    }
}

