// $Id: AnnotationConfiguration.java,v 1.31 2005/07/22 00:42:33 epbernard Exp $
package org.hibernate.cfg;

import java.io.File;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.HashSet;
import javax.persistence.EmbeddableSuperclass;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Attribute;
import org.dom4j.Element;
import org.hibernate.MappingException;
import org.hibernate.AnnotationException;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.Join;
import org.hibernate.mapping.Table;
import org.hibernate.mapping.UniqueKey;
import org.hibernate.util.ReflectHelper;

/**
 * Add JSR 175 configuration capability.
 * For now, only programmatic configuration is available. 
 * 
 * @author Emmanuel Bernard
 */
public class AnnotationConfiguration extends Configuration {
	private static Log log = LogFactory.getLog(AnnotationConfiguration.class);

	private Map namedGenerators;
	private Map<String, Map<String, Join>> joins;
	private Map<Class, AnnotatedClassType> classTypes;
	private Map<String,Properties> generatorTables;
	private Map<Table, List<String[]>> tableUniqueConstraints;
	private Map<String, String> mappedByResolver;
	private Map<String, String> propertyRefResolver;
	private List<Class> annotatedClasses;
	private List<CacheHolder> caches;

    public AnnotationConfiguration() {
		super();
	}

	public AnnotationConfiguration(SettingsFactory sf) {
		super(sf);
	}

	/**
	 * Add a list of classes to configuration.  This method will automatically sort classes based on any
	 * inheritance hierarchy
	 *
	 * @param classes
	 * @return return annotation configuration
	 * @deprecated use #addAnnotatedClass(Class)
	 */
	public AnnotationConfiguration addAnnotatedClasses(List<Class> classes) {
		List<Class> newList;
		newList = orderHierarchy(classes);
		for (Class clazz : newList) {
			addAnnotatedClass(clazz);
		}
		return this;
	}

	protected List<Class> orderHierarchy(List<Class> original) {
		List<Class> copy = new ArrayList<Class>(original);
		List<Class> newList = new ArrayList<Class>();
		while (copy.size() > 0)	{
			Class clazz = copy.get(0);
			orderHierarchy(copy, newList, original, clazz);
		}
		return newList;
	}

	private static void orderHierarchy(List<Class> copy, List<Class> newList, List<Class> original, Class clazz)	{
		if ( Object.class.equals(clazz) ) return;
		//process superclass first
		orderHierarchy( copy, newList, original, clazz.getSuperclass() );
		if ( original.contains(clazz) ) {
			if ( !newList.contains(clazz) )	{
				newList.add(clazz);
			}
			copy.remove(clazz);
		}
	}

	/**
	 * Read a mapping from the class annotation metadata (JSR 175).
	 * 
	 * @param persistentClass the mapped class
	 * 
	 * @return the configuration object
	 */
	public AnnotationConfiguration addAnnotatedClass(Class persistentClass) throws MappingException {
		try {
			annotatedClasses.add(persistentClass);
			return this;
		}
		catch (MappingException me) {
			log.error("Could not compile the mapping annotations", me);
			throw me;
		}
	}

	/**
	 * Read package level metadata
	 * 
	 * @param packageName java package name
	 * @return the configuration object
	 */
	public AnnotationConfiguration addPackage(String packageName) throws MappingException {
		log.info("Mapping package " + packageName);
		try {
			AnnotationBinder.bindPackage( packageName, createExtendedMappings() );
			return this;
		}
		catch (MappingException me) {
			log.error("Could not compile the mapping annotations", me);
			throw me;
		}
	}

	public ExtendedMappings createExtendedMappings() {
		return new ExtendedMappings(
				classes,
				collections,
				tables,
				namedQueries,
				namedSqlQueries,
				sqlResultSetMappings,
				imports,
				secondPasses,
				propertyReferences,
				namingStrategy,
				typeDefs,
				filterDefinitions,
				namedGenerators,
				joins,
				classTypes,
				extendsQueue,
				generatorTables,
				tableUniqueConstraints,
				mappedByResolver,
				propertyRefResolver
		);
	}

	@Override
	public void setCacheConcurrencyStrategy(String clazz, String concurrencyStrategy, String region) throws MappingException {
		caches.add( new CacheHolder(clazz, concurrencyStrategy, region, true) );
	}

	@Override
	public void setCollectionCacheConcurrencyStrategy(String collectionRole, String concurrencyStrategy, String region)
			throws MappingException {
		caches.add( new CacheHolder(collectionRole, concurrencyStrategy, region, false) );
	}

	protected void reset() {
		super.reset();
		namedGenerators = new HashMap();
		joins = new HashMap<String, Map<String, Join>>();
		classTypes = new HashMap<Class, AnnotatedClassType>();
		generatorTables = new HashMap<String, Properties>();
		tableUniqueConstraints = new HashMap<Table, List<String[]>>();
		mappedByResolver = new HashMap<String, String>();
		propertyRefResolver = new HashMap<String, String>();
		annotatedClasses = new ArrayList<Class>();
		caches = new ArrayList<CacheHolder>();
	}

	protected void secondPassCompile() throws MappingException {
		log.debug("Executing first pass of annotated classes");
		//bind classes in the correct order calculating some inheritance state
		List<Class> orderedClasses = orderHierarchy(annotatedClasses);
		orderedClasses = addImplicitEmbeddedSuperClasses(orderedClasses);
		Map<Class, InheritanceState> inheritanceStatePerClass = AnnotationBinder.buildInheritanceStates(orderedClasses);
		for (Class clazz : orderedClasses) {
			InheritanceState state = inheritanceStatePerClass.get( clazz);
			if (state.hasParents == false && state.hasEmbeddedSuperclass == false && state.isEmbeddableSuperclass ==false) {
				//might have an implicit superclass

			}
		}
		for (Class clazz : orderedClasses) {
			//todo use the same extended mapping
			AnnotationBinder.bindClass( clazz, inheritanceStatePerClass, createExtendedMappings() );
		}
        annotatedClasses.clear();
		int cacheNbr = caches.size();
		for (int index = 0 ; index < cacheNbr ; index++) {
			CacheHolder cacheHolder = caches.get(index);
			if (cacheHolder.isClass) {
				super.setCacheConcurrencyStrategy(cacheHolder.role, cacheHolder.usage, cacheHolder.region);
			}
			else {
				super.setCollectionCacheConcurrencyStrategy(cacheHolder.role, cacheHolder.usage, cacheHolder.region);
			}
		}
		caches.clear();

		log.debug("processing manytoone fk mappings");
		Iterator iter = secondPasses.iterator();
		while ( iter.hasNext() ) {
			HbmBinder.SecondPass sp = (HbmBinder.SecondPass) iter.next();
			//do the second pass of fk before the others and remove them
			if (sp instanceof FkSecondPass) {
				sp.doSecondPass(classes, Collections.EMPTY_MAP); // TODO: align meta-attributes with normal bind...
				iter.remove();
			}
		}
		super.secondPassCompile();
		Iterator tables = (Iterator<Map.Entry<Table, List<String[]>>>) tableUniqueConstraints.entrySet().iterator();
		Table table;
		Map.Entry entry;
		String keyName;
		int uniqueIndexPerTable;
		while ( tables.hasNext() ) {
			entry = (Map.Entry) tables.next();
			table = (Table) entry.getKey();
			List<String[]> uniqueConstraints = (List<String[]>) entry.getValue();
			uniqueIndexPerTable = 0;
			for (String[] columnNames : uniqueConstraints) {
				keyName = "key" + uniqueIndexPerTable++;
				buildUniqueKeyFromColumnNames(columnNames, table, keyName);
			}
		}
	}

	private List<Class> addImplicitEmbeddedSuperClasses(List<Class> orderedClasses) {
		List<Class> newOrderedClasses = new ArrayList<Class>(orderedClasses);
		for (Class clazz : orderedClasses) {
			Class superClazz = clazz.getSuperclass();
			if ( ! newOrderedClasses.contains( superClazz ) ) {
				//maybe an implicit embeddable superclass
				addEmbeddedSuperclasses(clazz, newOrderedClasses);
			}
		}
		return newOrderedClasses;
	}

	private void addEmbeddedSuperclasses(Class clazz, List<Class> newOrderedClasses) {
		Class superClass = clazz.getSuperclass();
		boolean hasToStop = false;
		while( ! hasToStop ) {
			if (superClass.isAnnotationPresent( EmbeddableSuperclass.class) ) {
				newOrderedClasses.add( 0, superClass );
				superClass = superClass.getSuperclass();
			}
			else {
				hasToStop = true;
			}
		}
	}

	private void buildUniqueKeyFromColumnNames(String[] columnNames, Table table, String keyName) {
		UniqueKey uc;
		int size = columnNames.length;
		Column[] columns = new Column[size];
		Set<Column> unbound = new HashSet<Column>();
		for (int index = 0 ;  index < size; index++) {
			columns[index] = new Column( columnNames[index] );
			unbound.add( columns[index] );
			//column equals and hashcode is based on column name
		}
		for (Column column : columns) {
			if ( table.containsColumn( column ) ) {
				uc = table.getOrCreateUniqueKey( keyName );
				uc.addColumn( table.getColumn( column ) );
				unbound.remove( column );
			}
		}
		if (unbound.size() > 0) {
			StringBuffer sb = new StringBuffer("Unable to create unique key constraint (");
			for (String columnName : columnNames) {
				sb.append( columnName ).append( ", ");
			}
			sb.setLength( sb.length() - 2 );
			sb.append(") on table ").append( table.getName() ).append(": ");
			for (Column column : unbound) {
				sb.append( column.getName() ).append( ", ");
			}
			sb.setLength( sb.length() - 2 );
			sb.append( " not found");
			throw new AnnotationException( sb.toString() );
		}
	}

	protected void parseMappingElement(Element subelement, String name) {
		Attribute rsrc = subelement.attribute( "resource" );
		Attribute file = subelement.attribute( "file" );
		Attribute jar = subelement.attribute( "jar" );
		Attribute pckg = subelement.attribute( "package" );
		Attribute clazz = subelement.attribute( "class" );
		if ( rsrc != null ) {
			log.debug( name + "<-" + rsrc );
			addResource( rsrc.getValue() );
		}
		else if ( jar != null ) {
			log.debug( name + "<-" + jar );
			addJar( new File( jar.getValue() ) );
		}
		else if ( file != null) {
			log.debug( name + "<-" + file );
			addFile( file.getValue() );
		}
		else if (pckg != null) {
			log.debug( name + "<-" + pckg );
			addPackage( pckg.getValue() );
		}
		else if (clazz != null) {
			log.debug( name + "<-" + clazz );
			Class loadedClass = null;
			try {
				loadedClass = ReflectHelper.classForName( clazz.getValue() );
			}
			catch (ClassNotFoundException cnf) {
				throw new MappingException("Unable to load class declared as <mapping class=\"" + clazz.getValue() + "\"/> in the configuration:", cnf);
			}
			addAnnotatedClass(loadedClass);
		}
		else {
			throw new MappingException( "<mapping> element in configuration specifies no attributes" );
		}
	}

	private class CacheHolder {
		public CacheHolder(String role, String usage, String region, boolean isClass) {
			this.role = role;
			this.usage = usage;
			this.region = region;
			this.isClass = isClass;
		}
		public String role;
		public String usage;
		public String region;
		public boolean isClass;
	}
}
