package org.hibernate.cfg.annotations;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.StringTokenizer;
import java.util.List;
import javax.persistence.FetchType;
import javax.persistence.MapKey;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.AnnotationException;
import org.hibernate.AssertionFailure;
import org.hibernate.FetchMode;
import org.hibernate.MappingException;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Cache;
import org.hibernate.annotations.OrderBy;
import org.hibernate.annotations.Sort;
import org.hibernate.annotations.SortType;
import org.hibernate.annotations.Where;
import org.hibernate.cfg.AnnotationBinder;
import org.hibernate.cfg.Ejb3JoinColumn;
import org.hibernate.cfg.ExtendedMappings;
import org.hibernate.cfg.IndexColumn;
import org.hibernate.cfg.PropertyHolder;
import org.hibernate.cfg.SecondPass;
import org.hibernate.cfg.PropertyInferredData;
import org.hibernate.cfg.BinderHelper;
import org.hibernate.mapping.Backref;
import org.hibernate.mapping.Collection;
import org.hibernate.mapping.DependantValue;
import org.hibernate.mapping.Join;
import org.hibernate.mapping.KeyValue;
import org.hibernate.mapping.ManyToOne;
import org.hibernate.mapping.OneToMany;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;
import org.hibernate.mapping.SimpleValue;
import org.hibernate.mapping.Table;
import org.hibernate.mapping.Column;
import org.hibernate.util.StringHelper;

/**
 * Collection binder
 *
 * @author inger
 * @author emmanuel
 */
public abstract class CollectionBinder {

	private static final Log log = LogFactory.getLog(CollectionBinder.class);

	protected Collection collection;
	protected String propertyName;
	FetchMode fetchMode;
	PropertyHolder propertyHolder;
	int batchSize;
	String where;
	private String mappedBy;
	private Table table;
	private Class collectionType;
	private String targetEntity;
	private ExtendedMappings mappings;
	private boolean unique;
	private Ejb3JoinColumn[] inverseJoinColumns;
	private String cascadeStrategy;
	String cacheConcurrencyStrategy;
	Map<String, String> filters = new HashMap<String, String>();
	String cacheRegionName;
	private boolean oneToMany;
	protected IndexColumn indexColumn;
	private String orderBy;
	protected String hqlOrderBy;
	private boolean isSorted;
	private Class comparator;
	private boolean hasToBeSorted;
	protected boolean cascadeDeleteEnabled;
	protected String mapKeyPropertyName;
	private boolean insertable = true;
	private boolean updatable = true;

	public void setUpdatable(boolean updatable) {
		this.updatable = updatable;
	}

	public void setInsertable(boolean insertable) {
		this.insertable = insertable;
	}


	public void setCascadeStrategy(String cascadeStrategy) {
		this.cascadeStrategy = cascadeStrategy;
	}

	public void setPropertyAccessorName(String propertyAccessorName) {
		this.propertyAccessorName = propertyAccessorName;
	}

	private String propertyAccessorName;

	public void setUnique(boolean unique) {
		this.unique = unique;
	}

	public void setInverseJoinColumns(Ejb3JoinColumn[] inverseJoinColumns) {
		this.inverseJoinColumns = inverseJoinColumns;
	}

	public void setJoinColumns(Ejb3JoinColumn[] joinColumns) {
		this.joinColumns = joinColumns;
	}

	private Ejb3JoinColumn[] joinColumns;

	public void setPropertyHolder(PropertyHolder propertyHolder) {
		this.propertyHolder = propertyHolder;
	}

	public void setBatchSize(BatchSize batchSize) {
		this.batchSize = batchSize == null ? -1 : batchSize.size();
	}

	public void setEjb3OrderBy(javax.persistence.OrderBy orderByAnn) {
		if (orderByAnn != null) {
			hqlOrderBy = orderByAnn.value();
		}
	}

	public void setSqlOrderBy(OrderBy orderByAnn) {
		if (orderByAnn != null) {
			if ( ! AnnotationBinder.isDefault( orderByAnn.clause() ) ) orderBy = orderByAnn.clause();
		}
	}

	public void setSort(Sort sortAnn) {
		if (sortAnn != null) {
			isSorted = ! SortType.UNSORTED.equals( sortAnn.type() );
			if (isSorted && SortType.COMPARATOR.equals( sortAnn.type() ) ) {
				comparator = sortAnn.comparator();
			}
		}
	}

	/** collection binder factory */
    public static CollectionBinder getCollectionBinder(String entityName, PropertyInferredData inferredData,
			boolean isIndexed) {
        CollectionBinder binder = null;
		Class returnedClass = inferredData.getReturnedClassOrElement();
        if ( inferredData.isArray() ) {
            binder = new ArrayBinder();
		}
        else if ( java.util.Set.class.equals(returnedClass) ) {
            binder = new SetBinder();
        }
		else if ( java.util.SortedSet.class.equals(returnedClass) ) {
			binder = new SetBinder(true);
		}
		else if ( java.util.Map.class.equals(returnedClass) ) {
			binder = new MapBinder();
		}
        else if ( java.util.Collection.class.equals(returnedClass)) {
            binder = new BagBinder();
        }
        else if ( java.util.List.class.equals(returnedClass) ) {
			if (isIndexed == true) {
            	binder = new ListBinder();
			}
			else {
				binder = new BagBinder();
			}
        }
        else {
            throw new AnnotationException(returnedClass.getName() + " collection not yet supported: "
					+ entityName + inferredData.getPropertyName() );
        }
        return binder;
    }

	protected CollectionBinder() {}

	protected CollectionBinder(boolean sorted) {
		this.hasToBeSorted = sorted;
	}

	public void setMappedBy(String mappedBy) {
		this.mappedBy = mappedBy;
	}

	public void setTable(Table table) {
		this.table = table;
	}

	public void setCollectionType(Class collectionType) {
		this.collectionType = collectionType;
	}

	public void setTargetEntity(Class targetEntity) {
		if ( AnnotationBinder.isDefault(targetEntity) ) {
			this.targetEntity = AnnotationBinder.ANNOTATION_STRING_DEFAULT;
		}
		else {
			this.targetEntity = targetEntity.getName();
		}
	}

	public void setMappings(ExtendedMappings mappings) {
		this.mappings = mappings;
	}

	protected abstract Collection createCollection(PersistentClass persistentClass);

	public Collection getCollection() {
		return collection;
	}

	public void setPropertyName(String propertyName) {
		this.propertyName = propertyName;
	}

	public void bind() {
		if ( log.isDebugEnabled() ) {
			if (! oneToMany) {
				if ( unique == false ) {
					log.debug("Binding as ManyToMany: " + propertyHolder.getEntityName() + "." + propertyName);
				}
				else {
					log.debug("Binding a OneToMany: " + propertyHolder.getEntityName() + "." + propertyName + " through an association table");
				}
			} else {
				log.debug("Binding a OneToMany: " + propertyHolder.getEntityName() + "." + propertyName + " through a foreign key");
			}
		}
		this.collection = createCollection( propertyHolder.getPersistentClass() );
		log.debug( "Collection role: " + StringHelper.qualify( propertyHolder.getPath(), propertyName) );
		collection.setRole( StringHelper.qualify( propertyHolder.getPath(), propertyName) );

		//set laziness
		collection.setFetchMode(fetchMode);
		collection.setLazy(fetchMode == FetchMode.SELECT);
		collection.setBatchSize(batchSize);
		if (orderBy != null && hqlOrderBy != null)
			throw new AnnotationException("Cannot use sql order by clause in conjunction of EJB3 order by clause: " + safeCollectionRole() );
		if (orderBy != null) collection.setOrderBy(orderBy);
		if (isSorted) {
			collection.setSorted(true);
			if (comparator != null) {
				try {
					collection.setComparator( (Comparator) comparator.newInstance() );
				}
				catch (Exception e) {
					throw new AnnotationException( "Could not instantiate comparator class: "
						+ comparator.getName() + "("  + safeCollectionRole() + ")" );
				}
			}
		}
		else {
			if (hasToBeSorted) throw new AnnotationException("A sorted collection has to define @Sort: "
					 + safeCollectionRole() );
		}

		if ( StringHelper.isNotEmpty( cacheConcurrencyStrategy ) ) {
			collection.setCacheConcurrencyStrategy(cacheConcurrencyStrategy);
			collection.setCacheRegionName(cacheRegionName);
		}
		Iterator<Map.Entry<String, String>> iter = filters.entrySet().iterator();
		if ( StringHelper.isNotEmpty(where) ) collection.setWhere(where);
		while (iter.hasNext()) {
			Map.Entry<String, String> filter = iter.next();
			collection.addFilter(filter.getKey(), filter.getValue());
		}
		boolean isMappedBy = ! AnnotationBinder.isDefault(mappedBy);
		collection.setInverse( isMappedBy );
		collection.setCollectionTable(table);
		String collType = getCollectionType();
		if (oneToMany) {
			org.hibernate.mapping.OneToMany oneToMany = new org.hibernate.mapping.OneToMany( collection.getOwner() );
			collection.setElement(oneToMany);
			oneToMany.setReferencedEntityName( collType );
		}
		else {
			//many to many may need some second pass informations
			if ( isMappedBy ) {
				mappings.addMappedBy( collType, mappedBy, propertyName );
			}
		}
		mappings.addSecondPass(
				getSecondPass(mappings, joinColumns, inverseJoinColumns, collType, fetchMode, unique),
				! isMappedBy
		);
		mappings.addCollection(collection);
		PropertyBinder binder = new PropertyBinder();
		binder.setName(propertyName);
		binder.setValue(collection);
		binder.setCascade(cascadeStrategy);
		binder.setPropertyAccessorName(propertyAccessorName);
		binder.setInsertable( insertable );
		binder.setUpdatable( updatable );
		Property prop = binder.make();
		//we don't care about the join stuffs because the column is on the association table.
		propertyHolder.addProperty(prop);
	}

	private String getCollectionType() {
		if ( AnnotationBinder.isDefault(targetEntity) ) {
			if (collectionType != null) {
				return collectionType.getName();
			}
			else {
				String errorMsg = "Collection has neither generic type or OneToMany.targetEntity() defined: "
						+ safeCollectionRole();
				throw new AnnotationException(errorMsg);
			}
		}
		else {
			return targetEntity;
		}
	}

	public SecondPass getSecondPass(final ExtendedMappings mappings,
											  final Ejb3JoinColumn[] keyColumns,
											  final Ejb3JoinColumn[] inverseColumns,
											  final String collType,
											  final FetchMode fetchMode,
											  final boolean unique) {
		if (inverseColumns != null) {
			return new SecondPass(mappings, collection) {
				public void secondPass(Map persistentClasses, Map inheritedMetas)
						throws MappingException {
					bindManyToManySecondPass(
							getCollection(),
							persistentClasses,
							keyColumns,
							inverseColumns,
							collType,
							fetchMode,
							unique,
							cascadeDeleteEnabled, (ExtendedMappings) getMappings()
					);
				}
			};
		}
		else {
			return new SecondPass(mappings, collection) {
				public void secondPass(Map persistentClasses, Map inheritedMetas)
						throws MappingException {
					bindCollectionSecondPass(
							getCollection(),
							persistentClasses,
							keyColumns,
							cascadeDeleteEnabled,
							hqlOrderBy,
							(ExtendedMappings) getMappings()
					);
				}
			};
		}
	}

	public void setCache(Cache cacheAnn) {
		if (cacheAnn != null) {
			cacheRegionName = AnnotationBinder.isDefault( cacheAnn.region() ) ? null : cacheAnn.region();
			cacheConcurrencyStrategy = EntityBinder.getCacheConcurrencyStrategy( cacheAnn.usage() );
		}
		else {
			cacheConcurrencyStrategy = null;
			cacheRegionName = null;
		}
	}

	public void setFetchType(FetchType fetch) {
		if (fetch == FetchType.EAGER) {
			fetchMode = FetchMode.JOIN;
		}
		else {
			fetchMode = FetchMode.SELECT;
		}
	}

	public void addFilter(String name, String condition) {
		filters.put(name, condition);
	}

	public void setWhere(Where whereAnn) {
		if (whereAnn != null) {
			where = whereAnn.clause();
		}
	}

	public void setOneToMany(boolean oneToMany) {
		this.oneToMany = oneToMany;
	}

	public void setIndexColumn(IndexColumn indexColumn) {
		this.indexColumn = indexColumn;
	}

	public void setMapKey(MapKey key) {
		if (key != null) {
			mapKeyPropertyName = key.name();
		}
	}

    protected static void bindCollectionSecondPass(
			Collection collValue, Map persistentClasses, Ejb3JoinColumn[] columns, boolean cascadeDeleteEnabled,
			String hqlOrderBy, ExtendedMappings mappings
			) throws MappingException {
		if ( collValue.isOneToMany() ) {
			org.hibernate.mapping.OneToMany oneToMany =
				(org.hibernate.mapping.OneToMany) collValue.getElement();
			String assocClass = oneToMany.getReferencedEntityName();
			PersistentClass associatedClass = (PersistentClass) persistentClasses.get(assocClass);
			String orderBy = buildOrderByClauseFromHql(hqlOrderBy, associatedClass, collValue.getRole() );
			if (orderBy != null) collValue.setOrderBy(orderBy);
			if (mappings == null) {
				throw new AssertionFailure("CollectionSecondPass for oneToMany should not be called with null mappings");
			}
			Map<String, Join> joins = mappings.getJoins(assocClass);
			if (associatedClass==null) throw new MappingException(
				"Association references unmapped class: " + assocClass
			);
			oneToMany.setAssociatedClass(associatedClass);
			//FIXME: find the appropriate table between second and primary
			for (Ejb3JoinColumn column : columns) {
				column.setPersistentClass(associatedClass);
				column.setJoins(joins);
				collValue.setCollectionTable( column.getTable() );
			}
			log.info("Mapping collection: " + collValue.getRole() + " -> " + collValue.getCollectionTable().getName() );
		}

		bindCollectionSecondPass(collValue, null, columns, cascadeDeleteEnabled, mappings );
		if ( collValue.isOneToMany()
			&& !collValue.isInverse()
			&& !collValue.getKey().isNullable() ) {
			// for non-inverse one-to-many, with a not-null fk, add a backref!
			String entityName = ( (OneToMany) collValue.getElement() ).getReferencedEntityName();
			PersistentClass referenced = mappings.getClass( entityName );
			Backref prop = new Backref();
			prop.setName( '_' + columns[0].getPropertyName() + "Backref" );
			prop.setUpdateable( false );
			prop.setSelectable( false );
			prop.setCollectionRole( collValue.getRole() );
			prop.setEntityName( collValue.getOwner().getEntityName() );
			prop.setValue( collValue.getKey() );
			referenced.addProperty( prop );
		}
	}

	private static String buildOrderByClauseFromHql(String hqlOrderBy, PersistentClass associatedClass, String role) {
		String orderByString = null;
		if (hqlOrderBy != null) {
			List<String> properties = new ArrayList<String>();
			List<String> ordering = new ArrayList<String>();
			StringBuffer orderByBuffer = new StringBuffer();
			if ( "".equals( hqlOrderBy ) ) {
				//order by id
				Iterator it = associatedClass.getIdentifier().getColumnIterator();
				while( it.hasNext() ) {
					Column col = (Column) it.next();
					orderByBuffer.append( col.getName() ).append(" asc").append(", ");
				}
			}
			else {
				StringTokenizer st = new StringTokenizer(hqlOrderBy, " ,", false);
				String currentOrdering = null;
				//FIXME make this code decent
				while( st.hasMoreTokens() ) {
					String token =  st.nextToken();
					if ( isNonPropertyToken(token) ) {
						if (currentOrdering != null)
							throw new AnnotationException("Error while parsing HQL orderBy clause: " + hqlOrderBy
								+ " (" + role + ")");
						currentOrdering = token;
					}
					else {
						//Add ordering of the previous
						if (currentOrdering == null) {
							//default ordering
							ordering.add("asc");
						}
						else {
							ordering.add(currentOrdering);
							currentOrdering = null;
						}
						properties.add(token);
					}
				}
				ordering.remove(0); //first one is the algorithm starter
				// add last one ordering
				if (currentOrdering == null) {
					//default ordering
					ordering.add("asc");
				}
				else {
					ordering.add(currentOrdering);
					currentOrdering = null;
				}
				int index = 0;

				for (String property : properties) {
					Property p = BinderHelper.findPropertyByName(associatedClass, property );
					if (p == null) {
						throw new AnnotationException("property from @OrderBy clause not found: "
							+ associatedClass.getEntityName() + "." + property);
					}

					Iterator propertyColumns = p.getColumnIterator();
					while ( propertyColumns.hasNext() ) {
						Column column = (Column) propertyColumns.next();
						orderByBuffer.append( column.getName() ).append(" ").append( ordering.get(index) ).append(", ");
					}
					index++;
				}
			}
			orderByString = orderByBuffer.substring(0, orderByBuffer.length() - 2);
		}
		return orderByString;
	}

	private static boolean isNonPropertyToken(String token) {
		if ( " ".equals(token) ) return true;
		if ( ",".equals(token) ) return true;
		if ( token.equalsIgnoreCase("desc") ) return true;
		if ( token.equalsIgnoreCase("asc") ) return true;
		return false;
	}

	private static SimpleValue buildCollectionKey(
			Collection collValue, Ejb3JoinColumn[] joinColumns, boolean cascadeDeleteEnabled,
			ExtendedMappings mappings
			) {
		//binding key reference using column
		KeyValue keyVal;
		//give a chance to override the referenced property name
		//has to do that here because the referencedProperty creation happens in a FKSecondPass for Many to one yuk!
		if (joinColumns.length > 0 && StringHelper.isNotEmpty( joinColumns[0].getMappedBy() ) ) {
			String propRef = mappings.getPropertyReferencedAssociation(
					joinColumns[0].getPropertyHolder().getEntityName(),
					joinColumns[0].getMappedBy()
			);
			if (propRef != null) {
				collValue.setReferencedPropertyName( propRef );
				mappings.addPropertyReference( collValue.getOwnerEntityName(), propRef );
			}
		}
		String propRef = collValue.getReferencedPropertyName();
		if (propRef==null) {
			keyVal = collValue.getOwner().getIdentifier();
		}
		else {
			keyVal = (KeyValue) collValue.getOwner()
				.getProperty(propRef)
				.getValue();
		}
		DependantValue key = new DependantValue( collValue.getCollectionTable(), keyVal );
		key.setTypeName(null);
		Ejb3JoinColumn.checkPropertyConsistency( joinColumns, collValue.getOwnerEntityName() );
		key.setNullable( joinColumns.length == 0 ? true : joinColumns[0].isNullable() );
		key.setUpdateable( joinColumns.length == 0 ? true : joinColumns[0].isUpdatable() );
		key.setCascadeDeleteEnabled(cascadeDeleteEnabled);
		collValue.setKey(key);
		return key;
	}

    protected static void bindManyToManySecondPass(
			Collection collValue,
			Map persistentClasses,
			Ejb3JoinColumn[] joinColumns,
			Ejb3JoinColumn[] inverseJoinColumns,
			String collType,
			FetchMode fetchMode,
			boolean unique,
			boolean cascadeDeleteEnabled, ExtendedMappings mappings
			) throws MappingException {

		PersistentClass collectionEntity = (PersistentClass) persistentClasses.get(collType);
		if (collectionEntity == null) {
			throw new MappingException(
				"Association references unmapped class: " + collType
			);
		}
		boolean mappedBy = ! AnnotationBinder.isDefault(joinColumns[0].getMappedBy() );
		if ( mappedBy ) {
			Property otherSideProperty;
			try {
				otherSideProperty = collectionEntity.getProperty( joinColumns[0].getMappedBy() );
			} catch (MappingException e) {
				throw new AnnotationException("mappedBy reference an unknown property: " + collType + "." + joinColumns[0].getMappedBy() );
			}
			Table table = ( (Collection) otherSideProperty.getValue() ).getCollectionTable();
			collValue.setCollectionTable(table);
			for (Ejb3JoinColumn column : joinColumns) {
				column.setDefaultColumnHeader( joinColumns[0].getMappedBy() );
			}
		}
		else {
			//TODO: only for implicit columns?
			for (Ejb3JoinColumn column : joinColumns) {
				String header = mappings.getFromMappedBy( collValue.getOwnerEntityName() , column.getPropertyName() );
				header = (header == null) ? collValue.getOwner().getTable().getName() : header;
				column.setDefaultColumnHeader( header );
			}
			if ( collValue.getCollectionTable() == null ) {
				//default value
				String tableName = collValue.getOwner().getTable().getName()
						+ "_"
						+ collectionEntity.getTable().getName();
				Table table = TableBinder.fillTable("", "", tableName, false, new ArrayList(), null, null, mappings);
				collValue.setCollectionTable(table);
			}
		}
		bindCollectionSecondPass(collValue, collectionEntity, joinColumns, cascadeDeleteEnabled, mappings );

		ManyToOne element =
			new ManyToOne( collValue.getCollectionTable() );
		collValue.setElement(element);
		element.setReferencedEntityName(collType);
		element.setFetchMode(fetchMode);
		element.setLazy(fetchMode!=FetchMode.JOIN);

		//for now it can't happen, but sometime soon...
		if ( ( collValue.getFilterMap().size() != 0 || StringHelper.isNotEmpty( collValue.getWhere() ) ) &&
			collValue.getFetchMode() == FetchMode.JOIN &&
			collValue.getElement().getFetchMode() != FetchMode.JOIN ) {
			throw new MappingException("@ManyToMany defining filter or where without join fetching "
				+ "not valid within collection using join fetching[" + collValue.getRole() + "]");
		}

		//FIXME: do optional = false
		if (collectionEntity==null) throw new MappingException(
			"Association references unmapped class: " + collType
		);
		TableBinder.bindManytoManyInverseFk(collectionEntity, inverseJoinColumns, element, unique, mappings );

	}

    private static void bindCollectionSecondPass(
			Collection collValue, PersistentClass collectionEntity, Ejb3JoinColumn[] joinColumns, boolean cascadeDeleteEnabled,
			ExtendedMappings mappings
			) {
		BinderHelper.createSyntheticPropertyReference( joinColumns, collValue.getOwner(), collValue, mappings );
		SimpleValue key = buildCollectionKey(collValue, joinColumns, cascadeDeleteEnabled, mappings );
		TableBinder.bindFk(collValue.getOwner(), collectionEntity, joinColumns, key, false );
	}

	public void setCascadeDeleteEnabled(boolean onDeleteCascade) {
		this.cascadeDeleteEnabled = onDeleteCascade;
	}

	private String safeCollectionRole() {
		if (propertyHolder != null) {
			return propertyHolder.getEntityName() + "." + propertyName;
		}
		else {
			return "";
		}
	}


}