//$Id: TableBinder.java,v 1.13 2005/07/26 04:57:08 epbernard Exp $
package org.hibernate.cfg.annotations;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.AnnotationException;
import org.hibernate.AssertionFailure;
import org.hibernate.annotations.Index;
import org.hibernate.cfg.AnnotationBinder;
import org.hibernate.cfg.Ejb3JoinColumn;
import org.hibernate.cfg.ExtendedMappings;
import org.hibernate.mapping.Collection;
import org.hibernate.mapping.Column;
import org.hibernate.mapping.DependantValue;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;
import org.hibernate.mapping.SimpleValue;
import org.hibernate.mapping.Table;
import org.hibernate.mapping.ToOne;
import org.hibernate.mapping.Value;
import org.hibernate.util.StringHelper;

/**
 * Table related operations
 * @author Emmanuel Bernard
 */
public abstract class TableBinder {
	//TODO move it to a getter/setter strategy
	private static Log log = LogFactory.getLog(TableBinder.class);

	public static Table fillTable(
			String schema, String catalog, String realTableName, boolean isAbstract,
			List uniqueConstraints, String constraints, Table denormalizedSuperTable, ExtendedMappings mappings
			) {
		schema = AnnotationBinder.isDefault(schema) ? schema = mappings.getSchemaName() : schema;
		catalog = AnnotationBinder.isDefault(catalog) ? catalog = mappings.getCatalogName() : catalog;
		Table table;
		if (denormalizedSuperTable != null) {
			table = mappings.addDenormalizedTable(
				schema,
				catalog,
				realTableName,
				isAbstract,
				null, //subselect
                denormalizedSuperTable
			);
		}
		else {
			table = mappings.addTable(
				schema,
				catalog,
				realTableName,
				null, //subselect
				isAbstract
			);
		}
		if (uniqueConstraints != null && uniqueConstraints.size() > 0)
			mappings.addUniqueConstraints(table, uniqueConstraints);
		if (constraints != null) table.addCheckConstraint(constraints);
		return table;
	}

	/**
	 * bind the inverse FK of a ManyToMany
	 * If we are in a mappedBy case, read the columns from the associated
	 * colletion element
	 * Otherwise delegates to the usual algorithm
	 */
	public static void bindManytoManyInverseFk(
			PersistentClass referencedEntity, Ejb3JoinColumn[] columns, SimpleValue value, boolean unique,
			ExtendedMappings mappings
			) {
		if ( ! StringHelper.isEmpty( columns[0].getMappedBy() ) ) {
			final Property property = referencedEntity.getProperty( columns[0].getMappedBy() );
			Iterator mappedByColumns = ( (Collection) property.getValue() ).getKey().getColumnIterator();
			while ( mappedByColumns.hasNext() ) {
				Column column = (Column) mappedByColumns.next();
				columns[0].linkValueUsingAColumnCopy(column, value);
			}
			value.createForeignKey();
		}
		else {
			bindFk(referencedEntity, null, columns, value, unique );
		}
	}

	public static void bindFk(
			PersistentClass referencedEntity, PersistentClass destinationEntity, Ejb3JoinColumn[] columns, SimpleValue value,
			boolean unique
			) {
		PersistentClass associatedClass;
		if (destinationEntity != null) {
			//overidden destination
			associatedClass = destinationEntity;
		}
		else {
			associatedClass = columns[0].getPropertyHolder() == null ? null : columns[0].getPropertyHolder().getPersistentClass();
		}
		final String mappedByProperty = columns[0].getMappedBy();
		if ( ! StringHelper.isEmpty( mappedByProperty ) ) {
			/**
			 * Get the columns of the mapped-by property
			 * copy them and link the copy to the actual value
			 */
			if ( log.isDebugEnabled() ) log.debug("Retrieving property " + associatedClass.getEntityName() + "." + mappedByProperty );

			final Property property = associatedClass.getProperty( columns[0].getMappedBy() );
			Iterator mappedByColumns;
			if ( property.getValue() instanceof Collection) {
				mappedByColumns = ( (Collection) property.getValue() ).getElement().getColumnIterator();
			}
			else {
				mappedByColumns = property.getValue().getColumnIterator();
			}
			while ( mappedByColumns.hasNext() ) {
				Column column = (Column) mappedByColumns.next();
				columns[0].linkValueUsingAColumnCopy(column, value);
			}
		} else if ( columns[0].isImplicit() ) {
			/**
			 * if columns are implicit, then create the columns based on the
			 * referenced entity id columns
			 */
            Iterator idColumns = referencedEntity.getIdentifier().getColumnIterator();
			while ( idColumns.hasNext() ) {
				Column column = (Column) idColumns.next();
				columns[0].linkValueUsingDefaultColumnNaming(column, value);
			}
		}
		else {
			int fkEnum = Ejb3JoinColumn.checkReferencedColumnsType(columns, referencedEntity);

			if ( Ejb3JoinColumn.NON_PK_REFERENCE == fkEnum ) {
				String referencedPropertyName;
				if (value instanceof ToOne) {
					referencedPropertyName = ( (ToOne) value ).getReferencedPropertyName();
				}
				else if (value instanceof DependantValue) {
					Collection collection = (Collection) referencedEntity.getProperty( columns[0].getPropertyName() ).getValue();
					referencedPropertyName = collection.getReferencedPropertyName();
				}
				else {
					throw new AssertionFailure("Do a property ref on an unexpected Value type: "
							+ value.getClass().getName() );
				}
				if (referencedPropertyName == null) throw new AssertionFailure("No property ref found while expected");
				Property synthProp = referencedEntity.getProperty( referencedPropertyName );
				if (synthProp == null) throw new AssertionFailure("Cannot find synthProp: " + referencedEntity.getEntityName() + "." + referencedPropertyName );
				linkColumnWithValueOverridingNameIfImplicit( synthProp.getColumnIterator(), columns, value );

			}
			else {
				if ( Ejb3JoinColumn.NO_REFERENCE == fkEnum ) {
					//implicit case, we hope PK and FK columns are in the same order
					if (columns.length != referencedEntity.getIdentifier().getColumnSpan() ) {
						throw new AnnotationException("A Foreign key refering " + referencedEntity.getEntityName()
							+ " has the wrong number of column. should be " + referencedEntity.getIdentifier().getColumnSpan()
						);
					}
					//Iterator idColumnsIt = referencedEntity.getIdentifier().getColumnIterator();
					linkColumnWithValueOverridingNameIfImplicit(
							referencedEntity.getIdentifier().getColumnIterator(),
							columns,
							value
					);
				}
				else {
					//explicit referencedColumnName
					Iterator idColItr = referencedEntity.getIdentifier().getColumnIterator();
					org.hibernate.mapping.Column col;
					if (! idColItr.hasNext()) log.debug("No column in the identifier!");
					while ( idColItr.hasNext() ) {
						boolean match = false;
						//for each PK column, find the associated FK column.
						col = (org.hibernate.mapping.Column) idColItr.next();
						for (Ejb3JoinColumn joinCol : columns) {
							if ( joinCol.getReferencedColumn().equals( col.getName() ) ) {
								//proper join column
								if ( joinCol.isNameDeferred() ) {
									joinCol.redefineColumnName( joinCol.getPropertyName() + "_" + joinCol.getReferencedColumn() );
								}
								joinCol.linkWithValue(value);
								match = true;
								break;
							}
						}
						if (match == false) {
							throw new AnnotationException("Column name " + col.getName() + " of "
									+ referencedEntity.getEntityName() + " not found in JoinColumns.referencedColumnName");
						}
					}
				}
			}
		}

// OLD original code
//			if (referencedEntity.getIdentifier().getColumnSpan() == 1) {
//				if (columns.length != 1) {
//					throw new AnnotationException("@JoinColumns with " + columns.length + " columns"
//						+ " refers to " + referencedEntity.getEntityName() + " which has a "
//						+ referencedEntity.getIdentifier().getColumnSpan() + " sized PK");
//				}
//				else {
//					if ( columns[0].isNameDeferred() ) {
//						//this has to be the default value
//						Column idCol = (Column) referencedEntity.getIdentifier().getColumnIterator().next();
//						columns[0].redefineColumnName( columns[0].getPropertyName() + "_" + idCol.getName() );
//					}
//					columns[0].linkWithValue(value);
//				}
//			}
//			else {
//				//Check if we have to do it the implicit way
//				//TODO: clarify the spec on it
//				boolean noReferencedColumn = true;
//				for (Ejb3JoinColumn joinCol : columns) {
//					if ( StringHelper.isNotEmpty( joinCol.getReferencedColumn() ) ) {
//						noReferencedColumn = false;
//						break;
//					}
//
//				}
//				if (noReferencedColumn) {
//					//implicit case, we hope PK and FK columns are in the same order
//					if (columns.length != referencedEntity.getIdentifier().getColumnSpan() ) {
//						throw new AnnotationException("A Foreign key refering " + referencedEntity.getEntityName()
//							+ " has the wrong number of column. should be " + referencedEntity.getIdentifier().getColumnSpan()
//						);
//					}
//					idColumnsIt = referencedEntity.getIdentifier().getColumnIterator();
//					for (Ejb3JoinColumn joinCol : columns) {
//						Column idCol = (Column) idColumnsIt.next();
//						if ( joinCol.isNameDeferred() ) {
//							//this has to be the default value
//							joinCol.redefineColumnName( joinCol.getPropertyName() + "_" + idCol.getName() );
//						}
//						joinCol.linkWithValue(value);
//					}
//				}
//				else {
//					//explicit referencedColumnName
//					Iterator idColItr = referencedEntity.getIdentifier().getColumnIterator();
//					org.hibernate.mapping.Column col;
//					if (! idColItr.hasNext()) log.debug("No column in the identifier!");
//					while ( idColItr.hasNext() ) {
//						boolean match = false;
//						//for each PK column, find the associated FK column.
//						col = (org.hibernate.mapping.Column) idColItr.next();
//						for (Ejb3JoinColumn joinCol : columns) {
//							if ( joinCol.getReferencedColumn().equals( col.getName() ) ) {
//								//proper join column
//								if ( joinCol.isNameDeferred() ) {
//									joinCol.redefineColumnName( joinCol.getPropertyName() + "_" + joinCol.getReferencedColumn() );
//								}
//								joinCol.linkWithValue(value);
//								match = true;
//								break;
//							}
//						}
//						if (match == false) {
//							throw new AnnotationException("Column name " + col.getName() + " of "
//									+ referencedEntity.getEntityName() + " not found in JoinColumns.referencedColumnName");
//						}
//					}
//				}
//			}
//		}
        value.createForeignKey();
		if ( unique == true ) {
			createUniqueConstraint( value );
		}
	}

	private static void linkColumnWithValueOverridingNameIfImplicit(Iterator columnIterator, Ejb3JoinColumn[] columns, SimpleValue value) {
		for (Ejb3JoinColumn joinCol : columns) {
			Column synthCol = (Column) columnIterator.next();
			if ( joinCol.isNameDeferred() ) {
				//this has to be the default value
				joinCol.redefineColumnName( joinCol.getPropertyName() + "_" + synthCol.getName() );
			}
			joinCol.linkWithValue(value);
		}
	}

	public static void createUniqueConstraint(Value value) {
		Iterator iter = value.getColumnIterator();
		ArrayList cols = new ArrayList();
		while ( iter.hasNext() ) {
			cols.add( iter.next() );
		}
		value.getTable().createUniqueKey( cols );
	}

	private static List<Property> findPropertiesByColumns(PersistentClass referencedEntity, Ejb3JoinColumn[] columns) {
		Map<Column,Set<Property>> columnsToProperty = new HashMap<Column,Set<Property>>();
		List<Column> orderedColumns = new ArrayList<Column>();
		for(int index = 0 ; index < columns.length ; index++) {
			Column column = new Column( columns[index].getReferencedColumn() );
			orderedColumns.add( column );
			columnsToProperty.put( column, new HashSet<Property>() );
		}
		Iterator it = referencedEntity.getPropertyIterator();
		while ( it.hasNext() ) {
			Property property = (Property) it.next();
			Iterator columnIt = property.getColumnIterator();
            while ( columnIt.hasNext() ) {
				Column column = (Column) columnIt.next();
				if ( columnsToProperty.containsKey( column ) ) {
					columnsToProperty.get( column ).add( property );
				}
			}
		}
		//first naive implementation
		//only check 1 columns properties
		//TODO make it smarter by checking correctly ordered multi column properties
		List<Property> orderedProperties = new ArrayList<Property>();
		for (Column column : orderedColumns) {
			boolean found = false;
			for (Property property : columnsToProperty.get( column ) ) {
				if (property.getColumnSpan() == 1) {
					orderedProperties.add(property);
					found = true;
					break;
				}
			}
			if (!found) return null; //have to find it the hard way
		}
		return orderedProperties;
	}

	public static void addIndexes(Table hibTable, Index[] indexes) {
		for ( Index index : indexes ) {
			String indexName = index.name();
			for ( String columnName : index.columnNames() ) {
				Column column = hibTable.getColumn( new Column(columnName) );
				if (column == null) throw new AnnotationException( "@Index references a unknown column: " + columnName );
                hibTable.getOrCreateIndex( indexName ).addColumn( column );
			}
		}
	}
}
