//$Id: ClassValidator.java,v 1.1 2005/05/27 08:58:53 epbernard Exp $
package org.hibernate.validator;

import java.beans.Introspector;
import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.StringTokenizer;

import org.hibernate.MappingException;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;


/**
 * Engine that take a bean and check every expressed annotation restrictions
 *
 * @author Gavin King
 */
public class ClassValidator<T> {
	
	private final Class<T> beanClass;
	private final List<Validator> beanValidators = new ArrayList<Validator>();
	private final List<Validator> propertyValidators = new ArrayList<Validator>();
	private final List<Method> propertyGetters = new ArrayList<Method>();
	private final Map<Validator, String> messages = new HashMap<Validator, String>();
	
	private final ResourceBundle messageBundle;

	/** create the validator engine for this bean type */
	public ClassValidator(Class<T> beanClass) {
		this(beanClass, null);
	}

	/**
	 * create the validator engine for a particular bean class, using a resource bundle
	 * for message rendering on violation *
	 */
	public ClassValidator(Class<T> beanClass, ResourceBundle resourceBundle) {
		this.beanClass = beanClass;
		this.messageBundle = resourceBundle;
		
		Annotation[] classAnnotations = beanClass.getAnnotations();
		for ( int i = 0; i < classAnnotations.length; i++ ) {
			Annotation classAnnotation = classAnnotations[i];
			Validator beanValidator = createValidator(classAnnotation);
			if (beanValidator!=null) beanValidators.add(beanValidator);
		}
		
		Method[] methods = beanClass.getMethods();
		for ( int i = 0; i < methods.length; i++ ) {
			Method method = methods[i];
			Annotation[] methodAnnotations = method.getAnnotations();
			for ( int j = 0; j < methodAnnotations.length; j++ ) {
				Annotation methodAnnotation = methodAnnotations[j];
				Validator propertyValidator = createValidator(methodAnnotation);
				if (propertyValidator!=null) {
					propertyValidators.add(propertyValidator);
					propertyGetters.add(method);
				}
			}
		}

		//TODO test fields also
	}


//	private static final Set<String> methodNames = new HashSet<String>();
//	static {
//		methodNames.add("toString");
//		methodNames.add("hashCode");
//		methodNames.add("equals");
//		methodNames.add("annotationType");
//	}

	private Validator createValidator(Annotation annotation)  {
		try {
			ValidatorClass validatorClass = annotation.annotationType().getAnnotation(ValidatorClass.class);
			if (validatorClass==null) return null;
			Validator beanValidator = validatorClass.value().newInstance();
			beanValidator.initialize(annotation);
			String messageTemplate = (String) annotation.getClass()
					.getMethod("message", null)
					.invoke(annotation);
			String message = replace(messageTemplate, annotation);
			messages.put( beanValidator, message );
			return beanValidator;
		}
		catch (Exception e) {
			throw new IllegalArgumentException("could not instantiate ClassValidator", e);
		}
	}

	/**
	 * apply constraints on a bean instance and return all the failures.
	 */
	public InvalidValue[] getInvalidValues(T bean) {
		if ( !beanClass.isInstance(bean) ) {
			throw new IllegalArgumentException( "not an instance of: " + bean.getClass() );
		}
		
		List<InvalidValue> results = new ArrayList<InvalidValue>();
		
		for ( int i=0; i < beanValidators.size(); i++ ) {
			Validator validator = beanValidators.get(i);
			if ( !validator.isValid(bean) ) {
				results.add( new InvalidValue( messages.get(validator), beanClass, null, bean, bean ) );
			}
		}
		
		for ( int i=0; i < propertyValidators.size(); i++ ) {
			Method getter = propertyGetters.get(i);
			Object value;
			try {
				value = getter.invoke(bean);
			}
			catch (Exception e) {
				throw new IllegalStateException("could not get property value", e);
			}
			Validator validator = propertyValidators.get(i);
			if ( !validator.isValid( value ) ) {
				String propertyName = getPropertyName( getter );
				results.add( new InvalidValue( messages.get(validator), beanClass, propertyName, value, bean ) );
			}
		}
		
		return results.toArray( new InvalidValue[results.size()] );
	}

	private static String getPropertyName(Method getter) {
		String name = getter.getName();
		if ( name.startsWith("is") ) {
			name = name.substring(2);
		}
		else if ( name.startsWith("get") ) {
			name = name.substring(3);
		}
		String propertyName = Introspector.decapitalize(name);
		return propertyName;
	}
	
	private String replace(String message, Annotation parameters) {
		StringTokenizer tokens = new StringTokenizer(message, "{}", true);
		StringBuffer buf = new StringBuffer();
		boolean escaped = false;
		while ( tokens.hasMoreTokens() ) {
			String token = tokens.nextToken();
			if ( "{".equals(token) ) {
				escaped = true;
			}
			else if ( "}".equals(token) ) {
				escaped = false;
			}
			else if (!escaped) {
				buf.append(token);
			}
			else {
				Method member;
				try {
					member = parameters.getClass().getMethod(token, null);
				}
				catch (NoSuchMethodException nsfme) {
					member = null;
				}
				if ( member!=null ) {
					try {
						buf.append( member.invoke(parameters) );
					}
					catch (Exception e) {
						throw new IllegalArgumentException("could not render message", e);
					}
				}
				else if ( messageBundle!=null ) {
					String string = messageBundle.getString(token);
					if (string!=null) buf.append( string );
				}
			}
		}
		return buf.toString();
	}

	/**
	 * apply the registred constraints rules on the hibernate metadata (to be applied on DB schema...)
	 * @param persistentClass hibernate metadata
	 */
	public void apply(PersistentClass persistentClass) {
		
		Iterator<Validator> validators = beanValidators.iterator();
		while ( validators.hasNext() ) {
			Validator validator = validators.next();
			if ( validator instanceof PersistentClassConstraint ) {
				( (PersistentClassConstraint) validator ).apply(persistentClass);
			}
		}
		
		validators = propertyValidators.iterator();
		Iterator<Method> getters = propertyGetters.iterator();
		while ( validators.hasNext() ) {
			Validator validator = validators.next();
			String propertyName = getPropertyName( getters.next() );
			if ( validator instanceof PropertyConstraint ) {
				try {
					Property property = persistentClass.getProperty( propertyName );
					( (PropertyConstraint) validator ).apply(property);
				}
				catch (MappingException pnfe) {
					//do nothing
				}	
			}
		}
		
	}
	
}
