//$Id: ClassValidator.java,v 1.11 2005/09/18 23:26:49 epbernard Exp $
package org.hibernate.validator;

import java.beans.Introspector;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.lang.annotation.Annotation;
import java.lang.reflect.AccessibleObject;
import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Member;
import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.Set;
import java.util.StringTokenizer;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.hibernate.AssertionFailure;
import org.hibernate.Hibernate;
import org.hibernate.MappingException;
import org.hibernate.mapping.PersistentClass;
import org.hibernate.mapping.Property;


/**
 * Engine that take a bean and check every expressed annotation restrictions
 *
 * @author Gavin King
 */
public class ClassValidator<T> implements Serializable {
	private static Log log = LogFactory.getLog( ClassValidator.class );
	private static final InvalidValue[] EMPTY_INVALID_VALUE_ARRAY = new InvalidValue[]{};
	private final Class<T> beanClass;
	private transient ResourceBundle messageBundle;

	private final transient Map<Class, ClassValidator> childClassValidators;
	private transient List<Validator> beanValidators;
	private transient List<Validator> memberValidators;
	private transient List<Member> memberGetters;
	private transient Map<Validator, String> messages;
	private transient List<Member> childGetters;


	/**
	 * create the validator engine for this bean type
	 */
	public ClassValidator(Class<T> beanClass) {
		this( beanClass, null );
	}

	/**
	 * create the validator engine for a particular bean class, using a resource bundle
	 * for message rendering on violation
	 */
	public ClassValidator(Class<T> beanClass, ResourceBundle resourceBundle) {
		this( beanClass, resourceBundle, new HashMap<Class, ClassValidator>() );
	}

	protected ClassValidator(
			Class<T> beanClass, ResourceBundle resourceBundle, Map<Class, ClassValidator> childClassValidators
	) {
		this.beanClass = beanClass;
		this.messageBundle = resourceBundle;
		this.childClassValidators = childClassValidators;
		initValidator( beanClass, childClassValidators, resourceBundle );

	}

	private void initValidator(
			Class<T> beanClass, Map<Class, ClassValidator> childClassValidators,
			ResourceBundle resourceBundle
	) {
		beanValidators = new ArrayList<Validator>();
		memberValidators = new ArrayList<Validator>();
		memberGetters = new ArrayList<Member>();
		messages = new HashMap<Validator, String>();
		childGetters = new ArrayList<Member>();

		childClassValidators.put( beanClass, this );
		Annotation[] classAnnotations = beanClass.getAnnotations();
		for ( int i = 0; i < classAnnotations.length ; i++ ) {
			Annotation classAnnotation = classAnnotations[i];
			Validator beanValidator = createValidator( classAnnotation );
			if ( beanValidator != null ) beanValidators.add( beanValidator );
		}

		// Ascend up the class tree finding all the fields and methods with validators
		for ( Class currClass = beanClass; currClass != null ; currClass = currClass.getSuperclass() ) {
			Method[] methods = currClass.getDeclaredMethods();
			for ( int i = 0; i < methods.length ; i++ ) {
				Method method = methods[i];
				createMemberValidator( method );
				Class clazz = method.getReturnType();
				createChildValidator( resourceBundle, method, clazz );
			}

			Field[] fields = currClass.getDeclaredFields();
			for ( int i = 0; i < fields.length ; i++ ) {
				Field field = fields[i];
				createMemberValidator( field );
				Class clazz = field.getType();
				createChildValidator( resourceBundle, field, clazz );
			}
		}
	}

	private void createChildValidator(ResourceBundle resourceBundle, Member member, Class clazz) {
		if ( ( (AnnotatedElement) member ).isAnnotationPresent( Valid.class ) ) {
			setAccessible( member );
			childGetters.add( member );
			if ( !childClassValidators.containsKey( clazz ) ) {
				new ClassValidator( clazz, resourceBundle, childClassValidators );
			}
		}
	}

	private void createMemberValidator(Member member) {
		Annotation[] memberAnnotations = ( (AnnotatedElement) member ).getAnnotations();
		for ( int j = 0; j < memberAnnotations.length ; j++ ) {
			Annotation methodAnnotation = memberAnnotations[j];
			Validator propertyValidator = createValidator( methodAnnotation );
			if ( propertyValidator != null ) {
				memberValidators.add( propertyValidator );
				setAccessible( member );
				memberGetters.add( member );
			}
		}
	}

	private static void setAccessible(Member member) {
		if ( !Modifier.isPublic( member.getModifiers() ) ) {
			( (AccessibleObject) member ).setAccessible( true );
		}
	}

//	private static final Set<String> methodNames = new HashSet<String>();
//	static {
//		methodNames.add("toString");
//		methodNames.add("hashCode");
//		methodNames.add("equals");
//		methodNames.add("annotationType");
//	}

	private Validator createValidator(Annotation annotation) {
		try {
			ValidatorClass validatorClass = annotation.annotationType().getAnnotation( ValidatorClass.class );
			if ( validatorClass == null ) {
				return null;
			}
			Validator beanValidator = validatorClass.value().newInstance();
			beanValidator.initialize( annotation );
			String messageTemplate = (String) annotation.getClass()
					.getMethod( "message", (Class[]) null )
					.invoke( annotation );
			String message = replace( messageTemplate, annotation );
			messages.put( beanValidator, message );
			return beanValidator;
		}
		catch (Exception e) {
			throw new IllegalArgumentException( "could not instantiate ClassValidator", e );
		}
	}

	public boolean hasValidationRules() {
		return beanValidators.size() != 0 || memberValidators.size() != 0;
	}

	/**
	 * apply constraints on a bean instance and return all the failures.
	 */
	public InvalidValue[] getInvalidValues(T bean) {
		return this.getInvalidValues( bean, new HashSet<Object>() );
	}

	/**
	 * apply constraints on a bean instance and return all the failures.
	 */
	protected InvalidValue[] getInvalidValues(T bean, Set<Object> circularityState) {
		if ( circularityState.contains( bean ) ) {
			return EMPTY_INVALID_VALUE_ARRAY; //Avoid circularity
		}
		else {
			circularityState.add( bean );
		}

		if ( !beanClass.isInstance( bean ) ) {
			throw new IllegalArgumentException( "not an instance of: " + bean.getClass() );
		}

		List<InvalidValue> results = new ArrayList<InvalidValue>();

		for ( int i = 0; i < beanValidators.size() ; i++ ) {
			Validator validator = beanValidators.get( i );
			if ( !validator.isValid( bean ) ) {
				results.add( new InvalidValue( messages.get( validator ), beanClass, null, bean, bean ) );
			}
		}

		for ( int i = 0; i < memberValidators.size() ; i++ ) {
			Member getter = memberGetters.get( i );
			Object value = getMemberValue( bean, getter );
			Validator validator = memberValidators.get( i );
			if ( !validator.isValid( value ) ) {
				String propertyName = getPropertyName( getter );
				results.add( new InvalidValue( messages.get( validator ), beanClass, propertyName, value, bean ) );
			}
		}

		for ( int i = 0; i < childGetters.size() ; i++ ) {
			Member getter = childGetters.get( i );
			Object value = getMemberValue( bean, getter );
			if ( value != null && Hibernate.isInitialized( value ) ) {
				InvalidValue[] invalidValues = getClassValidator( value )
						.getInvalidValues( value, circularityState );
				for ( InvalidValue invalidValue : invalidValues ) {
					results.add( invalidValue );
				}
			}
		}

		return results.toArray( new InvalidValue[results.size()] );
	}

	private ClassValidator getClassValidator(Object value) {
		Class clazz = value.getClass();
		ClassValidator validator = childClassValidators.get( clazz );
		if ( validator == null ) { //handles polymorphism
			validator = new ClassValidator( clazz );
		}
		return validator;
	}

	/**
	 * apply constraints of a particular property on a bean instance and return all the failures.
	 * Note this is not recursive.
	 */
	//TODO should it be recursive?
	public InvalidValue[] getInvalidValues(T bean, String propertyName) {
		List<InvalidValue> results = new ArrayList<InvalidValue>();

		for ( int i = 0; i < memberValidators.size() ; i++ ) {
			Member getter = memberGetters.get( i );
			if ( getPropertyName( getter ).equals( propertyName ) ) {
				Object value = getMemberValue( bean, getter );
				Validator validator = memberValidators.get( i );
				if ( !validator.isValid( value ) ) {
					results.add( new InvalidValue( messages.get( validator ), beanClass, propertyName, value, bean ) );
				}
			}
		}

		return results.toArray( new InvalidValue[results.size()] );
	}

	private Object getMemberValue(T bean, Member getter) {
		Object value;
		try {
			value = getValue( getter, bean );
		}
		catch (Exception e) {
			throw new IllegalStateException( "Could not get property value", e );
		}
		return value;
	}

	private Object getValue(Member member, T bean) throws IllegalAccessException, InvocationTargetException {
		if ( member instanceof Field ) {
			return ( (Field) member ).get( bean );
		}
		else if ( member instanceof Method ) {
			return ( (Method) member ).invoke( bean );
		}
		else {
			throw new AssertionFailure( "Unexpected member: " + member.getClass().getName() );
		}
	}

	private static String getPropertyName(Member member) {
		if ( member instanceof Field ) {
			return member.getName();
		}
		else if ( member instanceof Method ) {
			String name = member.getName();
			if ( name.startsWith( "is" ) ) {
				name = Introspector.decapitalize( name.substring( 2 ) );
			}
			else if ( name.startsWith( "get" ) ) {
				name = Introspector.decapitalize( name.substring( 3 ) );
			}
			//do nothing for non getter method, in case someone want to validate a PO Method
			return name;
		}
		else {
			throw new AssertionFailure( "Unexpected member: " + member.getClass().getName() );
		}
	}

	private String replace(String message, Annotation parameters) {
		StringTokenizer tokens = new StringTokenizer( message, "{}", true );
		StringBuilder buf = new StringBuilder( 30 );
		boolean escaped = false;
		while ( tokens.hasMoreTokens() ) {
			String token = tokens.nextToken();
			if ( "{".equals( token ) ) {
				escaped = true;
			}
			else if ( "}".equals( token ) ) {
				escaped = false;
			}
			else if ( !escaped ) {
				buf.append( token );
			}
			else {
				Method member;
				try {
					member = parameters.getClass().getMethod( token, (Class[]) null );
				}
				catch (NoSuchMethodException nsfme) {
					member = null;
				}
				if ( member != null ) {
					try {
						buf.append( member.invoke( parameters ) );
					}
					catch (Exception e) {
						throw new IllegalArgumentException( "could not render message", e );
					}
				}
				else if ( messageBundle != null ) {
					String string = messageBundle.getString( token );
					if ( string != null ) buf.append( replace( string, parameters ) );
				}
			}
		}
		return buf.toString();
	}

	/**
	 * apply the registred constraints rules on the hibernate metadata (to be applied on DB schema...)
	 *
	 * @param persistentClass hibernate metadata
	 */
	public void apply(PersistentClass persistentClass) {

		Iterator<Validator> validators = beanValidators.iterator();
		while ( validators.hasNext() ) {
			Validator validator = validators.next();
			if ( validator instanceof PersistentClassConstraint ) {
				( (PersistentClassConstraint) validator ).apply( persistentClass );
			}
		}

		validators = memberValidators.iterator();
		Iterator<Member> getters = memberGetters.iterator();
		while ( validators.hasNext() ) {
			Validator validator = validators.next();
			String propertyName = getPropertyName( getters.next() );
			if ( validator instanceof PropertyConstraint ) {
				try {
					Property property = persistentClass.getProperty( propertyName );
					( (PropertyConstraint) validator ).apply( property );
				}
				catch (MappingException pnfe) {
					//do nothing
				}
			}
		}

	}

	public void assertValid(T bean) {
		InvalidValue[] values = getInvalidValues( bean );
		if ( values.length > 0 ) {
			throw new InvalidStateException( values );
		}
	}

	private void writeObject(ObjectOutputStream oos) throws IOException {
		ResourceBundle rb = messageBundle;
		if ( rb != null && ! ( rb instanceof Serializable ) ) {
			messageBundle = null;
			log.warn( "Serializing a ClassValidator with a not serializable ResourceBundle: ResourceBundle ignored" );
		}
		oos.defaultWriteObject();
		messageBundle = rb;
	}

	private void readObject(ObjectInputStream ois) throws IOException, ClassNotFoundException {
		ois.defaultReadObject();
		initValidator( beanClass, new HashMap<Class, ClassValidator>(), null );
	}
}
