/*
 * Decompiled with CFR 0.152.
 */
package io.github.bonigarcia.seljup;

import com.google.common.collect.ImmutableMap;
import com.spotify.docker.client.DefaultDockerClient;
import com.spotify.docker.client.DockerClient;
import com.spotify.docker.client.LogMessage;
import com.spotify.docker.client.LogStream;
import com.spotify.docker.client.ProgressHandler;
import com.spotify.docker.client.exceptions.DockerCertificateException;
import com.spotify.docker.client.exceptions.DockerException;
import com.spotify.docker.client.messages.AttachedNetwork;
import com.spotify.docker.client.messages.ContainerConfig;
import com.spotify.docker.client.messages.HostConfig;
import com.spotify.docker.client.messages.PortBinding;
import com.spotify.docker.client.messages.ProgressMessage;
import io.github.bonigarcia.seljup.DockerContainer;
import io.github.bonigarcia.seljup.InternalPreferences;
import io.github.bonigarcia.seljup.SeleniumJupiterException;
import io.github.bonigarcia.seljup.config.Config;
import java.lang.invoke.MethodHandles;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import org.apache.commons.lang.SystemUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DockerService {
    final Logger log = LoggerFactory.getLogger(MethodHandles.lookup().lookupClass());
    private Config config;
    private String dockerDefaultSocket;
    private int dockerWaitTimeoutSec;
    private int dockerPollTimeMs;
    private DockerClient dockerClient;
    private InternalPreferences preferences;
    private boolean localDaemon = true;

    public DockerService(Config config, InternalPreferences preferences) {
        this.config = config;
        this.preferences = preferences;
        this.dockerDefaultSocket = this.getConfig().getDockerDefaultSocket();
        this.dockerWaitTimeoutSec = this.getConfig().getDockerWaitTimeoutSec();
        this.dockerPollTimeMs = this.getConfig().getDockerPollTimeMs();
        String dockerServerUrl = this.getConfig().getDockerServerUrl();
        DefaultDockerClient.Builder dockerClientBuilder = null;
        if (dockerServerUrl.isEmpty()) {
            try {
                dockerClientBuilder = DefaultDockerClient.fromEnv();
            }
            catch (DockerCertificateException e) {
                throw new SeleniumJupiterException(e);
            }
        } else {
            this.log.debug("Using Docker server URL {}", (Object)dockerServerUrl);
            dockerClientBuilder = DefaultDockerClient.builder().uri(dockerServerUrl);
        }
        this.dockerClient = dockerClientBuilder.build();
    }

    public String getHost(String containerId, String network) throws DockerException, InterruptedException {
        return SystemUtils.IS_OS_LINUX ? ((AttachedNetwork)this.dockerClient.inspectContainer(containerId).networkSettings().networks().get((Object)network)).gateway() : this.dockerClient.getHost();
    }

    public synchronized String startContainer(DockerContainer dockerContainer) throws DockerException, InterruptedException {
        Optional<List<String>> entryPoint;
        Optional<List<String>> cmd;
        Optional<List<String>> envs;
        Optional<List<String>> binds;
        Optional<Map<String, List<PortBinding>>> portBindings;
        Optional<String> network;
        String imageId = dockerContainer.getImageId();
        this.log.info("Starting Docker container {}", (Object)imageId);
        HostConfig.Builder hostConfigBuilder = HostConfig.builder();
        ContainerConfig.Builder containerConfigBuilder = ContainerConfig.builder();
        boolean privileged = dockerContainer.isPrivileged();
        if (privileged) {
            this.log.trace("Using privileged mode");
            hostConfigBuilder.privileged(Boolean.valueOf(true));
            hostConfigBuilder.capAdd(new String[]{"NET_ADMIN", "NET_RAW"});
        }
        if ((network = dockerContainer.getNetwork()).isPresent()) {
            this.log.trace("Using network: {}", (Object)network.get());
            hostConfigBuilder.networkMode(network.get());
        }
        if ((portBindings = dockerContainer.getPortBindings()).isPresent()) {
            this.log.trace("Using port bindings: {}", portBindings.get());
            hostConfigBuilder.portBindings(portBindings.get());
            containerConfigBuilder.exposedPorts(portBindings.get().keySet());
        }
        if ((binds = dockerContainer.getBinds()).isPresent()) {
            this.log.trace("Using binds: {}", binds.get());
            hostConfigBuilder.binds(binds.get());
        }
        if ((envs = dockerContainer.getEnvs()).isPresent()) {
            this.log.trace("Using envs: {}", envs.get());
            containerConfigBuilder.env(envs.get());
        }
        if ((cmd = dockerContainer.getCmd()).isPresent()) {
            this.log.trace("Using cmd: {}", cmd.get());
            containerConfigBuilder.cmd(cmd.get());
        }
        if ((entryPoint = dockerContainer.getEntryPoint()).isPresent()) {
            this.log.trace("Using entryPoint: {}", entryPoint.get());
            containerConfigBuilder.entrypoint(entryPoint.get());
        }
        ContainerConfig createContainer = containerConfigBuilder.image(imageId).hostConfig(hostConfigBuilder.build()).build();
        String containerId = this.dockerClient.createContainer(createContainer).id();
        this.dockerClient.startContainer(containerId);
        return containerId;
    }

    public String execCommandInContainer(String containerId, String ... command) throws DockerException, InterruptedException {
        String commandStr = Arrays.toString(command);
        this.log.trace("Running command {} in container {}", (Object)commandStr, (Object)containerId);
        String execId = this.dockerClient.execCreate(containerId, command, new DockerClient.ExecCreateParam[]{DockerClient.ExecCreateParam.attachStdout(), DockerClient.ExecCreateParam.attachStderr()}).id();
        String output = null;
        try (LogStream stream = this.dockerClient.execStart(execId, new DockerClient.ExecStartParameter[0]);){
            if (stream.hasNext()) {
                output = StandardCharsets.UTF_8.decode(((LogMessage)stream.next()).content()).toString();
            }
        }
        catch (Exception e) {
            this.log.trace("Exception executing command in container", (Throwable)e);
        }
        this.log.trace("Result of command {} in container {}: {}", new Object[]{commandStr, containerId, output});
        return output;
    }

    public String getBindPort(String containerId, String exposed) throws DockerException, InterruptedException {
        ImmutableMap ports = this.dockerClient.inspectContainer(containerId).networkSettings().ports();
        List exposedPort = (List)ports.get((Object)exposed);
        this.log.trace("Port list {} -- Exposed port {} = {}", new Object[]{ports, exposed, exposedPort});
        if (ports.isEmpty() || exposedPort.isEmpty()) {
            String dockerImage = this.dockerClient.inspectContainer(containerId).config().image();
            throw new SeleniumJupiterException("Port " + exposed + " is not bindable in container " + dockerImage);
        }
        return ((PortBinding)exposedPort.get(0)).hostPort();
    }

    public void pullImage(final String imageId) throws DockerException, InterruptedException {
        if (!(this.preferences.checkKeyInPreferences(imageId) && this.getConfig().isUsePreferences() && this.localDaemon)) {
            this.log.info("Pulling Docker image {}", (Object)imageId);
            this.dockerClient.pull(imageId, new ProgressHandler(){

                public void progress(ProgressMessage message) throws DockerException {
                    DockerService.this.log.trace("Pulling Docker image {} ... {}", (Object)imageId, (Object)message);
                }
            });
            this.log.trace("Docker image {} downloaded", (Object)imageId);
            if (this.getConfig().isUsePreferences() && this.localDaemon) {
                this.preferences.putValueInPreferencesIfEmpty(imageId, "pulled");
            }
        }
    }

    public boolean existsImage(String imageId) {
        boolean exists = true;
        try {
            this.dockerClient.inspectImage(imageId);
            this.log.trace("Docker image {} already exists", (Object)imageId);
        }
        catch (Exception e) {
            this.log.trace("Image {} does not exist", (Object)imageId);
            exists = false;
        }
        return exists;
    }

    public synchronized void stopAndRemoveContainer(String containerId, String imageId) {
        this.log.info("Stopping Docker container {}", (Object)imageId);
        try {
            this.stopContainer(containerId);
            this.removeContainer(containerId);
        }
        catch (Exception e) {
            this.log.warn("Exception stopping container {}", (Object)imageId, (Object)e);
        }
    }

    public synchronized void stopContainer(String containerId) throws DockerException, InterruptedException {
        int stopTimeoutSec = this.getConfig().getDockerStopTimeoutSec();
        if (stopTimeoutSec == 0) {
            this.log.trace("Killing container {}", (Object)containerId);
            this.dockerClient.killContainer(containerId, DockerClient.Signal.SIGKILL);
        } else {
            this.log.trace("Stopping container {} (timeout {} seconds)", (Object)containerId, (Object)stopTimeoutSec);
            this.dockerClient.stopContainer(containerId, stopTimeoutSec);
        }
    }

    public synchronized void removeContainer(String containerId) throws DockerException, InterruptedException {
        this.log.trace("Removing container {}", (Object)containerId);
        int stopTimeoutSec = this.getConfig().getDockerStopTimeoutSec();
        if (stopTimeoutSec == 0) {
            this.dockerClient.removeContainer(containerId, new DockerClient.RemoveContainerParam[]{DockerClient.RemoveContainerParam.forceKill()});
        } else {
            this.dockerClient.removeContainer(containerId);
        }
    }

    public String getDockerDefaultSocket() {
        return this.dockerDefaultSocket;
    }

    public int getDockerWaitTimeoutSec() {
        return this.dockerWaitTimeoutSec;
    }

    public int getDockerPollTimeMs() {
        return this.dockerPollTimeMs;
    }

    public DockerClient getDockerClient() {
        return this.dockerClient;
    }

    public void close() {
        this.dockerClient.close();
    }

    public void updateDockerClient(String url) {
        if (this.localDaemon) {
            this.log.debug("Updating Docker client using URL {}", (Object)url);
            this.dockerClient = DefaultDockerClient.builder().uri(url).build();
            this.localDaemon = false;
        }
    }

    public Config getConfig() {
        return this.config;
    }
}

