/*
 * Decompiled with CFR 0.152.
 */
package it.ozimov.springboot.mail.service.defaultimpl;

import com.google.common.base.Preconditions;
import it.ozimov.springboot.mail.configuration.EmailSchedulerProperties;
import it.ozimov.springboot.mail.logging.EmailLogRenderer;
import it.ozimov.springboot.mail.model.Email;
import it.ozimov.springboot.mail.model.EmailSchedulingData;
import it.ozimov.springboot.mail.model.InlinePicture;
import it.ozimov.springboot.mail.model.defaultimpl.DefaultEmailSchedulingData;
import it.ozimov.springboot.mail.model.defaultimpl.TemplateEmailSchedulingData;
import it.ozimov.springboot.mail.service.EmailSchedulerService;
import it.ozimov.springboot.mail.service.EmailService;
import it.ozimov.springboot.mail.service.PersistenceService;
import it.ozimov.springboot.mail.service.ServiceStatus;
import it.ozimov.springboot.mail.service.defaultimpl.PriorityQueueManager;
import it.ozimov.springboot.mail.service.exception.CannotSendEmailException;
import it.ozimov.springboot.mail.utils.TimeUtils;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.time.temporal.ChronoUnit;
import java.util.Collection;
import java.util.Comparator;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import javax.annotation.PreDestroy;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

@Service(value="priorityQueueEmailSchedulerService")
@ConditionalOnExpression(value="'${spring.mail.scheduler.enabled:false}' == 'true'")
public class PriorityQueueEmailSchedulerService
implements EmailSchedulerService {
    private static final Logger log = LoggerFactory.getLogger(PriorityQueueEmailSchedulerService.class);
    protected static final Duration CONSUMER_CYCLE_LENGTH = Duration.of(1L, ChronoUnit.SECONDS);
    protected static final Duration RESUMER_CYCLE_LENGTH = Duration.of(5L, ChronoUnit.SECONDS);
    private final int batchSize;
    private final int minInMemory;
    private final int maxInMemory;
    private volatile ServiceStatus serviceStatus = ServiceStatus.RUNNING;
    private AtomicLong timeOfNextScheduledMessage;
    private final PriorityQueueManager priorityQueueManager;
    private final EmailService emailService;
    private final Consumer consumer;
    private final Resumer resumer;
    private final ExecutorService executor = Executors.newFixedThreadPool(5);
    private Optional<PersistenceService> persistenceServiceOptional;
    private EmailLogRenderer emailLogRenderer;
    private final Lock schedulerLock = new ReentrantLock();

    @Autowired
    public PriorityQueueEmailSchedulerService(EmailService emailService, EmailSchedulerProperties emailSchedulerProperties, Optional<PersistenceService> persistenceServiceOptional, EmailLogRenderer emailLogRenderer) throws InterruptedException {
        this.emailService = emailService;
        this.persistenceServiceOptional = persistenceServiceOptional;
        this.emailLogRenderer = emailLogRenderer.registerLogger(log);
        this.timeOfNextScheduledMessage = new AtomicLong();
        this.batchSize = Objects.nonNull(emailSchedulerProperties.getPersistence()) ? emailSchedulerProperties.getPersistence().getDesiredBatchSize() : 0;
        this.minInMemory = Objects.nonNull(emailSchedulerProperties.getPersistence()) ? emailSchedulerProperties.getPersistence().getMinKeptInMemory() : 1;
        this.maxInMemory = Objects.nonNull(emailSchedulerProperties.getPersistence()) ? emailSchedulerProperties.getPersistence().getMaxKeptInMemory() : Integer.MAX_VALUE;
        int numberOfPriorityLevels = emailSchedulerProperties.getPriorityLevels();
        this.priorityQueueManager = new PriorityQueueManager(numberOfPriorityLevels, persistenceServiceOptional.isPresent(), this.maxInMemory, CONSUMER_CYCLE_LENGTH);
        this.consumer = new Consumer();
        this.startConsumer();
        if (this.persistenceServiceOptional.isPresent()) {
            this.resumer = new Resumer();
            this.startResumer();
        } else {
            this.resumer = null;
        }
    }

    @Override
    @Async
    public void schedule(@NonNull Email mimeEmail, int desiredPriorityLevel) {
        if (mimeEmail == null) {
            throw new NullPointerException("mimeEmail");
        }
        this.scheduleEmail(mimeEmail, TimeUtils.offsetDateTimeNow(), desiredPriorityLevel);
    }

    @Override
    @Async
    public void schedule(@NonNull Email mimeEmail, @NonNull OffsetDateTime scheduledDateTime, int desiredPriorityLevel) {
        if (mimeEmail == null) {
            throw new NullPointerException("mimeEmail");
        }
        if (scheduledDateTime == null) {
            throw new NullPointerException("scheduledDateTime");
        }
        this.scheduleEmail(mimeEmail, scheduledDateTime, desiredPriorityLevel);
    }

    @Override
    @Async
    public void schedule(@NonNull Email mimeEmail, int desiredPriorityLevel, @NonNull String template, @NonNull Map<String, Object> modelObject, InlinePicture ... inlinePictures) throws CannotSendEmailException {
        if (mimeEmail == null) {
            throw new NullPointerException("mimeEmail");
        }
        if (template == null) {
            throw new NullPointerException("template");
        }
        if (modelObject == null) {
            throw new NullPointerException("modelObject");
        }
        this.scheduleTemplateEmail(mimeEmail, TimeUtils.offsetDateTimeNow(), desiredPriorityLevel, template, modelObject, inlinePictures);
    }

    @Override
    @Async
    public void schedule(@NonNull Email mimeEmail, @NonNull OffsetDateTime scheduledDateTime, int desiredPriorityLevel, @NonNull String template, @NonNull Map<String, Object> modelObject, InlinePicture ... inlinePictures) throws CannotSendEmailException {
        if (mimeEmail == null) {
            throw new NullPointerException("mimeEmail");
        }
        if (scheduledDateTime == null) {
            throw new NullPointerException("scheduledDateTime");
        }
        if (template == null) {
            throw new NullPointerException("template");
        }
        if (modelObject == null) {
            throw new NullPointerException("modelObject");
        }
        this.scheduleTemplateEmail(mimeEmail, scheduledDateTime, desiredPriorityLevel, template, modelObject, inlinePictures);
    }

    private void scheduleEmail(Email mimeEmail, OffsetDateTime scheduledDateTime, int desiredPriorityLevel) {
        this.checkPriorityLevel(desiredPriorityLevel);
        int assignedPriorityLevel = this.normalizePriority(desiredPriorityLevel);
        EmailSchedulingData emailSchedulingData = this.buildEmailSchedulingData(mimeEmail, scheduledDateTime, desiredPriorityLevel, assignedPriorityLevel);
        this.schedule(emailSchedulingData);
        this.emailLogRenderer.info("Scheduled email {} at UTC time {} with priority {}", mimeEmail, scheduledDateTime, desiredPriorityLevel);
        this.notifyConsumerIfCouldFire(scheduledDateTime);
    }

    private void scheduleTemplateEmail(Email mimeEmail, OffsetDateTime scheduledDateTime, int desiredPriorityLevel, String template, Map<String, Object> modelObject, InlinePicture ... inlinePictures) throws CannotSendEmailException {
        this.checkPriorityLevel(desiredPriorityLevel);
        int assignedPriorityLevel = this.normalizePriority(desiredPriorityLevel);
        EmailSchedulingData emailTemplateSchedulingData = this.buildEmailSchedulingData(mimeEmail, scheduledDateTime, desiredPriorityLevel, template, modelObject, assignedPriorityLevel, inlinePictures);
        this.schedule(emailTemplateSchedulingData);
        this.emailLogRenderer.info("Scheduled email {} at UTC time {} with priority {} with template", mimeEmail, scheduledDateTime, desiredPriorityLevel);
        this.notifyConsumerIfCouldFire(scheduledDateTime);
    }

    protected EmailSchedulingData buildEmailSchedulingData(@NonNull Email mimeEmail, @NonNull OffsetDateTime scheduledDateTime, int desiredPriorityLevel, int assignedPriorityLevel) {
        if (mimeEmail == null) {
            throw new NullPointerException("mimeEmail");
        }
        if (scheduledDateTime == null) {
            throw new NullPointerException("scheduledDateTime");
        }
        return DefaultEmailSchedulingData.defaultEmailSchedulingDataBuilder().email(mimeEmail).scheduledDateTime(scheduledDateTime).assignedPriority(assignedPriorityLevel).desiredPriority(desiredPriorityLevel).build();
    }

    protected EmailSchedulingData buildEmailSchedulingData(@NonNull Email mimeEmail, @NonNull OffsetDateTime scheduledDateTime, int desiredPriorityLevel, @NonNull String template, @NonNull Map<String, Object> modelObject, int assignedPriorityLevel, InlinePicture[] inlinePictures) {
        if (mimeEmail == null) {
            throw new NullPointerException("mimeEmail");
        }
        if (scheduledDateTime == null) {
            throw new NullPointerException("scheduledDateTime");
        }
        if (template == null) {
            throw new NullPointerException("template");
        }
        if (modelObject == null) {
            throw new NullPointerException("modelObject");
        }
        return TemplateEmailSchedulingData.templateEmailSchedulingDataBuilder().email(mimeEmail).scheduledDateTime(scheduledDateTime).assignedPriority(assignedPriorityLevel).desiredPriority(desiredPriorityLevel).template(template).modelObject(modelObject).inlinePictures(inlinePictures).build();
    }

    protected synchronized void schedule(EmailSchedulingData emailSchedulingData) {
        this.enqueueFromScheduler(emailSchedulingData);
        this.addToPersistenceLayer(emailSchedulingData);
        this.completeEnqueue();
    }

    protected synchronized void startResumer() throws InterruptedException {
        this.startAndWaitForWaitingState(this.resumer);
    }

    protected synchronized void startConsumer() throws InterruptedException {
        this.startAndWaitForWaitingState(this.consumer);
    }

    private void startAndWaitForWaitingState(Thread thread) throws InterruptedException {
        thread.start();
        while (thread.getState() == Thread.State.RUNNABLE) {
            TimeUnit.MILLISECONDS.sleep(50L);
        }
    }

    private void notifyConsumerIfCouldFire(@NonNull OffsetDateTime scheduledDateTime) {
        boolean canFire;
        if (scheduledDateTime == null) {
            throw new NullPointerException("scheduledDateTime");
        }
        boolean bl = canFire = this.isTimeOfNextSchedulerMessageNotSet() || scheduledDateTime.toInstant().toEpochMilli() < this.timeOfNextScheduledMessage.get();
        if (canFire && this.consumer.enabled()) {
            this.executor.submit(() -> {
                Consumer consumer = this.consumer;
                synchronized (consumer) {
                    this.consumer.notify();
                }
            });
        }
    }

    private boolean enqueueFromScheduler(EmailSchedulingData emailSchedulingData) {
        return this.enqueue(emailSchedulingData, false);
    }

    private boolean enqueueFromPersistenceLayer(EmailSchedulingData emailSchedulingData) {
        return this.enqueue(emailSchedulingData, true);
    }

    private boolean enqueue(EmailSchedulingData emailSchedulingData, boolean isFromPersistenceLayer) {
        if (this.serviceStatus == ServiceStatus.RUNNING) {
            return this.priorityQueueManager.enqueue(emailSchedulingData, isFromPersistenceLayer);
        }
        return false;
    }

    private void completeEnqueue() {
        if (this.serviceStatus == ServiceStatus.RUNNING) {
            this.priorityQueueManager.completeEnqueue();
        }
    }

    protected void addToPersistenceLayer(EmailSchedulingData emailSchedulingData) {
        if (this.serviceStatus == ServiceStatus.RUNNING) {
            log.debug("Adding to persistence layer");
            this.persistenceServiceOptional.ifPresent(persistenceService -> persistenceService.add(emailSchedulingData));
        }
    }

    protected void deleteFromPersistenceLayer(EmailSchedulingData emailSchedulingData) {
        if (this.serviceStatus == ServiceStatus.RUNNING) {
            this.persistenceServiceOptional.ifPresent(persistenceService -> {
                persistenceService.remove(emailSchedulingData.getId());
                this.priorityQueueManager.completeDequeue();
            });
        }
    }

    protected void loadNextBatch() {
        if (this.serviceStatus == ServiceStatus.RUNNING) {
            this.persistenceServiceOptional.ifPresent(persistenceService -> {
                int expectedFromPersistenceLayer;
                Collection<EmailSchedulingData> emailSchedulingDataList;
                int currentlyInMemory = this.currentlyInMemory();
                if (currentlyInMemory < this.minInMemory && !(emailSchedulingDataList = persistenceService.getNextBatch(expectedFromPersistenceLayer = Math.min(currentlyInMemory + this.batchSize, this.maxInMemory))).isEmpty()) {
                    this.enqueueBatch(emailSchedulingDataList);
                }
            });
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected void enqueueBatch(Collection<EmailSchedulingData> emailSchedulingDataCollection) {
        if (!emailSchedulingDataCollection.isEmpty()) {
            EmailSchedulingData lastEmailSchedulingData = emailSchedulingDataCollection.stream().filter(Objects::nonNull).max(Comparator.comparing(EmailSchedulingData::getScheduledDateTime)).get();
            if (this.serviceStatus == ServiceStatus.RUNNING) {
                int countAdded = 0;
                for (EmailSchedulingData emailSchedulingData : emailSchedulingDataCollection) {
                    PriorityQueueEmailSchedulerService priorityQueueEmailSchedulerService = this;
                    synchronized (priorityQueueEmailSchedulerService) {
                        if (this.enqueueFromPersistenceLayer(emailSchedulingData)) {
                            ++countAdded;
                        }
                        this.completeEnqueue();
                    }
                }
                log.debug("Enqueued batch of {} emails of {} loaded from persistence layer.", (Object)countAdded, (Object)emailSchedulingDataCollection.size());
            }
            if (Objects.nonNull(lastEmailSchedulingData)) {
                this.notifyConsumerIfCouldFire(lastEmailSchedulingData.getScheduledDateTime());
            }
        }
    }

    protected int normalizePriority(int priorityLevel) {
        int maxLevel = this.priorityQueueManager.numberOfLevels();
        if (priorityLevel > maxLevel) {
            log.warn("Scheduled email with priority level {}, while priority level {} was requested. Reason: max level exceeded", (Object)maxLevel, (Object)priorityLevel);
        }
        return Math.max(1, Math.min(priorityLevel, maxLevel));
    }

    private Optional<EmailSchedulingData> dequeue() throws InterruptedException {
        Optional<EmailSchedulingData> emailSchedulingDataOptional = Optional.empty();
        this.timeOfNextScheduledMessage.set(0L);
        boolean consumerEnabled = this.consumer.enabled();
        while (consumerEnabled && !emailSchedulingDataOptional.isPresent()) {
            if (this.consumer.enabled()) {
                if (this.priorityQueueManager.hasElements()) {
                    emailSchedulingDataOptional = this.priorityQueueManager.dequeueNext(CONSUMER_CYCLE_LENGTH);
                }
                if (emailSchedulingDataOptional.isPresent()) continue;
                this.timeOfNextScheduledMessage.set(this.priorityQueueManager.millisToNextEmail());
                if (!this.consumer.enabled()) continue;
                if (this.isTimeOfNextSchedulerMessageNotSet()) {
                    this.consumer.waitForNotify();
                    continue;
                }
                long waitTime = this.timeOfNextScheduledMessage.get() - TimeUtils.now() - CONSUMER_CYCLE_LENGTH.toMillis();
                if (waitTime <= 0L) continue;
                this.consumer.waitForMillis(waitTime);
                continue;
            }
            consumerEnabled = false;
        }
        return emailSchedulingDataOptional;
    }

    private boolean isTimeOfNextSchedulerMessageNotSet() {
        return this.timeOfNextScheduledMessage.get() == 0L;
    }

    private void checkPriorityLevel(int priorityLevel) {
        Preconditions.checkArgument((priorityLevel > 0 ? 1 : 0) != 0, (Object)"The priority level index cannot be negative");
    }

    private int queueIndex(EmailSchedulingData emailSchedulingData) {
        return emailSchedulingData.getAssignedPriority() - 1;
    }

    private boolean canAddOneInMemory() {
        return !this.persistenceServiceOptional.isPresent() || this.currentlyInMemory() < this.maxInMemory;
    }

    private int currentlyInMemory() {
        return this.priorityQueueManager.currentlyInQueue();
    }

    @PreDestroy
    protected void cleanUp() throws Exception {
        log.info("Closing EmailScheduler");
        try {
            this.executor.shutdownNow();
            this.schedulerLock.lock();
            try {
                this.serviceStatus = ServiceStatus.CLOSING;
            }
            finally {
                this.schedulerLock.unlock();
            }
            log.debug("EMAIL SCHEDULER -- Closing PriorityQueueManager");
            this.priorityQueueManager.close();
            if (Objects.nonNull(this.resumer)) {
                log.debug("EMAIL SCHEDULER -- Closing Resumer");
                this.resumer.close();
            }
            log.debug("EMAIL SCHEDULER -- Closing Consumer");
            this.consumer.close();
        }
        catch (Exception e) {
            log.warn("An issue occurred while stopping EmailScheduler, it should be due to a thread interruption.", (Throwable)e);
        }
        finally {
            this.serviceStatus = ServiceStatus.CLOSED;
        }
        log.info("Closed EmailScheduler");
    }

    private class Resumer
    extends Thread {
        public Resumer() {
            super(PriorityQueueEmailSchedulerService.class.getSimpleName() + " -- " + Resumer.class.getSimpleName());
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void run() {
            if (PriorityQueueEmailSchedulerService.this.persistenceServiceOptional.isPresent()) {
                log.info("Email scheduler resumer started");
                while (this.enabled()) {
                    try {
                        if (!PriorityQueueEmailSchedulerService.this.canAddOneInMemory()) continue;
                        if (this.enabled()) {
                            PriorityQueueEmailSchedulerService.this.loadNextBatch();
                        }
                        if (!this.enabled()) continue;
                        Resumer resumer = this;
                        synchronized (resumer) {
                            this.wait(RESUMER_CYCLE_LENGTH.toMillis());
                        }
                    }
                    catch (InterruptedException e) {
                        log.error("Email scheduler consumer interrupted", (Throwable)e);
                    }
                }
                log.info("Email scheduler resumer stopped");
            } else {
                log.warn("Email scheduler resumer won't start because there is no email PersistenceService.");
            }
        }

        public boolean enabled() {
            return PriorityQueueEmailSchedulerService.this.serviceStatus == ServiceStatus.RUNNING && !this.isInterrupted();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public synchronized void close() throws InterruptedException {
            block6: {
                try {
                    if (!this.isInterrupted()) {
                        log.info("Interrupting email scheduler resumer");
                        this.interrupt();
                        Resumer resumer = this;
                        synchronized (resumer) {
                            this.notify();
                        }
                        this.join();
                        break block6;
                    }
                    log.info("Email scheduler resumer already interrupted");
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }
    }

    private class Consumer
    extends Thread {
        public Consumer() {
            super(PriorityQueueEmailSchedulerService.class.getSimpleName() + " -- " + Consumer.class.getSimpleName());
        }

        @Override
        public void run() {
            log.info("Email scheduler consumer started");
            while (this.enabled()) {
                try {
                    Optional emailSchedulingDataOptional = PriorityQueueEmailSchedulerService.this.dequeue();
                    if (!this.enabled() || !emailSchedulingDataOptional.isPresent()) continue;
                    EmailSchedulingData emailSchedulingData = (EmailSchedulingData)emailSchedulingDataOptional.get();
                    if (emailSchedulingData instanceof TemplateEmailSchedulingData) {
                        TemplateEmailSchedulingData emailTemplateSchedulingData = (TemplateEmailSchedulingData)emailSchedulingData;
                        try {
                            PriorityQueueEmailSchedulerService.this.emailService.send(emailTemplateSchedulingData.getEmail(), emailTemplateSchedulingData.getTemplate(), emailTemplateSchedulingData.getModelObject(), emailTemplateSchedulingData.getInlinePictures());
                        }
                        catch (CannotSendEmailException e) {
                            log.error("An error occurred while sending the email", (Throwable)e);
                        }
                    } else {
                        PriorityQueueEmailSchedulerService.this.emailService.send(emailSchedulingData.getEmail());
                    }
                    if (this.enabled() && !PriorityQueueEmailSchedulerService.this.persistenceServiceOptional.isPresent()) {
                        PriorityQueueEmailSchedulerService.this.priorityQueueManager.completeDequeue();
                    }
                    if (!this.enabled()) continue;
                    PriorityQueueEmailSchedulerService.this.deleteFromPersistenceLayer(emailSchedulingData);
                }
                catch (InterruptedException e) {
                    log.error("Email scheduler consumer interrupted", (Throwable)e);
                }
            }
            log.info("Email scheduler consumer stopped");
        }

        public boolean enabled() {
            return PriorityQueueEmailSchedulerService.this.serviceStatus == ServiceStatus.RUNNING && !this.isInterrupted();
        }

        public synchronized void waitForNotify() throws InterruptedException {
            if (this.enabled()) {
                log.debug("Email scheduler consumer starts waiting");
                this.wait();
            }
        }

        public synchronized void waitForMillis(long timeoutInMillis) throws InterruptedException {
            if (this.enabled()) {
                log.debug("Email scheduler consumer starts waiting for {} millis", (Object)timeoutInMillis);
                this.wait(timeoutInMillis);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public void close() throws InterruptedException {
            block6: {
                try {
                    if (!this.isInterrupted()) {
                        log.info("Interrupting email scheduler consumer");
                        this.interrupt();
                        Consumer consumer = this;
                        synchronized (consumer) {
                            this.notify();
                        }
                        this.join();
                        break block6;
                    }
                    log.info("Email scheduler consumer already interrupted");
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
            }
        }
    }
}

