/*
 * Decompiled with CFR 0.152.
 */
package it.ozimov.springboot.mail.service.defaultimpl;

import com.google.common.base.Preconditions;
import it.ozimov.springboot.mail.model.EmailSchedulingData;
import it.ozimov.springboot.mail.utils.TimeUtils;
import java.io.Closeable;
import java.io.IOException;
import java.time.Duration;
import java.time.OffsetDateTime;
import java.util.Arrays;
import java.util.Comparator;
import java.util.Objects;
import java.util.Optional;
import java.util.TreeSet;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Stream;
import lombok.NonNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PriorityQueueManager
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(PriorityQueueManager.class);
    private final boolean hasPersistence;
    private volatile int currentlyQueued;
    private final TreeSet<EmailSchedulingData>[] queues;
    private final int maxInMemory;
    private final Duration queuabilityDelta;
    private final Lock queueLock = new ReentrantLock();
    private final Condition notDequeuing = this.queueLock.newCondition();
    private final Condition notEnqueuing = this.queueLock.newCondition();
    private final ReentrantReadWriteLock currentOperationLock = new ReentrantReadWriteLock();
    private CurrentOperation currentOperation = CurrentOperation.NONE;

    PriorityQueueManager(int numberOfPriorityLevels, boolean hasPersistence, int maxInMemory, @NonNull Duration queuabilityDelta) {
        if (queuabilityDelta == null) {
            throw new NullPointerException("queuabilityDelta");
        }
        Preconditions.checkArgument((numberOfPriorityLevels > 0 ? 1 : 0) != 0, (String)"Number of priority levels should be a positive number, while %s was given", (int)numberOfPriorityLevels);
        Preconditions.checkArgument((maxInMemory > 0 ? 1 : 0) != 0, (String)"Number of max emails in memory should be a positive number, while %s was given", (int)maxInMemory);
        this.hasPersistence = hasPersistence;
        this.maxInMemory = maxInMemory;
        this.queuabilityDelta = queuabilityDelta;
        this.queues = new TreeSet[numberOfPriorityLevels];
        for (int i = 0; i < numberOfPriorityLevels; ++i) {
            this.queues[i] = new TreeSet();
        }
    }

    public int numberOfLevels() {
        return this.queues.length;
    }

    public boolean hasElements() {
        return this.currentlyQueued > 0;
    }

    public int currentlyInQueue() {
        return this.currentlyQueued;
    }

    public long millisToNextEmail() {
        Optional<OffsetDateTime> minScheduledTime = this.getStreamOfAllFirst().map(EmailSchedulingData::getScheduledDateTime).min(OffsetDateTime::compareTo);
        return !minScheduledTime.isPresent() ? 0L : minScheduledTime.get().toInstant().toEpochMilli();
    }

    public boolean enqueue(EmailSchedulingData emailSchedulingData, boolean isFromPersistenceLayer) {
        log.debug("Called Enqueue [currently queued = {}, isFromPersistenceLayer = {}]", (Object)this.currentlyInQueue(), (Object)isFromPersistenceLayer);
        this.queueLock.lock();
        try {
            int queueIndex;
            TreeSet<EmailSchedulingData> queue;
            while (this.isCurrentOperationDequeuing() || this.isCurrentOperationEnqueuing()) {
                this.notDequeuing.await();
                if (this.isCurrentOperationClosing()) continue;
                this.setCurrentOperationToEnqueuing();
            }
            if (this.isCurrentOperationNone() && !this.isCurrentOperationClosing()) {
                this.setCurrentOperationToEnqueuing();
            }
            if (this.isCurrentOperationEnqueuing() && !this.isCurrentOperationClosing() && !(queue = this.queues[queueIndex = this.queueIndex(emailSchedulingData)]).contains(emailSchedulingData)) {
                boolean dequeueLastLoaded;
                boolean isEnqueuable = isFromPersistenceLayer || this.beforeLastLoadedFromPersistenceLayer(emailSchedulingData);
                boolean bl = dequeueLastLoaded = isEnqueuable && !this.canAddOneInMemory() && this.hasElements();
                if (isEnqueuable) {
                    this.queues[queueIndex].add(emailSchedulingData);
                    ++this.currentlyQueued;
                } else {
                    log.debug("Email scheduling data {} not queued but should be persisted afterwards", (Object)emailSchedulingData);
                }
                if (dequeueLastLoaded) {
                    int queueIndexOfLatestOfAllLast = this.queueIndexOfLatestOfAllLast();
                    TreeSet<EmailSchedulingData> queueOfLatestOfAllLast = this.queues[queueIndexOfLatestOfAllLast];
                    queueOfLatestOfAllLast.pollLast();
                    --this.currentlyQueued;
                }
                return isEnqueuable;
            }
        }
        catch (InterruptedException e) {
            if (!this.isCurrentOperationClosing()) {
                log.error("Priority queue manager interrupted during dequeuing operation.", (Throwable)e);
            }
            this.completeEnqueue();
        }
        return false;
    }

    public void completeEnqueue() {
        try {
            if (!this.isCurrentOperationClosing()) {
                Preconditions.checkState((boolean)this.isCurrentOperationEnqueuing(), (String)"Cannot complete enqueue if current operation is %s.", (Object)((Object)this.currentOperation));
                log.debug("Completed Enqueue [currently queued = {}]", (Object)this.currentlyInQueue());
                this.setCurrentOperationToNone();
                this.notEnqueuing.signal();
            }
        }
        finally {
            this.queueLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Optional<EmailSchedulingData> dequeueNext(Duration consumerCycle) {
        log.debug("Called Dequeue [currently queued = {}]", (Object)this.currentlyInQueue());
        this.queueLock.lock();
        try {
            while (this.isCurrentOperationEnqueuing() || this.isCurrentOperationDequeuing()) {
                this.notDequeuing.await();
                if (this.isCurrentOperationClosing()) continue;
                this.setCurrentOperationToDequeuing();
            }
            if (this.isCurrentOperationNone() && !this.isCurrentOperationClosing()) {
                this.setCurrentOperationToDequeuing();
            }
            if (this.isCurrentOperationDequeuing() && !this.isCurrentOperationClosing()) {
                long now = TimeUtils.now();
                for (TreeSet<EmailSchedulingData> queue : this.queues) {
                    long time;
                    if (queue.isEmpty() || (time = queue.first().getScheduledDateTime().toInstant().toEpochMilli()) - now > consumerCycle.toMillis()) continue;
                    --this.currentlyQueued;
                    return Optional.of(queue.pollFirst());
                }
            }
        }
        catch (InterruptedException e) {
            if (!this.isCurrentOperationClosing()) {
                log.error("Priority queue manager interrupted during dequeuing operation.", (Throwable)e);
            }
            this.completeDequeue();
        }
        try {
            Optional<EmailSchedulingData> optional = Optional.empty();
            return optional;
        }
        finally {
            this.completeDequeue();
        }
    }

    public void completeDequeue() {
        try {
            if (!this.isCurrentOperationClosing()) {
                Preconditions.checkState((boolean)this.isCurrentOperationDequeuing(), (String)"Cannot complete dequeue if current operation is %s.", (Object)((Object)this.currentOperation));
                log.debug("Completed Dequeue [currently queued = {}]", (Object)this.currentlyInQueue());
                this.setCurrentOperationToNone();
                this.notDequeuing.signal();
            }
        }
        finally {
            this.queueLock.unlock();
        }
    }

    protected boolean isCurrentOperationNone() {
        return this.isCurrentOperation(CurrentOperation.NONE);
    }

    protected boolean isCurrentOperationClosing() {
        return this.isCurrentOperation(CurrentOperation.CLOSING);
    }

    protected boolean isCurrentOperationDequeuing() {
        return this.isCurrentOperation(CurrentOperation.DEQUEUING);
    }

    protected boolean isCurrentOperationEnqueuing() {
        return this.isCurrentOperation(CurrentOperation.ENQUEUING);
    }

    private boolean isCurrentOperation(CurrentOperation currentOperation) {
        this.currentOperationLock.readLock().lock();
        try {
            boolean bl = this.currentOperation == currentOperation;
            return bl;
        }
        finally {
            this.currentOperationLock.readLock().unlock();
        }
    }

    protected void setCurrentOperationToEnqueuing() throws InterruptedException {
        this.currentOperationLock.writeLock().lock();
        if (this.isCurrentOperationDequeuing()) {
            this.notDequeuing.await();
        }
        this.currentOperation = CurrentOperation.ENQUEUING;
        this.currentOperationLock.writeLock().unlock();
    }

    protected void setCurrentOperationToDequeuing() throws InterruptedException {
        this.currentOperationLock.writeLock().lock();
        if (this.isCurrentOperationEnqueuing()) {
            this.notEnqueuing.await();
        }
        this.currentOperation = CurrentOperation.DEQUEUING;
        this.currentOperationLock.writeLock().unlock();
    }

    protected void setCurrentOperationToNone() {
        this.currentOperationLock.writeLock().lock();
        this.currentOperation = CurrentOperation.NONE;
        this.currentOperationLock.writeLock().unlock();
    }

    private int queueIndex(EmailSchedulingData emailSchedulingData) {
        return emailSchedulingData.getAssignedPriority() - 1;
    }

    private boolean canAddOneInMemory() {
        return !this.hasPersistence || this.currentlyInQueue() < this.maxInMemory;
    }

    private Optional<EmailSchedulingData> getLeastOfAllLast() {
        return this.getStreamOfAllLast().min(Comparator.comparing(EmailSchedulingData::getScheduledDateTime));
    }

    private Optional<EmailSchedulingData> getLatestOfAllLast() {
        return this.getStreamOfAllLast().max(Comparator.comparing(EmailSchedulingData::getScheduledDateTime));
    }

    private boolean beforeLastLoadedFromPersistenceLayer(EmailSchedulingData emailSchedulingData) {
        if (!this.hasPersistence || !this.hasElements()) {
            return true;
        }
        EmailSchedulingData least = this.getLeastOfAllLast().get();
        int scheduledDateTimeComparison = emailSchedulingData.getScheduledDateTime().compareTo(least.getScheduledDateTime().plus(this.queuabilityDelta));
        return scheduledDateTimeComparison < 0 || scheduledDateTimeComparison == 0 && emailSchedulingData.getAssignedPriority() < least.getAssignedPriority();
    }

    private int queueIndexOfLatestOfAllLast() {
        Optional<EmailSchedulingData> latest = this.getLatestOfAllLast();
        Preconditions.checkState((boolean)latest.isPresent(), (Object)"Should not call queueIndexOfLatestOfAllLast() if no EmailSchedulingIsInQueue");
        return latest.get().getAssignedPriority() - 1;
    }

    private Stream<EmailSchedulingData> getStreamOfAllLast() {
        return Arrays.stream(this.queues).filter(queue -> !queue.isEmpty()).map(queue -> (EmailSchedulingData)queue.last()).filter(Objects::nonNull);
    }

    private Stream<EmailSchedulingData> getStreamOfAllFirst() {
        return Arrays.stream(this.queues).filter(queue -> !queue.isEmpty()).map(queue -> (EmailSchedulingData)queue.last()).filter(Objects::nonNull);
    }

    @Override
    public void close() throws IOException {
        if (this.currentOperationLock.isWriteLocked()) {
            this.currentOperationLock.writeLock().unlock();
        }
        this.currentOperationLock.writeLock().lock();
        this.currentOperation = CurrentOperation.CLOSING;
        this.currentOperationLock.writeLock().unlock();
        try {
            this.queueLock.unlock();
        }
        catch (IllegalMonitorStateException illegalMonitorStateException) {
            // empty catch block
        }
    }

    static enum CurrentOperation {
        DEQUEUING,
        ENQUEUING,
        NONE,
        CLOSING;

    }
}

